/* Copyright 2007 Openismus GmbH
 *
 * This program is free software; you can redistribute it and/or modify
 * it under the terms of the GNU General Public License version 2
 * as published by the Free Software Foundation.
 *
 * This program is distributed in the hope that it will be useful,
 * but WITHOUT ANY WARRANTY; without even the implied warranty of
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the
 * GNU General Public License for more details.
 *
 * You should have received a copy of the GNU General Public License
 * along with this program; if not, write to the Free Software
 * Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA 02111-1307 USA
 */

#include "triangle_actor.h"
#include <cogl/cogl.h>

namespace Tutorial
{

Glib::RefPtr<Triangle> Triangle::create()
{
  return Glib::RefPtr<Triangle>(new Triangle());
}

Glib::RefPtr<Triangle> Triangle::create(const Clutter::Color& color)
{
  return Glib::RefPtr<Triangle>(new Triangle(color));
}

Triangle::Triangle()
:
  color_ (0xFF, 0xFF, 0xFF, 0xFF)
{}

Triangle::Triangle(const Clutter::Color& color)
:
  color_ (color)
{}

Triangle::~Triangle()
{}

void Triangle::do_triangle_paint(const Clutter::Color& color)
{
  const Clutter::Geometry geom = get_geometry();

  cogl_push_matrix();
  cogl_color(color.gobj());

  ClutterFixed coords[6];

  // Paint a triangle.  The parent paint call will have translated us into
  // position so paint from 0, 0.
  coords[0] = CLUTTER_INT_TO_FIXED(0);
  coords[1] = CLUTTER_INT_TO_FIXED(0);

  coords[2] = CLUTTER_INT_TO_FIXED(0);
  coords[3] = CLUTTER_INT_TO_FIXED(geom.get_height());

  coords[4] = CLUTTER_INT_TO_FIXED(geom.get_width());
  coords[5] = coords[3];

  cogl_path_polygon(coords, G_N_ELEMENTS(coords) / 2);
  cogl_path_fill();

  cogl_pop_matrix();
}

void Triangle::on_paint()
{
  // Paint the triangle with the actor's color:
  Clutter::Color color(color_);
  color.set_alpha( get_opacity() );
  do_triangle_paint(color);
}

void Triangle::pick_vfunc(const Clutter::Color& color)
{
  // Paint the triangle with the pick color, offscreen.
  // This is used by Clutter to detect the actor under the cursor 
  // by identifying the unique color under the cursor.
  do_triangle_paint(color);
}

/**
 * Tutorial::Triangle::get_color:
 *
 * @returns the color of the triangle.
 */
Clutter::Color Triangle::get_color() const
{
  return color_;
}

/**
 * Tutorial::Triangle::set_color:
 * @color: a Clutter::Color
 *
 * Sets the color of the triangle.
 */
void Triangle::set_color(const Clutter::Color& color)
{
  color_ = color;
  set_opacity(color_.get_alpha());

  if(is_visible())
    queue_redraw();
}

} // namespace Tutorial
