/**
 * <copyright> 
 * 
 * Copyright (c) 2004-2005 IBM Corporation and others. 
 * All rights reserved.   This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License - v 1.0 
 * which accompanies this distribution, and is available at 
 * http://opensource.org/licenses/eclipse-1.0.txt 
 * 
 * Contributors: 
 *   IBM - Initial API and implementation 
 * 
 * </copyright> 
 *   
 * $Id: RDFXMLSaverImpl.java,v 1.2 2007/03/18 08:39:02 lzhang Exp $
 */

package org.eclipse.eodm.rdf.resource;

import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.Hashtable;
import java.util.Iterator;

import org.eclipse.eodm.rdf.rdfbase.PlainLiteral;
import org.eclipse.eodm.rdf.rdfbase.RDFGraph;
import org.eclipse.eodm.rdf.rdfbase.RDFSLiteral;
import org.eclipse.eodm.rdf.rdfbase.RDFXMLLiteral;
import org.eclipse.eodm.rdf.rdfbase.TypedLiteral;
import org.eclipse.eodm.rdf.rdfbase.URIReference;
import org.eclipse.eodm.rdf.rdfweb.Document;
import org.eclipse.eodm.rdf.rdfweb.NamespaceDefinition;
import org.eclipse.eodm.rdf.resource.parser.exception.ParserIOException;
import org.eclipse.eodm.util.Triple;
import org.eclipse.eodm.vocabulary.RDF;
import org.eclipse.eodm.vocabulary.RDFS;

/**
 * Save an EODM RDF document into a RDF/XML syntax resource.
 */
class RDFXMLSaverImpl {

    /**
     * The {@link java.io.Writer writer}to use for output
     */
    protected Writer writer;

    /**
     * The level of output indentation.
     */
    protected int indentLevel;

    /**
     * A flag that indicates the starting of an XML element, also the starting
     * of a type traversing process.
     */
    protected boolean isElementStarting;

    /**
     * The type string of the current XML element.
     */
    protected String elementType;

    /**
     * The graph to be serialized
     */
    protected RDFGraph graph;

    /**
     * Store and search all namespaces
     */
    protected Hashtable NamespaceHashTable = new Hashtable();;

    /**
     * Write a number of tabs according to the current indentation level
     * 
     */
    protected void writeIndents() {
        int len = indentLevel;
        while (len-- > 0)
            try {
                writer.write("\t");
            } catch (IOException e) {
                throw new ParserIOException("IO exception: " + e.getMessage());
            }
    }

    /**
     * Write the start part of a RDF resource.
     * 
     * @param triples
     *            the RDF triple to be outputed
     */
    protected void writeElementStart(Triple triple) {
        try {
            writeIndents();
            writer.write("<"
                         + elementType + " "
                         + getIDAttrString(triple, true, false) + ">\n");
            indentLevel++;
        } catch (IOException e) {
            throw new ParserIOException("IO exception: " + e.getMessage());
        }
    }

    /**
     * Write the end tag of the current XML element.
     * 
     */
    protected void writeElementEnding() {
        indentLevel--;
        try {
            writeIndents();
            writer.write("</" + elementType + ">\n");
        } catch (IOException e) {
            throw new ParserIOException("IO exception: " + e.getMessage());
        }
    }

    /**
     * Write the start part of a RDF resource
     * 
     * @param triple
     *            the RDF triple to be outputed
     */
    protected void writeElementStarting(Triple triple) {
        if (isElementStarting) {
            isElementStarting = false;
            writeElementStart(triple);
        }
    }

    /**
     * Write a RDF statement
     * 
     * @param triple
     *            the RDF triple to be outputed
     * 
     */
    protected void writeStatement(Triple triple) {
        String propertyName = getQName(triple.getPredicateUriRef(), false);
        try {
            writeIndents();
            writer.write("<" + propertyName);

            if (triple.isObjectLiteral()) {
                RDFSLiteral literal = (RDFSLiteral) triple.getObjectLiteral();
                if (literal instanceof RDFXMLLiteral) {
                    writer.write(" rdf:parseType=\"Literal\">");
                    writer.write(replaceKeywords(literal.getLexicalForm()));
                    writer.write("\n");
                    writeIndents();
                    writer.write("</" + propertyName + ">\n");
                    return;
                } else if (literal instanceof TypedLiteral) {
                    String datatypeURI = ((TypedLiteral) literal)
                            .getDatatypeURI().getURIString();
                    writer.write(" rdf:datatype=\"" + datatypeURI + "\">");
                    writer.write(replaceKeywords(literal.getLexicalForm()));
                    writer.write("</" + propertyName + ">\n");
                    return;
                } else {
                    if (literal instanceof PlainLiteral) {
                        PlainLiteral plainLiteral = (PlainLiteral) literal;
                        String lang = plainLiteral.getLanguage();
                        if (lang != null)
                            if (lang.length() > 0) {
                                writer.write(" xml:lang=\"" + lang + "\"");
                            }
                    }
                    writer.write(">"
                                 + replaceKeywords(literal.getLexicalForm()));
                    writer.write("</" + propertyName + ">\n");
                    return;
                }
            } else if (triple.isObjectBlankNode())
                writer.write(" rdf:nodeID=\""
                             + triple.getObjectNodeID() + "\"/>\n");
            else
                writer.write(" "
                             + getIDAttrString(triple, false, true) + "/>\n");
        } catch (IOException e) {
            throw new ParserIOException("Parser IO Exception: "
                                        + e.getMessage());
        }
    }

    /**
     * Get the XML QName representation of a RDF resource's URI.
     * 
     * @param uri
     *            the uri of the RDF resource
     * @param isEntityRef
     *            whether the returned representation is in the form of XML
     *            entity reference
     * @return the XML QName representation of a RDF resource's URI
     */
    protected String getQName(URIReference uri, boolean isEntityRef) {
        String namespace = null;
        if (uri.getNamespace() != null)
            namespace = getNamespaceAbbr(uri.getNamespace().getNamespaceURIRef().getURIString());

        if (isEntityRef) {
            if (namespace != null) {
                System.out.println("&"
                                   + namespace
                                   + ";"
                                   + replaceKeywords(uri
                                           .getFragmentIdentifier().getName()));
                return "&"
                       + namespace + ";"
                       + replaceKeywords(uri.getFragmentIdentifier().getName());
            } else {

                if (uri.getURIString().startsWith(RDF.NAMESPACE))
                    return "&"
                           + getNamespaceAbbr(RDF.NAMESPACE)
                           + ";"
                           + uri.getURIString().substring(
                                   RDF.NAMESPACE.length(),
                                   uri.getURIString().length());
                else if (uri.getURIString().startsWith(RDFS.NAMESPACE))
                    return "&"
                           + getNamespaceAbbr(RDFS.NAMESPACE)
                           + ";"
                           + uri.getURIString().substring(
                                   RDFS.NAMESPACE.length(),
                                   uri.getURIString().length());
                else
                    return replaceKeywords(uri.getURIString());
            }
        } else {
            if (namespace != null) {
                System.out.println(namespace
                                   + ":"
                                   + replaceKeywords(uri
                                           .getFragmentIdentifier().getName()));
                return namespace
                       + ":"
                       + replaceKeywords(uri.getFragmentIdentifier().getName());
            } else {
                if (uri.getURIString().startsWith(RDF.NAMESPACE))
                    return getNamespaceAbbr(RDF.NAMESPACE)
                           + ":"
                           + uri.getURIString().substring(
                                   RDF.NAMESPACE.length(),
                                   uri.getURIString().length());
                else if (uri.getURIString().startsWith(RDFS.NAMESPACE))
                    return getNamespaceAbbr(RDFS.NAMESPACE)
                           + ":"
                           + uri.getURIString().substring(
                                   RDFS.NAMESPACE.length(),
                                   uri.getURIString().length());
                else
                    return replaceKeywords(uri.getURIString());
            }
        }
    }

    /**
     * 
     * @param triple
     *            the triple to be processed
     * @param sub
     *            a flag to denote which part of a triple, the subject or the
     *            object, will be processed
     * @param isPropAttr
     *            for a RDF URI resource, whether to obtain as a XML attribute
     *            on a property XML element ("rdf:resource=...") or as a normal
     *            "rdf:about=...".
     * @return the XML attribute string of the resource's URI or blank node ID.
     */
    protected String getIDAttrString(Triple triple, boolean sub,
            boolean isPropAttr) {
        if (sub) {
            if (triple.getSubjectNodeID() != null)
                return "rdf:nodeID=\"" + triple.getSubjectNodeID() + "\"";

            String leftSide = isPropAttr ? "rdf:resource=" : "rdf:about=";

            return leftSide
                   + "\"" + getQName(triple.getSubjectUriRef(), true) + "\"";
        } else {
            if (triple.getObjectNodeID() != null)
                return "rdf:nodeID=\"" + triple.getSubjectNodeID() + "\"";

            String leftSide = isPropAttr ? "rdf:resource=" : "rdf:about=";

            return leftSide
                   + "\"" + getQName(triple.getObjectUriRef(), true) + "\"";
        }
    }

    /**
     * Construct an RDFXMLSaverImpl object with a writer and a RDFS document
     * 
     * @param writer
     *            the writer to use
     * @param document
     *            the document to save
     */
    public RDFXMLSaverImpl(Writer writer, Document document) {
        this.writer = writer;
        isElementStarting = false;
        indentLevel = 0;
        elementType = "";
        graph = document.getComplementalGraph();
        for (Iterator iter = document.getNamespaceDefinition().iterator(); iter
                .hasNext();) {
            NamespaceDefinition namespace = (NamespaceDefinition) iter.next();
            NamespaceHashTable.put(namespace.getNamespace().getNamespaceURIRef().getURIString(), namespace.getNamespacePrefix());
        }
        if (NamespaceHashTable.get(RDF.NAMESPACE)==null)            
            NamespaceHashTable.put(RDF.NAMESPACE, "rdf");
        if (NamespaceHashTable.get(RDFS.NAMESPACE)==null)   
            NamespaceHashTable.put(RDFS.NAMESPACE, "rdfs");
        if (NamespaceHashTable.get(RDFS.XSD_NAMESPACE)==null)   
            NamespaceHashTable.put(RDFS.XSD_NAMESPACE, "xsd");
    }

    public String getNamespaceAbbr(String uri) {
        String abbr = (String) NamespaceHashTable.get(uri);
        if (abbr == null)
            return null;
        else
            return abbr;
    }

    /**
     * Save a document
     * 
     */
    public void save() {
        ArrayList triples = graph.exportTriples();
        Triple triple;
        String subjectURI = null;
        String currentURI = null;
        ArrayList object = new ArrayList();
        for (Iterator iterator = triples.iterator(); iterator.hasNext();) {
            triple = (Triple) iterator.next();
            if (subjectURI == null)
                if (triple.isSubjectURI())
                    subjectURI = triple.getSubjectURI();
                else
                    subjectURI = triple.getSubjectNodeID();
            if (triple.isSubjectURI())
                currentURI = triple.getSubjectURI();
            else
                currentURI = triple.getSubjectNodeID();
            if (subjectURI.equals(currentURI)) {
                object.add(triple);
            } else {
                saveResource(object);
                object.clear();
                subjectURI = currentURI;
                object.add(triple);
            }
        }
        saveResource(object);
        object.clear();
    }

    /**
     * Save a bunch of rdf triples with the same subject.
     * 
     * @param triples
     *            the RDF triples to be processed
     */
    protected void saveResource(ArrayList triples) {
        indentLevel = 0;
        isElementStarting = true;
        URIReference type = null;
        Triple triple, typeTriple = null;
        int number = 0;
        Iterator iterator;
        for (iterator = triples.iterator(); iterator.hasNext();) {
            triple = (Triple) iterator.next();
            if (triple.getPredicate().equals(RDF.P_TYPE_STR)) {
                if (!triple.getObjectURI().equals(RDFS.C_RESOURCE_STR)) {
                    type = triple.getObjectUriRef();
                    number++;
                    typeTriple = triple;
                }
            }
        }
        if (number == 1) {
            elementType = getQName(type, false);
            writeElementStarting(typeTriple);
            triples.remove(typeTriple);
            iterator = triples.iterator();
            while (iterator.hasNext()) {
                writeStatement((Triple) iterator.next());
            }
            writeElementEnding();
        } else {
            elementType = "rdf:Description";
            iterator = triples.iterator();
            writeElementStarting((Triple) iterator.next());
            iterator = triples.iterator();
            while (iterator.hasNext()) {
                writeStatement((Triple) iterator.next());
            }
            writeElementEnding();
        }
    }

    /**
     * Replace invalid characters in an XML stream
     * 
     * @param str
     *            the string to be processed
     * @return a valid XML string
     */
    private String replaceKeywords(String str) {
        str = str.replaceAll("&", "&amp;");
        str = str.replaceAll("<", "&lt;");
        str = str.replaceAll(">", "&gt;");
        str = str.replaceAll("'", "&apos;");
        str = str.replaceAll("\"", "&quot;");
        return str;
    }
}
