/**
 * <copyright> 
 * 
 * Copyright (c) 2004-2005 IBM Corporation and others. 
 * All rights reserved. This program and the accompanying materials 
 * are made available under the terms of the Eclipse Public License - v 1.0 
 * which accompanies this distribution, and is available at 
 * http://opensource.org/licenses/eclipse-1.0.txt 
 * 
 * Contributors: 
 *   IBM - Initial API and implementation 
 * 
 * </copyright> 
 * 
 * $Id: OWLParserTripleAnalyzer.java,v 1.1 2007/03/18 09:07:04 lzhang Exp $
 */

package org.eclipse.eodm.owl.resource.parser.impl;

import java.net.URISyntaxException;
import java.util.ArrayList;
import java.util.List;

import org.eclipse.eodm.OWLFactory;
import org.eclipse.eodm.RDFFactory;
import org.eclipse.eodm.exceptions.AddingTripleException;
import org.eclipse.eodm.exceptions.InvalidLexicalFormException;
import org.eclipse.eodm.exceptions.UnsupportedViewTypeException;
import org.eclipse.eodm.owl.owlbase.OWLGraph;
import org.eclipse.eodm.owl.resource.parser.TripleAnalyzer;
import org.eclipse.eodm.owl.resource.parser.exception.OWLParserException;
import org.eclipse.eodm.rdf.rdfweb.Document;
import org.eclipse.eodm.rdf.rdfweb.NamespaceDefinition;
import org.eclipse.eodm.rdf.resource.parser.element.RDFBlankNode;
import org.eclipse.eodm.rdf.resource.parser.element.RDFLiteralElement;
import org.eclipse.eodm.rdf.resource.parser.element.RDFResourceElement;
import org.eclipse.eodm.rdf.resource.parser.element.RDFValue;
import org.eclipse.eodm.rdf.resource.parser.element.URIReference;
import org.eclipse.eodm.rdf.resource.parser.impl.RDFNamespaceMap;
import org.eclipse.eodm.util.Triple;
import org.eclipse.eodm.vocabulary.OWL;
import org.eclipse.eodm.vocabulary.RDF;
import org.eclipse.eodm.vocabulary.RDFS;

/**
 * OWLTripleAnalyzer analyze triples from an RDF triple parser
 */
public class OWLParserTripleAnalyzer implements TripleAnalyzer {
    /* The namespace map */
    private RDFNamespaceMap namespaceMap = new RDFNamespaceMap();

    /**
     * Get the namespaces map
     * 
     * @return the namespaces map
     */
    public RDFNamespaceMap getNamespaceMap() {
        return namespaceMap;
    }

    private Document document;

    private OWLGraph graph;

    OWLOntologyGraph ontologygraph;
    
    private List specialTriples=new ArrayList();
    /**
     * Get the EODM document.
     * 
     * @return EODM document
     */
    public Document getDocument() {
        return document;
    }

    /**
     * Get the generated ontology graph
     * 
     * @return
     */
    public OWLOntologyGraph getOWLOntologyGraph() {
        return ontologygraph;
    }

    /**
     * Initialize the analyser. Add default namespaces to a hashtable
     * 
     * @param documentURI
     */
    public void initialize(String documentURI) {

        try {
            document = RDFFactory.eINSTANCE.createDocument(documentURI);
            graph = OWLFactory.eINSTANCE.createOWLGraph(documentURI);
            document.setComplementalGraph(graph);
            ontologygraph = new OWLOntologyGraph();
            ontologygraph.setDocument(document);
            ontologygraph.setgraph(graph);
            specialTriples.clear() ;
        } catch (URISyntaxException e) {
            e.printStackTrace();
            throw new OWLParserException("URI error:", e);
        }
        handleNamespace("rdf", RDF.NAMESPACE);
        handleNamespace("rdfs", RDFS.NAMESPACE);
        handleNamespace("owl", OWL.NAMESPACE_URI);
        handleNamespace("xsd", RDFS.XSD_NAMESPACE);
        handleNamespace("bnode", RDFS.BNODE_NAMESPACE);
    }

    /**
     * Handle namespaces generated during RDF parsing.
     * 
     * @param name
     *            The abbreviation of the namespace
     * @param uri
     *            The namespace full uri
     * @return the Namespace handled
     */
    public NamespaceDefinition handleNamespace(String name, String uri) {
        NamespaceDefinition namespace = namespaceMap.getNamespace(uri);
        /*
         * if the namespace is not found or it's not the BNode namespace &&
         * names are not equivalent, then add it
         */
        if (namespace == null
            || !name.equals(namespace.getNamespacePrefix())
            && !uri.equals(RDFS.BNODE_NAMESPACE)
            && !namespace.getNamespacePrefix().startsWith("ns_")) {
            namespace = createNamespace(name, uri);
            // if the namespace's prefix is null, it will be set a default value
            namespaceMap.addNamespace(namespace);
            document.getNamespaceDefinition().add(namespace);
        } else if (namespace != null
                   && name != null && name.length() > 0
                   && namespace.getNamespacePrefix().startsWith("ns_")
                   && !name.startsWith("ns_"))
            namespace.setNamespacePrefix(name);
        return namespace;
    }

    /**
     * Create a new namespace with the name and uri designated.
     * 
     * @param name
     *            The namespace name
     * @param uri
     *            The namespace uri
     * @return The namespace created
     */
    private NamespaceDefinition createNamespace(String name, String uri) {
        NamespaceDefinition namespace = null;
        try {
            namespace = RDFFactory.eINSTANCE.createNamespaceDefinition(name,
                    RDFFactory.eINSTANCE.createNamespace(uri));
        } catch (URISyntaxException e) {
            e.printStackTrace();
            throw new OWLParserException("URI error:", e);
        }
        namespace.setNamespacePrefix(name);
        return namespace;
    }

    /**
     * Get a namespace by the specific uri.
     * 
     * @param uri
     *            The namespace uri
     * @return The namespace object
     */
    public NamespaceDefinition getNamespaceByURI(String uri) {
        NamespaceDefinition namespace = (NamespaceDefinition) namespaceMap
                .getNamespace(uri);
        if (namespace == null)
            namespace = handleNamespace(null, uri);
        return namespace;
    }

    /**
     * The analyzing process ends.
     */
    public void endAnalyzing() {
    }

    /**
     * Clear all the temporary resources created during parsing process.
     */
    public void clearResources() {
        // remove all namespacces
        namespaceMap.removeAllNamespaces();
    }

    /**
     * Analyse an RDF triple {subject, predicate, object}, it's the first step.
     * 
     * @param subject
     *            The subject of the triple
     * @param predicate
     *            The predicate of the triple
     * @param object
     *            The object of the triple
     */
    public void analyseStatement(RDFResourceElement subject,
            URIReference predicate, RDFValue object) {
               
        Triple triple = new Triple();

        if (subject instanceof RDFBlankNode)
            triple.setSubjectNodeID(subject.getLocalName());
        else{
            triple.setSubjectURI(getNamespaceByURI(subject.getNamespace())
                    .getNamespace(), subject.getLocalName());
            if (subject.getLocalName().indexOf("Medium")!=-1)
                System.out.println( "read a triple whose subject includes substring Medium "+subject.getFullURI());
        }

        triple.setPredicateUriRef(getNamespaceByURI(predicate.getNamespace())
                .getNamespace(), predicate.getLocalName());
        if (predicate.getLocalName().indexOf("comment")!=-1)
            System.out.println( "read rdfs:comment "+subject.getFullURI());
        if (object instanceof RDFBlankNode)
            triple.setObjectNodeID(((RDFBlankNode) object).getLocalName());
        else if (object instanceof URIReference) {
            URIReference uri = (URIReference) object;
            triple.setObjectURI(getNamespaceByURI(uri.getNamespace())
                    .getNamespace(), uri.getLocalName());
            if (uri.getLocalName().indexOf("TransitiveProperty")!=-1)
                System.out.println( "read transitiveProperty "+subject.getFullURI());
        } else {
            RDFLiteralElement literal = (RDFLiteralElement) object;
            URIReference typeRef = literal.getDatatype();
            if (typeRef != null) {
                if (typeRef.getFullURI().equals(
                        RDF.NAMESPACE + RDF.C_XMLLITERAL))
                    try {
                        triple.setObjectLiteral(RDFFactory.eINSTANCE
                                .createRDFXMLLiteral(literal.getText()));
                    } catch (InvalidLexicalFormException e) {
                        e.printStackTrace();
                        throw new OWLParserException("Invalid literals:", e);
                    }
                else
                    try {
                        triple.setObjectLiteral(RDFFactory.eINSTANCE
                                .createTypedLiteral(literal.getText(), literal
                                        .getDatatype().getFullURI()));
                    } catch (URISyntaxException e) {
                        e.printStackTrace();
                        throw new OWLParserException("URI error:", e);
                    } catch (InvalidLexicalFormException e) {
                        e.printStackTrace();
                        throw new OWLParserException("Invalid literals:", e);
                    }
            } else if (literal.getLanguage() != null) {
                triple.setObjectLiteral(RDFFactory.eINSTANCE
                        .createPlainLiteral(literal.getText(), literal
                                .getLanguage()));
            } else {
                if (predicate.getFullURI().equals(RDFS.P_COMMENT_STR)
                    || predicate.getFullURI().equals(RDFS.P_LABEL_STR))
                    triple.setObjectLiteral(RDFFactory.eINSTANCE
                            .createPlainLiteral(literal.getText(), null));
                else
                    triple.setObjectLiteral(RDFFactory.eINSTANCE
                            .createRDFSLiteral(literal.getText()));
            }
        }
        try {
            if (!predicate.getFullURI().equals(RDF.P_TYPE_STR) ||
                    predicate.getFullURI().equals(RDF.P_TYPE_STR) && triple.isObjectURI() && !(triple.getObjectURI().startsWith(OWL.NAMESPACE_URI) || triple.getObjectURI().startsWith(RDFS.NAMESPACE) || triple.getObjectURI().startsWith(RDF.NAMESPACE))
                    || predicate.getFullURI().equals(RDF.P_TYPE_STR) && triple.isObjectBlankNode()){
                specialTriples.add(triple);
            }
            else
                this.graph.addTriple(triple);
        } catch (UnsupportedViewTypeException e) {
            e.printStackTrace();
            throw new OWLParserException("Parser error:", e);
        } catch (URISyntaxException e) {
            e.printStackTrace();
            throw new OWLParserException("URI error:", e);
        } catch (AddingTripleException e ) {
            e.printStackTrace();
            throw new OWLParserException("Adding triple error:", e);        	
        }
    }
    
    /**
     * Process unhandled triples, such as triples with OWLAllValuesFrom as property
     *
     */
    public void postProcessing(){        
        for(int i=0;i<specialTriples.size();i++){
            try {
                this.graph.addTriple((Triple)specialTriples.get(i));
            } catch (UnsupportedViewTypeException e) {
                e.printStackTrace();
                throw new OWLParserException("Parser error:", e);
            } catch (URISyntaxException e) {
                e.printStackTrace();
                throw new OWLParserException("URI error:", e);
            } catch (AddingTripleException e) {
                e.printStackTrace();
                throw new OWLParserException("Adding triple:", e);            	
            }
        }
    }
}