/**
 * Copyright (c) 2016, 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package org.eclipse.gemoc.executionframework.debugger;

import com.google.common.base.Objects;
import java.lang.reflect.Field;
import java.lang.reflect.Method;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Collections;
import java.util.HashMap;
import java.util.HashSet;
import java.util.LinkedHashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;
import java.util.function.BiConsumer;
import java.util.function.Consumer;
import java.util.function.Supplier;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.impl.EObjectImpl;
import org.eclipse.gemoc.executionframework.debugger.IMutableFieldExtractor;
import org.eclipse.gemoc.executionframework.debugger.MutableField;
import org.eclipse.gemoc.executionframework.engine.commons.DslHelper;
import org.eclipse.gemoc.executionframework.engine.commons.K3DslHelper;
import org.eclipse.xtext.xbase.lib.Conversions;
import org.eclipse.xtext.xbase.lib.Exceptions;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.InputOutput;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.Pair;
import org.osgi.framework.Bundle;

@SuppressWarnings("all")
public class IntrospectiveMutableFieldExtractor implements IMutableFieldExtractor {
  private String languageName;
  
  private Map<EObject, List<MutableField>> eObjects = new HashMap<EObject, List<MutableField>>();
  
  private Map<EClass, List<Pair<Class<?>, Class<?>>>> aspectClasses = new HashMap<EClass, List<Pair<Class<?>, Class<?>>>>();
  
  public IntrospectiveMutableFieldExtractor(final String languageName) {
    this.languageName = languageName;
  }
  
  private String decapitalize(final String string) {
    final char[] c = string.toCharArray();
    char _get = c[0];
    char _lowerCase = Character.toLowerCase(_get);
    c[0] = _lowerCase;
    return new String(c);
  }
  
  private String findName(final Class<?> cls, final EObject eObject) {
    try {
      Field[] _declaredFields = cls.getDeclaredFields();
      final Function1<Field, Boolean> _function = new Function1<Field, Boolean>() {
        @Override
        public Boolean apply(final Field f) {
          String _name = f.getName();
          return Boolean.valueOf(_name.equals("name"));
        }
      };
      final Iterable<Field> name = IterableExtensions.<Field>filter(((Iterable<Field>)Conversions.doWrapArray(_declaredFields)), _function);
      boolean _isEmpty = IterableExtensions.isEmpty(name);
      if (_isEmpty) {
        Class<?> _superclass = cls.getSuperclass();
        boolean _notEquals = (!Objects.equal(_superclass, EObjectImpl.class));
        if (_notEquals) {
          Class<?> _superclass_1 = cls.getSuperclass();
          return this.findName(_superclass_1, eObject);
        }
        return null;
      } else {
        final Field f = ((Field[])Conversions.unwrapArray(name, Field.class))[0];
        f.setAccessible(true);
        Object _get = f.get(eObject);
        return _get.toString();
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private String findId(final Class<?> cls, final EObject eObject) {
    try {
      Field[] _declaredFields = cls.getDeclaredFields();
      final Function1<Field, Boolean> _function = new Function1<Field, Boolean>() {
        @Override
        public Boolean apply(final Field f) {
          String _name = f.getName();
          return Boolean.valueOf(_name.equals("id"));
        }
      };
      final Iterable<Field> id = IterableExtensions.<Field>filter(((Iterable<Field>)Conversions.doWrapArray(_declaredFields)), _function);
      boolean _isEmpty = IterableExtensions.isEmpty(id);
      if (_isEmpty) {
        Class<?> _superclass = cls.getSuperclass();
        boolean _notEquals = (!Objects.equal(_superclass, EObjectImpl.class));
        if (_notEquals) {
          Class<?> _superclass_1 = cls.getSuperclass();
          return this.findId(_superclass_1, eObject);
        }
        return null;
      } else {
        final Field f = ((Field[])Conversions.unwrapArray(id, Field.class))[0];
        f.setAccessible(true);
        Object _get = f.get(eObject);
        return _get.toString();
      }
    } catch (Throwable _e) {
      throw Exceptions.sneakyThrow(_e);
    }
  }
  
  private String findDataName(final EObject eObject) {
    Class<? extends EObject> _class = eObject.getClass();
    final String name = this.findName(_class, eObject);
    if ((name == null)) {
      Class<? extends EObject> _class_1 = eObject.getClass();
      final String id = this.findId(_class_1, eObject);
      if ((id == null)) {
        return eObject.toString();
      } else {
        EClass _eClass = eObject.eClass();
        String _name = _eClass.getName();
        String _decapitalize = this.decapitalize(_name);
        String _plus = (_decapitalize + " ");
        return (_plus + id);
      }
    } else {
      return name;
    }
  }
  
  private List<MutableField> getMutableFieldsFromAspect(final EObject eObject, final Class<?> properties, final Class<?> aspect) {
    final ArrayList<MutableField> result = new ArrayList<MutableField>();
    final Field[] fields = properties.getFields();
    boolean _isEmpty = ((List<Field>)Conversions.doWrapArray(fields)).isEmpty();
    boolean _not = (!_isEmpty);
    if (_not) {
      final Consumer<Field> _function = new Consumer<Field>() {
        @Override
        public void accept(final Field f) {
          Method[] _methods = aspect.getMethods();
          final Function1<Method, Boolean> _function = new Function1<Method, Boolean>() {
            @Override
            public Boolean apply(final Method m) {
              String _name = m.getName();
              String _name_1 = f.getName();
              return Boolean.valueOf(_name.equals(_name_1));
            }
          };
          final Iterable<Method> methods = IterableExtensions.<Method>filter(((Iterable<Method>)Conversions.doWrapArray(_methods)), _function);
          final Function1<Method, Boolean> _function_1 = new Function1<Method, Boolean>() {
            @Override
            public Boolean apply(final Method m) {
              int _parameterCount = m.getParameterCount();
              return Boolean.valueOf((_parameterCount == 1));
            }
          };
          final Method getter = IterableExtensions.<Method>findFirst(methods, _function_1);
          final Function1<Method, Boolean> _function_2 = new Function1<Method, Boolean>() {
            @Override
            public Boolean apply(final Method m) {
              int _parameterCount = m.getParameterCount();
              return Boolean.valueOf((_parameterCount == 2));
            }
          };
          final Method setter = IterableExtensions.<Method>findFirst(methods, _function_2);
          if (((getter != null) && (setter != null))) {
            String _findDataName = IntrospectiveMutableFieldExtractor.this.findDataName(eObject);
            final Supplier<Object> _function_3 = new Supplier<Object>() {
              @Override
              public Object get() {
                try {
                  return getter.invoke(null, eObject);
                } catch (Throwable _e) {
                  throw Exceptions.sneakyThrow(_e);
                }
              }
            };
            final Consumer<Object> _function_4 = new Consumer<Object>() {
              @Override
              public void accept(final Object t) {
                try {
                  setter.invoke(null, eObject, t);
                } catch (Throwable _e) {
                  throw Exceptions.sneakyThrow(_e);
                }
              }
            };
            final MutableField data = new MutableField(_findDataName, eObject, _function_3, _function_4);
            result.add(data);
          }
        }
      };
      ((List<Field>)Conversions.doWrapArray(fields)).forEach(_function);
    }
    return result;
  }
  
  @Override
  public List<MutableField> extractMutableField(final EObject eObject) {
    boolean _containsKey = this.eObjects.containsKey(eObject);
    boolean _not = (!_containsKey);
    if (_not) {
      final ArrayList<MutableField> datas = new ArrayList<MutableField>();
      EClass _eClass = eObject.eClass();
      boolean _containsKey_1 = this.aspectClasses.containsKey(_eClass);
      boolean _not_1 = (!_containsKey_1);
      if (_not_1) {
        final Map<Class<?>, List<Class<?>>> classes = this.getStaticHelperClasses(eObject);
        if ((classes != null)) {
          final ArrayList<Pair<Class<?>, Class<?>>> list = new ArrayList<Pair<Class<?>, Class<?>>>();
          final BiConsumer<Class<?>, List<Class<?>>> _function = new BiConsumer<Class<?>, List<Class<?>>>() {
            @Override
            public void accept(final Class<?> i, final List<Class<?>> l) {
              final Consumer<Class<?>> _function = new Consumer<Class<?>>() {
                @Override
                public void accept(final Class<?> c) {
                  try {
                    Bundle _dslBundle = DslHelper.getDslBundle(IntrospectiveMutableFieldExtractor.this.languageName);
                    String _name = c.getName();
                    String _simpleName = i.getSimpleName();
                    String _plus = (_name + _simpleName);
                    String _plus_1 = (_plus + "AspectProperties");
                    final Class<?> properties = _dslBundle.loadClass(_plus_1);
                    final Pair<Class<?>, Class<?>> pair = new Pair<Class<?>, Class<?>>(c, properties);
                    list.add(pair);
                    List<MutableField> _mutableFieldsFromAspect = IntrospectiveMutableFieldExtractor.this.getMutableFieldsFromAspect(eObject, properties, c);
                    datas.addAll(_mutableFieldsFromAspect);
                  } catch (final Throwable _t) {
                    if (_t instanceof ClassNotFoundException) {
                      final ClassNotFoundException e = (ClassNotFoundException)_t;
                    } else {
                      throw Exceptions.sneakyThrow(_t);
                    }
                  }
                }
              };
              l.forEach(_function);
            }
          };
          classes.forEach(_function);
          EClass _eClass_1 = eObject.eClass();
          this.aspectClasses.put(_eClass_1, list);
        } else {
          EClass _eClass_2 = eObject.eClass();
          this.aspectClasses.put(_eClass_2, Collections.EMPTY_LIST);
        }
      } else {
        EClass _eClass_3 = eObject.eClass();
        final List<Pair<Class<?>, Class<?>>> list_1 = this.aspectClasses.get(_eClass_3);
        final Consumer<Pair<Class<?>, Class<?>>> _function_1 = new Consumer<Pair<Class<?>, Class<?>>>() {
          @Override
          public void accept(final Pair<Class<?>, Class<?>> p) {
            Class<?> _value = p.getValue();
            Class<?> _key = p.getKey();
            List<MutableField> _mutableFieldsFromAspect = IntrospectiveMutableFieldExtractor.this.getMutableFieldsFromAspect(eObject, _value, _key);
            datas.addAll(_mutableFieldsFromAspect);
          }
        };
        list_1.forEach(_function_1);
      }
      this.eObjects.put(eObject, datas);
      return datas;
    } else {
      return this.eObjects.get(eObject);
    }
  }
  
  private void getSuperInterfacesOfInterface(final Class<?> c, final HashSet<Class<?>> set) {
    Class<?>[] _interfaces = c.getInterfaces();
    List<Class<?>> _asList = Arrays.<Class<?>>asList(_interfaces);
    final Function1<Class<?>, Boolean> _function = new Function1<Class<?>, Boolean>() {
      @Override
      public Boolean apply(final Class<?> i) {
        boolean _equals = i.equals(EObject.class);
        return Boolean.valueOf((!_equals));
      }
    };
    final Iterable<Class<?>> possibleSuperInterfaces = IterableExtensions.<Class<?>>filter(_asList, _function);
    final Consumer<Class<?>> _function_1 = new Consumer<Class<?>>() {
      @Override
      public void accept(final Class<?> i) {
        boolean _add = set.add(i);
        if (_add) {
          IntrospectiveMutableFieldExtractor.this.getSuperInterfacesOfInterface(i, set);
        }
      }
    };
    possibleSuperInterfaces.forEach(_function_1);
  }
  
  private List<Class<?>> getSuperInterfacesOfInterface(final Class<?> c) {
    if ((c == null)) {
      return Collections.EMPTY_LIST;
    }
    final LinkedHashSet<Class<?>> interfacesFound = new LinkedHashSet<Class<?>>();
    this.getSuperInterfacesOfInterface(c, interfacesFound);
    return new ArrayList<Class<?>>(interfacesFound);
  }
  
  private List<Class<?>> getInterfacesOfEObject(final EObject o) {
    final List<Class<?>> possibleInterfaces = new ArrayList<Class<?>>();
    Class<? extends EObject> _class = o.getClass();
    final List<Class<?>> interfaces = this.getAllInterfaces(_class);
    final Function1<Class<?>, Boolean> _function = new Function1<Class<?>, Boolean>() {
      @Override
      public Boolean apply(final Class<?> i) {
        String _simpleName = i.getSimpleName();
        EClass _eClass = o.eClass();
        String _name = _eClass.getName();
        return Boolean.valueOf(_simpleName.equals(_name));
      }
    };
    final Class<?> baseInterface = IterableExtensions.<Class<?>>findFirst(interfaces, _function);
    if ((baseInterface != null)) {
      possibleInterfaces.add(baseInterface);
      List<Class<?>> _superInterfacesOfInterface = this.getSuperInterfacesOfInterface(baseInterface);
      possibleInterfaces.addAll(_superInterfacesOfInterface);
    }
    InputOutput.<List<Class<?>>>println(possibleInterfaces);
    return possibleInterfaces;
  }
  
  private List<Class<?>> getAllInterfaces(final Class<? extends EObject> cls) {
    if ((cls == null)) {
      return Collections.EMPTY_LIST;
    }
    final LinkedHashSet<Class<?>> interfacesFound = new LinkedHashSet<Class<?>>();
    this.getAllInterfaces(cls, interfacesFound);
    final ArrayList<Class<?>> res = new ArrayList<Class<?>>(interfacesFound);
    return res;
  }
  
  private void getAllInterfaces(final Class<?> cls, final HashSet<Class<?>> interfacesFound) {
    Class<?> currCls = cls;
    while ((currCls != null)) {
      {
        Class<?>[] _interfaces = currCls.getInterfaces();
        final Consumer<Class<?>> _function = new Consumer<Class<?>>() {
          @Override
          public void accept(final Class<?> i) {
            boolean _add = interfacesFound.add(i);
            if (_add) {
              IntrospectiveMutableFieldExtractor.this.getAllInterfaces(i, interfacesFound);
            }
          }
        };
        ((List<Class<?>>)Conversions.doWrapArray(_interfaces)).forEach(_function);
        Class<?> _superclass = currCls.getSuperclass();
        currCls = _superclass;
      }
    }
  }
  
  private Map<Class<?>, List<Class<?>>> getStaticHelperClasses(final EObject target) {
    final List<Class<?>> allPossibleInterfaces = this.getInterfacesOfEObject(target);
    final Map<Class<?>, List<Class<?>>> res = new HashMap<Class<?>, List<Class<?>>>();
    final Set<Class<?>> allAspects = K3DslHelper.getAspects(this.languageName);
    final Consumer<Class<?>> _function = new Consumer<Class<?>>() {
      @Override
      public void accept(final Class<?> i) {
        final Function1<Class<?>, Boolean> _function = new Function1<Class<?>, Boolean>() {
          @Override
          public Boolean apply(final Class<?> asp) {
            Class<?> _target = K3DslHelper.getTarget(asp);
            return Boolean.valueOf(Objects.equal(_target, i));
          }
        };
        final Iterable<Class<?>> appliedAspects = IterableExtensions.<Class<?>>filter(allAspects, _function);
        List<Class<?>> _list = IterableExtensions.<Class<?>>toList(appliedAspects);
        res.put(i, _list);
      }
    };
    allPossibleInterfaces.forEach(_function);
    return res;
  }
}
