/**
 * Copyright (c) 2016, 2017 Inria and others.
 * All rights reserved. This program and the accompanying materials
 * are made available under the terms of the Eclipse Public License v1.0
 * which accompanies this distribution, and is available at
 * http://www.eclipse.org/legal/epl-v10.html
 * 
 * Contributors:
 *     Inria - initial API and implementation
 */
package org.eclipse.gemoc.executionframework.debugger;

import com.google.common.base.Objects;
import com.google.common.collect.Iterables;
import fr.inria.diverse.melange.resource.MelangeResourceImpl;
import java.util.ArrayDeque;
import java.util.ArrayList;
import java.util.Deque;
import java.util.Iterator;
import java.util.List;
import java.util.function.BiPredicate;
import org.eclipse.emf.common.util.EList;
import org.eclipse.emf.ecore.EClass;
import org.eclipse.emf.ecore.EObject;
import org.eclipse.emf.ecore.EOperation;
import org.eclipse.emf.ecore.EcorePackage;
import org.eclipse.emf.ecore.resource.Resource;
import org.eclipse.emf.ecore.resource.ResourceSet;
import org.eclipse.gemoc.dsl.debug.ide.IDSLDebugger;
import org.eclipse.gemoc.dsl.debug.ide.event.IDSLDebugEventProcessor;
import org.eclipse.gemoc.executionframework.debugger.AbstractGemocDebugger;
import org.eclipse.gemoc.executionframework.debugger.GemocBreakpoint;
import org.eclipse.gemoc.executionframework.engine.core.EngineStoppedException;
import org.eclipse.gemoc.trace.commons.model.helper.StepHelper;
import org.eclipse.gemoc.trace.commons.model.trace.MSE;
import org.eclipse.gemoc.trace.commons.model.trace.MSEOccurrence;
import org.eclipse.gemoc.trace.commons.model.trace.ParallelStep;
import org.eclipse.gemoc.trace.commons.model.trace.Step;
import org.eclipse.gemoc.xdsmlframework.api.core.IExecutionContext;
import org.eclipse.gemoc.xdsmlframework.api.core.IExecutionEngine;
import org.eclipse.xtend2.lib.StringConcatenation;
import org.eclipse.xtext.naming.DefaultDeclarativeQualifiedNameProvider;
import org.eclipse.xtext.naming.QualifiedName;
import org.eclipse.xtext.xbase.lib.Functions.Function1;
import org.eclipse.xtext.xbase.lib.IterableExtensions;
import org.eclipse.xtext.xbase.lib.ListExtensions;

@SuppressWarnings("all")
public class GenericSequentialModelDebugger extends AbstractGemocDebugger {
  public static class MSEFrameInformation {
    public final EObject caller;
    
    public final String prettyLabel;
    
    public MSEFrameInformation(final EObject caller, final String prettyLabel) {
      this.caller = caller;
      this.prettyLabel = prettyLabel;
    }
  }
  
  private static class ToPushPop {
    public Step<?> step;
    
    public boolean push;
    
    public ToPushPop(final Step<?> step, final boolean push) {
      this.step = step;
      this.push = push;
    }
  }
  
  /**
   * A fake instruction to prevent the stepping return to stop on each event.
   */
  private final static EObject FAKE_INSTRUCTION = EcorePackage.eINSTANCE;
  
  private List<GenericSequentialModelDebugger.ToPushPop> toPushPop = new ArrayList<GenericSequentialModelDebugger.ToPushPop>();
  
  protected final String threadName = "Model debugging";
  
  protected int nbStackFrames = 0;
  
  protected boolean executionTerminated = false;
  
  public GenericSequentialModelDebugger(final IDSLDebugEventProcessor target, final IExecutionEngine engine) {
    super(target, engine);
  }
  
  /**
   * This method is eventually called within a new engine thread. (non-Javadoc)
   * 
   * @see IDSLDebugger#start()
   */
  @Override
  public void start() {
    this.engine.start();
  }
  
  @Override
  public void disconnect() {
    return;
  }
  
  protected void setupStepReturnPredicateBreak() {
    abstract class __GenericSequentialModelDebugger_1 implements BiPredicate<IExecutionEngine, Step<?>> {
      Step<?> steppedReturn;
    }
    
    final IExecutionEngine seqEngine = ((IExecutionEngine) this.engine);
    final Deque<Step<?>> stack = seqEngine.getCurrentStack();
    int _size = stack.size();
    boolean _greaterThan = (_size > 1);
    if (_greaterThan) {
      final Iterator<Step<?>> it = stack.iterator();
      it.next();
      __GenericSequentialModelDebugger_1 ___GenericSequentialModelDebugger_1 = new __GenericSequentialModelDebugger_1() {
        {
          steppedReturn = it.next();
        }
        @Override
        public boolean test(final IExecutionEngine t, final Step<?> u) {
          Deque<Step<?>> _currentStack = seqEngine.getCurrentStack();
          boolean _contains = _currentStack.contains(this.steppedReturn);
          return (!_contains);
        }
      };
      this.addPredicateBreak(___GenericSequentialModelDebugger_1);
    }
  }
  
  @Override
  public void steppingReturn(final String threadName) {
    super.steppingReturn(threadName);
    this.setupStepReturnPredicateBreak();
  }
  
  protected void setupStepOverPredicateBreak() {
    abstract class __GenericSequentialModelDebugger_2 implements BiPredicate<IExecutionEngine, Step<?>> {
      IExecutionEngine seqEngine;
      
      Step<?> steppedOver;
    }
    
    __GenericSequentialModelDebugger_2 ___GenericSequentialModelDebugger_2 = new __GenericSequentialModelDebugger_2() {
      {
        seqEngine = ((IExecutionEngine) GenericSequentialModelDebugger.this.engine);
        
        steppedOver = this.seqEngine.getCurrentStep();
      }
      @Override
      public boolean test(final IExecutionEngine t, final Step<?> u) {
        Deque<Step<?>> _currentStack = this.seqEngine.getCurrentStack();
        boolean _contains = _currentStack.contains(this.steppedOver);
        return (!_contains);
      }
    };
    this.addPredicateBreak(___GenericSequentialModelDebugger_2);
  }
  
  @Override
  public void steppingOver(final String threadName) {
    super.steppingOver(threadName);
    this.setupStepOverPredicateBreak();
  }
  
  @Override
  public boolean canStepInto(final String threadName, final EObject instruction) {
    final EObject currentInstruction = this.currentInstructions.get(threadName);
    final Step<?> currentStep = this.engine.getCurrentStep();
    final boolean correctObject = Objects.equal(currentInstruction, instruction);
    final boolean canStepInto = (!(currentStep instanceof ParallelStep<?, ?>));
    return (correctObject && canStepInto);
  }
  
  @Override
  public void steppingInto(final String threadName) {
    super.steppingInto(threadName);
    this.addPredicateBreak(new BiPredicate<IExecutionEngine, Step<?>>() {
      @Override
      public boolean test(final IExecutionEngine t, final Step<?> u) {
        return true;
      }
    });
  }
  
  @Override
  public void pushStackFrame(final String threadName, final String frameName, final EObject context, final EObject instruction) {
    super.pushStackFrame(threadName, frameName, context, instruction);
    this.nbStackFrames++;
  }
  
  @Override
  public void popStackFrame(final String threadName) {
    super.popStackFrame(threadName);
    this.nbStackFrames--;
  }
  
  protected final DefaultDeclarativeQualifiedNameProvider nameprovider = new DefaultDeclarativeQualifiedNameProvider();
  
  protected String prettyFrameName(final MSE mse, final boolean implicit) {
    if ((mse != null)) {
      EObject caller = mse.getCaller();
      final QualifiedName qname = this.nameprovider.getFullyQualifiedName(caller);
      String _xifexpression = null;
      if ((qname != null)) {
        _xifexpression = qname.toString();
      } else {
        _xifexpression = caller.toString();
      }
      final String objectName = _xifexpression;
      String _xifexpression_1 = null;
      if (implicit) {
        EOperation _action = mse.getAction();
        String _name = null;
        if (_action!=null) {
          _name=_action.getName();
        }
        _xifexpression_1 = (_name + "_implicitStep");
      } else {
        EOperation _action_1 = mse.getAction();
        String _name_1 = null;
        if (_action_1!=null) {
          _name_1=_action_1.getName();
        }
        _xifexpression_1 = _name_1;
      }
      final String opName = _xifexpression_1;
      EClass _eClass = caller.eClass();
      final String callerType = _eClass.getName();
      final String prettyName = (((((("(" + callerType) + ") ") + objectName) + " -> ") + opName) + "()");
      return prettyName;
    }
    return null;
  }
  
  protected GenericSequentialModelDebugger.MSEFrameInformation getMSEFrameInformation(final Step<?> step) {
    final MSEOccurrence mseOccurrence = step.getMseoccurrence();
    final EObject container = step.eContainer();
    String prettyName = "";
    EObject caller = null;
    if ((mseOccurrence != null)) {
      MSE _mse = mseOccurrence.getMse();
      EObject _caller = _mse.getCaller();
      caller = _caller;
      MSE _mse_1 = mseOccurrence.getMse();
      String _prettyFrameName = this.prettyFrameName(_mse_1, false);
      prettyName = _prettyFrameName;
    } else {
      if ((((container != null) && (container instanceof Step<?>)) && (((Step<?>) container).getMseoccurrence() != null))) {
        MSEOccurrence _mseoccurrence = ((Step<?>) container).getMseoccurrence();
        final MSE parentMSE = _mseoccurrence.getMse();
        EObject _caller_1 = parentMSE.getCaller();
        caller = _caller_1;
        String _prettyFrameName_1 = this.prettyFrameName(parentMSE, true);
        prettyName = _prettyFrameName_1;
      } else {
        if ((step instanceof ParallelStep<?, ?>)) {
          caller = step;
          StringConcatenation _builder = new StringConcatenation();
          _builder.append(" ");
          String _stepName = StepHelper.getStepName(step);
          _builder.append(_stepName, " ");
          _builder.append(" (");
          List<MSE> _mSEs = StepHelper.getMSEs(step);
          final Function1<MSE, String> _function = new Function1<MSE, String>() {
            @Override
            public String apply(final MSE it) {
              return it.getName();
            }
          };
          List<String> _map = ListExtensions.<MSE, String>map(_mSEs, _function);
          String _join = IterableExtensions.join(_map, ", ");
          _builder.append(_join, " ");
          _builder.append(")");
          prettyName = _builder.toString();
        } else {
          caller = step;
          prettyName = "Unknown step";
        }
      }
    }
    return new GenericSequentialModelDebugger.MSEFrameInformation(caller, prettyName);
  }
  
  @Override
  public void updateStack(final String threadName, final EObject instruction) {
    final Deque<Step<?>> virtualStack = new ArrayDeque<Step<?>>();
    for (final GenericSequentialModelDebugger.ToPushPop m : this.toPushPop) {
      if (m.push) {
        virtualStack.push(m.step);
      } else {
        boolean _isEmpty = virtualStack.isEmpty();
        if (_isEmpty) {
          this.popStackFrame(threadName);
        } else {
          virtualStack.pop();
        }
      }
    }
    final Iterator<Step<?>> iterator = virtualStack.descendingIterator();
    while (iterator.hasNext()) {
      {
        final Step<?> step = iterator.next();
        final GenericSequentialModelDebugger.MSEFrameInformation info = this.getMSEFrameInformation(step);
        this.pushStackFrame(threadName, info.prettyLabel, info.caller, info.caller);
      }
    }
    this.setCurrentInstruction(threadName, instruction);
    this.toPushPop.clear();
  }
  
  @Override
  public void updateData(final String threadName, final EObject instruction) {
    EObject realInstruction = instruction;
    if ((instruction == null)) {
      this.updateVariables(threadName);
      this.updateStack(threadName, null);
      return;
    }
    if ((instruction instanceof Step<?>)) {
      final Step<?> step = ((Step<?>) instruction);
      MSEOccurrence _mseoccurrence = step.getMseoccurrence();
      boolean _tripleNotEquals = (_mseoccurrence != null);
      if (_tripleNotEquals) {
        MSEOccurrence _mseoccurrence_1 = step.getMseoccurrence();
        MSE _mse = _mseoccurrence_1.getMse();
        EObject _caller = _mse.getCaller();
        realInstruction = _caller;
      }
    } else {
      if ((instruction instanceof MSEOccurrence)) {
        MSE _mse_1 = ((MSEOccurrence)instruction).getMse();
        EObject _caller_1 = _mse_1.getCaller();
        realInstruction = _caller_1;
      }
    }
    super.updateData(threadName, realInstruction);
  }
  
  @Override
  public boolean shouldBreak(final EObject instruction) {
    if ((instruction instanceof Step<?>)) {
      return this.shouldBreakStep(((Step<?>) instruction));
    } else {
      boolean _equals = Objects.equal(instruction, GenericSequentialModelDebugger.FAKE_INSTRUCTION);
      if (_equals) {
        return true;
      }
    }
    return false;
  }
  
  private boolean hasRegularBreakpointTrue(final EObject o) {
    EObject target = o;
    final Resource res = o.eResource();
    if ((res != null)) {
      ResourceSet _resourceSet = res.getResourceSet();
      EList<Resource> _resources = _resourceSet.getResources();
      Iterable<MelangeResourceImpl> _filter = Iterables.<MelangeResourceImpl>filter(_resources, MelangeResourceImpl.class);
      final MelangeResourceImpl mr = IterableExtensions.<MelangeResourceImpl>head(_filter);
      if ((mr != null)) {
        final String uriFragment = res.getURIFragment(o);
        Resource _wrappedResource = mr.getWrappedResource();
        EObject _eObject = _wrappedResource.getEObject(uriFragment);
        target = _eObject;
      }
    }
    return (super.shouldBreak(target) && ((Boolean.valueOf(((String) this.getBreakpointAttributes(target, GemocBreakpoint.BREAK_ON_LOGICAL_STEP)))).booleanValue() || 
      (Boolean.valueOf(((String) this.getBreakpointAttributes(target, GemocBreakpoint.BREAK_ON_MSE_OCCURRENCE)))).booleanValue()));
  }
  
  private boolean shouldBreakStep(final Step<?> step) {
    boolean _shouldBreakPredicates = this.shouldBreakPredicates(this.engine, step);
    if (_shouldBreakPredicates) {
      return true;
    }
    MSEOccurrence _mseoccurrence = step.getMseoccurrence();
    boolean _tripleNotEquals = (_mseoccurrence != null);
    if (_tripleNotEquals) {
      MSEOccurrence _mseoccurrence_1 = step.getMseoccurrence();
      final MSE mse = _mseoccurrence_1.getMse();
      boolean _hasRegularBreakpointTrue = this.hasRegularBreakpointTrue(mse);
      if (_hasRegularBreakpointTrue) {
        return true;
      }
      final EObject caller = mse.getCaller();
      boolean _hasRegularBreakpointTrue_1 = this.hasRegularBreakpointTrue(caller);
      if (_hasRegularBreakpointTrue_1) {
        return true;
      }
    }
    return false;
  }
  
  @Override
  public EObject getNextInstruction(final String threadName, final EObject currentInstruction, final IDSLDebugger.Stepping stepping) {
    return GenericSequentialModelDebugger.FAKE_INSTRUCTION;
  }
  
  @Override
  public void engineStarted(final IExecutionEngine executionEngine) {
    IExecutionContext _executionContext = this.engine.getExecutionContext();
    Resource _resourceModel = _executionContext.getResourceModel();
    EList<EObject> _contents = _resourceModel.getContents();
    EObject _get = _contents.get(0);
    this.spawnRunningThread(this.threadName, _get);
  }
  
  @Override
  public void engineStopped(final IExecutionEngine engine) {
    boolean _isTerminated = this.isTerminated(this.threadName);
    boolean _not = (!_isTerminated);
    if (_not) {
      this.terminated(this.threadName);
    }
  }
  
  @Override
  public void aboutToExecuteStep(final IExecutionEngine executionEngine, final Step<?> step) {
    final GenericSequentialModelDebugger.ToPushPop stackModification = new GenericSequentialModelDebugger.ToPushPop(step, true);
    this.toPushPop.add(stackModification);
    final boolean shallcontinue = this.control(this.threadName, step);
    if ((!shallcontinue)) {
      throw new EngineStoppedException("Debug thread has stopped.");
    }
  }
  
  @Override
  public void stepExecuted(final IExecutionEngine engine, final Step<?> step) {
    final GenericSequentialModelDebugger.ToPushPop stackModification = new GenericSequentialModelDebugger.ToPushPop(step, false);
    this.toPushPop.add(stackModification);
  }
  
  @Override
  public void engineAboutToStop(final IExecutionEngine engine) {
    this.executionTerminated = true;
    this.control(this.threadName, GenericSequentialModelDebugger.FAKE_INSTRUCTION);
  }
  
  @Override
  public void terminate() {
    super.terminate();
    this.engine.stop();
  }
}
