package org.apache.torque.om;

/*
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *   http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

import java.sql.Types;
import java.util.Date;

/**
 * This class can be used as an ObjectKey to uniquely identify an
 * object within an application where the id is a Date.
 *
 * @author <a href="mailto:jmcnally@apache.org">John McNally</a>
 * @version $Id: DateKey.java 1351125 2012-06-17 16:51:03Z tv $
 */
public class DateKey extends SimpleKey
{
    /**
     * Serial version
     */
    private static final long serialVersionUID = 3102583536685348517L;

    /**
     * Creates an DateKey whose internal representation will be
     * set later, through a set method
     */
    public DateKey()
    {
        // empty
    }

    /**
     * Creates a DateKey whose internal representation is a Date
     * given by the long number given by the String
     *
     * @param key the key value
     * @throws NumberFormatException if key is not valid
     */
    public DateKey(String key)
    {
        if (key != null)
        {
            this.key = new Date(Long.parseLong(key));
        }
        else
        {
            this.key = null;
        }
    }

    /**
     * Creates a DateKey
     *
     * @param key the key value
     */
    public DateKey(Date key)
    {
        this.key = key;
    }

    /**
     * Creates a DateKey that is equivalent to key.
     *
     * @param key the key value
     */
    public DateKey(DateKey key)
    {
        if (key != null)
        {
            this.key = key.getValue();
        }
        else
        {
            this.key = null;
        }
    }

    /**
     * Sets the internal representation to a String
     *
     * @param key the key value
     */
    public void setValue(String key)
    {
        if (key != null)
        {
            this.key = new Date(Long.parseLong(key));
        }
        else
        {
            this.key = null;
        }
    }

    /**
     * Sets the internal representation to the same object used by key.
     *
     * @param key the key value
     */
    public void setValue(DateKey key)
    {
        if (key != null)
        {
            this.key = key.getValue();
        }
        else
        {
            this.key = null;
        }
    }

    /**
     * Returns the JDBC type of the key
     * as defined in <code>java.sql.Types</code>.
     *
     * @return <code>Types.TIMESTAMP</code>.
     */
    public int getJdbcType()
    {
        return Types.TIMESTAMP;
    }

    /**
     * Access the underlying Date object.
     *
     * @return a <code>Date</code> value
     */
    public Date getDate()
    {
        return (Date) key;
    }

    /**
     * Get a String representation for this key.
     *
     * @return a String representation of the Date or an empty String if the
     *          Date is null
     */
    @Override
    public String toString()
    {
        Date dt = getDate();
        return (dt == null ? "" : Long.toString(dt.getTime()));
    }
}
