/*
 *
 *    Artistic License
 *
 *    Preamble
 *
 *    The intent of this document is to state the conditions under which a Package may be copied, such that
 *    the Copyright Holder maintains some semblance of artistic control over the development of the
 *    package, while giving the users of the package the right to use and distribute the Package in a
 *    more-or-less customary fashion, plus the right to make reasonable modifications.
 *
 *    Definitions:
 *
 *    "Package" refers to the collection of files distributed by the Copyright Holder, and derivatives
 *    of that collection of files created through textual modification.
 *
 *    "Standard Version" refers to such a Package if it has not been modified, or has been modified
 *    in accordance with the wishes of the Copyright Holder.
 *
 *    "Copyright Holder" is whoever is named in the copyright or copyrights for the package.
 *
 *    "You" is you, if you're thinking about copying or distributing this Package.
 *
 *    "Reasonable copying fee" is whatever you can justify on the basis of media cost, duplication
 *    charges, time of people involved, and so on. (You will not be required to justify it to the
 *    Copyright Holder, but only to the computing community at large as a market that must bear the
 *    fee.)
 *
 *    "Freely Available" means that no fee is charged for the item itself, though there may be fees
 *    involved in handling the item. It also means that recipients of the item may redistribute it under
 *    the same conditions they received it.
 *
 *    1. You may make and give away verbatim copies of the source form of the Standard Version of this
 *    Package without restriction, provided that you duplicate all of the original copyright notices and
 *    associated disclaimers.
 *
 *    2. You may apply bug fixes, portability fixes and other modifications derived from the Public Domain
 *    or from the Copyright Holder. A Package modified in such a way shall still be considered the
 *    Standard Version.
 *
 *    3. You may otherwise modify your copy of this Package in any way, provided that you insert a
 *    prominent notice in each changed file stating how and when you changed that file, and provided that
 *    you do at least ONE of the following:
 *
 *        a) place your modifications in the Public Domain or otherwise make them Freely
 *        Available, such as by posting said modifications to Usenet or an equivalent medium, or
 *        placing the modifications on a major archive site such as ftp.uu.net, or by allowing the
 *        Copyright Holder to include your modifications in the Standard Version of the Package.
 *
 *        b) use the modified Package only within your corporation or organization.
 *
 *        c) rename any non-standard executables so the names do not conflict with standard
 *        executables, which must also be provided, and provide a separate manual page for each
 *        non-standard executable that clearly documents how it differs from the Standard
 *        Version.
 *
 *        d) make other distribution arrangements with the Copyright Holder.
 *
 *    4. You may distribute the programs of this Package in object code or executable form, provided that
 *    you do at least ONE of the following:
 *
 *        a) distribute a Standard Version of the executables and library files, together with
 *        instructions (in the manual page or equivalent) on where to get the Standard Version.
 *
 *        b) accompany the distribution with the machine-readable source of the Package with
 *        your modifications.
 *
 *        c) accompany any non-standard executables with their corresponding Standard Version
 *        executables, giving the non-standard executables non-standard names, and clearly
 *        documenting the differences in manual pages (or equivalent), together with instructions
 *        on where to get the Standard Version.
 *
 *        d) make other distribution arrangements with the Copyright Holder.
 *
 *    5. You may charge a reasonable copying fee for any distribution of this Package. You may charge
 *    any fee you choose for support of this Package. You may not charge a fee for this Package itself.
 *    However, you may distribute this Package in aggregate with other (possibly commercial) programs as
 *    part of a larger (possibly commercial) software distribution provided that you do not advertise this
 *    Package as a product of your own.
 *
 *    6. The scripts and library files supplied as input to or produced as output from the programs of this
 *    Package do not automatically fall under the copyright of this Package, but belong to whomever
 *    generated them, and may be sold commercially, and may be aggregated with this Package.
 *
 *    7. C or perl subroutines supplied by you and linked into this Package shall not be considered part of
 *    this Package.
 *
 *    8. The name of the Copyright Holder may not be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 *    9. THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED
 *    WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 *    MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 */
package org.chiba.xml.xforms.ui;

import org.apache.log4j.Category;
import org.apache.commons.jxpath.JXPathContext;
import org.chiba.xml.xforms.*;
import org.chiba.xml.xforms.config.Config;
import org.chiba.xml.xforms.events.EventFactory;
import org.chiba.xml.xforms.exception.XFormsException;
import org.w3c.dom.Element;

/**
 * Manages the chiba:data Element which is created for every bound XForms Control and for Container Elements
 * such as 'group', 'repeat' and 'switch'. This acts as a proxy for the status and value of the associated
 * instance data item and is created and updated by the Processor.
 * <br><br>
 * This class is a convenience function of the processor and has no correspondence in the XForms set of
 * elements. It eases the generation of the user interface by pulling all instance data information down
 * to the controls and thereby simplifying the task of writing XSL tranformations for the output DOM.
 */
public class DataElement implements ModelItemListener {
	private static final Category LOGGER = Category.getInstance(DataElement.class);
	private static final String IGNORE_PROPERTY = "chiba.ui.ignoreInitialValidity";
	private static final String IGNORE_DEFAULT = "true";
	private BoundElement boundElement = null;
	private Element dataElement = null;

	/**
	 * Creates a new DataElement object.
	 *
	 * @param boundElement the parent form control or container element
	 */
	public DataElement(BoundElement boundElement) {
		this.boundElement = boundElement;
		this.dataElement = this.boundElement.getElement().getOwnerDocument().createElementNS(NamespaceCtx.CHIBA_NS,
		                                                                                     NamespaceCtx.CHIBA_PREFIX +
		                                                                                     ":data");
		this.boundElement.getElement().appendChild(this.dataElement);
		this.dataElement.setAttributeNS(NamespaceCtx.CHIBA_NS, NamespaceCtx.CHIBA_PREFIX + ":valid",
		                                String.valueOf(true));
		this.dataElement.setAttributeNS(NamespaceCtx.CHIBA_NS, NamespaceCtx.CHIBA_PREFIX + ":readonly",
		                                String.valueOf(false));
		this.dataElement.setAttributeNS(NamespaceCtx.CHIBA_NS, NamespaceCtx.CHIBA_PREFIX + ":required",
		                                String.valueOf(false));
		this.dataElement.setAttributeNS(NamespaceCtx.CHIBA_NS, NamespaceCtx.CHIBA_PREFIX + ":enabled",
		                                String.valueOf(true));
	}

	/**
	 * Initializes the control properties of this form control.
	 * <p>
	 * The <code>valid</code>, <code>readonly</code>, <code>required</code>
	 * and <code>enabled</code> attributes are overwritten in order to reflect
	 * the current state of this form control.
	 */
	public void initializeControlProperties() {
		Model model = this.boundElement.getModel();
		String instanceId = this.boundElement.getInstanceId();
		String locationPath = this.boundElement.getLocationPath();

		if (model.getInstance(instanceId).existsNode(locationPath)) {
			ModelItemProperties item = model.getInstance(instanceId).getDataItem(locationPath);

			// set the initial validity to true to prevent alerts when the user views the form for the first time !!!
			boolean valid = ignoreInitialValidity()
			                ? true
			                : item.isValid();

            //jt: start
            Instance instance = boundElement.getModel().getInstance(instanceId);
            JXPathContext instContext = instance.getInstanceContext();
            String xpath = instContext.getPointer(locationPath).asPath();

            this.dataElement.setAttributeNS(NamespaceCtx.CHIBA_NS,NamespaceCtx.CHIBA_PREFIX + ":xpath",
                                            xpath);

            //jt: end
			this.dataElement.setAttributeNS(NamespaceCtx.CHIBA_NS, NamespaceCtx.CHIBA_PREFIX + ":valid",
			                                String.valueOf(valid));
			this.dataElement.setAttributeNS(NamespaceCtx.CHIBA_NS, NamespaceCtx.CHIBA_PREFIX + ":readonly",
			                                String.valueOf(item.isReadonly()));
			this.dataElement.setAttributeNS(NamespaceCtx.CHIBA_NS, NamespaceCtx.CHIBA_PREFIX + ":required",
			                                String.valueOf(item.isRequired()));
			this.dataElement.setAttributeNS(NamespaceCtx.CHIBA_NS, NamespaceCtx.CHIBA_PREFIX + ":enabled",
			                                String.valueOf(item.isEnabled()));

            // register with model item as listener
            item.register(this);
		} else {
			// disable controls which are bound to a non-existing instance node
			this.dataElement.setAttributeNS(NamespaceCtx.CHIBA_NS, NamespaceCtx.CHIBA_PREFIX + ":enabled",
			                                String.valueOf(false));
		}
	}

	/**
	 * Initializes the control value of this form control.
	 * <p>
	 * The text value of the <code>data</code> child element is created in
	 * order to reflect the initial value of the instance data node this form
	 * control is bound to.
	 */
	public void initializeControlValue() {
		this.dataElement.appendChild(this.dataElement.getOwnerDocument().createTextNode(this.boundElement.getInstanceValue()));
	}

	/**
	 * Initializes the external name of this bound element. If there's a ChibaContext it will be called
     * to provide the name otherwise the id of the control is used.
	 * <p>
	 * The <code>name</code> attribute is created in order to hold environment
	 * information about this bound element.
	 */
	public void initializeExternalName() {
		// add external name
        ChibaContext chibaContext = this.boundElement.getContainerObject().getProcessor().getContext();
        String externalName;
        if(chibaContext!=null){
            externalName = this.boundElement.getContainerObject().getProcessor().getContext().getExternalName(this.boundElement);

        }else{
            externalName = boundElement.getId();
        }
        this.dataElement.setAttributeNS(NamespaceCtx.CHIBA_NS, NamespaceCtx.CHIBA_PREFIX + ":name",
                                        externalName);
	}

	/**
	 * Initializes the type declarations of this form control.
	 * <p>
	 * The <code>type</code> and <code>p3ptype</code> attributes are
	 * created in order to hold type information of this form control.
	 */
	public void initializeTypeDeclarations() {
		Model model = this.boundElement.getModel();
		String instanceId = this.boundElement.getInstanceId();
		String locationPath = this.boundElement.getLocationPath();

		if (model.getInstance(instanceId).existsNode(locationPath)) {
			// add schema datatype
			ModelItemProperties item = model.getInstance(instanceId).getDataItem(locationPath);
			String dataType = item.getDatatype();
            if(dataType==null){
                dataType="string";
            }
			this.dataElement.setAttributeNS(NamespaceCtx.CHIBA_NS, NamespaceCtx.CHIBA_PREFIX + ":type",
			                                dataType);

			String p3pType = item.getP3PType();

			if (p3pType != null) {
				// add p3p type
				this.dataElement.setAttributeNS(NamespaceCtx.CHIBA_NS,
				                                NamespaceCtx.CHIBA_PREFIX + ":p3ptype", p3pType);
			}
		}
	}

	/**
	 * Updates the control properties of this form control.
	 * <p>
	 * The <code>valid</code>, <code>readonly</code>, <code>required</code>
	 * and <code>enabled</code> attributes are overwritten in order to reflect
	 * the current state of this form control.
	 */
	public void updateControlProperties() {
		Model model = this.boundElement.getModel();
		String instanceId = this.boundElement.getInstanceId();
		String locationPath = this.boundElement.getLocationPath();

		if (model.getInstance(instanceId).existsNode(locationPath)) {
			ModelItemProperties item = model.getInstance(instanceId).getDataItem(locationPath);
			this.dataElement.setAttributeNS(NamespaceCtx.CHIBA_NS, NamespaceCtx.CHIBA_PREFIX + ":valid",
			                                String.valueOf(item.isValid()));
			this.dataElement.setAttributeNS(NamespaceCtx.CHIBA_NS, NamespaceCtx.CHIBA_PREFIX + ":readonly",
			                                String.valueOf(item.isReadonly()));
			this.dataElement.setAttributeNS(NamespaceCtx.CHIBA_NS, NamespaceCtx.CHIBA_PREFIX + ":required",
			                                String.valueOf(item.isRequired()));
			this.dataElement.setAttributeNS(NamespaceCtx.CHIBA_NS, NamespaceCtx.CHIBA_PREFIX + ":enabled",
			                                String.valueOf(item.isEnabled()));
		} else {
			// disable controls which are bound to a non-existing instance node
			this.dataElement.setAttributeNS(NamespaceCtx.CHIBA_NS, NamespaceCtx.CHIBA_PREFIX + ":valid",
			                                String.valueOf(true));
			this.dataElement.setAttributeNS(NamespaceCtx.CHIBA_NS, NamespaceCtx.CHIBA_PREFIX + ":readonly",
			                                String.valueOf(false));
			this.dataElement.setAttributeNS(NamespaceCtx.CHIBA_NS, NamespaceCtx.CHIBA_PREFIX + ":required",
			                                String.valueOf(false));
			this.dataElement.setAttributeNS(NamespaceCtx.CHIBA_NS, NamespaceCtx.CHIBA_PREFIX + ":enabled",
			                                String.valueOf(false));
		}
	}

	/**
	 * Updates the control value of this form control.
	 * <p>
	 * The text value of the <code>data</code> child element is overwritten in
	 * order to reflect the current value of the instance data node this form
	 * control is bound to.
	 */
	public void updateControlValue() {
		this.dataElement.getFirstChild().setNodeValue(this.boundElement.getInstanceValue());
	}

    // implementation of 'org.chiba.xml.xforms.ModelItemListener'

    /**
     * Dispatches <code>xforms-enabled</code> / <code>xforms-disabled</code>
     * events depending on the current <code>enabled</code> state.
     *
     * @param enabled the current <code>enabled</code> state.
     */
    public void enabledChanged (boolean enabled) {
        try {
            if (enabled) {
                this.boundElement.getContainerObject().dispatch(this.boundElement.getTarget(), EventFactory.ENABLED, null);
            }
            else {
                this.boundElement.getContainerObject().dispatch(this.boundElement.getTarget(), EventFactory.DISABLED, null);
            }
        }
        catch (XFormsException e) {
            // todo: error handling
            e.printStackTrace();
        }
    }

    /**
     * Dispatches <code>xforms-readonly</code> / <code>xforms-readwrite</code>
     * events depending on the current <code>readonly</code> state.
     *
     * @param readonly the current <code>readonly</code> state.
     */
    public void readonlyChanged (boolean readonly) {
        try {
            if (readonly) {
                this.boundElement.getContainerObject().dispatch(this.boundElement.getTarget(), EventFactory.READONLY, null);
            }
            else {
                this.boundElement.getContainerObject().dispatch(this.boundElement.getTarget(), EventFactory.READWRITE, null);
            }
        }
        catch (XFormsException e) {
            // todo: error handling
            e.printStackTrace();
        }
    }

    /**
     * Dispatches <code>xforms-required</code> / <code>xforms-optional</code>
     * events depending on the current <code>required</code> state.
     *
     * @param required the current <code>required</code> state.
     */
    public void requiredChanged (boolean required) {
        try {
            if (required) {
                this.boundElement.getContainerObject().dispatch(this.boundElement.getTarget(), EventFactory.REQUIRED, null);
            }
            else {
                this.boundElement.getContainerObject().dispatch(this.boundElement.getTarget(), EventFactory.OPTIONAL, null);
            }
        }
        catch (XFormsException e) {
            // todo: error handling
            e.printStackTrace();
        }
    }

    /**
     * Dispatches <code>xforms-valid</code> / <code>xforms-invalid</code>
     * events depending on the current <code>valid</code> state.
     *
     * @param valid the current <code>valid</code> state.
     */
    public void validChanged (boolean valid) {
        try {
            if (valid) {
                this.boundElement.getContainerObject().dispatch(this.boundElement.getTarget(), EventFactory.VALID, null);
            }
            else {
                this.boundElement.getContainerObject().dispatch(this.boundElement.getTarget(), EventFactory.INVALID, null);
            }
        }
        catch (XFormsException e) {
            // todo: error handling
            e.printStackTrace();
        }
    }

	private boolean ignoreInitialValidity() {
		try {
			return Boolean.valueOf(Config.getInstance().getProperty(IGNORE_PROPERTY, IGNORE_DEFAULT))
			              .booleanValue();
		} catch (Exception e) {
			return Boolean.valueOf(IGNORE_DEFAULT).booleanValue();
		}
	}
}
//end of class

