/*
 *
 *    Artistic License
 *
 *    Preamble
 *
 *    The intent of this document is to state the conditions under which a Package may be copied, such that
 *    the Copyright Holder maintains some semblance of artistic control over the development of the
 *    package, while giving the users of the package the right to use and distribute the Package in a
 *    more-or-less customary fashion, plus the right to make reasonable modifications.
 *
 *    Definitions:
 *
 *    "Package" refers to the collection of files distributed by the Copyright Holder, and derivatives
 *    of that collection of files created through textual modification.
 *
 *    "Standard Version" refers to such a Package if it has not been modified, or has been modified
 *    in accordance with the wishes of the Copyright Holder.
 *
 *    "Copyright Holder" is whoever is named in the copyright or copyrights for the package.
 *
 *    "You" is you, if you're thinking about copying or distributing this Package.
 *
 *    "Reasonable copying fee" is whatever you can justify on the basis of media cost, duplication
 *    charges, time of people involved, and so on. (You will not be required to justify it to the
 *    Copyright Holder, but only to the computing community at large as a market that must bear the
 *    fee.)
 *
 *    "Freely Available" means that no fee is charged for the item itself, though there may be fees
 *    involved in handling the item. It also means that recipients of the item may redistribute it under
 *    the same conditions they received it.
 *
 *    1. You may make and give away verbatim copies of the source form of the Standard Version of this
 *    Package without restriction, provided that you duplicate all of the original copyright notices and
 *    associated disclaimers.
 *
 *    2. You may apply bug fixes, portability fixes and other modifications derived from the Public Domain
 *    or from the Copyright Holder. A Package modified in such a way shall still be considered the
 *    Standard Version.
 *
 *    3. You may otherwise modify your copy of this Package in any way, provided that you insert a
 *    prominent notice in each changed file stating how and when you changed that file, and provided that
 *    you do at least ONE of the following:
 *
 *        a) place your modifications in the Public Domain or otherwise make them Freely
 *        Available, such as by posting said modifications to Usenet or an equivalent medium, or
 *        placing the modifications on a major archive site such as ftp.uu.net, or by allowing the
 *        Copyright Holder to include your modifications in the Standard Version of the Package.
 *
 *        b) use the modified Package only within your corporation or organization.
 *
 *        c) rename any non-standard executables so the names do not conflict with standard
 *        executables, which must also be provided, and provide a separate manual page for each
 *        non-standard executable that clearly documents how it differs from the Standard
 *        Version.
 *
 *        d) make other distribution arrangements with the Copyright Holder.
 *
 *    4. You may distribute the programs of this Package in object code or executable form, provided that
 *    you do at least ONE of the following:
 *
 *        a) distribute a Standard Version of the executables and library files, together with
 *        instructions (in the manual page or equivalent) on where to get the Standard Version.
 *
 *        b) accompany the distribution with the machine-readable source of the Package with
 *        your modifications.
 *
 *        c) accompany any non-standard executables with their corresponding Standard Version
 *        executables, giving the non-standard executables non-standard names, and clearly
 *        documenting the differences in manual pages (or equivalent), together with instructions
 *        on where to get the Standard Version.
 *
 *        d) make other distribution arrangements with the Copyright Holder.
 *
 *    5. You may charge a reasonable copying fee for any distribution of this Package. You may charge
 *    any fee you choose for support of this Package. You may not charge a fee for this Package itself.
 *    However, you may distribute this Package in aggregate with other (possibly commercial) programs as
 *    part of a larger (possibly commercial) software distribution provided that you do not advertise this
 *    Package as a product of your own.
 *
 *    6. The scripts and library files supplied as input to or produced as output from the programs of this
 *    Package do not automatically fall under the copyright of this Package, but belong to whomever
 *    generated them, and may be sold commercially, and may be aggregated with this Package.
 *
 *    7. C or perl subroutines supplied by you and linked into this Package shall not be considered part of
 *    this Package.
 *
 *    8. The name of the Copyright Holder may not be used to endorse or promote products derived from
 *    this software without specific prior written permission.
 *
 *    9. THIS PACKAGE IS PROVIDED "AS IS" AND WITHOUT ANY EXPRESS OR IMPLIED
 *    WARRANTIES, INCLUDING, WITHOUT LIMITATION, THE IMPLIED WARRANTIES OF
 *    MERCHANTIBILITY AND FITNESS FOR A PARTICULAR PURPOSE.
 *
 */
package org.chiba.xml.xforms;

import org.apache.log4j.Category;
import org.w3c.dom.Attr;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.NamedNodeMap;
import org.w3c.dom.Node;
import org.w3c.dom.traversal.DocumentTraversal;
import org.w3c.dom.traversal.NodeFilter;
import org.w3c.dom.traversal.NodeIterator;

import java.util.ArrayList;

/**
 *
 * preprocesses the input XForms document to customize some of the XML representations e.g.
 * adjust attributes to always use a prefix.
 *
 * @version $Id: PreProcessor.java,v 1.17 2004/02/19 00:11:59 joernt Exp $
 * @author joern turner (joernt@chiba.soureforge.net)
 */
public class PreProcessor {
	private static Category cat = Category.getInstance(PreProcessor.class);
	private Document doc = null;

	/**
	 * Creates a new PreProcessor object.
	 *
	 * @param aInput the DOM input Document
	 */
	public PreProcessor(Document aInput) {
		this.doc = aInput;
	}

	/**
	 * fetches the result of the preprocessing as DOM Document
	 *
	 * @return the result of the preprocessing as DOM Document
	 */
	public Document getResult() {
		return this.doc;
	}

	/**
	 * responsible for transforming the input document into an
	 * canonicalized version.
	 *
	 * public for testing
	 */
    private void canonicalize() {
		importChibaNamespace();
		adjustAttributeNamespace();
	}

	/**
	 * triggers preprocessing
	 */
	public void process() {
		canonicalize();
	}

	/*
	 * @author mark
	 *
	 * Adjust attributes so they are all namespaced
	 */
	private Document adjustAttributeNamespace() {
		DocumentTraversal parentDoc = (DocumentTraversal)doc;
		Node root = doc.getDocumentElement();

		NodeIterator mainWalker = parentDoc.createNodeIterator(root, NodeFilter.SHOW_ELEMENT,
		                                                       new NodeFilter() {
				public short acceptNode(Node n) {
					if ((n.getNodeType() == Node.ELEMENT_NODE) &&
						    NamespaceCtx.XFORMS_NS.equals(n.getNamespaceURI())) {
						return FILTER_ACCEPT;
					} else {
						return FILTER_SKIP;
					}
				}
			}, false);

		while (true) {
			Element node = (Element)mainWalker.nextNode();

			if (node == null) {
				break;
			}

            String xformsPrefix = NamespaceCtx.getPrefix(node, NamespaceCtx.XFORMS_NS);
            if (xformsPrefix == null || xformsPrefix.length() == 0) {
                // declare xforms namespace locally
                xformsPrefix = NamespaceCtx.DEFAULT_XFORMS_PREFIX;
                node.setAttributeNS(NamespaceCtx.XMLNS_NS,
                                    "xmlns:" + xformsPrefix,
                                    NamespaceCtx.XFORMS_NS);
            }

			conditionXFormAttributes(node, xformsPrefix);
            fixInstanceExpressions(node);
		}

		return doc;
	}

    private void fixInstanceExpressions(Element element) {
        NamedNodeMap attributes = element.getAttributes();

        for (int index = 0; index < attributes.getLength(); index++) {
            Attr attr = (Attr) attributes.item(index);

            if (NamespaceCtx.XFORMS_NS.equals(attr.getNamespaceURI())) {
                attr.setValue(PreProcessor.fixInstanceExpression(attr.getValue()));
            }
        }
    }

    private void conditionXFormAttributes(Element element, String xformsPrefix) {
		NamedNodeMap attrs = element.getAttributes();
		ArrayList list = new ArrayList();

		// get list of all attributes that need to be changed
		for (int c = 0; c < attrs.getLength(); c++) {
			Node attr = attrs.item(c);
			String name = attr.getNodeName();

            if (attr.getNamespaceURI() == null
                    && !name.equals("id")
                    && !name.equals("class")) {
                list.add(name);
            }
		}

        // change them
        for (int c = 0; c < list.size(); c++) {
            String name = (String)list.get(c);
            String value = element.getAttribute(name);
            element.removeAttribute(name);

            if (cat.isDebugEnabled()) {
                cat.debug(this + " setting Attribute: " + xformsPrefix + ":" + name + " value=" +
                          value);
            }

            element.setAttributeNS(NamespaceCtx.XFORMS_NS, xformsPrefix + ":" + name, value);
        }
	}

	private void importChibaNamespace() {
		this.doc.getDocumentElement().setAttributeNS(NamespaceCtx.XMLNS_NS,
		                                             "xmlns:" + NamespaceCtx.CHIBA_PREFIX,
		                                             NamespaceCtx.CHIBA_NS);
	}

    /**
     * Replaces any occurrence of <code>instance('&lt;instance-id&gt;')</code> with
     * <code>instance('&lt;instance-id&gt;')/.</code>. This is needed for correct
     * JXPath operation.
     *
     * @param value the value of an attribute to be fixed.
     * @return the fixed value.
     */
    public static String fixInstanceExpression (final String value) {
        int start = 0;
        int end = 0;
        int index = value.indexOf("instance('");
        StringBuffer buffer = new StringBuffer();

        while (index > -1) {
            end = value.indexOf("')", index) + 2;
            buffer.append(value.substring(start, end)).append("/.");
            start = end;
            index = value.indexOf("instance('", start);
        }

        if (start < value.length()) {
            buffer.append(value.substring(start));
        }

        return buffer.toString();
    }

}
//end of class
