/******************************************************************************
  (c) Copyright 2003-2004, 1060 Research Ltd                                   

  This Software is licensed to You, the licensee, for use under the terms of   
  the 1060 Public License v1.0. Please read and agree to the 1060 Public       
  License v1.0 [www.1060research.com/license] before using or redistributing   
  this software.                                                               

  In summary the 1060 Public license has the following conditions.             
  A. You may use the Software free of charge provided you agree to the terms   
  laid out in the 1060 Public License v1.0                                     
  B. You are only permitted to use the Software with components or applications
  that provide you with OSI Certified Open Source Code [www.opensource.org], or
  for which licensing has been approved by 1060 Research Limited.              
  You may write your own software for execution by this Software provided any  
  distribution of your software with this Software complies with terms set out 
  in section 2 of the 1060 Public License v1.0                                 
  C. You may redistribute the Software provided you comply with the terms of   
  the 1060 Public License v1.0 and that no warranty is implied or given.       
  D. If you find you are unable to comply with this license you may seek to    
  obtain an alternative license from 1060 Research Limited by contacting       
  license@1060research.com or by visiting www.1060research.com                 

  NO WARRANTY:  THIS SOFTWARE IS NOT COVERED BY ANY WARRANTY. SEE 1060 PUBLIC  
  LICENSE V1.0 FOR DETAILS                                                     

  THIS COPYRIGHT NOTICE IS *NOT* THE 1060 PUBLIC LICENSE v1.0. PLEASE READ     
  THE DISTRIBUTED 1060_Public_License.txt OR www.1060research.com/license      

  File:          $RCSfile: SMTPAccessor.java,v $
  Version:       $Name:  $ $Revision: 1.3 $
  Last Modified: $Date: 2004/03/04 09:01:24 $
 *****************************************************************************/

package org.ten60.smtp.accessor;

import org.ten60.netkernel.xml.xahelper.*;
import org.ten60.netkernel.xml.xda.*;
import org.ten60.netkernel.xml.representation.*;
import com.ten60.netkernel.urii.*;

import org.ten60.netkernel.layer1.representation.*;
import org.ten60.netkernel.layer1.meta.*;
import org.ten60.netkernel.layer1.util.*;

import com.ten60.netkernel.urii.aspect.*;

import java.io.*;
import java.util.*;
import org.w3c.dom.*;

import java.sql.*;
import java.security.*;
import java.net.*;

import javax.mail.*;
import javax.mail.internet.*;
import javax.activation.*;


/**
 * An SMTP post service that holds an Queue of mail messages.
 * A mail processor thread periodically sends queued messages.
 *
 * @author  pjr
 */
public class SMTPAccessor extends XAccessor
{
	final static String MESSAGE_ARGUMENT="part";
	final static int DEFAULT_MAX_RETRYS = 3;
	
	private Session mSession;
	private MailProcessor mMailProcessor;
	private ArrayList mOutBox;
	private String mMailAddress;
	private String mSMTPGateway;
	private String mSMTPUser;
	private String mSMTPPassword;
	private int	mMaxRetrys=DEFAULT_MAX_RETRYS;
	private int mPollInterval=10000;
	
	private boolean mConfigured=false;
	
	/** Creates a new instance of SMTPAccessor */
	public SMTPAccessor()
	{	declareDisableArgumentChecking();
	}
	
	/**Configure common Email settings
	 * @param aHelper The helper class that provides access to the config,
	 */
	private void configure(XAHelper aHelper) throws Throwable
	{	URI configURI=URI.create("ffcpl:/etc/SMTPConfig.xml");
		IURRepresentation rep=aHelper.getResource(configURI, IAspectXDA.class);
		IAspectXDA config=(IAspectXDA)rep.getAspect(IAspectXDA.class);
		IXDAReadOnly aConfig=config.getXDA();

		mSMTPGateway=aConfig.getText("/SMTPConfig/SMTPGateway", true);
		mSMTPUser=aConfig.getText("/SMTPConfig/SMTPUser", true);
		mSMTPPassword=aConfig.getText("/SMTPConfig/SMTPPassword", true);
		mMailAddress=aConfig.getText("/SMTPConfig/sender", true);
		try{ mPollInterval=Integer.parseInt(aConfig.getText("/SMTPConfig/pollInterval", true));
		} catch (Exception e){};
		try{ mMaxRetrys=Integer.parseInt(aConfig.getText("/SMTPConfig/maxRetrys", true));
		}catch (Exception e) {};

		//Create mail session
		StringBuffer mailsettings=new StringBuffer("mail.store.protocol=pop3\n");
		mailsettings.append("mail.transport.protocol=smtp\n");
		/*Debug*/
		//mailsettings.append("mail.debug=true\n");
		ByteArrayInputStream bis=new ByteArrayInputStream(mailsettings.toString().getBytes());
		java.util.Properties ps=new java.util.Properties();
		ps.load(bis);
		mSession=Session.getInstance(ps);
		mOutBox=new ArrayList(10);
		mMailProcessor=new MailProcessor();

		//Start mailprocessor thread
		mMailProcessor.start();
	}
	
	
	/** Abstract method to be implemented in subclass.
	 * @return Either the result as a Document or a Reader.
	 * @param aHelper The helper class that provides access to the operator,
	 * operand, and parameters.
	 * @throws AccessViolationException Thrown if we fail to access any of the beans
	 * @throws DocumentAccessorException Thrown if we fail for any reason.
	 *
	 */
	public IURRepresentation source(XAHelper aHelper) throws Throwable
	{	if(!mConfigured || (aHelper.hasOperator() && aHelper.getOperator().getXDA().isTrue("/reconfigure")))
		{	if(mMailProcessor!=null)
			{	mMailProcessor.terminate();
			}
			configure(aHelper);
		}
		if(!aHelper.hasOperator())
		{	IXDAReadOnly opd=aHelper.getOperand().getXDA();

			MimeMessage message=new MimeMessage(mSession);
			message.setSubject(opd.getText("/email/subject",true));
			String from=null;
			if(opd.isTrue("/email/from"))
			{	from=opd.getText("/email/from",true);
			}
			else from=mMailAddress;
			InternetAddress IAfrom=new InternetAddress(from);
			message.setSender(IAfrom);
			message.setFrom(IAfrom);
			Map map=new HashMap(5);
			IXDAReadOnlyIterator ir=opd.readOnlyIterator("/email/to");
			while(ir.hasNext())
			{	ir.next();
				String to=ir.getText(".", true);
				if(!map.containsKey(to))
				{	message.addRecipient(Message.RecipientType.TO, new InternetAddress(to));
					map.put(to, new Boolean(true));
				}
			}
			ir=opd.readOnlyIterator("/email/cc");
			map=new HashMap(5);
			while(ir.hasNext())
			{	ir.next();
				String cc=ir.getText(".", true);
				if(!map.containsKey(cc))
				{	message.addRecipient(Message.RecipientType.CC, new InternetAddress(cc));
					map.put(cc, new Boolean(true));
				}
			}
			ir=opd.readOnlyIterator("/email/bcc");
			map=new HashMap(5);
			while(ir.hasNext())
			{	ir.next();
				String bcc=ir.getText(".", true);
				if(!map.containsKey(bcc))
				{	message.addRecipient(Message.RecipientType.BCC, new InternetAddress(bcc));
					map.put(bcc, new Boolean(true));
				}
			}
			
			//Add Messages
			MimeMultipart replymp=new MimeMultipart();
			XAHelperExtra extra=(XAHelperExtra)aHelper;
			CompoundURIdentifier curi=new CompoundURIdentifier(extra.getRequest().getURI());
			Iterator i=curi.getArgs();
			boolean messageflag=false;
			while(i.hasNext())
			{	CompoundURIdentifier.CompoundURIStruct struct= (CompoundURIdentifier.CompoundURIStruct)i.next();
				if(!struct.getKey().equals("operand"))
				{	messageflag=true;		
					IURRepresentation rep=aHelper.getResource(URI.create(struct.getURI()), IAspectBinaryStream.class);
					IAspectBinaryStream bsAspect=(IAspectBinaryStream)rep.getAspect(IAspectBinaryStream.class);
					ByteArrayOutputStream baos=new ByteArrayOutputStream();
					bsAspect.write(baos);
					baos.flush();
					MimeBodyPart mbp=new MimeBodyPart();
					DataHandler dh=null;
					if(rep.getMeta().getMimeType().startsWith("text"))
					{	dh=new DataHandler(baos.toString(),rep.getMeta().getMimeType());
					}
					else
					{	SMTPAccessor.ByteArrayDataSource bads=new SMTPAccessor.ByteArrayDataSource(baos.toByteArray(), rep.getMeta().getMimeType());
						dh=new DataHandler(bads);
						try
						{	mbp.setFileName(aHelper.getOperand().getXDA().getText("//attachment[name='"+struct.getKey()+"']/filename",true));
						}
						catch(Exception e)
						{	mbp.setFileName(struct.getKey());
						}
					}
					mbp.setDataHandler(dh);
					replymp.addBodyPart(mbp);
				}
			}
			if(!messageflag) throw new Exception("Must provide at least one message part argument");
			message.setContent(replymp);
			message.saveChanges();

			synchronized(mOutBox)
			{	mOutBox.add(new MessageWrapper(message));
			}
		}
		return org.ten60.netkernel.layer1.representation.VoidAspect.create();
	}
	
	private boolean sendMessage(MimeMessage message)
	{	try
		{	Transport t=mSession.getTransport();
			t.connect(mSMTPGateway,-1,mSMTPUser,mSMTPPassword);
			if(!t.isConnected())
			{	//SysLogger.log(SysLogger.WARNING, this, "Failed to get SMTP Connection");
				return false;
			}
			message.setSentDate(new java.util.Date());
			message.saveChanges();
			t.sendMessage(message, message.getAllRecipients());
			return true;
		}
		catch(Exception e)
		{	//SysLogger.log(SysLogger.WARNING, this, "SMTP message not delivered:"+e.getMessage());
			e.printStackTrace();
			return false;
		}
	}
	
	protected void finalize()
	{	mMailProcessor.terminate();
	}	
	
	/**
	 * Message Wrapper to hold MimeMessage and Send attempts counter
	 */
	class MessageWrapper
	{	private MimeMessage mMessage;
		private int trys;
		
		public MessageWrapper(MimeMessage message)
		{	mMessage=message;
		}
		
		public MimeMessage getMessage()
		{	return mMessage;
		}
		
		public void incrementTrys(){ trys++; };
		
		public int getTrys() { return trys; };
	}
	
	/*Outbox Mail Processor Thread*/
	class MailProcessor extends Thread
	{
		boolean mRun=true;
		
		public MailProcessor()
		{
		}
		
		public void terminate()
		{	interrupt();
			while(!mOutBox.isEmpty()){}
			mRun=false;
			interrupt();
		}
		
		/** 
		 *	Mailbox Poller
		 *
		 * @see     java.lang.Thread#run()
		 *
		 */
		public void run()
		{	while(mRun)
			{	synchronized(mOutBox)
				{	Iterator i=mOutBox.iterator();
					while(i.hasNext())
					{	MessageWrapper message=(MessageWrapper)i.next();
						if(sendMessage(message.getMessage()))
						{	//Log message success;
							i.remove();
						}
						else
						{	if(message.getTrys() == mMaxRetrys)
							{	//Log message failure
								System.err.println("Error sending message");
								i.remove();
							}
							else message.incrementTrys();
						}
					}
				}
				try
				{	sleep(mPollInterval);
				}
				catch(InterruptedException e){};
			}
		}
	}
	
	/**
	 * ByteArrayDataSource required for our email attachments as they don't
	 * originate from Files
	 */
	class ByteArrayDataSource implements DataSource {
		private byte[] mData;	
		private String mType;
		private String mName;	

		/* Create a DataSource from an input stream */
		public ByteArrayDataSource(InputStream is, String type)
		{   mType = type;
			try
			{	ByteArrayOutputStream os = new ByteArrayOutputStream();
				int ch;
				while ((ch = is.read()) != -1)	os.write(ch);
				mData = os.toByteArray();
			}
			catch (IOException e) { }
		}

		/* Create a DataSource from a byte array */
		public ByteArrayDataSource(byte[] data, String type)
		{	mData = data;
			mType = type;
		}

		/* Create a DataSource from a String */
		public ByteArrayDataSource(String data, String type)
		{	try
			{    mData = data.getBytes("iso-8859-1");
			}
			catch (UnsupportedEncodingException uex) { }
			mType = type;
		}

		/**
		 * Return an InputStream for the data.
		 * Note - a new stream must be returned each time.
		 */
		public InputStream getInputStream() throws IOException
		{	if (mData == null) throw new IOException("no data");
			return new ByteArrayInputStream(mData);
		}

		public OutputStream getOutputStream() throws IOException
		{	throw new IOException("not supported");
		}

		public String getContentType() {
			return mType;
		}

		public String getName() {
			return mName;
		}
		
		public void setName(String name) {
			mName=name;
		}
	}

}
