/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License (the "License").
 * You may not use this file except in compliance with the License.
 *
 * You can obtain a copy of the license at usr/src/OPENSOLARIS.LICENSE
 * or http://www.opensolaris.org/os/licensing.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at usr/src/OPENSOLARIS.LICENSE.
 * If applicable, add the following below this CDDL HEADER, with the
 * fields enclosed by brackets "[]" replaced with your own identifying
 * information: Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 */

/*
 * Copyright 2006 Sun Microsystems, Inc.  All rights reserved.
 * Use is subject to license terms.
 */

#pragma ident	"@(#)__vpow.c	1.13	06/01/31 SMI"

#include <sys/isa_defs.h>

#ifdef _LITTLE_ENDIAN
#define HI(x)	*(1+(int*)x)
#define LO(x)	*(unsigned*)x
#else
#define HI(x)	*(int*)x
#define LO(x)	*(1+(unsigned*)x)
#endif

#ifdef __RESTRICT
#define restrict _Restrict
#else
#define restrict
#endif

/* double pow(double x, double y)
 *
 * Method :
 *	1. Special cases:
 *		for (anything) ** 0				=> 1
 *		for (anything) ** NaN				=> QNaN + invalid
 *		for NaN ** (anything)				=> QNaN + invalid
 *		for +-1 ** +-Inf				=> QNaN + invalid
 *		for +-(|x| < 1) ** +Inf				=> +0
 *		for +-(|x| < 1) ** -Inf				=> +Inf
 *		for +-(|x| > 1) ** +Inf				=> +Inf
 *		for +-(|x| > 1) ** -Inf				=> +0
 *		for +Inf ** (negative)				=> +0
 *		for +Inf ** (positive)				=> +Inf
 *		for -Inf ** (negative except odd integer)	=> +0
 *		for -Inf ** (negative odd integer)		=> -0
 *		for -Inf ** (positive except odd integer)	=> +Inf
 *		for -Inf ** (positive odd integer)		=> -Inf
 *		for (negative) ** (non-integer)			=> QNaN + invalid
 *		for +0 ** (negative)				=> +Inf + overflow
 *		for +0 ** (positive)				=> +0
 *		for -0 ** (negative except odd integer)		=> +Inf + overflow
 *		for -0 ** (negative odd integer)		=> -Inf + overflow
 *		for -0 ** (positive except odd integer)		=> +0
 *		for -0 ** (positive odd integer)		=> -0
 *	2. Computes x**y from:
 *		x**y = 2**(y*log2(x)) = 2**(w/256), where w = 256*log2(|x|)*y.
 *	3. Computes w = 256*log2(|x|)*y from
 *		|x| = m * 2**n => log2(|x|) = n + log2(m).
 *	Let m = m0 + dm, where m0 = 1 + k / 256,
 *		k = [0, 255],
 *		dm = [-1/512, 1/512].
 *	Then 256*log2(m) = 256*log2(m0 + dm) = 256*log2(m0) + 256*log2((1+z)/(1-z)),
 *		where z = (m-m0)/(m+m0), z = [-1/1025, 1/1025].
 *	Then
 *		256*log2(m0) is looked up in a table of 256*log2(1), 256*log2(1+1/128),
 *			..., 256*log2(1+128/128).
 *		256*log2((1+z)/(1-z)) is computed using
 *			approximation: 256*log2((1+z)/(1-z)) = a0 * z + a1 * z**3 + a1 * z**5.
 *	Perform w = 256*log2(|x|)*y = w1 + w2 by simulating muti-precision arithmetic.
 *	3. For w >= 262144
 *		then for (negative) ** (odd integer)	=> -Inf + overflow
 *		else					=> +Inf + overflow
 *	For w <= -275200
 *		then for (negative) ** (odd integer)	=> -0 + underflow
 *		else					=> +0 + underflow
 *	4. Computes 2 ** (w/256) from:
 *		2 ** (w/256) = 2**a  *  2**(k/256)  *  2**(r/256)
 *	Where:
 *		a    =    int  ( w ) >> 8;
 *		k    =    int  ( w ) & 0xFF;
 *		r    =    frac ( w ).
 *	Note that:
 *		k = 0, 1, ..., 255;
 *		r = (-1, 1).
 *	Then:
 *		2**(k/256) is looked up in a table of 2**0, 2**1/256, ...
 *		2**(r/256) is computed using approximation:
 *			2**(r/256) =  ((((b5 * r + b4) * r + b3) * r + b2) * r + b1) * r + b0
 *		Multiplication by 2**a is done by adding "a" to
 *		the biased exponent.
 *	Perform 2 ** (w/256) by simulating muti-precision arithmetic.
 *	5. For (negative) ** (odd integer)		=> -(2**(w/256))
 *	otherwise					=>   2**(w/256)
 *
 * Accuracy:
 *	Max. relative aproximation error < 2**(-67.94) for 256*log2((1+z)/(1-z)).
 *	Max. relative aproximation error < 2**(-63.15) for 2**(r/256).
 *	Maximum error observed: less than 0.761 ulp after 1.300.000.000
 *	results.
 */

static void
__vpowx( int n, double * restrict px, double * restrict py,
	int stridey, double * restrict pz, int stridez );

static const double __TBL_exp2[] = {
	/* __TBL_exp2[2*i] = high order bits 2^(i/256), i = [0, 255] */
	/* __TBL_exp2[2*i+1] = least bits 2^(i/256), i = [0, 255]     */
 1.000000000000000000e+00, 0.000000000000000000e+00, 1.002711275050202522e+00,
-3.636615928692263944e-17, 1.005429901112802726e+00, 9.499186535455031757e-17,
 1.008155898118417548e+00,-3.252058756084308061e-17, 1.010889286051700475e+00,
-1.523477860336857718e-17, 1.013630084951489430e+00, 9.283599768183567587e-18,
 1.016378314910953096e+00,-5.772170073199660028e-17, 1.019133996077737914e+00,
 3.601904982259661106e-17, 1.021897148654116627e+00, 5.109225028973443894e-17,
 1.024667792897135721e+00,-7.561607868487779440e-17, 1.027445949118763746e+00,
-4.956074174645370440e-17, 1.030231637686040980e+00, 3.319830041080812944e-17,
 1.033024879021228415e+00, 7.600838874027088489e-18, 1.035825693601957198e+00,
-7.806782391337636167e-17, 1.038634101961378731e+00, 5.996273788852510618e-17,
 1.041450124688316103e+00, 3.784830480287576210e-17, 1.044273782427413755e+00,
 8.551889705537964892e-17, 1.047105095879289793e+00, 7.277077243104314749e-17,
 1.049944085800687210e+00, 5.592937848127002586e-17, 1.052790773004626423e+00,
-9.629482899026935739e-17, 1.055645178360557157e+00, 1.759325738772091599e-18,
 1.058507322794512762e+00,-7.152651856637780738e-17, 1.061377227289262093e+00,
-1.197353708536565756e-17, 1.064254912884464499e+00, 5.078754198611230394e-17,
 1.067140400676823697e+00,-7.899853966841582122e-17, 1.070033711820241873e+00,
-9.937162711288919381e-17, 1.072934867525975555e+00,-3.839668843358823807e-18,
 1.075843889062791048e+00,-1.000271615114413611e-17, 1.078760797757119860e+00,
-6.656660436056592603e-17, 1.081685614993215250e+00,-4.782623902997086266e-17,
 1.084618362213309206e+00, 3.166152845816346116e-17, 1.087559060917769660e+00,
 5.409349307820290759e-18, 1.090507732665257690e+00,-3.046782079812471147e-17,
 1.093464399072885840e+00, 1.441395814726920934e-17, 1.096429081816376883e+00,
-5.919933484449315824e-17, 1.099401802630221914e+00, 7.170459599701923225e-17,
 1.102382583307840891e+00, 5.266036871570694387e-17, 1.105371445701741173e+00,
 8.239288760500213590e-17, 1.108368411723678726e+00,-8.786813845180526616e-17,
 1.111373503344817548e+00, 5.563945026669697643e-17, 1.114386742595892432e+00,
 1.041027845684557095e-16, 1.117408151567369279e+00,-7.976805902628220456e-17,
 1.120437752409606746e+00,-6.201085906554178750e-17, 1.123475567333019898e+00,
-9.699737588987042995e-17, 1.126521618608241848e+00, 5.165856758795456737e-17,
 1.129575928566288079e+00, 6.712805858726256588e-17, 1.132638519598719196e+00,
 3.237356166738000264e-17, 1.135709414157805464e+00, 5.066599926126155859e-17,
 1.138788634756691565e+00, 8.912812676025407778e-17, 1.141876203969561576e+00,
 4.651091177531412387e-17, 1.144972144431804173e+00, 4.641289892170010657e-17,
 1.148076478840178938e+00, 6.897740236627191770e-17, 1.151189229952982673e+00,
 3.250710218863827212e-17, 1.154310420590215935e+00, 1.041712894627326619e-16,
 1.157440073633751121e+00,-9.123871231134400287e-17, 1.160578212027498779e+00,
-3.261040205417393722e-17, 1.163724858777577476e+00, 3.829204836924093499e-17,
 1.166880036952481658e+00,-8.791879579999169742e-17, 1.170043769683250190e+00,
-1.847744201790004694e-18, 1.173216080163637320e+00,-7.287562586584994479e-17,
 1.176396991650281221e+00, 5.554203254218078963e-17, 1.179586527462875845e+00,
 1.009231277510039044e-16, 1.182784710984341014e+00, 1.542975430079076058e-17,
 1.185991565660993841e+00,-9.209506835293105905e-18, 1.189207115002721027e+00,
 3.982015231465646111e-17, 1.192431382583151178e+00, 4.397551415609721443e-17,
 1.195664392039827328e+00, 4.616603670481481397e-17, 1.198906167074380580e+00,
-9.809193356008423118e-17, 1.202156731452703076e+00, 6.644981499252301245e-17,
 1.205416109005123859e+00,-3.357272193267529634e-17, 1.208684323626581625e+00,
-4.746725945228984097e-17, 1.211961399276801243e+00,-4.890611077521118357e-17,
 1.215247359980468955e+00,-7.712630692681488131e-17, 1.218542229827408452e+00,
-9.006726958363837675e-17, 1.221846032972757623e+00,-1.061102121140269116e-16,
 1.225158793637145527e+00,-8.903533814269983429e-17, 1.228480536106870025e+00,
-1.898781631302529953e-17, 1.231811284734075862e+00, 7.389382471610050247e-17,
 1.235151063936933413e+00,-1.075524434430784138e-16, 1.238499898199816540e+00,
 2.767702055573967430e-17, 1.241857812073484002e+00, 4.658027591836936791e-17,
 1.245224830175257980e+00,-4.677240449846727500e-17, 1.248600977189204819e+00,
-8.261810999021963550e-17, 1.251986277866316222e+00, 4.834167152469897600e-17,
 1.255380757024691096e+00,-6.711389821296878419e-18, 1.258784439549716527e+00,
-8.421782587730599357e-17, 1.262197350394250739e+00,-3.084464887473846465e-17,
 1.265619514578806282e+00, 4.250577003450868637e-17, 1.269050957191733220e+00,
 2.667932131342186095e-18, 1.272491703389402762e+00,-1.057791626721242103e-17,
 1.275941778396392001e+00, 9.915430244214290330e-17, 1.279401207505669325e+00,
-9.759095008356062210e-17, 1.282870016078778264e+00, 1.713594918243560968e-17,
 1.286348229546025568e+00,-3.416955706936181976e-17, 1.289835873406665723e+00,
 8.949257530897591722e-17, 1.293332973229089466e+00,-2.974590443132751646e-17,
 1.296839554651009641e+00, 2.538250279488831496e-17, 1.300355643379650594e+00,
 5.678728102802217422e-17, 1.303881265191935812e+00, 8.647675598267871179e-17,
 1.307416445934677318e+00,-7.336645652878868892e-17, 1.310961211524764414e+00,
-7.181536135519453857e-17, 1.314515587949354636e+00, 2.267543315104585645e-17,
 1.318079601266064049e+00,-5.457955827149153502e-17, 1.321653277603157539e+00,
-2.480638245913021742e-17, 1.325236643159741323e+00,-2.858731210038861373e-17,
 1.328829724205954355e+00, 4.089086223910160052e-17, 1.332432547083161500e+00,
-5.101586630916743959e-17, 1.336045138204145832e+00,-5.891866356388801353e-17,
 1.339667524053302916e+00, 8.927282594831731984e-17, 1.343299731186835322e+00,
-5.802580890201437751e-17, 1.346941786232945804e+00, 3.224065101254679169e-17,
 1.350593715892034474e+00,-8.287110381462416533e-17, 1.354255546936892651e+00,
 7.700948379802989462e-17, 1.357927306212901142e+00,-9.529635744825188867e-17,
 1.361609020638224754e+00, 1.533787661270668046e-18, 1.365300717204011915e+00,
-1.000536312597476517e-16, 1.369002422974590516e+00, 9.593797919118848773e-17,
 1.372714165087668414e+00,-4.495960595234841262e-17, 1.376435970754530169e+00,
-6.898588935871801042e-17, 1.380167867260237990e+00, 1.051031457996998395e-16,
 1.383909881963832023e+00,-6.770511658794786287e-17, 1.387662042298529075e+00,
 8.422984274875415318e-17, 1.391424375771926236e+00,-4.906174865288989325e-17,
 1.395196909966200272e+00,-9.329336224225496552e-17, 1.398979672538311236e+00,
-9.614213209051323072e-17, 1.402772691220204759e+00,-5.295783249407989223e-17,
 1.406575993819015435e+00, 7.034914812136422188e-18, 1.410389608217270663e+00,
 4.166548728435062259e-17, 1.414213562373095145e+00,-9.667293313452913451e-17,
 1.418047884320415175e+00, 2.274438542185529452e-17, 1.421892602169165576e+00,
-1.607782891589024413e-17, 1.425747744105494208e+00, 9.880690758500607284e-17,
 1.429613338391970023e+00,-1.203164248905365518e-17, 1.433489413367788901e+00,
-5.802454243926826103e-17, 1.437375997448982368e+00,-4.204034016467556612e-17,
 1.441273119128625657e+00, 5.602503650878985675e-18, 1.445180806977046650e+00,
-3.023758134993987319e-17, 1.449099089642035043e+00,-6.259405000819309254e-17,
 1.453027995849052623e+00,-5.779948609396106102e-17, 1.456967554401443765e+00,
 5.648679453876998140e-17, 1.460917794180647045e+00,-5.600377186075215800e-17,
 1.464878744146405731e+00, 9.530767543587157319e-17, 1.468850433336981842e+00,
 8.465882756533627608e-17, 1.472832890869367528e+00, 6.691774081940589372e-17,
 1.476826145939499346e+00,-3.483994556892795796e-17, 1.480830227822471867e+00,
-9.686952102630618578e-17, 1.484845165872752393e+00, 1.078008676440748076e-16,
 1.488870989524397004e+00, 6.155367157742871330e-17, 1.492907728291264835e+00,
 1.419292015428403577e-17, 1.496955411767235455e+00,-2.861663253899158211e-17,
 1.501014069626425584e+00,-6.413767275790235039e-17, 1.505083731623406473e+00,
 7.074710613582846364e-17, 1.509164427593422841e+00,-1.016455327754295039e-16,
 1.513256187452609813e+00, 8.884497851338712091e-17, 1.517359041198214742e+00,
-4.308699472043340801e-17, 1.521473018908814590e+00,-5.996387675945683420e-18,
 1.525598150744538417e+00,-1.102494171234256094e-16, 1.529734466947286986e+00,
 3.785792115157219653e-17, 1.533881997840955913e+00, 8.875226844438446141e-17,
 1.538040773831656827e+00, 1.017467235116135806e-16, 1.542210825407940744e+00,
 7.949834809697620856e-17, 1.546392183141021448e+00, 1.068396000565721980e-16,
 1.550584877684999974e+00,-1.460070659068938518e-17, 1.554788939777088652e+00,
-8.003161350116035641e-17, 1.559004400237836929e+00, 3.781207053357527502e-17,
 1.563231289971357629e+00, 7.484777645590734389e-17, 1.567469639965552997e+00,
-1.035206176884972199e-16, 1.571719481292341403e+00,-3.342984004687200069e-17,
 1.575980845107886497e+00,-1.013691647127830398e-17, 1.580253762652824578e+00,
-5.163402929554468062e-17, 1.584538265252493749e+00,-1.933771703458570293e-17,
 1.588834384317163950e+00,-5.994950118824479401e-18, 1.593142151342266999e+00,
-1.009440654231196372e-16, 1.597461597908627073e+00, 2.486839279622099613e-17,
 1.601792755682693414e+00,-6.054917453527784343e-17, 1.606135656416771029e+00,
-1.035454528805999526e-16, 1.610490331949254283e+00, 2.470719256979788785e-17,
 1.614856814204860713e+00,-7.316663399125123263e-17, 1.619235135194863728e+00,
 2.094133415422909241e-17, 1.623625327017328868e+00,-3.584512851414474710e-17,
 1.628027421857347834e+00,-6.712955084707084086e-17, 1.632441451987274972e+00,
 9.852819230429992964e-17, 1.636867449766964411e+00, 7.698325071319875575e-17,
 1.641305447644006321e+00,-9.247568737640705508e-17, 1.645755478153964946e+00,
-1.012567991367477260e-16, 1.650217573920617742e+00, 9.133279588729904190e-18,
 1.654691767656194301e+00, 9.643294303196028661e-17, 1.659178092161616158e+00,
-7.275545550823050654e-17, 1.663676580326736376e+00, 5.890992696713099670e-17,
 1.668187265130582464e+00, 4.269178019570615091e-17, 1.672710179641596628e+00,
-5.476715964599563076e-17, 1.677245357017878469e+00, 8.303949509950732785e-17,
 1.681792830507429004e+00, 8.199010020581496520e-17, 1.686352633448393368e+00,
-7.181463278358010675e-17, 1.690924799269305279e+00,-9.669671474394880166e-17,
 1.695509361489332623e+00, 7.238416872845166641e-17, 1.700106353718523478e+00,
-8.023719370397700246e-18, 1.704715809658051251e+00,-2.728883284797281563e-17,
 1.709337763100462926e+00,-9.868779456632931076e-17, 1.713972247929925974e+00,
 6.473975107753367064e-17, 1.718619298122477934e+00,-1.851380418263110988e-17,
 1.723278947746273992e+00,-9.522123800393799963e-17, 1.727951230961837670e+00,
-1.075098186120464245e-16, 1.732636182022311067e+00,-1.698051074315415494e-18,
 1.737333835273706217e+00, 3.164389299292956947e-17, 1.742044225155156445e+00,
-1.525959118950788792e-18, 1.746767386199169048e+00,-1.075229048350751450e-16,
 1.751503353031878207e+00,-5.124450420596724659e-17, 1.756252160373299454e+00,
 2.960140695448873307e-17, 1.761013843037583904e+00,-7.943253125039227711e-17,
 1.765788435933272726e+00, 9.461315018083267867e-17, 1.770575974063554714e+00,
 5.961794510040555848e-17, 1.775376492526521188e+00, 6.429731796556572034e-17,
 1.780190026515424462e+00,-5.284627289091617365e-17, 1.785016611318934965e+00,
 1.533040012103131382e-17, 1.789856282321401038e+00,-4.154354660683350387e-17,
 1.794709075003107168e+00, 1.822745842791208677e-17, 1.799575024940535117e+00,
-2.526889233358897644e-17, 1.804454167806623932e+00,-5.177222408793317883e-17,
 1.809346539371031959e+00,-9.032641402450029682e-17, 1.814252175500398856e+00,
-9.969531538920348820e-17, 1.819171112158608494e+00, 7.402676901145838890e-17,
 1.824103385407053413e+00,-1.015962786227708306e-16, 1.829049031404897274e+00,
 6.889192908835695637e-17, 1.834008086409342431e+00, 3.283107224245627204e-17,
 1.838980586775893711e+00, 6.918969740272511942e-18, 1.843966568958625984e+00,
-5.939742026949964550e-17, 1.848966069510450838e+00, 9.027580446261089288e-17,
 1.853979125083385471e+00, 9.761887490727593538e-17, 1.859005772428820480e+00,
-9.528705461989940687e-17, 1.864046048397788979e+00, 6.540912680620571711e-17,
 1.869099989941238604e+00,-9.938505214255067083e-17, 1.874167634110299963e+00,
-6.122763413004142562e-17, 1.879249018056560194e+00,-1.622631555783584478e-17,
 1.884344179032334532e+00,-8.226593125533710906e-17, 1.889453154390939194e+00,
-9.005168285059126718e-17, 1.894575981586965607e+00, 3.403403535216529671e-17,
 1.899712698176555303e+00,-3.859739769378514323e-17, 1.904863341817674138e+00,
 6.533857514718278629e-17, 1.910027950270389852e+00,-5.909688006744060237e-17,
 1.915206561397147400e+00,-1.061994605619596264e-16, 1.920399213163047403e+00,
 7.116681540630314186e-17, 1.925605943636125028e+00,-9.914963769693740927e-17,
 1.930826790987627106e+00, 6.167149706169109553e-17, 1.936061793492294347e+00,
 1.033238596067632574e-16, 1.941310989528640452e+00,-6.638029891621487990e-17,
 1.946574417579233218e+00, 6.811022349533877184e-17, 1.951852116230978318e+00,
-2.199016969979351086e-17, 1.957144124175400179e+00, 8.960767791036667768e-17,
 1.962450480208927317e+00, 1.097684400091354695e-16, 1.967771223233175881e+00,
-1.031492801153113151e-16, 1.973106392255234320e+00,-7.451617863956037486e-18,
 1.978456026387950928e+00, 4.038875310927816657e-17, 1.983820164850219392e+00,
-2.203454412391062657e-17, 1.989198846967266343e+00, 8.205132638369199416e-18,
 1.994592112170940235e+00, 1.790971035200264509e-17
};

static const double __TBL_log2[] = {
	/* __TBL_log2[2*i] = high order rounded 32 bits log2(1+i/256)*256, i = [0, 255] */
	/* __TBL_log2[2*i+1] = low order least bits log2(1+i/256)*256, i = [0, 255]     */
 0.000000000000000000e+00, 0.000000000000000000e+00, 1.439884185791015625e+00,
 4.078417797464839152e-07, 2.874177932739257812e+00,-5.443862030060025621e-07,
 4.302921295166015625e+00, 3.525917800357419922e-07, 5.726161956787109375e+00,
-1.821502755258614180e-06, 7.143936157226562500e+00,-1.035336134691423741e-06,
 8.556289672851562500e+00,-1.279264291071495652e-06, 9.963264465332031250e+00,
-3.206502629414843101e-06, 1.136489105224609375e+01, 3.503517986289194222e-06,
 1.276123046875000000e+01,-1.809406249049319022e-06, 1.415230560302734375e+01,
-2.114722805833714926e-06, 1.553816223144531250e+01,-3.719431504776986979e-06,
 1.691883850097656250e+01,-5.743786819670105240e-06, 1.829435729980468750e+01,
 7.514691093524705578e-06, 1.966479492187500000e+01,-2.076862291588726520e-06,
 2.103015136718750000e+01, 3.219403619538604258e-06, 2.239048767089843750e+01,
-3.108115489869591032e-07, 2.374583435058593750e+01,-6.275103710481114264e-06,
 2.509620666503906250e+01, 6.572855776743687178e-06, 2.644168090820312500e+01,
-1.954725505303359537e-06, 2.778225708007812500e+01, 3.855133152759458770e-06,
 2.911799621582031250e+01,-1.707228100041815487e-06, 3.044891357421875000e+01,
 1.042999152333371737e-06, 3.177505493164062500e+01, 8.966313933586820042e-07,
 3.309646606445312500e+01,-1.372654171244005427e-05, 3.441314697265625000e+01,
-8.996099168734074844e-06, 3.572515869140625000e+01,-1.247731510027211536e-05,
 3.703250122070312500e+01, 8.944258749129049106e-06, 3.833526611328125000e+01,
-3.520082642279872716e-06, 3.963342285156250000e+01, 1.306577612991810031e-05,
 4.092706298828125000e+01,-7.730135593513790229e-07, 4.221618652343750000e+01,
-1.329446142304436745e-05, 4.350079345703125000e+01, 6.912200714904314733e-06,
 4.478097534179687500e+01,-6.216230979739182064e-07, 4.605673217773437500e+01,
-5.133911151040936670e-06, 4.732809448242187500e+01,-6.697901206512330627e-06,
 4.859509277343750000e+01,-5.700153089154811841e-06, 4.985775756835937500e+01,
-2.836263919120346801e-06, 5.111611938476562500e+01, 8.933436604624454391e-07,
 5.237020874023437500e+01, 4.187561748309498307e-06, 5.362005615234375000e+01,
 5.448667394155597532e-06, 5.486569213867187500e+01, 2.786324169943508531e-06,
 5.610714721679687500e+01,-5.978483512667373796e-06, 5.734442138671875000e+01,
 7.207996138368885843e-06, 5.857757568359375000e+01, 9.083351754561760127e-06,
 5.980664062500000000e+01,-3.374516276140515786e-06, 6.103161621093750000e+01,
-2.943717299925017200e-06, 6.225253295898437500e+01, 6.810091060168101732e-06,
 6.346945190429687500e+01,-8.462738988588859704e-06, 6.468237304687500000e+01,
-2.233961135216831566e-05, 6.589129638671875000e+01,-8.657399896582645111e-06,
 6.709625244140625000e+01, 2.797335967336006296e-05, 6.829736328125000000e+01,
-8.863355250907819214e-06, 6.949450683593750000e+01, 2.830758238800374038e-05,
 7.068786621093750000e+01,-1.846073268549083018e-05, 7.187731933593750000e+01,
-2.182503249464459606e-06, 7.306298828125000000e+01,-2.025251442448625989e-05,
 7.424481201171875000e+01, 1.280303154355201204e-05, 7.542291259765625000e+01,
-8.813997363590295654e-07, 7.659722900390625000e+01, 2.370323712746426047e-05,
 7.776788330078125000e+01,-1.176744290134661421e-05, 7.893481445312500000e+01,
-2.273743674288609119e-05, 8.009802246093750000e+01, 1.409185747234803696e-05,
 8.125762939453125000e+01,-2.707246895087010889e-07, 8.241357421875000000e+01,
 1.807241476105480180e-05, 8.356597900390625000e+01,-3.030059664889450720e-05,
 8.471472167968750000e+01,-8.823455531875539245e-07, 8.585992431640625000e+01,
 6.485238524924182146e-06, 8.700158691406250000e+01, 1.382440142980862947e-05,
 8.813977050781250000e+01,-1.808136338482881111e-05, 8.927441406250000000e+01,
-6.579344146543672011e-06, 9.040557861328125000e+01, 8.714227880222726313e-06,
 9.153332519531250000e+01,-1.201308307454951138e-05, 9.265759277343750000e+01,
 1.330278431878087205e-05, 9.377850341796875000e+01,-1.657103990890600482e-05,
 9.489599609375000000e+01,-1.995110226941163424e-05, 9.601007080078125000e+01,
 2.362403148762806632e-05, 9.712084960937500000e+01, 1.236086810905991142e-05,
 9.822827148437500000e+01, 2.738898236946465744e-05, 9.933239746093750000e+01,
 2.758741700388469572e-05, 1.004332885742187500e+02,-2.834285611604269955e-05,
 1.015308227539062500e+02, 1.228649517068771375e-06, 1.026251220703125000e+02,
 1.361792668612316888e-05, 1.037161865234375000e+02, 2.803946653578170389e-05,
 1.048040771484375000e+02, 2.502814149567842806e-06, 1.058887329101562500e+02,
 1.692003190104140317e-05, 1.069702148437500000e+02, 2.896703985131545672e-05,
 1.080485839843750000e+02,-3.844135045484567362e-06, 1.091237792968750000e+02,
-2.093137927645659717e-06, 1.101958618164062500e+02,-8.590030211185738579e-06,
 1.112648315429687500e+02,-5.267967244023324300e-06, 1.123306884765625000e+02,
 2.578347229232600646e-05, 1.133935546875000000e+02,-1.975022555464358195e-05,
 1.144533081054687500e+02,-2.195797778964440179e-06, 1.155100708007812500e+02,
-2.617170507638525077e-05, 1.165637817382812500e+02,-1.334031370958194516e-05,
 1.176145019531250000e+02,-7.581976902412963145e-06, 1.186622314453125000e+02,
 8.112109654298731037e-06, 1.197070312500000000e+02,-1.042875265529314613e-05,
 1.207488403320312500e+02, 1.455233211877492951e-05, 1.217877807617187500e+02,
-2.243432092472914265e-05, 1.228237304687500000e+02, 1.712269952247034061e-05,
 1.238568115234375000e+02, 2.745621214456745937e-05, 1.248870239257812500e+02,
 2.473291989440979066e-05, 1.259143676757812500e+02, 2.498461547595911484e-05,
 1.269389038085937500e+02,-1.692547797717771941e-05, 1.279605712890625000e+02,
-2.419576192770340594e-05, 1.289793701171875000e+02, 1.880972467762623192e-05,
 1.299954833984375000e+02,-5.550757125543327248e-05, 1.310086669921875000e+02,
 1.237226167189998996e-05, 1.320191650390625000e+02,-6.438347630770959254e-06,
 1.330268554687500000e+02, 2.525911246920619613e-05, 1.340318603515625000e+02,
 3.990327953073019333e-07, 1.350340576171875000e+02, 5.593427389035480335e-05,
 1.360336914062500000e+02,-3.751407409478960320e-05, 1.370305175781250000e+02,
-2.116319935859897563e-05, 1.380246582031250000e+02,-2.559468964093475045e-06,
 1.390161132812500000e+02, 3.270409087092109593e-05, 1.400050048828125000e+02,
-2.315157751389992129e-05, 1.409912109375000000e+02,-3.387938973438343638e-05,
 1.419747314453125000e+02, 1.458416266727572812e-05, 1.429556884765625000e+02,
 1.412021555596584681e-05, 1.439340820312500000e+02,-2.143065540113838312e-05,
 1.449097900390625000e+02, 4.373273697503468317e-05, 1.458830566406250000e+02,
-2.090790235253405790e-05, 1.468536376953125000e+02, 4.230297794089183646e-05,
 1.478217773437500000e+02, 2.633401664450247309e-06, 1.487873535156250000e+02,
-4.542835986281740771e-06, 1.497503662109375000e+02, 3.397367848245215483e-05,
 1.507109375000000000e+02, 9.209059510146982590e-06, 1.516689453125000000e+02,
 5.622812858742714859e-05, 1.526246337890625000e+02,-5.621609346274134244e-05,
 1.535776367187500000e+02, 5.088115468603551539e-05, 1.545283203125000000e+02,
 2.400396513473623342e-05, 1.554765625000000000e+02,-2.180099663431456814e-06,
 1.564223632812500000e+02,-1.517056781617965675e-05, 1.573657226562500000e+02,
-2.562756696989711716e-06, 1.583066406250000000e+02, 4.795320325388065854e-05,
 1.592452392578125000e+02, 2.652301982429665372e-05, 1.601815185546875000e+02,
-5.473018439029181240e-05, 1.611152343750000000e+02, 6.036538006249134820e-05,
 1.620467529296875000e+02, 1.753890969321481711e-05, 1.629759521484375000e+02,
-4.928926339732922490e-05, 1.639027099609375000e+02,-6.288016979631557560e-06,
 1.648271484375000000e+02, 3.614482952210960361e-05, 1.657493896484375000e+02,
-3.247597790375142114e-05, 1.666691894531250000e+02, 4.348868072528205213e-05,
 1.675867919921875000e+02, 3.131097214651595330e-05, 1.685021972656250000e+02,
-5.768116554728405733e-05, 1.694151611328125000e+02, 3.189681619086343127e-05,
 1.703260498046875000e+02,-5.500528238559059116e-05, 1.712344970703125000e+02,
 5.890184674174263693e-05, 1.721408691406250000e+02, 1.840407787096519837e-05,
 1.730450439453125000e+02,-4.351222480150346831e-05, 1.739468994140625000e+02,
 6.059331686505290421e-06, 1.748465576171875000e+02, 5.580532332169584454e-05,
 1.757441406250000000e+02,-5.666096094448416139e-06, 1.766395263671875000e+02,
-4.568380948624016041e-05, 1.775327148437500000e+02,-5.372392273978838048e-05,
 1.784237060546875000e+02,-1.933871000131713187e-05, 1.793126220703125000e+02,
-5.422619290693841471e-05, 1.801993408203125000e+02,-2.601847861521447132e-05,
 1.810839843750000000e+02,-4.656229401600182454e-05, 1.819664306640625000e+02,
 1.636297150881445295e-05, 1.828468017578125000e+02, 5.076471489501210225e-05,
 1.837252197265625000e+02,-5.542156510357154555e-05, 1.846014404296875000e+02,
-4.812064810565531807e-05, 1.854755859375000000e+02,-3.953879286781995545e-05,
 1.863476562500000000e+02,-1.988182101010412125e-05, 1.872176513671875000e+02,
 2.057522891062264376e-05, 1.880856933593750000e+02,-3.058156040982771239e-05,
 1.889516601562500000e+02,-4.169340446171797184e-05, 1.898155517578125000e+02,
-3.239118881346662872e-06, 1.906774902343750000e+02,-2.783449132689922134e-05,
 1.915373535156250000e+02, 1.597927683340914293e-05, 1.923952636718750000e+02,
 1.545493412281261116e-05, 1.932512207031250000e+02,-2.014927705264352875e-05,
 1.941051025390625000e+02, 4.043097907577914080e-05, 1.949571533203125000e+02,
-3.781452579504048975e-05, 1.958071289062500000e+02,-1.677810793588779092e-06,
 1.966551513671875000e+02, 3.577570564777057149e-05, 1.975013427734375000e+02,
-3.858128431828155999e-05, 1.983454589843750000e+02, 2.827352539329734468e-05,
 1.991877441406250000e+02, 1.020426695132691908e-06, 2.000280761718750000e+02,
 1.049043785864183866e-05, 2.008665771484375000e+02,-5.668571223208539910e-05,
 2.017030029296875000e+02, 5.227451898157462205e-05, 2.025377197265625000e+02,
-2.025647781341857894e-05, 2.033704833984375000e+02,-2.161281037339224341e-05,
 2.042012939453125000e+02, 5.667325008632565576e-05, 2.050303955078125000e+02,
-2.112821448834358837e-05, 2.058575439453125000e+02,-2.522383155215216853e-06,
 2.066828613281250000e+02,-1.281378348494855858e-06, 2.075063476562500000e+02,
-9.162516382743561384e-06, 2.083280029296875000e+02,-1.797812601298608335e-05,
 2.091478271484375000e+02,-1.959505997696247453e-05, 2.099658203125000000e+02,
-5.934211946670452627e-06, 2.107819824218750000e+02, 3.102996118252714271e-05,
 2.115964355468750000e+02,-2.280040076415178584e-05, 2.124090576171875000e+02,
-3.743515649437846729e-05, 2.132198486328125000e+02,-5.006638631136701490e-06,
 2.140289306640625000e+02,-3.976919665668718942e-05, 2.148361816406250000e+02,
-1.188780735169185652e-05, 2.156417236328125000e+02,-3.571887766413048520e-05,
 2.164454345703125000e+02, 1.847144755636210490e-05, 2.172474365234375000e+02,
 3.622647302213163157e-05, 2.180477294921875000e+02, 2.511032323154433900e-05,
 2.188463134765625000e+02,-7.361941985081681848e-06, 2.196431884765625000e+02,
-5.372390403709574017e-05, 2.204382324218750000e+02, 1.551294579696132803e-05,
 2.212316894531250000e+02,-3.642162925932327343e-05, 2.220233154296875000e+02,
 4.193598594979618241e-05, 2.228133544921875000e+02, 1.372116405796589833e-05,
 2.236016845703125000e+02, 8.233623894335039537e-06, 2.243883056640625000e+02,
 3.265657742833052654e-05, 2.251733398437500000e+02,-2.794287750390687326e-05,
 2.259566650390625000e+02,-4.440243113774530265e-05, 2.267382812500000000e+02,
-9.675114830058622014e-06, 2.275183105468750000e+02,-3.882892066889445600e-05,
 2.282966308593750000e+02,-2.835487591479255673e-06, 2.290733642578125000e+02,
-1.685097895998181422e-05, 2.298483886718750000e+02, 4.806553595480019518e-05,
 2.306219482421875000e+02,-4.539911586906436716e-05, 2.313937988281250000e+02,
-4.631966285757620260e-05, 2.321639404296875000e+02, 5.204609324350696002e-05,
 2.329326171875000000e+02, 1.225763073721718197e-05, 2.336997070312500000e+02,
-3.695637982554016382e-05, 2.344650878906250000e+02, 3.309133292926460016e-05,
 2.352290039062500000e+02,-1.516395380482592629e-05, 2.359913330078125000e+02,
-5.311674305290968619e-05, 2.367519531250000000e+02, 4.779807991226078768e-05,
 2.375111083984375000e+02, 4.989464209345647548e-05, 2.382687988281250000e+02,
-4.041202611322311408e-05, 2.390247802734375000e+02, 2.739433433590848536e-05,
 2.397792968750000000e+02, 1.550965806406508966e-05, 2.405322265625000000e+02,
 5.230206142425020257e-05, 2.412836914062500000e+02, 2.196059540790264514e-05,
 2.420335693359375000e+02, 5.277680785141730338e-05, 2.427819824218750000e+02,
 2.886380247947272558e-05, 2.435289306640625000e+02,-4.363251767645384661e-05,
 2.442742919921875000e+02,-3.653314744654563199e-05, 2.450180664062500000e+02,
 5.623369525922526825e-05, 2.457604980468750000e+02,-3.437446279919778004e-06,
 2.465013427734375000e+02, 3.459290119679066472e-05, 2.472407226562500000e+02,
 5.421724428316440202e-05, 2.479787597656250000e+02,-6.070765164808318435e-05,
 2.487152099609375000e+02,-6.014953987030989107e-05, 2.494501953125000000e+02,
-6.032228506450037554e-05, 2.501837158203125000e+02,-5.540433388359054134e-05,
 2.509157714843750000e+02,-3.960875078622925214e-05, 2.516463623046875000e+02,
-7.182944107105660894e-06, 2.523754882812500000e+02, 4.759160516857532540e-05,
 2.531032714843750000e+02, 8.329299458439681639e-06, 2.538295898437500000e+02,
 2.751627995643241118e-06, 2.545544433593750000e+02, 3.647649263201999678e-05,
 2.552779541015625000e+02,-6.981531437649667064e-06
};

static const unsigned long long LCONST[] = {
0x3c90000000000000ULL,	/* 2**(-54) = 5.551115123125782702e-17		*/
0x3ff0000000000000ULL,	/* DONE = 1.0					*/
0x4330000000000000ULL,	/* DVAIN52 = 2**52 = 4.503599627370496e15	*/
0xffffffff00000000ULL,	/* 0xffffffff00000000				*/
0x000fffffffffffffULL,	/* 0x000fffffffffffff				*/
0x0000080000000000ULL,	/* 0x0000080000000000				*/
0xfffff00000000000ULL,	/* 0xfffff00000000000				*/
0x0000000000000000ULL,	/* DZERO   = 0.0				*/
0x4062776d8ce329bdULL,	/* KA5     = 5.77078604860893737986e-01*256	*/
0x406ec709dc39fc99ULL,	/* KA3     = 9.61796693925765549423e-01*256	*/
0x3f6d94ae0bf85de6ULL,	/* KA1_LO  = 1.41052154268147309568e-05*256	*/
0x4087154000000000ULL,	/* KA1_HI  = 2.8853759765625e+00*256		*/
0x40871547652b82feULL,	/* KA1     = 2.885390081777926774e+00*256	*/
0x4110000000000000ULL,	/* HTHRESH = 262144.0				*/
0xc110cc0000000000ULL,	/* LTHRESH = -275200.0				*/
0x3cd5d52893bc7fecULL,	/* KB5     = 1.21195555854068860923e-15		*/
0x3d83b2abc07c93d0ULL,	/* KB4     = 2.23939573811855104311e-12		*/
0x3e2c6b08d71f5d1eULL,	/* KB3     = 3.30830268126604677436e-09		*/
0x3ecebfbdff82c4edULL,	/* KB2     = 3.66556559691003767877e-06		*/
0x3f662e42fefa39efULL,	/* KB1     = 2.70760617406228636578e-03		*/
0x01a56e1fc2f8f359ULL,	/* _TINY   = 1.0e-300				*/
0x7e37e43c8800759cULL	/* _HUGE   = 1.0e+300				*/
};

#define SCALE_ARR	((double*)LCONST + 1)
#define _TINY		((double*)LCONST)[20]	/* 1.0e-300	*/
#define _HUGE		((double*)LCONST)[21]	/* 1.0e+300	*/

#define RET_SC(I)										\
	px += stridex;										\
	py += stridey;										\
	pz += stridez;										\
	if ( --n <= 0 )										\
		break;										\
	goto start##I;

#define RETURN(I, ret)										\
{												\
	pz[0] = (ret);										\
	RET_SC(I)										\
}

#define PREP(I)											\
hx = HI(px);									\
lx = LO(px);									\
hy = HI(py);									\
ly = LO(py);									\
sx = hx >> 31;											\
sy = hy >> 31;											\
hx &= 0x7fffffff;										\
hy &= 0x7fffffff;										\
ull_y0 = *(unsigned long long*)px;								\
												\
if ( hy < 0x3bf00000 )		/* |Y| < 2^(-64) */						\
{												\
	y0 = *px;										\
	if ( (hy | ly) == 0 )	/* pow(X,0) */							\
		RETURN (I, DONE)								\
	if ( hx > 0x7ff00000 || (hx == 0x7ff00000 && lx != 0) )		/* |X| = Nan */		\
		*pz = y0 + y0;									\
	else if ( (hx | lx) == 0 || (hx == 0x7ff00000 && lx == 0) )	/* X = 0 or Inf */	\
	{											\
		HI(pz) = hx;								\
		LO(pz) = lx;								\
		if ( sy )									\
			*pz = DONE / *pz;							\
	}											\
	else											\
		*pz = (sx) ? DZERO / DZERO : DONE;						\
	RET_SC(I)										\
}												\
yisint##I = 0;	/* Y - non-integer */								\
exp = hy >> 20;	/* Y exponent */								\
ull_y0 &= LMMANT;										\
ull_x##I = ull_y0 | LDONE;									\
x##I = *(double*)&ull_x##I;									\
ull_ax##I = (ull_x##I + LMROUND) & LMHI20;							\
ax##I = *(double*)&ull_ax##I;									\
if ( hx >= 0x7ff00000 || exp >= 0x43e )		/* X=Inf,Nan or |Y|>2^63,Inf,Nan */		\
{												\
	y0 = *px;										\
	if ( hx > 0x7ff00000 || (hx == 0x7ff00000 && lx != 0) ||				\
	hy > 0x7ff00000 || (hy == 0x7ff00000 && ly != 0) )	/* |X| or |Y| = Nan */		\
		RETURN (I, y0 + *py)								\
	if ( hy == 0x7ff00000 && (ly == 0) )			/* |Y| = Inf */			\
	{											\
		if ( hx == 0x3ff00000 && (lx == 0) )		/* +-1 ** +-Inf */		\
			*pz = *py - *py;							\
		else if ( (hx < 0x3ff00000) != sy )						\
			*pz = DZERO;								\
		else										\
		{										\
			HI(pz) = hy;							\
			LO(pz) = ly;							\
		}										\
		RET_SC(I)									\
	}											\
	if ( exp < 0x43e )	/* |Y| < 2^63   */						\
	{											\
		if ( sx )	/* X = -Inf     */						\
		{										\
			if ( exp >= 0x434 )	/* |Y| >= 2^53  */				\
				yisint##I = 2;	/* Y - even     */				\
			else									\
			{									\
				if ( exp >= 0x3ff )	/* |Y| >= 1     */			\
				{								\
					if ( exp > (20 + 0x3ff) )				\
					{							\
						i0 = ly >> (52 - (exp - 0x3ff));		\
						if ( (i0 << (52 - (exp - 0x3ff))) == ly )	\
							yisint##I = 2 - (i0 & 1);		\
					}							\
					else if ( ly == 0 )					\
					{							\
						i0 = hy >> (20 - (exp - 0x3ff));		\
						if ( (i0 << (20 - (exp - 0x3ff))) == hy )	\
							yisint##I = 2 - (i0 & 1);		\
					}							\
				}								\
			}									\
		}										\
		if ( sy )									\
			hx = lx = 0;								\
		hx += yisint##I << 31;								\
		HI(pz) = hx;								\
		LO(pz) = lx;								\
		RET_SC(I)									\
	}											\
	else	/* |Y| >= 2^63   */								\
	{											\
		/* |X| = 0, 1, Inf */								\
		if ( lx == 0 && (hx == 0 || hx == 0x3ff00000 || hx == 0x7ff00000) )		\
		{										\
			HI(pz) = hx;							\
			LO(pz) = lx;							\
			if ( sy )								\
				*pz = DONE / *pz;						\
		}										\
		else										\
		{										\
			y0 = ( (hx < 0x3ff00000) != sy ) ? _TINY : _HUGE;			\
			*pz = y0 * y0;								\
		}										\
		RET_SC(I)									\
	}											\
}												\
if ( sx || (hx | lx) == 0 )	/* X <= 0      */						\
{												\
	if ( exp >= 0x434 )	/* |Y| >= 2^53 */						\
		yisint##I = 2;	/* Y - even    */						\
	else											\
	{											\
		if ( exp >= 0x3ff )	/* |Y| >= 1    */					\
		{										\
			if ( exp > (20 + 0x3ff) )						\
			{									\
				i0 = ly >> (52 - (exp - 0x3ff));				\
				if ( (i0 << (52 - (exp - 0x3ff))) == ly )			\
					yisint##I = 2 - (i0 & 1);				\
			}									\
			else if ( ly == 0 )							\
			{									\
				i0 = hy >> (20 - (exp - 0x3ff));				\
				if ( (i0 << (20 - (exp - 0x3ff))) == hy )			\
					yisint##I = 2 - (i0 & 1);				\
			}									\
		}										\
	}											\
	if ( (hx | lx) == 0 )		/* X == 0  */						\
	{											\
		y0 = DZERO;									\
		if ( sy )									\
			y0 = DONE / y0;								\
		if ( sx & yisint##I )								\
			y0 = -y0;								\
		RETURN (I, y0)									\
	}											\
	if ( yisint##I == 0 )			/* pow(neg,non-integer) */			\
		RETURN (I, DZERO / DZERO)	/* NaN */					\
}												\
exp = (hx >> 20);										\
exp##I = exp - 2046;										\
py##I = py;											\
pz##I = pz;											\
ux##I = x##I + ax##I;										\
if ( !exp )											\
{												\
	ax##I = (double) ull_y0;								\
	ull_ax##I = *(unsigned long long*)&ax##I;						\
	ull_x##I = ull_ax##I & LMMANT | LDONE;							\
	x##I = *(double*)&ull_x##I;								\
	exp##I = ((unsigned int*) & ull_ax##I)[0];						\
	exp##I = (exp##I >> 20) - (2046 + 1023 + 51);						\
	ull_ax##I = ull_x##I + LMROUND & LMHI20;						\
	ax##I = *(double*)&ull_ax##I;								\
	ux##I = x##I + ax##I;									\
}												\
ull_x##I = *(unsigned long long *)&ux##I;							\
hx##I = HI(&ull_ax##I);									\
yd##I = DONE / ux##I;

void
__vpow( int n, double * restrict px, int stridex, double * restrict py,
	int stridey, double * restrict pz, int stridez )
{
	double			*py0, *py1, *py2;
	double			*pz0, *pz1, *pz2;
	double			y0, yd0, u0, s0, s_l0, m_h0;
	double			y1, yd1, u1, s1, s_l1, m_h1;
	double			y2, yd2, u2, s2, s_l2, m_h2;
	double			ax0, x0, s_h0, ux0;
	double			ax1, x1, s_h1, ux1;
	double			ax2, x2, s_h2, ux2;
	int			eflag0, gflag0, ind0, i0;
	int			eflag1, gflag1, ind1, i1;
	int			eflag2, gflag2, ind2, i2;
	int			hx0, yisint0, exp0;
	int			hx1, yisint1, exp1;
	int			hx2, yisint2, exp2;
	int			exp, i = 0;
	unsigned		hx, lx, sx, hy, ly, sy;
	unsigned long long	ull_y0, ull_x0, ull_x1, ull_x2, ull_ax0, ull_ax1, ull_ax2;
	unsigned long long	LDONE = ((unsigned long long*)LCONST)[1];	/* 1.0					*/
	unsigned long long	LMMANT = ((unsigned long long*)LCONST)[4];	/* 0x000fffffffffffff			*/
	unsigned long long	LMROUND = ((unsigned long long*)LCONST)[5];	/* 0x0000080000000000			*/
	unsigned long long	LMHI20 = ((unsigned long long*)LCONST)[6];	/* 0xfffff00000000000			*/
	double			DONE = ((double*)LCONST)[1];			/* 1.0					*/
	double			DZERO = ((double*)LCONST)[7];			/* 0.0					*/
	double			KA5 = ((double*)LCONST)[8];			/* 5.77078604860893737986e-01*256	*/
	double			KA3 = ((double*)LCONST)[9];			/* 9.61796693925765549423e-01*256	*/
	double			KA1_LO = ((double*)LCONST)[10];			/* 1.41052154268147309568e-05*256	*/
	double			KA1_HI = ((double*)LCONST)[11];			/* 2.8853759765625e+00*256		*/
	double			KA1 = ((double*)LCONST)[12];			/* 2.885390081777926774e+00*256		*/
	double			HTHRESH = ((double*)LCONST)[13];		/*  262144.0				*/
	double			LTHRESH = ((double*)LCONST)[14];		/* -275200.0				*/
	double			KB5 = ((double*)LCONST)[15];			/* 1.21195555854068860923e-15		*/
	double			KB4 = ((double*)LCONST)[16];			/* 2.23939573811855104311e-12		*/
	double			KB3 = ((double*)LCONST)[17];			/* 3.30830268126604677436e-09		*/
	double			KB2 = ((double*)LCONST)[18];			/* 3.66556559691003767877e-06		*/
	double			KB1 = ((double*)LCONST)[19];			/* 2.70760617406228636578e-03		*/

	if (stridex == 0)
	{
		unsigned	hx = HI(px);
		unsigned	lx = LO(px);

		/* if x is a positive normal number not equal to one,
		   call __vpowx */
		if (hx >= 0x00100000 && hx < 0x7ff00000 &&
			(hx != 0x3ff00000 || lx != 0))
		{
			__vpowx( n, px, py, stridey, pz, stridez );
			return;
		}
	}

	do
	{
		/* perform si + ydi = 256*log2(xi)*yi */
start0:
		PREP(0)
		px += stridex;
		py += stridey;
		pz += stridez;
		i = 1;
		if ( --n <= 0 )
			break;

start1:
		PREP(1)
		px += stridex;
		py += stridey;
		pz += stridez;
		i = 2;
		if ( --n <= 0 )
			break;

start2:
		PREP(2)

		u0 = x0 - ax0;
		u1 = x1 - ax1;
		u2 = x2 - ax2;

		s0 = u0 * yd0;
		LO(&ux0) = 0;
		s1 = u1 * yd1;
		LO(&ux1) = 0;
		s2 = u2 * yd2;
		LO(&ux2) = 0;

		y0 = s0 * s0;
		s_h0 = s0;
		LO(&s_h0) = 0;
		y1 = s1 * s1;
		s_h1 = s1;
		LO(&s_h1) = 0;
		y2 = s2 * s2;
		s_h2 = s2;
		LO(&s_h2) = 0;

		s0 = (KA5 * y0 + KA3) * y0 * s0;
		s1 = (KA5 * y1 + KA3) * y1 * s1;
		s2 = (KA5 * y2 + KA3) * y2 * s2;

		s_l0 = (x0 - (ux0 - ax0));
		s_l1 = (x1 - (ux1 - ax1));
		s_l2 = (x2 - (ux2 - ax2));

		s_l0 = u0 - s_h0 * ux0 - s_h0 * s_l0;
		s_l1 = u1 - s_h1 * ux1 - s_h1 * s_l1;
		s_l2 = u2 - s_h2 * ux2 - s_h2 * s_l2;

		s_l0 = KA1 * yd0 * s_l0;
		i0 = (hx0 >> 8) & 0xff0;
		exp0 += (hx0 >> 20);

		s_l1 = KA1 * yd1 * s_l1;
		i1 = (hx1 >> 8) & 0xff0;
		exp1 += (hx1 >> 20);

		s_l2 = KA1 * yd2 * s_l2;
		i2 = (hx2 >> 8) & 0xff0;
		exp2 += (hx2 >> 20);

		yd0 = KA1_HI * s_h0;
		yd1 = KA1_HI * s_h1;
		yd2 = KA1_HI * s_h2;

		y0 = *(double *)((char*)__TBL_log2 + i0);
		y1 = *(double *)((char*)__TBL_log2 + i1);
		y2 = *(double *)((char*)__TBL_log2 + i2);

		y0 += (double)(exp0 << 8);
		y1 += (double)(exp1 << 8);
		y2 += (double)(exp2 << 8);

		m_h0 = y0 + yd0;
		m_h1 = y1 + yd1;
		m_h2 = y2 + yd2;

		y0 = s0 - ((m_h0 - y0 - yd0) - s_l0);
		y1 = s1 - ((m_h1 - y1 - yd1) - s_l1);
		y2 = s2 - ((m_h2 - y2 - yd2) - s_l2);

		y0 += *(double *)((char*)__TBL_log2 + i0 + 8) + KA1_LO * s_h0;
		y1 += *(double *)((char*)__TBL_log2 + i1 + 8) + KA1_LO * s_h1;
		y2 += *(double *)((char*)__TBL_log2 + i2 + 8) + KA1_LO * s_h2;

		s_h0 = y0 + m_h0;
		s_h1 = y1 + m_h1;
		s_h2 = y2 + m_h2;

		LO(&s_h0) = 0;
		LO(&s_h1) = 0;
		LO(&s_h2) = 0;

		yd0 = *py0;
		yd1 = *py1;
		yd2 = *py2;
		s0 = yd0;
		s1 = yd1;
		s2 = yd2;
		LO(&s0) = 0;
		LO(&s1) = 0;
		LO(&s2) = 0;

		y0 = y0 - (s_h0 - m_h0);
		y1 = y1 - (s_h1 - m_h1);
		y2 = y2 - (s_h2 - m_h2);

		yd0 = (yd0 - s0) * s_h0 + yd0 * y0;
		yd1 = (yd1 - s1) * s_h1 + yd1 * y1;
		yd2 = (yd2 - s2) * s_h2 + yd2 * y2;

		s0 = s_h0 * s0;
		s1 = s_h1 * s1;
		s2 = s_h2 * s2;

		/* perform 2 ** ((si+ydi)/256) */
		if ( s0 > HTHRESH )
		{
			s0 = HTHRESH;
			yd0 = DZERO;
		}
		if ( s1 > HTHRESH )
		{
			s1 = HTHRESH;
			yd1 = DZERO;
		}
		if ( s2 > HTHRESH )
		{
			s2 = HTHRESH;
			yd2 = DZERO;
		}

		if ( s0 < LTHRESH )
		{
			s0 = LTHRESH;
			yd0 = DZERO;
		}
		ind0 = (int) (s0 + yd0);
		if ( s1 < LTHRESH )
		{
			s1 = LTHRESH;
			yd1 = DZERO;
		}
		ind1 = (int) (s1 + yd1);
		if ( s2 < LTHRESH )
		{
			s2 = LTHRESH;
			yd2 = DZERO;
		}
		ind2 = (int) (s2 + yd2);

		i0 = (ind0 & 0xff) << 4;
		u0 = (double) ind0;
		ind0 >>= 8;

		i1 = (ind1 & 0xff) << 4;
		u1 = (double)ind1;
		ind1 >>= 8;

		i2 = (ind2 & 0xff) << 4;
		u2 = (double) ind2;
		ind2 >>= 8;

		y0 = s0 - u0 + yd0;
		y1 = s1 - u1 + yd1;
		y2 = s2 - u2 + yd2;

		u0 = *(double*)((char*)__TBL_exp2 + i0);
		y0 = ((((KB5 * y0 + KB4) * y0 + KB3) * y0 + KB2) * y0 + KB1) * y0;
		u1 = *(double*)((char*)__TBL_exp2 + i1);
		y1 = ((((KB5 * y1 + KB4) * y1 + KB3) * y1 + KB2) * y1 + KB1) * y1;
		u2 = *(double*)((char*)__TBL_exp2 + i2);
		y2 = ((((KB5 * y2 + KB4) * y2 + KB3) * y2 + KB2) * y2 + KB1) * y2;

		eflag0 = (ind0 + 1021) >> 31;
		gflag0 = (1022 - ind0) >> 31;
		eflag1 = (ind1 + 1021) >> 31;
		gflag1 = (1022 - ind1) >> 31;
		eflag2 = (ind2 + 1021) >> 31;
		gflag2 = (1022 - ind2) >> 31;

		ind0 = (yisint0 << 11) + ind0 + (54 & eflag0) - (52 & gflag0);
		ind0 <<= 20;
		ind1 = (yisint1 << 11) + ind1 + (54 & eflag1) - (52 & gflag1);
		ind1 <<= 20;
		ind2 = (yisint2 << 11) + ind2 + (54 & eflag2) - (52 & gflag2);
		ind2 <<= 20;

		u0 = *(double*)((char*)__TBL_exp2 + i0 + 8) + u0 * y0 + u0;
		u1 = *(double*)((char*)__TBL_exp2 + i1 + 8) + u1 * y1 + u1;
		u2 = *(double*)((char*)__TBL_exp2 + i2 + 8) + u2 * y2 + u2;

		ull_x0 = *(unsigned long long*)&u0;
		HI(&ull_x0) += ind0;
		u0 = *(double*)&ull_x0;

		ull_x1 = *(unsigned long long*)&u1;
		HI(&ull_x1) += ind1;
		u1 = *(double*)&ull_x1;

		ull_x2 = *(unsigned long long*)&u2;
		HI(&ull_x2) += ind2;
		u2 = *(double*)&ull_x2;

		*pz0 = u0 * SCALE_ARR[eflag0 - gflag0];
		*pz1 = u1 * SCALE_ARR[eflag1 - gflag1];
		*pz2 = u2 * SCALE_ARR[eflag2 - gflag2];

		px += stridex;
		py += stridey;
		pz += stridez;
		i = 0;

	} while ( --n > 0 );

	if ( i > 0 )
	{
		/* perform si + ydi = 256*log2(xi)*yi */
		u0 = x0 - ax0;
		s0 = u0 * yd0;
		LO(&ux0) = 0;
		y0 = s0 * s0;
		s_h0 = s0;
		LO(&s_h0) = 0;
		s0 = (KA5 * y0 + KA3) * y0 * s0;
		s_l0 = (x0 - (ux0 - ax0));
		s_l0 = u0 - s_h0 * ux0 - s_h0 * s_l0;
		s_l0 = KA1 * yd0 * s_l0;
		i0 = (hx0 >> 8) & 0xff0;
		exp0 += (hx0 >> 20);
		yd0 = KA1_HI * s_h0;
		y0 = *(double *)((char*)__TBL_log2 + i0);
		y0 += (double)(exp0 << 8);
		m_h0 = y0 + yd0;
		y0 = s0 - ((m_h0 - y0 - yd0) - s_l0);
		y0 += *(double *)((char*)__TBL_log2 + i0 + 8) + KA1_LO * s_h0;
		s_h0 = y0 + m_h0;
		LO(&s_h0) = 0;
		y0 = y0 - (s_h0 - m_h0);
		s0 = yd0 = *py0;
		LO(&s0) = 0;
		yd0 = (yd0 - s0) * s_h0 + yd0 * y0;
		s0 = s_h0 * s0;

		/* perform 2 ** ((si+ydi)/256) */
		if ( s0 > HTHRESH )
		{
			s0 = HTHRESH;
			yd0 = DZERO;
		}
		if ( s0 < LTHRESH )
		{
			s0 = LTHRESH;
			yd0 = DZERO;
		}
		ind0 = (int) (s0 + yd0);
		i0 = (ind0 & 0xff) << 4;
		u0 = (double) ind0;
		ind0 >>= 8;
		y0 = s0 - u0 + yd0;
		u0 = *(double*)((char*)__TBL_exp2 + i0);
		y0 = ((((KB5 * y0 + KB4) * y0 + KB3) * y0 + KB2) * y0 + KB1) * y0;
		eflag0 = (ind0 + 1021) >> 31;
		gflag0 = (1022 - ind0) >> 31;
		u0 = *(double*)((char*)__TBL_exp2 + i0 + 8) + u0 * y0 + u0;
		ind0 = (yisint0 << 11) + ind0 + (54 & eflag0) - (52 & gflag0);
		ind0 <<= 20;
		ull_x0 = *(unsigned long long*)&u0;
		HI(&ull_x0) += ind0;
		u0 = *(double*)&ull_x0;

		*pz0 = u0 * SCALE_ARR[eflag0 - gflag0];

		if ( i > 1 )
		{
			/* perform si + ydi = 256*log2(xi)*yi */
			u0 = x1 - ax1;
			s0 = u0 * yd1;
			LO(&ux1) = 0;
			y0 = s0 * s0;
			s_h0 = s0;
			LO(&s_h0) = 0;
			s0 = (KA5 * y0 + KA3) * y0 * s0;
			s_l0 = (x1 - (ux1 - ax1));
			s_l0 = u0 - s_h0 * ux1 - s_h0 * s_l0;
			s_l0 = KA1 * yd1 * s_l0;
			i0 = (hx1 >> 8) & 0xff0;
			exp1 += (hx1 >> 20);
			yd0 = KA1_HI * s_h0;
			y0 = *(double *)((char*)__TBL_log2 + i0);
			y0 += (double)(exp1 << 8);
			m_h0 = y0 + yd0;
			y0 = s0 - ((m_h0 - y0 - yd0) - s_l0);
			y0 += *(double *)((char*)__TBL_log2 + i0 + 8) + KA1_LO * s_h0;
			s_h0 = y0 + m_h0;
			LO(&s_h0) = 0;
			y0 = y0 - (s_h0 - m_h0);
			s0 = yd0 = *py1;
			LO(&s0) = 0;
			yd0 = (yd0 - s0) * s_h0 + yd0 * y0;
			s0 = s_h0 * s0;
			/* perform 2 ** ((si+ydi)/256) */
			if ( s0 > HTHRESH )
			{
				s0 = HTHRESH;
				yd0 = DZERO;
			}
			if ( s0 < LTHRESH )
			{
				s0 = LTHRESH;
				yd0 = DZERO;
			}
			ind0 = (int) (s0 + yd0);
			i0 = (ind0 & 0xff) << 4;
			u0 = (double) ind0;
			ind0 >>= 8;
			y0 = s0 - u0 + yd0;
			u0 = *(double*)((char*)__TBL_exp2 + i0);
			y0 = ((((KB5 * y0 + KB4) * y0 + KB3) * y0 + KB2) * y0 + KB1) * y0;
			eflag0 = (ind0 + 1021) >> 31;
			gflag0 = (1022 - ind0) >> 31;
			u0 = *(double*)((char*)__TBL_exp2 + i0 + 8) + u0 * y0 + u0;
			ind0 = (yisint1 << 11) + ind0 + (54 & eflag0) - (52 & gflag0);
			ind0 <<= 20;
			ull_x0 = *(unsigned long long*)&u0;
			HI(&ull_x0) += ind0;
			u0 = *(double*)&ull_x0;
			*pz1 = u0 * SCALE_ARR[eflag0 - gflag0];
		}
	}
}

#undef RET_SC
#define RET_SC(I)							\
	py += stridey;							\
	pz += stridez;							\
	if ( --n <= 0 )							\
		break;							\
	goto start##I;

#define PREP_X(I)							\
hy = HI(py);						\
ly = LO(py);						\
sy = hy >> 31;								\
hy &= 0x7fffffff;							\
py##I = py;								\
									\
if ( hy < 0x3bf00000 )		/* |Y| < 2^(-64) */			\
	RETURN (I, DONE)						\
pz##I = pz;								\
if ( hy >= 0x43e00000 )		/* |Y|>2^63,Inf,Nan */			\
{									\
	if ( hy >= 0x7ff00000 )		/* |Y|=Inf,Nan */		\
	{								\
		if ( hy == 0x7ff00000 && ly == 0 )	/* |Y|=Inf */	\
		{							\
			if ( (hx < 0x3ff00000) != sy )			\
				*pz = DZERO;				\
			else						\
			{						\
				HI(pz) = hy;			\
				LO(pz) = ly;			\
			}						\
		}							\
		else							\
			*pz = *px + *py;	/* |Y|=Nan */		\
	}								\
	else	/* |Y|>2^63 */						\
	{								\
		y0 = ( (hx < 0x3ff00000) != sy ) ? _TINY : _HUGE;	\
		*pz = y0 * y0;						\
	}								\
	RET_SC(I)							\
}									\

#define LMMANT	((unsigned long long*)LCONST)[4]	/* 0x000fffffffffffff			*/
#define LMROUND	((unsigned long long*)LCONST)[5]	/* 0x0000080000000000			*/
#define LMHI20	((unsigned long long*)LCONST)[6]	/* 0xfffff00000000000			*/
#define MMANT	((double*)LCONST)[4]			/* 0x000fffffffffffff			*/
#define MROUND	((double*)LCONST)[5]			/* 0x0000080000000000			*/
#define MHI20	((double*)LCONST)[6]			/* 0xfffff00000000000			*/
#define KA5	((double*)LCONST)[8]			/* 5.77078604860893737986e-01*256	*/
#define KA3	((double*)LCONST)[9]			/* 9.61796693925765549423e-01*256	*/
#define KA1_LO	((double*)LCONST)[10]			/* 1.41052154268147309568e-05*256	*/
#define KA1_HI	((double*)LCONST)[11]			/* 2.8853759765625e+00*256		*/
#define KA1	((double*)LCONST)[12]			/* 2.885390081777926774e+00*256		*/


static void
__vpowx( int n, double * restrict px, double * restrict py,
	int stridey, double * restrict pz, int stridez )
{
	double			*py0, *py1, *py2;
	double			*pz0, *pz1, *pz2;
	double			ux0, y0, yd0, u0, s0;
	double			y1, yd1, u1, s1;
	double			y2, yd2, u2, s2;
	double			yr, s_h0, s_l0, m_h0, x0, ax0;
	unsigned long long	ull_y0, ull_x0, ull_x1, ull_x2, ull_ax0;
	int			eflag0, gflag0, ind0, i0, exp0;
	int			eflag1, gflag1, ind1, i1;
	int			eflag2, gflag2, ind2, i2;
	int			i = 0;
	unsigned		hx, hx0, hy, ly, sy;
	double			DONE = ((double*)LCONST)[1];			/*  1.0				*/
	unsigned long long	LDONE = ((unsigned long long*)LCONST)[1];	/*  1.0				*/
	double			DZERO = ((double*)LCONST)[7];			/*  0.0				*/
	double			HTHRESH = ((double*)LCONST)[13];		/*  262144.0			*/
	double			LTHRESH = ((double*)LCONST)[14];		/* -275200.0			*/
	double			KB5 = ((double*)LCONST)[15];			/*  1.21195555854068860923e-15	*/
	double			KB4 = ((double*)LCONST)[16];			/*  2.23939573811855104311e-12	*/
	double			KB3 = ((double*)LCONST)[17];			/*  3.30830268126604677436e-09	*/
	double			KB2 = ((double*)LCONST)[18];			/*  3.66556559691003767877e-06	*/
	double			KB1 = ((double*)LCONST)[19];			/*  2.70760617406228636578e-03	*/

	/* perform s_h + yr = 256*log2(x) */
	ull_y0 = *(unsigned long long*)px;
	hx = HI(px);
	ull_x0 = ull_y0 & LMMANT | LDONE;
	x0 = *(double*)&ull_x0;
	exp0 = (hx >> 20) - 2046;
	ull_ax0 = ull_x0 + LMROUND & LMHI20;
	ax0 = *(double*)&ull_ax0;
	hx0 = HI(&ax0);
	ux0 = x0 + ax0;
	yd0 = DONE / ux0;
	u0 = x0 - ax0;
	s0 = u0 * yd0;
	LO(&ux0) = 0;
	y0 = s0 * s0;
	s_h0 = s0;
	LO(&s_h0) = 0;
	s0 = (KA5 * y0 + KA3) * y0 * s0;
	s_l0 = (x0 - (ux0 - ax0));
	s_l0 = u0 - s_h0 * ux0 - s_h0 * s_l0;
	s_l0 = KA1 * yd0 * s_l0;
	i0 = (hx0 >> 8) & 0xff0;
	exp0 += (hx0 >> 20);
	yd0 = KA1_HI * s_h0;
	y0 = *(double *)((char*)__TBL_log2 + i0);
	y0 += (double)(exp0 << 8);
	m_h0 = y0 + yd0;
	y0 = s0 - ((m_h0 - y0 - yd0) - s_l0);
	y0 += *(double *)((char*)__TBL_log2 + i0 + 8) + KA1_LO * s_h0;
	s_h0 = y0 + m_h0;
	LO(&s_h0) = 0;
	yr = y0 - (s_h0 - m_h0);

	do
	{
	/* perform 2 ** ((s_h0+yr)*yi/256) */
start0:
		PREP_X(0)
		py += stridey;
		pz += stridez;
		i = 1;
		if ( --n <= 0 )
			break;

start1:
		PREP_X(1)
		py += stridey;
		pz += stridez;
		i = 2;
		if ( --n <= 0 )
			break;

start2:
		PREP_X(2)

		s0 = yd0 = *py0;
		s1 = yd1 = *py1;
		s2 = yd2 = *py2;

		LO(&s0) = 0;
		LO(&s1) = 0;
		LO(&s2) = 0;

		yd0 = (yd0 - s0) * s_h0 + yd0 * yr;
		yd1 = (yd1 - s1) * s_h0 + yd1 * yr;
		yd2 = (yd2 - s2) * s_h0 + yd2 * yr;

		s0 = s_h0 * s0;
		s1 = s_h0 * s1;
		s2 = s_h0 * s2;

		if ( s0 > HTHRESH )
		{
			s0 = HTHRESH;
			yd0 = DZERO;
		}
		if ( s1 > HTHRESH )
		{
			s1 = HTHRESH;
			yd1 = DZERO;
		}
		if ( s2 > HTHRESH )
		{
			s2 = HTHRESH;
			yd2 = DZERO;
		}

		if ( s0 < LTHRESH )
		{
			s0 = LTHRESH;
			yd0 = DZERO;
		}
		ind0 = (int) (s0 + yd0);
		if ( s1 < LTHRESH )
		{
			s1 = LTHRESH;
			yd1 = DZERO;
		}
		ind1 = (int) (s1 + yd1);
		if ( s2 < LTHRESH )
		{
			s2 = LTHRESH;
			yd2 = DZERO;
		}
		ind2 = (int) (s2 + yd2);

		i0 = (ind0 & 0xff) << 4;
		u0 = (double) ind0;
		ind0 >>= 8;

		i1 = (ind1 & 0xff) << 4;
		u1 = (double) ind1;
		ind1 >>= 8;

		i2 = (ind2 & 0xff) << 4;
		u2 = (double) ind2;
		ind2 >>= 8;

		y0 = s0 - u0 + yd0;
		y1 = s1 - u1 + yd1;
		y2 = s2 - u2 + yd2;

		u0 = *(double*)((char*)__TBL_exp2 + i0);
		y0 = ((((KB5 * y0 + KB4) * y0 + KB3) * y0 + KB2) * y0 + KB1) * y0;
		u1 = *(double*)((char*)__TBL_exp2 + i1);
		y1 = ((((KB5 * y1 + KB4) * y1 + KB3) * y1 + KB2) * y1 + KB1) * y1;
		u2 = *(double*)((char*)__TBL_exp2 + i2);
		y2 = ((((KB5 * y2 + KB4) * y2 + KB3) * y2 + KB2) * y2 + KB1) * y2;

		eflag0 = (ind0 + 1021) >> 31;
		gflag0 = (1022 - ind0) >> 31;
		eflag1 = (ind1 + 1021) >> 31;
		gflag1 = (1022 - ind1) >> 31;
		eflag2 = (ind2 + 1021) >> 31;
		gflag2 = (1022 - ind2) >> 31;

		u0 = *(double*)((char*)__TBL_exp2 + i0 + 8) + u0 * y0 + u0;
		ind0 = ind0 + (54 & eflag0) - (52 & gflag0);
		ind0 <<= 20;
		ull_x0 = *(unsigned long long*)&u0;
		HI(&ull_x0) += ind0;
		u0 = *(double*)&ull_x0;

		u1 = *(double*)((char*)__TBL_exp2 + i1 + 8) + u1 * y1 + u1;
		ind1 = ind1 + (54 & eflag1) - (52 & gflag1);
		ind1 <<= 20;
		ull_x1 = *(unsigned long long*)&u1;
		HI(&ull_x1) += ind1;
		u1 = *(double*)&ull_x1;

		u2 = *(double*)((char*)__TBL_exp2 + i2 + 8) + u2 * y2 + u2;
		ind2 = ind2 + (54 & eflag2) - (52 & gflag2);
		ind2 <<= 20;
		ull_x2 = *(unsigned long long*)&u2;
		HI(&ull_x2) += ind2;
		u2 = *(double*)&ull_x2;

		*pz0 = u0 * SCALE_ARR[eflag0 - gflag0];
		*pz1 = u1 * SCALE_ARR[eflag1 - gflag1];
		*pz2 = u2 * SCALE_ARR[eflag2 - gflag2];

		py += stridey;
		pz += stridez;
		i = 0;

	} while ( --n > 0 );

	if ( i > 0 )
	{
		/* perform 2 ** ((s_h0+yr)*yi/256) */
		s0 = y0 = *py0;
		LO(&s0) = 0;
		yd0 = (y0 - s0) * s_h0 + y0 * yr;
		s0 = s_h0 * s0;
		if ( s0 > HTHRESH )
		{
			s0 = HTHRESH;
			yd0 = DZERO;
		}
		if ( s0 < LTHRESH )
		{
			s0 = LTHRESH;
			yd0 = DZERO;
		}
		ind0 = (int) (s0 + yd0);
		i0 = (ind0 & 0xff) << 4;
		u0 = (double) ind0;
		ind0 >>= 8;
		y0 = s0 - u0 + yd0;
		u0 = *(double*)((char*)__TBL_exp2 + i0);
		y0 = ((((KB5 * y0 + KB4) * y0 + KB3) * y0 + KB2) * y0 + KB1) * y0;
		eflag0 = (ind0 + 1021) >> 31;
		gflag0 = (1022 - ind0) >> 31;
		u0 = *(double*)((char*)__TBL_exp2 + i0 + 8) + u0 * y0 + u0;
		ind0 = ind0 + (54 & eflag0) - (52 & gflag0);
		ind0 <<= 20;
		ull_x0 = *(unsigned long long*)&u0;
		HI(&ull_x0) += ind0;
		u0 = *(double*)&ull_x0;
		*pz0 = u0 * SCALE_ARR[eflag0 - gflag0];

		if ( i > 1 )
		{
			/* perform 2 ** ((s_h0+yr)*yi/256) */
			s0 = y0 = *py1;
			LO(&s0) = 0;
			yd0 = (y0 - s0) * s_h0 + y0 * yr;
			s0 = s_h0 * s0;
			if ( s0 > HTHRESH )
			{
				s0 = HTHRESH;
				yd0 = DZERO;
			}
			if ( s0 < LTHRESH )
			{
				s0 = LTHRESH;
				yd0 = DZERO;
			}
			ind0 = (int) (s0 + yd0);
			i0 = (ind0 & 0xff) << 4;
			u0 = (double) ind0;
			ind0 >>= 8;
			y0 = s0 - u0 + yd0;
			u0 = *(double*)((char*)__TBL_exp2 + i0);
			y0 = ((((KB5 * y0 + KB4) * y0 + KB3) * y0 + KB2) * y0 + KB1) * y0;
			eflag0 = (ind0 + 1021) >> 31;
			gflag0 = (1022 - ind0) >> 31;
			u0 = *(double*)((char*)__TBL_exp2 + i0 + 8) + u0 * y0 + u0;
			ind0 = ind0 + (54 & eflag0) - (52 & gflag0);
			ind0 <<= 20;
			ull_x0 = *(unsigned long long*)&u0;
			HI(&ull_x0) += ind0;
			u0 = *(double*)&ull_x0;
			*pz1 = u0 * SCALE_ARR[eflag0 - gflag0];
		}
	}
}
