/* MakeTeXTFM.c -- Make a TFM file
   Time-stamp: "97/08/24 12:18:08 mik"

   Copyright (C) 1995, 96, 97
	Christian Schenk  <cschenk@berlin.snafu.de>

   This file is part of MiKTeX.

   MiKTeX is free software; you can redistribute it and/or modify it
   under the terms of the GNU General Public License as published by
   the Free Software Foundation; either version 2, or (at your option)
   any later version.
   
   MiKTeX is distributed in the hope that it will be useful, but
   WITHOUT ANY WARRANTY; without even the implied warranty of
   MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
   General Public License for more details.
   
   You should have received a copy of the GNU General Public License
   along with MiKTeX; if not, write to the Free Software Foundation,
   59 Temple Place - Suite 330, Boston, MA 02111-1307, USA.  */

#include <stdlib.h>
#include <stdio.h>
#include <getopt.h>
#include <io.h>
#include <direct.h>
#include <process.h>

#include "miktex.h"
#include "MakeTeXTFM.rc"

#if defined (_WIN32)
#  define WIN32_LEAN_AND_MEAN
#  include <windows.h>
#endif /* _WIN32 */

#define PATH_SEP_CHAR ';'
#define isslash(c) ((c) == '\\' || (c) == '/')

static int		verbose_p;
static int		debug_p;
static int		quiet_p;
static int		print_only_p;

static const char *	destdir;

static char *		name;

/* _________________________________________________________________________

   USAGE / HELP
   _________________________________________________________________________ */


static void
show_version (void)

{
  printf ("This is MakeTeXTFM version %s (%s %s)\n",
	  VER_FILEVERSION_STR, VER_PRODUCTNAME_STR,
	  VER_PRODUCTVERSION_STR);
}

static void
usage (void)

{
  puts ("Usage: MakeTeXTFM [OPTION]... name");
  puts ("");
  puts ("This program makes a TeX font metric file.");
  puts ("");
  puts ("NAME is the name of the font, such as `cmr10'.");
  puts ("");
  puts ("Options:");
  puts ("--debug, -d                         Print debugging information.");
  puts ("--dest-dir DIRECTORY, -D DIRECTORY  Specify the destination font directory.");
  puts ("--help, -h                          Print this help screen and exit.");
  puts ("--print-only, -n                    Print what commands would be executed.");
  puts ("--verbose, -v                       Print information on what is being done.");
  puts ("--version, -V                       Print the version number and exit.");
  puts ("");
  puts ("Email problem reports to cschenk@berlin.snafu.de");
}

/* _________________________________________________________________________

   GETTING COMMAND LINE OPTIONS.
   _________________________________________________________________________ */


static struct option long_options[] =

{
  "debug",		no_argument,		0,	'd',
  "help",		no_argument,		0,	'h',
  "print-only",		no_argument,		0,	'n',
  "dest-dir",		required_argument,	0,	'D',
  "quiet",		no_argument,		0,	'q',
  "verbose",		no_argument,		0,	'v',
  "version",		no_argument,		0,	'V',
  0,			no_argument,		0,	0,
};

static void
get_options (int	argc,
	     char **	argv)

{
  int option_index;
  int c;

  destdir = 0;
  print_only_p = 0;
  quiet_p = 0;
#if defined (DEBUG)
  verbose_p = 1;
  debug_p = 1;
#else
  verbose_p = 0;
  debug_p = 0;
#endif

  c4pargv[0] = argv[0];
  while ((c=getopt_long (argc, argv, "dD:hnqvV", long_options, &option_index))
	 != EOF)
    {
      switch (c)
	{
	case 'h':
	  usage ();
	  exit (0);
	  break;
	case 'n':
	  print_only_p = 1;
	  break;
	case 'D':
	  destdir = strdup (optarg);
	  break;
	case 'd':
	  debug_p = 1;
	  break;
	case 'v':
	  verbose_p = 1;
	  break;
	case 'q':
	  quiet_p = 1;
	  break;
	case 'V':
	  show_version ();
	  exit (0);
	  break;
	default:
	  usage ();
	  exit (1);
	  break;
	}
    }

  if (argc - optind < 1)
    {
      usage ();
      exit (1);
    }

  name = strdup (argv[optind++]);

  if (optind != argc)
    {
      usage ();
      exit (1);
    }
}

/* _________________________________________________________________________

   CREATING THE DESTINATION DIRECTORY.
   _________________________________________________________________________ */


static char *
create_dest_directory (const char *name)

{
  char dest_directories[_MAX_PATH];
  char name_template[_MAX_FNAME];
  char supplier[_MAX_FNAME];
  char typeface[_MAX_FNAME];
  char *cp_dest_directories;
  const char *cp_name_template;
  char *destdir;
  char *next_destdir;
  char *result;

  if (! get_font_info (name, supplier, typeface))
    {
      strcpy (supplier, "public");
      strcpy (typeface, "xxx");
    }

  get_cfg_value ("TeX", "Font Metric Temp Dir",
		 name_template, sizeof (name_template),
		 "%R\\fonts\\tfm\\%s\\%t");

  cp_name_template = name_template;
  cp_dest_directories = dest_directories;

  while (*cp_name_template)
    {
      if (*cp_name_template == '%')
	{
	  switch (cp_name_template[1])
	    {
	    default:
	      *cp_dest_directories = 0;
	      break;
	    case '%':
	      *cp_dest_directories++ = '%';
	      *cp_dest_directories = 0;
	      break;
	    case 'R':
	      *cp_dest_directories++ = '%';
	      *cp_dest_directories++ = 'R';
	      *cp_dest_directories = 0;
	      break;
	    case 's':
	      strcpy (cp_dest_directories, supplier);
	      break;
	    case 't':
	      strcpy (cp_dest_directories, typeface);
	      break;
	    }
	  cp_dest_directories += strlen (cp_dest_directories);
	  cp_name_template += 2;
	}
      else
	*cp_dest_directories++ = *cp_name_template++;
    }
  *cp_dest_directories = 0;
  
  /* Initialize search. */
  destdir = dest_directories;
  next_destdir = strchr (destdir, PATH_SEP_CHAR);
  if (next_destdir)
    *next_destdir++ = 0; 

  result = 0;
  while (destdir && result == 0)
    {
      if (destdir[0] == '%' && destdir[1] == 'R' && isslash (destdir[2]))
	{
	  char alt_destdir[_MAX_PATH];
	  size_t i;

	  for (i = 0; i < get_number_of_texmf_roots () && result == 0; i++)
	    {
	      xassert (get_root_directory (i) != 0);
	      _makepath (alt_destdir, 0, get_root_directory (i),
			 &destdir[3], 0);
	      if (print_only_p)
		{
		  printf ("mkdir %s\n", alt_destdir);
		  result = strdup (alt_destdir);
		}
	      else if (make_path_ex (alt_destdir, 0))
		result = strdup (alt_destdir);
	    }
	}
      else if (print_only_p)
	{
	  printf ("mkdir %s\n", destdir);
	  result = strdup (destdir);
	}
      else if (make_path_ex (destdir, 0))
	result = strdup (destdir);
      
      if (next_destdir && *next_destdir)
	{
	  destdir = next_destdir;
	  next_destdir = strchr (next_destdir, PATH_SEP_CHAR);
	  if (next_destdir)
	    *next_destdir++ = 0;
	}
      else
	destdir = 0;
    }

  if (result == 0)
    {
      fprintf (stderr, "Don't know where to install the font!\n");
      exit (1);
    }

  return (result);
}

/* _________________________________________________________________________

   MAKING FILE NAMES.
   _________________________________________________________________________ */


static char *
make_tfm_filename (const char *	name)

{
  char result[_MAX_PATH];
  sprintf (result, "%s.tfm", name);
  return (strdup (result));
}

static char *
make_dest_path (const char *	destdir,
		const char *	tfmname)

{
  char result[_MAX_PATH];
  sprintf (result, "%s\\%s", destdir, tfmname);
  return (strdup (result));
}

/* _________________________________________________________________________

   INVOKING METAFONT.
   _________________________________________________________________________ */


static int
invoke_metafont (const char *	name)

{
  char command_line[2 * _MAX_PATH + 100];
  int pargc;
  const char *pargv[20];
  int rc;

  if (! find_executable ("virmf.exe", command_line))
    {
      fprintf (stderr, "virmf.exe not found!\n");
      if (debug_p)
	fprintf (stderr, "Intermediate files were *not* removed!\n");
      else
	leave_temporary_directory ();
      exit (1);
    }

  pargc = 0;
  pargv[pargc++] = command_line;
  pargv[pargc++] = "\\mode:=cx;";
  pargv[pargc++] = "batchmode;";
  pargv[pargc++] = "input";
  pargv[pargc++] = name;
  pargv[pargc++] = 0;
  
  if (! quiet_p || print_only_p)
    printf ("%s \\mode:=cx; batchmode; input %s\n", command_line, name);

  if (print_only_p)
    rc = 0;
  else
    {
      _flushall ();
      rc = _spawnv (_P_WAIT, command_line, pargv);
    }

  if (rc < 0)
    {
      perror (command_line);
      leave_temporary_directory ();
      exit (1);
    }

  if (rc > 0)
    {
      fprintf (stderr, "METAFONT failed on %s! Return code: %d\n", name, rc);
      if (debug_p)
	fprintf (stderr, "Intermediate files were *not* removed!\n");
      else
	leave_temporary_directory ();
      exit (1);
    }

  return (rc);
}

/* _________________________________________________________________________

   INSTALLING TFM FONTS.
   _________________________________________________________________________ */


static void
install_tfm_font (const char *	tfmname,
		  const char *	destdir,
		  const char *	destname)

{
  if (! print_only_p && _access (tfmname, 0) < 0)
    {
      fprintf (stderr, "Cannot access TFM file %s\n", tfmname);
      if (debug_p)
	fprintf (stderr, "Intermediate files were *not* removed!\n");
      else
	leave_temporary_directory ();
      exit (1);
    }
  if (print_only_p || verbose_p)
    printf ("move %s %s\n", tfmname, destname);
  if (! print_only_p)
    {
      if (make_path (destdir))
	{
	  if (rename (tfmname, destname) != 0)
	    {
	      perror (tfmname);
	      leave_temporary_directory ();
	      exit (1);
	    }
	}
    }
}

/* _________________________________________________________________________

   MAIN.
   _________________________________________________________________________ */


int
main (int	argc,
      char **	argv)

{
  char *tfmname;
  char *destname;

  /* Get command line options. */
  c4pargv[0] = argv[0];
  get_options (argc, argv);

  /* Make destination directory name if none was specified. */
  if (destdir == 0)
    destdir = create_dest_directory (name);

  /* Make TFM file name. */
  tfmname = make_tfm_filename (name);

  /* Make full destination path name. */
  destname = make_dest_path (destdir, tfmname);

  /* Quit, if destination font file already exists. */
  if (_access (destname, 0) == 0)
    {
      if (! quiet_p)
	printf ("%s already exists!\n", destname);
      exit (0);
    }

  if (verbose_p && ! print_only_p)
    printf ("Creating %s...\n", tfmname);

  /* Create temporary directory; make it current. */
  enter_temporary_directory ("tfm", print_only_p);

  /* Invoke METAFONT to make a TFM font. */
  invoke_metafont (name);

  /* Install TFM font file in destination directory. */
  install_tfm_font (tfmname, destdir, destname);

  /* Remove temporary directory. */
  leave_temporary_directory ();

  return (0);
}

/* MakeTeXTFM.c ends here */
