import java.io.*;
import java.lang.*;

/**
 * Class to store information for one BibTeX field.  This class
 * stores the information for a field, can read it from a variety
 * of input streams, and output it to a variety of output streams.
 * It also sets a flag if the end of an entry has been encountered.
 * @author John Janmaat
 * @version 0.1
 */
public class BibTeXField
{
    // static members, set globally
    static boolean unicode = false;  // set true if using unicode.

    // define data members
    String field_name;
    String field_text;
    StringBuffer buffer;
    boolean isName;
    boolean isBrace;
    boolean skipNext;
    boolean isText;
    boolean lastField;
    boolean endOfField;
    boolean endOfFile;
    int numBrace;
    private static boolean m_Debug;  // flag controlling debugging output
    private long m_DataStart;  // start of data field
    private long m_DataEnd;  // end of data field
    
    /**
     * Constructor, create a blank field object.
     */
    public BibTeXField() {
        initAddChar();
        m_Debug = false;
        m_DataStart = -1;
        m_DataEnd = -1;
    }

    /**
     * Constructor, create a field object, with passed strings
     * @parm name Name of field.
     * @parm text Text contained by field
     */
    public BibTeXField(String name, String text) {
        initAddChar();
        field_name = name;
        field_text = text;
        m_Debug = false;
        m_DataStart = -1;
        m_DataEnd = -1;
    }

    /**
     * Constructor, copy an existing field object into new object
     * @parm original Field object to be copied.
     */
    public BibTeXField(BibTeXField original) {
        initAddChar();
        field_name = original.getName();
        field_text = original.getText();
        m_Debug = false;
        m_DataStart = -1;
        m_DataEnd = -1;
    }
    
    /**
     * Return field name
     * @return Name of field.
     */
    public String getName() {
        return field_name;}

    /**
     * Return field text
     * @return Text contained in field object
     */
    public String getText() {
        return field_text;}

    /**
     Check to see if this field is equal to another one
     */
    public boolean equals(BibTeXField compare) {
        return field_name.equals(compare.getName())
            && field_text.equals(compare.getText());
    }

    /**
     Get the bare text, after removing quote marks or brace marks if
     they exist at both ends of the text
     */
    public String getBareText() {
        // test for the boundaries on the block of text
        if((field_text.startsWith("{") &&
            field_text.endsWith("}")) ||
           (field_text.startsWith("\"") &&
            field_text.endsWith("\""))) {
            // just carve them off
            return field_text.substring(1, field_text.length() - 1);}
        // return the unaltered text
        return field_text;
    }

    /**
     * Return boolean flag, end of record
     * @return Returns true if this field is the end of the record
     */
    public boolean isEndOfRecord() {
        return lastField;}

    /**
     * Return boolean flag, end of field.  Used internally to track progress,
     * @return Returns true if the end of the field has been reached.
     */
    private boolean isEndOfField() {
        return endOfField;}

    /**
     * Return boolean flag, end of file.  Set when an end of file is encountered
     * @return Returns true if the end of the file has been reached.
     */
    public boolean isEndOfFile() {
        return endOfFile;}

    /**
     * Set field name.
     * @param name Name of field.
     */
    public void setName(String name) {
        field_name = name;}

    /**
     * Set field text.
     * @param text Text value in field.
     */
    public void setText(String text) {
        field_text = text;}

    /**
     Set the text field from passed bare text
     @param text value to set in the text field
     */
    public void setBareText(String text) {
        // Just check to see
        // if we have delimiters already.
        if((text.startsWith("{") &&
            text.endsWith("}")) ||
           (text.startsWith("\"") &&
            text.endsWith("\""))) {
            // just set it
            field_text = text;}
        else {
            // add braces
            field_text = "{" + text + "}";}
    }

    /**
     * Set true if using Unicode characters.
     * @param flag True if unicode characters being used.
     */
    public static void setUnicode(boolean flag) {
        unicode = flag;}

    /**
     * Get the value of the unicode flag
     * @return True if unicode characters are being used.
     */
    public static boolean getUnicode() {
        return unicode;}
    
    /**
     * Initialize character addition.  Clears the input buffer, storage
     * space, and flags used to manage input
     */
    public void initAddChar()
    {
        field_name = new String();
        field_text = new String();
        buffer = new StringBuffer();
        isName = true;
        isText = false;
        isBrace = true;
        skipNext = false;
        lastField = false;
        endOfField = false;
        endOfFile = false;
        numBrace = 0;
    }

    /**
     * Adds one character to either the field name or field
     * text for a BibTeX file.
     * @param input Character to add to either the name or text field.
     */
    private void addChar(char input) {
        // which part of the data are we looking at
        if(isName) {
            // looking at first part, so after a field name
            if(Character.isWhitespace(input)) {
                // is a space, have we already got field name?
                if(buffer.length() > 0) {
                    // yes, so flag that we are looking for data now
                    isName = false;
                    field_name = buffer.toString();
                    buffer = new StringBuffer();}}
            else {if(input == '=') {
                    // have found an equals, check if have name
                    if(buffer.length() > 0) {
                        // yes, have name, so flag getting data
                        isName = false;
                        isText = true;
                        field_name = buffer.toString();
                        buffer = new StringBuffer();}
                    else {
                        // problem, no data yet, print error and exit
                        System.err.println("Field name missing!");
                        System.exit(1);}}
                else {
                    // have neither a space nor an equals, must keep
                    buffer.append(input);}}}
        else {if(!isText) {
                // not collecting data yet, look for '='
                if(input == '=') {
                    isText = true;}}
            else {
                // must be looking for data
                if(Character.isWhitespace(input)) {
                    // it is a space
                    if(buffer.length() > 0) {
                        // have been adding characters, so add this
                        buffer.append(input);}}
                else {if(numBrace == 0) {
                        // looking for a starting brace. or a comma or
                        // end of line
                        if((input == '}') || (input == ')')) {
                            // end of entry
                            lastField = true;
                            endOfField = true;}
                        else {if(input == ',') {
                                // end of this field
                                endOfField = true;}
                            else {if(input == '\"') {
                                    // found a quote, so set brace and level
                                    numBrace = 1;
                                    isBrace = false;}
                                else {if(input == '{') {
                                        // found opening brace, set
                                        numBrace = 1;
                                        isBrace = true;}}
                                // not space, end of record or end of field,
                                // so add the character
                                buffer.append(input);}}}
                    else {
                        // we are between braces, so add everything
                        buffer.append(input);
                        if((input == '\\') && (!skipNext)) {
                            // found a backslash, so skip next character
                            skipNext = true;}
                        else {
                            if(!skipNext) {
                                // not a backslash, and not to be skipped
                                if((input == '{') && isBrace) {
                                    // increase brace level
                                    numBrace++;}
                                if((((input == '}') && isBrace)
                                   || ((input == '\"') && !isBrace))
                                    && (numBrace > 0)) {
                                    // reduce brace level
                                    numBrace--;}}
                            else{
                                // turn off skip next
                                skipNext = false;}
                        }}}
                if(lastField || endOfField) {
                    // end of field or line, set string
                    field_text = buffer.toString();}}}
    }

    /**
     * Read one field from a random access file.  Based on the value
     * of unicode, it reads a byte which is then cast to a character,
     * or directly reads a unicode character.
     * @param reader RandomAccessFile for reading from
     * @exception IOException passed along for capture later.
     */
    public void read(RandomAccessFile reader) throws IOException
    {
        initAddChar();
        int input = 0;
        while((!endOfField) && (input != -1)) {
            // catch EOFexception
            if(unicode) {
                // directly read a character
                addChar(reader.readChar());}
            else {
                // read byte and convert to character
                input = reader.read();
                if(input != -1) {
                    // add character if not end of file
                    addChar((char)input);}}
            // check to see if we need to take the start position
            if((numBrace > 0) && (m_DataStart == -1)) {
                m_DataStart = reader.getFilePointer();}}
        if(input == -1) {
            // have an end of file, throw EOFException and set flag
            endOfFile = true;
            endOfField = true;
            lastField = true;
            EOFException e = new EOFException("End of file encountered in byte read.");
            throw e;}
//        debug(field_text);
        field_text = chomp(field_text);
    }

    /**
     * Read just the last part of the field.  This is here
     * principally to deal with the abbreviation, where the first
     * part of the record is read where the key is supposed to
     * be read.
     * @param reader StringReader from which to read data
     * @param name String with name of field
     */
    public void readText(RandomAccessFile reader, String name) throws IOException {
        initAddChar();
        setName(name);
        isName = false;
        isText = true;
        int input = 0;
        while((!endOfField) && (input > -1)) {
            // catch EOFexception
            if(unicode) {
                // directly read a character
                addChar(reader.readChar());}
            else {
                // read byte and convert to character
                input = reader.read();
                if(input != -1) {
                    // add character if not end of file
                    addChar((char)input);}}}
//            input = reader.read();
//            if(input > -1) {
                // add character if not end of file
//                addChar((char)input);}}
        if(input == -1) {
            // have an end of file, throw EOFException and set flag
            endOfFile = true;
            endOfField = true;
            lastField = true;
            EOFException e = new EOFException("End of file encountered.");
            throw e;}
//        debug(field_text);
        field_text = chomp(field_text);
    }
    
    /**
     * Read just the last part of the field.  This is here
     * principally to deal with the abbreviation, where the first
     * part of the record is read where the key is supposed to
     * be read.
     * @param reader StringReader from which to read data
     * @param name String with name of field
     */
    public void readText(Reader reader, String name) throws IOException {
        initAddChar();
        setName(name);
        isName = false;
        isText = true;
        int input = 0;
        while((!endOfField) && (input > -1)) {
            input = reader.read();
            if(input > -1) {
                // add character if not end of file
                addChar((char)input);}}
        if(input == -1) {
            // have an end of file, throw EOFException and set flag
            endOfFile = true;
            endOfField = true;
            lastField = true;
            EOFException e = new EOFException("End of file encountered.");
            throw e;}
//        debug(field_text);
        field_text = chomp(field_text);
    }

    /**
     * Read one field from a generic Reader.  Using a reader ensures
     * that character encoding is done properly, but does not permit
     * random access.
     * @param reader StringReader from which data is read
     * @exception IOException passed along for processing later.
     */
    public void read(Reader reader) throws IOException
    {
        initAddChar();
        int input = 0;
        while((!endOfField) && (input > -1)) {
            input = reader.read();
            if(input > -1) {
                // add character if not end of file
                addChar((char)input);}}
        if(input == -1) {
            // have an end of file, throw EOFException and set flag
            endOfFile = true;
            endOfField = true;
            lastField = true;
            EOFException e = new EOFException("End of file encountered in byte read.");
            throw e;}
//        debug(field_text);
    }

    /**
     * Write output to a PrintStream.  Note that this method does not
     * print an end of line character or a comma.
     * @param output Printstream to send data to.
     */
    public void write(PrintStream output)
    {
        output.print(toString());
    }

    /**
     * Write output to a RandomAccessFile.  Note that this method does not
     * print and end of line character or a comma.
     * @param output RandomAccessFile to send data to.
     */
    public void write(RandomAccessFile output) throws IOException
    {
        if(unicode) {
            output.writeChars("   ");
            output.writeChars(field_name);
            output.writeChars(" = ");
            output.writeChars(field_text);}
        else {
            StringReader r = new StringReader(toString());
            try {
                int letter = r.read();
                while(letter != -1) {
                    output.write((byte)letter);
                    letter = r.read();}
                r.close();}
            catch(IOException e) {
                System.err.println("StringReader error writing random access file\n" + e.toString());
                System.exit(1);}}
    }

    /**
     * Write output to a generic Writer.  This method writes to a sequential
     * stream, appropriately dealing with the character coding..
     * @param output RandomAccessFile to send data to.
     */
    public void write(Writer output) throws IOException
    {
        output.write(toString());
    }
            
    /**
     * Remove the whitespaces from each side of a string.
     * @param string String that is to be chomped.
     * @return String with whitespace removed from both ends.
     */
    public String chomp(String string) {
        // variables to point at parts to keep
        int begin, end;
        begin = 0;
        end = string.length();
        // remove beginning whitespaces
        while((Character.isWhitespace(string.charAt(begin)))
              && (begin < end)) {
            begin++;}
        // remove ending whitespaces
        while((Character.isWhitespace(string.charAt(end - 1)))
              && (end > begin)) {
            end--;}
        // ditch the stuff that is not wanted
        return string.toString().substring(begin, end);
    }

    /**
     Return true if this field contains the passed text.
     */
    public boolean containsText(String test) {
        return (field_text.indexOf(test) != -1);}

    /**
     Return true if this field contains the passed text.  For this method,
     ignore the case of the arguement and the data.
     */
    public boolean containsTextIgnoreCase(String test) {
        return (field_text.toUpperCase().indexOf(test.toUpperCase()) != -1);}

    /**
     * Generate a string representation of the Field object
     * @return String representation of object
     */
    public String toString()
    {
        return ("   " + field_name + " = " + field_text);
    }
    
    /**
     Method to output a string when a debug flag is set
     */
    public static void debug(String message)
    {
        if(m_Debug) System.out.println("BibTeXField:"+message);}

    /**
     Method to set the debug flag
     */
    public static void setDebug(boolean flag) {m_Debug = flag;}
}