﻿/**
 * Copyright Amazon.com, Inc. or its affiliates. All Rights Reserved.
 * SPDX-License-Identifier: Apache-2.0.
 */

#pragma once
#include <aws/core/utils/memory/stl/AWSString.h>
#include <aws/core/utils/memory/stl/AWSVector.h>
#include <aws/ecs/ECSRequest.h>
#include <aws/ecs/ECS_EXPORTS.h>
#include <aws/ecs/model/ServiceField.h>

#include <utility>

namespace Aws {
namespace ECS {
namespace Model {

/**
 */
class DescribeServicesRequest : public ECSRequest {
 public:
  AWS_ECS_API DescribeServicesRequest() = default;

  // Service request name is the Operation name which will send this request out,
  // each operation should has unique request name, so that we can get operation's name from this request.
  // Note: this is not true for response, multiple operations may have the same response name,
  // so we can not get operation's name from response.
  inline virtual const char* GetServiceRequestName() const override { return "DescribeServices"; }

  AWS_ECS_API Aws::String SerializePayload() const override;

  AWS_ECS_API Aws::Http::HeaderValueCollection GetRequestSpecificHeaders() const override;

  ///@{
  /**
   * <p>The short name or full Amazon Resource Name (ARN)the cluster that hosts the
   * service to describe. If you do not specify a cluster, the default cluster is
   * assumed. This parameter is required if the service or services you are
   * describing were launched in any cluster other than the default cluster.</p>
   */
  inline const Aws::String& GetCluster() const { return m_cluster; }
  inline bool ClusterHasBeenSet() const { return m_clusterHasBeenSet; }
  template <typename ClusterT = Aws::String>
  void SetCluster(ClusterT&& value) {
    m_clusterHasBeenSet = true;
    m_cluster = std::forward<ClusterT>(value);
  }
  template <typename ClusterT = Aws::String>
  DescribeServicesRequest& WithCluster(ClusterT&& value) {
    SetCluster(std::forward<ClusterT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>A list of services to describe. You may specify up to 10 services to describe
   * in a single operation.</p>
   */
  inline const Aws::Vector<Aws::String>& GetServices() const { return m_services; }
  inline bool ServicesHasBeenSet() const { return m_servicesHasBeenSet; }
  template <typename ServicesT = Aws::Vector<Aws::String>>
  void SetServices(ServicesT&& value) {
    m_servicesHasBeenSet = true;
    m_services = std::forward<ServicesT>(value);
  }
  template <typename ServicesT = Aws::Vector<Aws::String>>
  DescribeServicesRequest& WithServices(ServicesT&& value) {
    SetServices(std::forward<ServicesT>(value));
    return *this;
  }
  template <typename ServicesT = Aws::String>
  DescribeServicesRequest& AddServices(ServicesT&& value) {
    m_servicesHasBeenSet = true;
    m_services.emplace_back(std::forward<ServicesT>(value));
    return *this;
  }
  ///@}

  ///@{
  /**
   * <p>Determines whether you want to see the resource tags for the service. If
   * <code>TAGS</code> is specified, the tags are included in the response. If this
   * field is omitted, tags aren't included in the response.</p>
   */
  inline const Aws::Vector<ServiceField>& GetInclude() const { return m_include; }
  inline bool IncludeHasBeenSet() const { return m_includeHasBeenSet; }
  template <typename IncludeT = Aws::Vector<ServiceField>>
  void SetInclude(IncludeT&& value) {
    m_includeHasBeenSet = true;
    m_include = std::forward<IncludeT>(value);
  }
  template <typename IncludeT = Aws::Vector<ServiceField>>
  DescribeServicesRequest& WithInclude(IncludeT&& value) {
    SetInclude(std::forward<IncludeT>(value));
    return *this;
  }
  inline DescribeServicesRequest& AddInclude(ServiceField value) {
    m_includeHasBeenSet = true;
    m_include.push_back(value);
    return *this;
  }
  ///@}
 private:
  Aws::String m_cluster;

  Aws::Vector<Aws::String> m_services;

  Aws::Vector<ServiceField> m_include;
  bool m_clusterHasBeenSet = false;
  bool m_servicesHasBeenSet = false;
  bool m_includeHasBeenSet = false;
};

}  // namespace Model
}  // namespace ECS
}  // namespace Aws
