"""Font metrics for the Adobe core 14 fonts.

Font metrics are used to compute the boundary of each character
written with a proportional font.

The following data were extracted from the AFM files:

  http://www.ctan.org/tex-archive/fonts/adobe/afm/

"""

###  BEGIN Verbatim copy of the license part

#
# Adobe Core 35 AFM Files with 314 Glyph Entries - ReadMe
#
# This file and the 35 PostScript(R) AFM files it accompanies may be
# used, copied, and distributed for any purpose and without charge,
# with or without modification, provided that all copyright notices
# are retained; that the AFM files are not distributed without this
# file; that all modifications to this file or any of the AFM files
# are prominently noted in the modified file(s); and that this
# paragraph is not modified. Adobe Systems has no responsibility or
# obligation to support the use of the AFM files.
#

###  END Verbatim copy of the license part

# flake8: noqa
from typing import Dict


def convert_font_metrics(path: str) -> None:
    """Convert an AFM file to a mapping of font metrics.

    See below for the output.
    """
    fonts = {}
    with open(path) as fileinput:
        for line in fileinput.readlines():
            f = line.strip().split(" ")
            if not f:
                continue
            k = f[0]
            if k == "FontName":
                fontname = f[1]
                props = {"FontName": fontname, "Flags": 0}
                chars: Dict[int, int] = {}
                fonts[fontname] = (props, chars)
            elif k == "C":
                cid = int(f[1])
                if 0 <= cid and cid <= 255:
                    width = int(f[4])
                    chars[cid] = width
            elif k in ("CapHeight", "XHeight", "ItalicAngle", "Ascender", "Descender"):
                k = {"Ascender": "Ascent", "Descender": "Descent"}.get(k, k)
                props[k] = float(f[1])
            elif k in ("FontName", "FamilyName", "Weight"):
                k = {"FamilyName": "FontFamily", "Weight": "FontWeight"}.get(k, k)
                props[k] = f[1]
            elif k == "IsFixedPitch":
                if f[1].lower() == "true":
                    props["Flags"] = 64
            elif k == "FontBBox":
                props[k] = tuple(map(float, f[1:5]))
        print("# -*- python -*-")
        print("FONT_METRICS = {")
        for fontname, (props, chars) in fonts.items():
            print(f" {fontname!r}: {(props, chars)!r},")
        print("}")


FONT_METRICS = {
    "Courier": (
        {
            "FontName": "Courier",
            "Descent": -194.0,
            "FontBBox": (-6.0, -249.0, 639.0, 803.0),
            "FontWeight": "Medium",
            "CapHeight": 572.0,
            "FontFamily": "Courier",
            "Flags": 64,
            "XHeight": 434.0,
            "ItalicAngle": 0.0,
            "Ascent": 627.0,
        },
        {
            " ": 600,
            "!": 600,
            '"': 600,
            "#": 600,
            "$": 600,
            "%": 600,
            "&": 600,
            "'": 600,
            "(": 600,
            ")": 600,
            "*": 600,
            "+": 600,
            ",": 600,
            "-": 600,
            ".": 600,
            "/": 600,
            "0": 600,
            "1": 600,
            "2": 600,
            "3": 600,
            "4": 600,
            "5": 600,
            "6": 600,
            "7": 600,
            "8": 600,
            "9": 600,
            ":": 600,
            ";": 600,
            "<": 600,
            "=": 600,
            ">": 600,
            "?": 600,
            "@": 600,
            "A": 600,
            "B": 600,
            "C": 600,
            "D": 600,
            "E": 600,
            "F": 600,
            "G": 600,
            "H": 600,
            "I": 600,
            "J": 600,
            "K": 600,
            "L": 600,
            "M": 600,
            "N": 600,
            "O": 600,
            "P": 600,
            "Q": 600,
            "R": 600,
            "S": 600,
            "T": 600,
            "U": 600,
            "V": 600,
            "W": 600,
            "X": 600,
            "Y": 600,
            "Z": 600,
            "[": 600,
            "\\": 600,
            "]": 600,
            "^": 600,
            "_": 600,
            "`": 600,
            "a": 600,
            "b": 600,
            "c": 600,
            "d": 600,
            "e": 600,
            "f": 600,
            "g": 600,
            "h": 600,
            "i": 600,
            "j": 600,
            "k": 600,
            "l": 600,
            "m": 600,
            "n": 600,
            "o": 600,
            "p": 600,
            "q": 600,
            "r": 600,
            "s": 600,
            "t": 600,
            "u": 600,
            "v": 600,
            "w": 600,
            "x": 600,
            "y": 600,
            "z": 600,
            "{": 600,
            "|": 600,
            "}": 600,
            "~": 600,
            "\xa1": 600,
            "\xa2": 600,
            "\xa3": 600,
            "\xa4": 600,
            "\xa5": 600,
            "\xa6": 600,
            "\xa7": 600,
            "\xa8": 600,
            "\xa9": 600,
            "\xaa": 600,
            "\xab": 600,
            "\xac": 600,
            "\xae": 600,
            "\xaf": 600,
            "\xb0": 600,
            "\xb1": 600,
            "\xb2": 600,
            "\xb3": 600,
            "\xb4": 600,
            "\xb5": 600,
            "\xb6": 600,
            "\xb7": 600,
            "\xb8": 600,
            "\xb9": 600,
            "\xba": 600,
            "\xbb": 600,
            "\xbc": 600,
            "\xbd": 600,
            "\xbe": 600,
            "\xbf": 600,
            "\xc0": 600,
            "\xc1": 600,
            "\xc2": 600,
            "\xc3": 600,
            "\xc4": 600,
            "\xc5": 600,
            "\xc6": 600,
            "\xc7": 600,
            "\xc8": 600,
            "\xc9": 600,
            "\xca": 600,
            "\xcb": 600,
            "\xcc": 600,
            "\xcd": 600,
            "\xce": 600,
            "\xcf": 600,
            "\xd0": 600,
            "\xd1": 600,
            "\xd2": 600,
            "\xd3": 600,
            "\xd4": 600,
            "\xd5": 600,
            "\xd6": 600,
            "\xd7": 600,
            "\xd8": 600,
            "\xd9": 600,
            "\xda": 600,
            "\xdb": 600,
            "\xdc": 600,
            "\xdd": 600,
            "\xde": 600,
            "\xdf": 600,
            "\xe0": 600,
            "\xe1": 600,
            "\xe2": 600,
            "\xe3": 600,
            "\xe4": 600,
            "\xe5": 600,
            "\xe6": 600,
            "\xe7": 600,
            "\xe8": 600,
            "\xe9": 600,
            "\xea": 600,
            "\xeb": 600,
            "\xec": 600,
            "\xed": 600,
            "\xee": 600,
            "\xef": 600,
            "\xf0": 600,
            "\xf1": 600,
            "\xf2": 600,
            "\xf3": 600,
            "\xf4": 600,
            "\xf5": 600,
            "\xf6": 600,
            "\xf7": 600,
            "\xf8": 600,
            "\xf9": 600,
            "\xfa": 600,
            "\xfb": 600,
            "\xfc": 600,
            "\xfd": 600,
            "\xfe": 600,
            "\xff": 600,
            "\u0100": 600,
            "\u0101": 600,
            "\u0102": 600,
            "\u0103": 600,
            "\u0104": 600,
            "\u0105": 600,
            "\u0106": 600,
            "\u0107": 600,
            "\u010c": 600,
            "\u010d": 600,
            "\u010e": 600,
            "\u010f": 600,
            "\u0110": 600,
            "\u0111": 600,
            "\u0112": 600,
            "\u0113": 600,
            "\u0116": 600,
            "\u0117": 600,
            "\u0118": 600,
            "\u0119": 600,
            "\u011a": 600,
            "\u011b": 600,
            "\u011e": 600,
            "\u011f": 600,
            "\u0122": 600,
            "\u0123": 600,
            "\u012a": 600,
            "\u012b": 600,
            "\u012e": 600,
            "\u012f": 600,
            "\u0130": 600,
            "\u0131": 600,
            "\u0136": 600,
            "\u0137": 600,
            "\u0139": 600,
            "\u013a": 600,
            "\u013b": 600,
            "\u013c": 600,
            "\u013d": 600,
            "\u013e": 600,
            "\u0141": 600,
            "\u0142": 600,
            "\u0143": 600,
            "\u0144": 600,
            "\u0145": 600,
            "\u0146": 600,
            "\u0147": 600,
            "\u0148": 600,
            "\u014c": 600,
            "\u014d": 600,
            "\u0150": 600,
            "\u0151": 600,
            "\u0152": 600,
            "\u0153": 600,
            "\u0154": 600,
            "\u0155": 600,
            "\u0156": 600,
            "\u0157": 600,
            "\u0158": 600,
            "\u0159": 600,
            "\u015a": 600,
            "\u015b": 600,
            "\u015e": 600,
            "\u015f": 600,
            "\u0160": 600,
            "\u0161": 600,
            "\u0162": 600,
            "\u0163": 600,
            "\u0164": 600,
            "\u0165": 600,
            "\u016a": 600,
            "\u016b": 600,
            "\u016e": 600,
            "\u016f": 600,
            "\u0170": 600,
            "\u0171": 600,
            "\u0172": 600,
            "\u0173": 600,
            "\u0178": 600,
            "\u0179": 600,
            "\u017a": 600,
            "\u017b": 600,
            "\u017c": 600,
            "\u017d": 600,
            "\u017e": 600,
            "\u0192": 600,
            "\u0218": 600,
            "\u0219": 600,
            "\u02c6": 600,
            "\u02c7": 600,
            "\u02d8": 600,
            "\u02d9": 600,
            "\u02da": 600,
            "\u02db": 600,
            "\u02dc": 600,
            "\u02dd": 600,
            "\u2013": 600,
            "\u2014": 600,
            "\u2018": 600,
            "\u2019": 600,
            "\u201a": 600,
            "\u201c": 600,
            "\u201d": 600,
            "\u201e": 600,
            "\u2020": 600,
            "\u2021": 600,
            "\u2022": 600,
            "\u2026": 600,
            "\u2030": 600,
            "\u2039": 600,
            "\u203a": 600,
            "\u2044": 600,
            "\u2122": 600,
            "\u2202": 600,
            "\u2206": 600,
            "\u2211": 600,
            "\u2212": 600,
            "\u221a": 600,
            "\u2260": 600,
            "\u2264": 600,
            "\u2265": 600,
            "\u25ca": 600,
            "\uf6c3": 600,
            "\ufb01": 600,
            "\ufb02": 600,
        },
    ),
    "Courier-Bold": (
        {
            "FontName": "Courier-Bold",
            "Descent": -194.0,
            "FontBBox": (-88.0, -249.0, 697.0, 811.0),
            "FontWeight": "Bold",
            "CapHeight": 572.0,
            "FontFamily": "Courier",
            "Flags": 64,
            "XHeight": 434.0,
            "ItalicAngle": 0.0,
            "Ascent": 627.0,
        },
        {
            " ": 600,
            "!": 600,
            '"': 600,
            "#": 600,
            "$": 600,
            "%": 600,
            "&": 600,
            "'": 600,
            "(": 600,
            ")": 600,
            "*": 600,
            "+": 600,
            ",": 600,
            "-": 600,
            ".": 600,
            "/": 600,
            "0": 600,
            "1": 600,
            "2": 600,
            "3": 600,
            "4": 600,
            "5": 600,
            "6": 600,
            "7": 600,
            "8": 600,
            "9": 600,
            ":": 600,
            ";": 600,
            "<": 600,
            "=": 600,
            ">": 600,
            "?": 600,
            "@": 600,
            "A": 600,
            "B": 600,
            "C": 600,
            "D": 600,
            "E": 600,
            "F": 600,
            "G": 600,
            "H": 600,
            "I": 600,
            "J": 600,
            "K": 600,
            "L": 600,
            "M": 600,
            "N": 600,
            "O": 600,
            "P": 600,
            "Q": 600,
            "R": 600,
            "S": 600,
            "T": 600,
            "U": 600,
            "V": 600,
            "W": 600,
            "X": 600,
            "Y": 600,
            "Z": 600,
            "[": 600,
            "\\": 600,
            "]": 600,
            "^": 600,
            "_": 600,
            "`": 600,
            "a": 600,
            "b": 600,
            "c": 600,
            "d": 600,
            "e": 600,
            "f": 600,
            "g": 600,
            "h": 600,
            "i": 600,
            "j": 600,
            "k": 600,
            "l": 600,
            "m": 600,
            "n": 600,
            "o": 600,
            "p": 600,
            "q": 600,
            "r": 600,
            "s": 600,
            "t": 600,
            "u": 600,
            "v": 600,
            "w": 600,
            "x": 600,
            "y": 600,
            "z": 600,
            "{": 600,
            "|": 600,
            "}": 600,
            "~": 600,
            "\xa1": 600,
            "\xa2": 600,
            "\xa3": 600,
            "\xa4": 600,
            "\xa5": 600,
            "\xa6": 600,
            "\xa7": 600,
            "\xa8": 600,
            "\xa9": 600,
            "\xaa": 600,
            "\xab": 600,
            "\xac": 600,
            "\xae": 600,
            "\xaf": 600,
            "\xb0": 600,
            "\xb1": 600,
            "\xb2": 600,
            "\xb3": 600,
            "\xb4": 600,
            "\xb5": 600,
            "\xb6": 600,
            "\xb7": 600,
            "\xb8": 600,
            "\xb9": 600,
            "\xba": 600,
            "\xbb": 600,
            "\xbc": 600,
            "\xbd": 600,
            "\xbe": 600,
            "\xbf": 600,
            "\xc0": 600,
            "\xc1": 600,
            "\xc2": 600,
            "\xc3": 600,
            "\xc4": 600,
            "\xc5": 600,
            "\xc6": 600,
            "\xc7": 600,
            "\xc8": 600,
            "\xc9": 600,
            "\xca": 600,
            "\xcb": 600,
            "\xcc": 600,
            "\xcd": 600,
            "\xce": 600,
            "\xcf": 600,
            "\xd0": 600,
            "\xd1": 600,
            "\xd2": 600,
            "\xd3": 600,
            "\xd4": 600,
            "\xd5": 600,
            "\xd6": 600,
            "\xd7": 600,
            "\xd8": 600,
            "\xd9": 600,
            "\xda": 600,
            "\xdb": 600,
            "\xdc": 600,
            "\xdd": 600,
            "\xde": 600,
            "\xdf": 600,
            "\xe0": 600,
            "\xe1": 600,
            "\xe2": 600,
            "\xe3": 600,
            "\xe4": 600,
            "\xe5": 600,
            "\xe6": 600,
            "\xe7": 600,
            "\xe8": 600,
            "\xe9": 600,
            "\xea": 600,
            "\xeb": 600,
            "\xec": 600,
            "\xed": 600,
            "\xee": 600,
            "\xef": 600,
            "\xf0": 600,
            "\xf1": 600,
            "\xf2": 600,
            "\xf3": 600,
            "\xf4": 600,
            "\xf5": 600,
            "\xf6": 600,
            "\xf7": 600,
            "\xf8": 600,
            "\xf9": 600,
            "\xfa": 600,
            "\xfb": 600,
            "\xfc": 600,
            "\xfd": 600,
            "\xfe": 600,
            "\xff": 600,
            "\u0100": 600,
            "\u0101": 600,
            "\u0102": 600,
            "\u0103": 600,
            "\u0104": 600,
            "\u0105": 600,
            "\u0106": 600,
            "\u0107": 600,
            "\u010c": 600,
            "\u010d": 600,
            "\u010e": 600,
            "\u010f": 600,
            "\u0110": 600,
            "\u0111": 600,
            "\u0112": 600,
            "\u0113": 600,
            "\u0116": 600,
            "\u0117": 600,
            "\u0118": 600,
            "\u0119": 600,
            "\u011a": 600,
            "\u011b": 600,
            "\u011e": 600,
            "\u011f": 600,
            "\u0122": 600,
            "\u0123": 600,
            "\u012a": 600,
            "\u012b": 600,
            "\u012e": 600,
            "\u012f": 600,
            "\u0130": 600,
            "\u0131": 600,
            "\u0136": 600,
            "\u0137": 600,
            "\u0139": 600,
            "\u013a": 600,
            "\u013b": 600,
            "\u013c": 600,
            "\u013d": 600,
            "\u013e": 600,
            "\u0141": 600,
            "\u0142": 600,
            "\u0143": 600,
            "\u0144": 600,
            "\u0145": 600,
            "\u0146": 600,
            "\u0147": 600,
            "\u0148": 600,
            "\u014c": 600,
            "\u014d": 600,
            "\u0150": 600,
            "\u0151": 600,
            "\u0152": 600,
            "\u0153": 600,
            "\u0154": 600,
            "\u0155": 600,
            "\u0156": 600,
            "\u0157": 600,
            "\u0158": 600,
            "\u0159": 600,
            "\u015a": 600,
            "\u015b": 600,
            "\u015e": 600,
            "\u015f": 600,
            "\u0160": 600,
            "\u0161": 600,
            "\u0162": 600,
            "\u0163": 600,
            "\u0164": 600,
            "\u0165": 600,
            "\u016a": 600,
            "\u016b": 600,
            "\u016e": 600,
            "\u016f": 600,
            "\u0170": 600,
            "\u0171": 600,
            "\u0172": 600,
            "\u0173": 600,
            "\u0178": 600,
            "\u0179": 600,
            "\u017a": 600,
            "\u017b": 600,
            "\u017c": 600,
            "\u017d": 600,
            "\u017e": 600,
            "\u0192": 600,
            "\u0218": 600,
            "\u0219": 600,
            "\u02c6": 600,
            "\u02c7": 600,
            "\u02d8": 600,
            "\u02d9": 600,
            "\u02da": 600,
            "\u02db": 600,
            "\u02dc": 600,
            "\u02dd": 600,
            "\u2013": 600,
            "\u2014": 600,
            "\u2018": 600,
            "\u2019": 600,
            "\u201a": 600,
            "\u201c": 600,
            "\u201d": 600,
            "\u201e": 600,
            "\u2020": 600,
            "\u2021": 600,
            "\u2022": 600,
            "\u2026": 600,
            "\u2030": 600,
            "\u2039": 600,
            "\u203a": 600,
            "\u2044": 600,
            "\u2122": 600,
            "\u2202": 600,
            "\u2206": 600,
            "\u2211": 600,
            "\u2212": 600,
            "\u221a": 600,
            "\u2260": 600,
            "\u2264": 600,
            "\u2265": 600,
            "\u25ca": 600,
            "\uf6c3": 600,
            "\ufb01": 600,
            "\ufb02": 600,
        },
    ),
    "Courier-BoldOblique": (
        {
            "FontName": "Courier-BoldOblique",
            "Descent": -194.0,
            "FontBBox": (-49.0, -249.0, 758.0, 811.0),
            "FontWeight": "Bold",
            "CapHeight": 572.0,
            "FontFamily": "Courier",
            "Flags": 64,
            "XHeight": 434.0,
            "ItalicAngle": -11.0,
            "Ascent": 627.0,
        },
        {
            " ": 600,
            "!": 600,
            '"': 600,
            "#": 600,
            "$": 600,
            "%": 600,
            "&": 600,
            "'": 600,
            "(": 600,
            ")": 600,
            "*": 600,
            "+": 600,
            ",": 600,
            "-": 600,
            ".": 600,
            "/": 600,
            "0": 600,
            "1": 600,
            "2": 600,
            "3": 600,
            "4": 600,
            "5": 600,
            "6": 600,
            "7": 600,
            "8": 600,
            "9": 600,
            ":": 600,
            ";": 600,
            "<": 600,
            "=": 600,
            ">": 600,
            "?": 600,
            "@": 600,
            "A": 600,
            "B": 600,
            "C": 600,
            "D": 600,
            "E": 600,
            "F": 600,
            "G": 600,
            "H": 600,
            "I": 600,
            "J": 600,
            "K": 600,
            "L": 600,
            "M": 600,
            "N": 600,
            "O": 600,
            "P": 600,
            "Q": 600,
            "R": 600,
            "S": 600,
            "T": 600,
            "U": 600,
            "V": 600,
            "W": 600,
            "X": 600,
            "Y": 600,
            "Z": 600,
            "[": 600,
            "\\": 600,
            "]": 600,
            "^": 600,
            "_": 600,
            "`": 600,
            "a": 600,
            "b": 600,
            "c": 600,
            "d": 600,
            "e": 600,
            "f": 600,
            "g": 600,
            "h": 600,
            "i": 600,
            "j": 600,
            "k": 600,
            "l": 600,
            "m": 600,
            "n": 600,
            "o": 600,
            "p": 600,
            "q": 600,
            "r": 600,
            "s": 600,
            "t": 600,
            "u": 600,
            "v": 600,
            "w": 600,
            "x": 600,
            "y": 600,
            "z": 600,
            "{": 600,
            "|": 600,
            "}": 600,
            "~": 600,
            "\xa1": 600,
            "\xa2": 600,
            "\xa3": 600,
            "\xa4": 600,
            "\xa5": 600,
            "\xa6": 600,
            "\xa7": 600,
            "\xa8": 600,
            "\xa9": 600,
            "\xaa": 600,
            "\xab": 600,
            "\xac": 600,
            "\xae": 600,
            "\xaf": 600,
            "\xb0": 600,
            "\xb1": 600,
            "\xb2": 600,
            "\xb3": 600,
            "\xb4": 600,
            "\xb5": 600,
            "\xb6": 600,
            "\xb7": 600,
            "\xb8": 600,
            "\xb9": 600,
            "\xba": 600,
            "\xbb": 600,
            "\xbc": 600,
            "\xbd": 600,
            "\xbe": 600,
            "\xbf": 600,
            "\xc0": 600,
            "\xc1": 600,
            "\xc2": 600,
            "\xc3": 600,
            "\xc4": 600,
            "\xc5": 600,
            "\xc6": 600,
            "\xc7": 600,
            "\xc8": 600,
            "\xc9": 600,
            "\xca": 600,
            "\xcb": 600,
            "\xcc": 600,
            "\xcd": 600,
            "\xce": 600,
            "\xcf": 600,
            "\xd0": 600,
            "\xd1": 600,
            "\xd2": 600,
            "\xd3": 600,
            "\xd4": 600,
            "\xd5": 600,
            "\xd6": 600,
            "\xd7": 600,
            "\xd8": 600,
            "\xd9": 600,
            "\xda": 600,
            "\xdb": 600,
            "\xdc": 600,
            "\xdd": 600,
            "\xde": 600,
            "\xdf": 600,
            "\xe0": 600,
            "\xe1": 600,
            "\xe2": 600,
            "\xe3": 600,
            "\xe4": 600,
            "\xe5": 600,
            "\xe6": 600,
            "\xe7": 600,
            "\xe8": 600,
            "\xe9": 600,
            "\xea": 600,
            "\xeb": 600,
            "\xec": 600,
            "\xed": 600,
            "\xee": 600,
            "\xef": 600,
            "\xf0": 600,
            "\xf1": 600,
            "\xf2": 600,
            "\xf3": 600,
            "\xf4": 600,
            "\xf5": 600,
            "\xf6": 600,
            "\xf7": 600,
            "\xf8": 600,
            "\xf9": 600,
            "\xfa": 600,
            "\xfb": 600,
            "\xfc": 600,
            "\xfd": 600,
            "\xfe": 600,
            "\xff": 600,
            "\u0100": 600,
            "\u0101": 600,
            "\u0102": 600,
            "\u0103": 600,
            "\u0104": 600,
            "\u0105": 600,
            "\u0106": 600,
            "\u0107": 600,
            "\u010c": 600,
            "\u010d": 600,
            "\u010e": 600,
            "\u010f": 600,
            "\u0110": 600,
            "\u0111": 600,
            "\u0112": 600,
            "\u0113": 600,
            "\u0116": 600,
            "\u0117": 600,
            "\u0118": 600,
            "\u0119": 600,
            "\u011a": 600,
            "\u011b": 600,
            "\u011e": 600,
            "\u011f": 600,
            "\u0122": 600,
            "\u0123": 600,
            "\u012a": 600,
            "\u012b": 600,
            "\u012e": 600,
            "\u012f": 600,
            "\u0130": 600,
            "\u0131": 600,
            "\u0136": 600,
            "\u0137": 600,
            "\u0139": 600,
            "\u013a": 600,
            "\u013b": 600,
            "\u013c": 600,
            "\u013d": 600,
            "\u013e": 600,
            "\u0141": 600,
            "\u0142": 600,
            "\u0143": 600,
            "\u0144": 600,
            "\u0145": 600,
            "\u0146": 600,
            "\u0147": 600,
            "\u0148": 600,
            "\u014c": 600,
            "\u014d": 600,
            "\u0150": 600,
            "\u0151": 600,
            "\u0152": 600,
            "\u0153": 600,
            "\u0154": 600,
            "\u0155": 600,
            "\u0156": 600,
            "\u0157": 600,
            "\u0158": 600,
            "\u0159": 600,
            "\u015a": 600,
            "\u015b": 600,
            "\u015e": 600,
            "\u015f": 600,
            "\u0160": 600,
            "\u0161": 600,
            "\u0162": 600,
            "\u0163": 600,
            "\u0164": 600,
            "\u0165": 600,
            "\u016a": 600,
            "\u016b": 600,
            "\u016e": 600,
            "\u016f": 600,
            "\u0170": 600,
            "\u0171": 600,
            "\u0172": 600,
            "\u0173": 600,
            "\u0178": 600,
            "\u0179": 600,
            "\u017a": 600,
            "\u017b": 600,
            "\u017c": 600,
            "\u017d": 600,
            "\u017e": 600,
            "\u0192": 600,
            "\u0218": 600,
            "\u0219": 600,
            "\u02c6": 600,
            "\u02c7": 600,
            "\u02d8": 600,
            "\u02d9": 600,
            "\u02da": 600,
            "\u02db": 600,
            "\u02dc": 600,
            "\u02dd": 600,
            "\u2013": 600,
            "\u2014": 600,
            "\u2018": 600,
            "\u2019": 600,
            "\u201a": 600,
            "\u201c": 600,
            "\u201d": 600,
            "\u201e": 600,
            "\u2020": 600,
            "\u2021": 600,
            "\u2022": 600,
            "\u2026": 600,
            "\u2030": 600,
            "\u2039": 600,
            "\u203a": 600,
            "\u2044": 600,
            "\u2122": 600,
            "\u2202": 600,
            "\u2206": 600,
            "\u2211": 600,
            "\u2212": 600,
            "\u221a": 600,
            "\u2260": 600,
            "\u2264": 600,
            "\u2265": 600,
            "\u25ca": 600,
            "\uf6c3": 600,
            "\ufb01": 600,
            "\ufb02": 600,
        },
    ),
    "Courier-Oblique": (
        {
            "FontName": "Courier-Oblique",
            "Descent": -194.0,
            "FontBBox": (-49.0, -249.0, 749.0, 803.0),
            "FontWeight": "Medium",
            "CapHeight": 572.0,
            "FontFamily": "Courier",
            "Flags": 64,
            "XHeight": 434.0,
            "ItalicAngle": -11.0,
            "Ascent": 627.0,
        },
        {
            " ": 600,
            "!": 600,
            '"': 600,
            "#": 600,
            "$": 600,
            "%": 600,
            "&": 600,
            "'": 600,
            "(": 600,
            ")": 600,
            "*": 600,
            "+": 600,
            ",": 600,
            "-": 600,
            ".": 600,
            "/": 600,
            "0": 600,
            "1": 600,
            "2": 600,
            "3": 600,
            "4": 600,
            "5": 600,
            "6": 600,
            "7": 600,
            "8": 600,
            "9": 600,
            ":": 600,
            ";": 600,
            "<": 600,
            "=": 600,
            ">": 600,
            "?": 600,
            "@": 600,
            "A": 600,
            "B": 600,
            "C": 600,
            "D": 600,
            "E": 600,
            "F": 600,
            "G": 600,
            "H": 600,
            "I": 600,
            "J": 600,
            "K": 600,
            "L": 600,
            "M": 600,
            "N": 600,
            "O": 600,
            "P": 600,
            "Q": 600,
            "R": 600,
            "S": 600,
            "T": 600,
            "U": 600,
            "V": 600,
            "W": 600,
            "X": 600,
            "Y": 600,
            "Z": 600,
            "[": 600,
            "\\": 600,
            "]": 600,
            "^": 600,
            "_": 600,
            "`": 600,
            "a": 600,
            "b": 600,
            "c": 600,
            "d": 600,
            "e": 600,
            "f": 600,
            "g": 600,
            "h": 600,
            "i": 600,
            "j": 600,
            "k": 600,
            "l": 600,
            "m": 600,
            "n": 600,
            "o": 600,
            "p": 600,
            "q": 600,
            "r": 600,
            "s": 600,
            "t": 600,
            "u": 600,
            "v": 600,
            "w": 600,
            "x": 600,
            "y": 600,
            "z": 600,
            "{": 600,
            "|": 600,
            "}": 600,
            "~": 600,
            "\xa1": 600,
            "\xa2": 600,
            "\xa3": 600,
            "\xa4": 600,
            "\xa5": 600,
            "\xa6": 600,
            "\xa7": 600,
            "\xa8": 600,
            "\xa9": 600,
            "\xaa": 600,
            "\xab": 600,
            "\xac": 600,
            "\xae": 600,
            "\xaf": 600,
            "\xb0": 600,
            "\xb1": 600,
            "\xb2": 600,
            "\xb3": 600,
            "\xb4": 600,
            "\xb5": 600,
            "\xb6": 600,
            "\xb7": 600,
            "\xb8": 600,
            "\xb9": 600,
            "\xba": 600,
            "\xbb": 600,
            "\xbc": 600,
            "\xbd": 600,
            "\xbe": 600,
            "\xbf": 600,
            "\xc0": 600,
            "\xc1": 600,
            "\xc2": 600,
            "\xc3": 600,
            "\xc4": 600,
            "\xc5": 600,
            "\xc6": 600,
            "\xc7": 600,
            "\xc8": 600,
            "\xc9": 600,
            "\xca": 600,
            "\xcb": 600,
            "\xcc": 600,
            "\xcd": 600,
            "\xce": 600,
            "\xcf": 600,
            "\xd0": 600,
            "\xd1": 600,
            "\xd2": 600,
            "\xd3": 600,
            "\xd4": 600,
            "\xd5": 600,
            "\xd6": 600,
            "\xd7": 600,
            "\xd8": 600,
            "\xd9": 600,
            "\xda": 600,
            "\xdb": 600,
            "\xdc": 600,
            "\xdd": 600,
            "\xde": 600,
            "\xdf": 600,
            "\xe0": 600,
            "\xe1": 600,
            "\xe2": 600,
            "\xe3": 600,
            "\xe4": 600,
            "\xe5": 600,
            "\xe6": 600,
            "\xe7": 600,
            "\xe8": 600,
            "\xe9": 600,
            "\xea": 600,
            "\xeb": 600,
            "\xec": 600,
            "\xed": 600,
            "\xee": 600,
            "\xef": 600,
            "\xf0": 600,
            "\xf1": 600,
            "\xf2": 600,
            "\xf3": 600,
            "\xf4": 600,
            "\xf5": 600,
            "\xf6": 600,
            "\xf7": 600,
            "\xf8": 600,
            "\xf9": 600,
            "\xfa": 600,
            "\xfb": 600,
            "\xfc": 600,
            "\xfd": 600,
            "\xfe": 600,
            "\xff": 600,
            "\u0100": 600,
            "\u0101": 600,
            "\u0102": 600,
            "\u0103": 600,
            "\u0104": 600,
            "\u0105": 600,
            "\u0106": 600,
            "\u0107": 600,
            "\u010c": 600,
            "\u010d": 600,
            "\u010e": 600,
            "\u010f": 600,
            "\u0110": 600,
            "\u0111": 600,
            "\u0112": 600,
            "\u0113": 600,
            "\u0116": 600,
            "\u0117": 600,
            "\u0118": 600,
            "\u0119": 600,
            "\u011a": 600,
            "\u011b": 600,
            "\u011e": 600,
            "\u011f": 600,
            "\u0122": 600,
            "\u0123": 600,
            "\u012a": 600,
            "\u012b": 600,
            "\u012e": 600,
            "\u012f": 600,
            "\u0130": 600,
            "\u0131": 600,
            "\u0136": 600,
            "\u0137": 600,
            "\u0139": 600,
            "\u013a": 600,
            "\u013b": 600,
            "\u013c": 600,
            "\u013d": 600,
            "\u013e": 600,
            "\u0141": 600,
            "\u0142": 600,
            "\u0143": 600,
            "\u0144": 600,
            "\u0145": 600,
            "\u0146": 600,
            "\u0147": 600,
            "\u0148": 600,
            "\u014c": 600,
            "\u014d": 600,
            "\u0150": 600,
            "\u0151": 600,
            "\u0152": 600,
            "\u0153": 600,
            "\u0154": 600,
            "\u0155": 600,
            "\u0156": 600,
            "\u0157": 600,
            "\u0158": 600,
            "\u0159": 600,
            "\u015a": 600,
            "\u015b": 600,
            "\u015e": 600,
            "\u015f": 600,
            "\u0160": 600,
            "\u0161": 600,
            "\u0162": 600,
            "\u0163": 600,
            "\u0164": 600,
            "\u0165": 600,
            "\u016a": 600,
            "\u016b": 600,
            "\u016e": 600,
            "\u016f": 600,
            "\u0170": 600,
            "\u0171": 600,
            "\u0172": 600,
            "\u0173": 600,
            "\u0178": 600,
            "\u0179": 600,
            "\u017a": 600,
            "\u017b": 600,
            "\u017c": 600,
            "\u017d": 600,
            "\u017e": 600,
            "\u0192": 600,
            "\u0218": 600,
            "\u0219": 600,
            "\u02c6": 600,
            "\u02c7": 600,
            "\u02d8": 600,
            "\u02d9": 600,
            "\u02da": 600,
            "\u02db": 600,
            "\u02dc": 600,
            "\u02dd": 600,
            "\u2013": 600,
            "\u2014": 600,
            "\u2018": 600,
            "\u2019": 600,
            "\u201a": 600,
            "\u201c": 600,
            "\u201d": 600,
            "\u201e": 600,
            "\u2020": 600,
            "\u2021": 600,
            "\u2022": 600,
            "\u2026": 600,
            "\u2030": 600,
            "\u2039": 600,
            "\u203a": 600,
            "\u2044": 600,
            "\u2122": 600,
            "\u2202": 600,
            "\u2206": 600,
            "\u2211": 600,
            "\u2212": 600,
            "\u221a": 600,
            "\u2260": 600,
            "\u2264": 600,
            "\u2265": 600,
            "\u25ca": 600,
            "\uf6c3": 600,
            "\ufb01": 600,
            "\ufb02": 600,
        },
    ),
    "Helvetica": (
        {
            "FontName": "Helvetica",
            "Descent": -207.0,
            "FontBBox": (-166.0, -225.0, 1000.0, 931.0),
            "FontWeight": "Medium",
            "CapHeight": 718.0,
            "FontFamily": "Helvetica",
            "Flags": 0,
            "XHeight": 523.0,
            "ItalicAngle": 0.0,
            "Ascent": 718.0,
        },
        {
            " ": 278,
            "!": 278,
            '"': 355,
            "#": 556,
            "$": 556,
            "%": 889,
            "&": 667,
            "'": 191,
            "(": 333,
            ")": 333,
            "*": 389,
            "+": 584,
            ",": 278,
            "-": 333,
            ".": 278,
            "/": 278,
            "0": 556,
            "1": 556,
            "2": 556,
            "3": 556,
            "4": 556,
            "5": 556,
            "6": 556,
            "7": 556,
            "8": 556,
            "9": 556,
            ":": 278,
            ";": 278,
            "<": 584,
            "=": 584,
            ">": 584,
            "?": 556,
            "@": 1015,
            "A": 667,
            "B": 667,
            "C": 722,
            "D": 722,
            "E": 667,
            "F": 611,
            "G": 778,
            "H": 722,
            "I": 278,
            "J": 500,
            "K": 667,
            "L": 556,
            "M": 833,
            "N": 722,
            "O": 778,
            "P": 667,
            "Q": 778,
            "R": 722,
            "S": 667,
            "T": 611,
            "U": 722,
            "V": 667,
            "W": 944,
            "X": 667,
            "Y": 667,
            "Z": 611,
            "[": 278,
            "\\": 278,
            "]": 278,
            "^": 469,
            "_": 556,
            "`": 333,
            "a": 556,
            "b": 556,
            "c": 500,
            "d": 556,
            "e": 556,
            "f": 278,
            "g": 556,
            "h": 556,
            "i": 222,
            "j": 222,
            "k": 500,
            "l": 222,
            "m": 833,
            "n": 556,
            "o": 556,
            "p": 556,
            "q": 556,
            "r": 333,
            "s": 500,
            "t": 278,
            "u": 556,
            "v": 500,
            "w": 722,
            "x": 500,
            "y": 500,
            "z": 500,
            "{": 334,
            "|": 260,
            "}": 334,
            "~": 584,
            "\xa1": 333,
            "\xa2": 556,
            "\xa3": 556,
            "\xa4": 556,
            "\xa5": 556,
            "\xa6": 260,
            "\xa7": 556,
            "\xa8": 333,
            "\xa9": 737,
            "\xaa": 370,
            "\xab": 556,
            "\xac": 584,
            "\xae": 737,
            "\xaf": 333,
            "\xb0": 400,
            "\xb1": 584,
            "\xb2": 333,
            "\xb3": 333,
            "\xb4": 333,
            "\xb5": 556,
            "\xb6": 537,
            "\xb7": 278,
            "\xb8": 333,
            "\xb9": 333,
            "\xba": 365,
            "\xbb": 556,
            "\xbc": 834,
            "\xbd": 834,
            "\xbe": 834,
            "\xbf": 611,
            "\xc0": 667,
            "\xc1": 667,
            "\xc2": 667,
            "\xc3": 667,
            "\xc4": 667,
            "\xc5": 667,
            "\xc6": 1000,
            "\xc7": 722,
            "\xc8": 667,
            "\xc9": 667,
            "\xca": 667,
            "\xcb": 667,
            "\xcc": 278,
            "\xcd": 278,
            "\xce": 278,
            "\xcf": 278,
            "\xd0": 722,
            "\xd1": 722,
            "\xd2": 778,
            "\xd3": 778,
            "\xd4": 778,
            "\xd5": 778,
            "\xd6": 778,
            "\xd7": 584,
            "\xd8": 778,
            "\xd9": 722,
            "\xda": 722,
            "\xdb": 722,
            "\xdc": 722,
            "\xdd": 667,
            "\xde": 667,
            "\xdf": 611,
            "\xe0": 556,
            "\xe1": 556,
            "\xe2": 556,
            "\xe3": 556,
            "\xe4": 556,
            "\xe5": 556,
            "\xe6": 889,
            "\xe7": 500,
            "\xe8": 556,
            "\xe9": 556,
            "\xea": 556,
            "\xeb": 556,
            "\xec": 278,
            "\xed": 278,
            "\xee": 278,
            "\xef": 278,
            "\xf0": 556,
            "\xf1": 556,
            "\xf2": 556,
            "\xf3": 556,
            "\xf4": 556,
            "\xf5": 556,
            "\xf6": 556,
            "\xf7": 584,
            "\xf8": 611,
            "\xf9": 556,
            "\xfa": 556,
            "\xfb": 556,
            "\xfc": 556,
            "\xfd": 500,
            "\xfe": 556,
            "\xff": 500,
            "\u0100": 667,
            "\u0101": 556,
            "\u0102": 667,
            "\u0103": 556,
            "\u0104": 667,
            "\u0105": 556,
            "\u0106": 722,
            "\u0107": 500,
            "\u010c": 722,
            "\u010d": 500,
            "\u010e": 722,
            "\u010f": 643,
            "\u0110": 722,
            "\u0111": 556,
            "\u0112": 667,
            "\u0113": 556,
            "\u0116": 667,
            "\u0117": 556,
            "\u0118": 667,
            "\u0119": 556,
            "\u011a": 667,
            "\u011b": 556,
            "\u011e": 778,
            "\u011f": 556,
            "\u0122": 778,
            "\u0123": 556,
            "\u012a": 278,
            "\u012b": 278,
            "\u012e": 278,
            "\u012f": 222,
            "\u0130": 278,
            "\u0131": 278,
            "\u0136": 667,
            "\u0137": 500,
            "\u0139": 556,
            "\u013a": 222,
            "\u013b": 556,
            "\u013c": 222,
            "\u013d": 556,
            "\u013e": 299,
            "\u0141": 556,
            "\u0142": 222,
            "\u0143": 722,
            "\u0144": 556,
            "\u0145": 722,
            "\u0146": 556,
            "\u0147": 722,
            "\u0148": 556,
            "\u014c": 778,
            "\u014d": 556,
            "\u0150": 778,
            "\u0151": 556,
            "\u0152": 1000,
            "\u0153": 944,
            "\u0154": 722,
            "\u0155": 333,
            "\u0156": 722,
            "\u0157": 333,
            "\u0158": 722,
            "\u0159": 333,
            "\u015a": 667,
            "\u015b": 500,
            "\u015e": 667,
            "\u015f": 500,
            "\u0160": 667,
            "\u0161": 500,
            "\u0162": 611,
            "\u0163": 278,
            "\u0164": 611,
            "\u0165": 317,
            "\u016a": 722,
            "\u016b": 556,
            "\u016e": 722,
            "\u016f": 556,
            "\u0170": 722,
            "\u0171": 556,
            "\u0172": 722,
            "\u0173": 556,
            "\u0178": 667,
            "\u0179": 611,
            "\u017a": 500,
            "\u017b": 611,
            "\u017c": 500,
            "\u017d": 611,
            "\u017e": 500,
            "\u0192": 556,
            "\u0218": 667,
            "\u0219": 500,
            "\u02c6": 333,
            "\u02c7": 333,
            "\u02d8": 333,
            "\u02d9": 333,
            "\u02da": 333,
            "\u02db": 333,
            "\u02dc": 333,
            "\u02dd": 333,
            "\u2013": 556,
            "\u2014": 1000,
            "\u2018": 222,
            "\u2019": 222,
            "\u201a": 222,
            "\u201c": 333,
            "\u201d": 333,
            "\u201e": 333,
            "\u2020": 556,
            "\u2021": 556,
            "\u2022": 350,
            "\u2026": 1000,
            "\u2030": 1000,
            "\u2039": 333,
            "\u203a": 333,
            "\u2044": 167,
            "\u2122": 1000,
            "\u2202": 476,
            "\u2206": 612,
            "\u2211": 600,
            "\u2212": 584,
            "\u221a": 453,
            "\u2260": 549,
            "\u2264": 549,
            "\u2265": 549,
            "\u25ca": 471,
            "\uf6c3": 250,
            "\ufb01": 500,
            "\ufb02": 500,
        },
    ),
    "Helvetica-Bold": (
        {
            "FontName": "Helvetica-Bold",
            "Descent": -207.0,
            "FontBBox": (-170.0, -228.0, 1003.0, 962.0),
            "FontWeight": "Bold",
            "CapHeight": 718.0,
            "FontFamily": "Helvetica",
            "Flags": 0,
            "XHeight": 532.0,
            "ItalicAngle": 0.0,
            "Ascent": 718.0,
        },
        {
            " ": 278,
            "!": 333,
            '"': 474,
            "#": 556,
            "$": 556,
            "%": 889,
            "&": 722,
            "'": 238,
            "(": 333,
            ")": 333,
            "*": 389,
            "+": 584,
            ",": 278,
            "-": 333,
            ".": 278,
            "/": 278,
            "0": 556,
            "1": 556,
            "2": 556,
            "3": 556,
            "4": 556,
            "5": 556,
            "6": 556,
            "7": 556,
            "8": 556,
            "9": 556,
            ":": 333,
            ";": 333,
            "<": 584,
            "=": 584,
            ">": 584,
            "?": 611,
            "@": 975,
            "A": 722,
            "B": 722,
            "C": 722,
            "D": 722,
            "E": 667,
            "F": 611,
            "G": 778,
            "H": 722,
            "I": 278,
            "J": 556,
            "K": 722,
            "L": 611,
            "M": 833,
            "N": 722,
            "O": 778,
            "P": 667,
            "Q": 778,
            "R": 722,
            "S": 667,
            "T": 611,
            "U": 722,
            "V": 667,
            "W": 944,
            "X": 667,
            "Y": 667,
            "Z": 611,
            "[": 333,
            "\\": 278,
            "]": 333,
            "^": 584,
            "_": 556,
            "`": 333,
            "a": 556,
            "b": 611,
            "c": 556,
            "d": 611,
            "e": 556,
            "f": 333,
            "g": 611,
            "h": 611,
            "i": 278,
            "j": 278,
            "k": 556,
            "l": 278,
            "m": 889,
            "n": 611,
            "o": 611,
            "p": 611,
            "q": 611,
            "r": 389,
            "s": 556,
            "t": 333,
            "u": 611,
            "v": 556,
            "w": 778,
            "x": 556,
            "y": 556,
            "z": 500,
            "{": 389,
            "|": 280,
            "}": 389,
            "~": 584,
            "\xa1": 333,
            "\xa2": 556,
            "\xa3": 556,
            "\xa4": 556,
            "\xa5": 556,
            "\xa6": 280,
            "\xa7": 556,
            "\xa8": 333,
            "\xa9": 737,
            "\xaa": 370,
            "\xab": 556,
            "\xac": 584,
            "\xae": 737,
            "\xaf": 333,
            "\xb0": 400,
            "\xb1": 584,
            "\xb2": 333,
            "\xb3": 333,
            "\xb4": 333,
            "\xb5": 611,
            "\xb6": 556,
            "\xb7": 278,
            "\xb8": 333,
            "\xb9": 333,
            "\xba": 365,
            "\xbb": 556,
            "\xbc": 834,
            "\xbd": 834,
            "\xbe": 834,
            "\xbf": 611,
            "\xc0": 722,
            "\xc1": 722,
            "\xc2": 722,
            "\xc3": 722,
            "\xc4": 722,
            "\xc5": 722,
            "\xc6": 1000,
            "\xc7": 722,
            "\xc8": 667,
            "\xc9": 667,
            "\xca": 667,
            "\xcb": 667,
            "\xcc": 278,
            "\xcd": 278,
            "\xce": 278,
            "\xcf": 278,
            "\xd0": 722,
            "\xd1": 722,
            "\xd2": 778,
            "\xd3": 778,
            "\xd4": 778,
            "\xd5": 778,
            "\xd6": 778,
            "\xd7": 584,
            "\xd8": 778,
            "\xd9": 722,
            "\xda": 722,
            "\xdb": 722,
            "\xdc": 722,
            "\xdd": 667,
            "\xde": 667,
            "\xdf": 611,
            "\xe0": 556,
            "\xe1": 556,
            "\xe2": 556,
            "\xe3": 556,
            "\xe4": 556,
            "\xe5": 556,
            "\xe6": 889,
            "\xe7": 556,
            "\xe8": 556,
            "\xe9": 556,
            "\xea": 556,
            "\xeb": 556,
            "\xec": 278,
            "\xed": 278,
            "\xee": 278,
            "\xef": 278,
            "\xf0": 611,
            "\xf1": 611,
            "\xf2": 611,
            "\xf3": 611,
            "\xf4": 611,
            "\xf5": 611,
            "\xf6": 611,
            "\xf7": 584,
            "\xf8": 611,
            "\xf9": 611,
            "\xfa": 611,
            "\xfb": 611,
            "\xfc": 611,
            "\xfd": 556,
            "\xfe": 611,
            "\xff": 556,
            "\u0100": 722,
            "\u0101": 556,
            "\u0102": 722,
            "\u0103": 556,
            "\u0104": 722,
            "\u0105": 556,
            "\u0106": 722,
            "\u0107": 556,
            "\u010c": 722,
            "\u010d": 556,
            "\u010e": 722,
            "\u010f": 743,
            "\u0110": 722,
            "\u0111": 611,
            "\u0112": 667,
            "\u0113": 556,
            "\u0116": 667,
            "\u0117": 556,
            "\u0118": 667,
            "\u0119": 556,
            "\u011a": 667,
            "\u011b": 556,
            "\u011e": 778,
            "\u011f": 611,
            "\u0122": 778,
            "\u0123": 611,
            "\u012a": 278,
            "\u012b": 278,
            "\u012e": 278,
            "\u012f": 278,
            "\u0130": 278,
            "\u0131": 278,
            "\u0136": 722,
            "\u0137": 556,
            "\u0139": 611,
            "\u013a": 278,
            "\u013b": 611,
            "\u013c": 278,
            "\u013d": 611,
            "\u013e": 400,
            "\u0141": 611,
            "\u0142": 278,
            "\u0143": 722,
            "\u0144": 611,
            "\u0145": 722,
            "\u0146": 611,
            "\u0147": 722,
            "\u0148": 611,
            "\u014c": 778,
            "\u014d": 611,
            "\u0150": 778,
            "\u0151": 611,
            "\u0152": 1000,
            "\u0153": 944,
            "\u0154": 722,
            "\u0155": 389,
            "\u0156": 722,
            "\u0157": 389,
            "\u0158": 722,
            "\u0159": 389,
            "\u015a": 667,
            "\u015b": 556,
            "\u015e": 667,
            "\u015f": 556,
            "\u0160": 667,
            "\u0161": 556,
            "\u0162": 611,
            "\u0163": 333,
            "\u0164": 611,
            "\u0165": 389,
            "\u016a": 722,
            "\u016b": 611,
            "\u016e": 722,
            "\u016f": 611,
            "\u0170": 722,
            "\u0171": 611,
            "\u0172": 722,
            "\u0173": 611,
            "\u0178": 667,
            "\u0179": 611,
            "\u017a": 500,
            "\u017b": 611,
            "\u017c": 500,
            "\u017d": 611,
            "\u017e": 500,
            "\u0192": 556,
            "\u0218": 667,
            "\u0219": 556,
            "\u02c6": 333,
            "\u02c7": 333,
            "\u02d8": 333,
            "\u02d9": 333,
            "\u02da": 333,
            "\u02db": 333,
            "\u02dc": 333,
            "\u02dd": 333,
            "\u2013": 556,
            "\u2014": 1000,
            "\u2018": 278,
            "\u2019": 278,
            "\u201a": 278,
            "\u201c": 500,
            "\u201d": 500,
            "\u201e": 500,
            "\u2020": 556,
            "\u2021": 556,
            "\u2022": 350,
            "\u2026": 1000,
            "\u2030": 1000,
            "\u2039": 333,
            "\u203a": 333,
            "\u2044": 167,
            "\u2122": 1000,
            "\u2202": 494,
            "\u2206": 612,
            "\u2211": 600,
            "\u2212": 584,
            "\u221a": 549,
            "\u2260": 549,
            "\u2264": 549,
            "\u2265": 549,
            "\u25ca": 494,
            "\uf6c3": 250,
            "\ufb01": 611,
            "\ufb02": 611,
        },
    ),
    "Helvetica-BoldOblique": (
        {
            "FontName": "Helvetica-BoldOblique",
            "Descent": -207.0,
            "FontBBox": (-175.0, -228.0, 1114.0, 962.0),
            "FontWeight": "Bold",
            "CapHeight": 718.0,
            "FontFamily": "Helvetica",
            "Flags": 0,
            "XHeight": 532.0,
            "ItalicAngle": -12.0,
            "Ascent": 718.0,
        },
        {
            " ": 278,
            "!": 333,
            '"': 474,
            "#": 556,
            "$": 556,
            "%": 889,
            "&": 722,
            "'": 238,
            "(": 333,
            ")": 333,
            "*": 389,
            "+": 584,
            ",": 278,
            "-": 333,
            ".": 278,
            "/": 278,
            "0": 556,
            "1": 556,
            "2": 556,
            "3": 556,
            "4": 556,
            "5": 556,
            "6": 556,
            "7": 556,
            "8": 556,
            "9": 556,
            ":": 333,
            ";": 333,
            "<": 584,
            "=": 584,
            ">": 584,
            "?": 611,
            "@": 975,
            "A": 722,
            "B": 722,
            "C": 722,
            "D": 722,
            "E": 667,
            "F": 611,
            "G": 778,
            "H": 722,
            "I": 278,
            "J": 556,
            "K": 722,
            "L": 611,
            "M": 833,
            "N": 722,
            "O": 778,
            "P": 667,
            "Q": 778,
            "R": 722,
            "S": 667,
            "T": 611,
            "U": 722,
            "V": 667,
            "W": 944,
            "X": 667,
            "Y": 667,
            "Z": 611,
            "[": 333,
            "\\": 278,
            "]": 333,
            "^": 584,
            "_": 556,
            "`": 333,
            "a": 556,
            "b": 611,
            "c": 556,
            "d": 611,
            "e": 556,
            "f": 333,
            "g": 611,
            "h": 611,
            "i": 278,
            "j": 278,
            "k": 556,
            "l": 278,
            "m": 889,
            "n": 611,
            "o": 611,
            "p": 611,
            "q": 611,
            "r": 389,
            "s": 556,
            "t": 333,
            "u": 611,
            "v": 556,
            "w": 778,
            "x": 556,
            "y": 556,
            "z": 500,
            "{": 389,
            "|": 280,
            "}": 389,
            "~": 584,
            "\xa1": 333,
            "\xa2": 556,
            "\xa3": 556,
            "\xa4": 556,
            "\xa5": 556,
            "\xa6": 280,
            "\xa7": 556,
            "\xa8": 333,
            "\xa9": 737,
            "\xaa": 370,
            "\xab": 556,
            "\xac": 584,
            "\xae": 737,
            "\xaf": 333,
            "\xb0": 400,
            "\xb1": 584,
            "\xb2": 333,
            "\xb3": 333,
            "\xb4": 333,
            "\xb5": 611,
            "\xb6": 556,
            "\xb7": 278,
            "\xb8": 333,
            "\xb9": 333,
            "\xba": 365,
            "\xbb": 556,
            "\xbc": 834,
            "\xbd": 834,
            "\xbe": 834,
            "\xbf": 611,
            "\xc0": 722,
            "\xc1": 722,
            "\xc2": 722,
            "\xc3": 722,
            "\xc4": 722,
            "\xc5": 722,
            "\xc6": 1000,
            "\xc7": 722,
            "\xc8": 667,
            "\xc9": 667,
            "\xca": 667,
            "\xcb": 667,
            "\xcc": 278,
            "\xcd": 278,
            "\xce": 278,
            "\xcf": 278,
            "\xd0": 722,
            "\xd1": 722,
            "\xd2": 778,
            "\xd3": 778,
            "\xd4": 778,
            "\xd5": 778,
            "\xd6": 778,
            "\xd7": 584,
            "\xd8": 778,
            "\xd9": 722,
            "\xda": 722,
            "\xdb": 722,
            "\xdc": 722,
            "\xdd": 667,
            "\xde": 667,
            "\xdf": 611,
            "\xe0": 556,
            "\xe1": 556,
            "\xe2": 556,
            "\xe3": 556,
            "\xe4": 556,
            "\xe5": 556,
            "\xe6": 889,
            "\xe7": 556,
            "\xe8": 556,
            "\xe9": 556,
            "\xea": 556,
            "\xeb": 556,
            "\xec": 278,
            "\xed": 278,
            "\xee": 278,
            "\xef": 278,
            "\xf0": 611,
            "\xf1": 611,
            "\xf2": 611,
            "\xf3": 611,
            "\xf4": 611,
            "\xf5": 611,
            "\xf6": 611,
            "\xf7": 584,
            "\xf8": 611,
            "\xf9": 611,
            "\xfa": 611,
            "\xfb": 611,
            "\xfc": 611,
            "\xfd": 556,
            "\xfe": 611,
            "\xff": 556,
            "\u0100": 722,
            "\u0101": 556,
            "\u0102": 722,
            "\u0103": 556,
            "\u0104": 722,
            "\u0105": 556,
            "\u0106": 722,
            "\u0107": 556,
            "\u010c": 722,
            "\u010d": 556,
            "\u010e": 722,
            "\u010f": 743,
            "\u0110": 722,
            "\u0111": 611,
            "\u0112": 667,
            "\u0113": 556,
            "\u0116": 667,
            "\u0117": 556,
            "\u0118": 667,
            "\u0119": 556,
            "\u011a": 667,
            "\u011b": 556,
            "\u011e": 778,
            "\u011f": 611,
            "\u0122": 778,
            "\u0123": 611,
            "\u012a": 278,
            "\u012b": 278,
            "\u012e": 278,
            "\u012f": 278,
            "\u0130": 278,
            "\u0131": 278,
            "\u0136": 722,
            "\u0137": 556,
            "\u0139": 611,
            "\u013a": 278,
            "\u013b": 611,
            "\u013c": 278,
            "\u013d": 611,
            "\u013e": 400,
            "\u0141": 611,
            "\u0142": 278,
            "\u0143": 722,
            "\u0144": 611,
            "\u0145": 722,
            "\u0146": 611,
            "\u0147": 722,
            "\u0148": 611,
            "\u014c": 778,
            "\u014d": 611,
            "\u0150": 778,
            "\u0151": 611,
            "\u0152": 1000,
            "\u0153": 944,
            "\u0154": 722,
            "\u0155": 389,
            "\u0156": 722,
            "\u0157": 389,
            "\u0158": 722,
            "\u0159": 389,
            "\u015a": 667,
            "\u015b": 556,
            "\u015e": 667,
            "\u015f": 556,
            "\u0160": 667,
            "\u0161": 556,
            "\u0162": 611,
            "\u0163": 333,
            "\u0164": 611,
            "\u0165": 389,
            "\u016a": 722,
            "\u016b": 611,
            "\u016e": 722,
            "\u016f": 611,
            "\u0170": 722,
            "\u0171": 611,
            "\u0172": 722,
            "\u0173": 611,
            "\u0178": 667,
            "\u0179": 611,
            "\u017a": 500,
            "\u017b": 611,
            "\u017c": 500,
            "\u017d": 611,
            "\u017e": 500,
            "\u0192": 556,
            "\u0218": 667,
            "\u0219": 556,
            "\u02c6": 333,
            "\u02c7": 333,
            "\u02d8": 333,
            "\u02d9": 333,
            "\u02da": 333,
            "\u02db": 333,
            "\u02dc": 333,
            "\u02dd": 333,
            "\u2013": 556,
            "\u2014": 1000,
            "\u2018": 278,
            "\u2019": 278,
            "\u201a": 278,
            "\u201c": 500,
            "\u201d": 500,
            "\u201e": 500,
            "\u2020": 556,
            "\u2021": 556,
            "\u2022": 350,
            "\u2026": 1000,
            "\u2030": 1000,
            "\u2039": 333,
            "\u203a": 333,
            "\u2044": 167,
            "\u2122": 1000,
            "\u2202": 494,
            "\u2206": 612,
            "\u2211": 600,
            "\u2212": 584,
            "\u221a": 549,
            "\u2260": 549,
            "\u2264": 549,
            "\u2265": 549,
            "\u25ca": 494,
            "\uf6c3": 250,
            "\ufb01": 611,
            "\ufb02": 611,
        },
    ),
    "Helvetica-Oblique": (
        {
            "FontName": "Helvetica-Oblique",
            "Descent": -207.0,
            "FontBBox": (-171.0, -225.0, 1116.0, 931.0),
            "FontWeight": "Medium",
            "CapHeight": 718.0,
            "FontFamily": "Helvetica",
            "Flags": 0,
            "XHeight": 523.0,
            "ItalicAngle": -12.0,
            "Ascent": 718.0,
        },
        {
            " ": 278,
            "!": 278,
            '"': 355,
            "#": 556,
            "$": 556,
            "%": 889,
            "&": 667,
            "'": 191,
            "(": 333,
            ")": 333,
            "*": 389,
            "+": 584,
            ",": 278,
            "-": 333,
            ".": 278,
            "/": 278,
            "0": 556,
            "1": 556,
            "2": 556,
            "3": 556,
            "4": 556,
            "5": 556,
            "6": 556,
            "7": 556,
            "8": 556,
            "9": 556,
            ":": 278,
            ";": 278,
            "<": 584,
            "=": 584,
            ">": 584,
            "?": 556,
            "@": 1015,
            "A": 667,
            "B": 667,
            "C": 722,
            "D": 722,
            "E": 667,
            "F": 611,
            "G": 778,
            "H": 722,
            "I": 278,
            "J": 500,
            "K": 667,
            "L": 556,
            "M": 833,
            "N": 722,
            "O": 778,
            "P": 667,
            "Q": 778,
            "R": 722,
            "S": 667,
            "T": 611,
            "U": 722,
            "V": 667,
            "W": 944,
            "X": 667,
            "Y": 667,
            "Z": 611,
            "[": 278,
            "\\": 278,
            "]": 278,
            "^": 469,
            "_": 556,
            "`": 333,
            "a": 556,
            "b": 556,
            "c": 500,
            "d": 556,
            "e": 556,
            "f": 278,
            "g": 556,
            "h": 556,
            "i": 222,
            "j": 222,
            "k": 500,
            "l": 222,
            "m": 833,
            "n": 556,
            "o": 556,
            "p": 556,
            "q": 556,
            "r": 333,
            "s": 500,
            "t": 278,
            "u": 556,
            "v": 500,
            "w": 722,
            "x": 500,
            "y": 500,
            "z": 500,
            "{": 334,
            "|": 260,
            "}": 334,
            "~": 584,
            "\xa1": 333,
            "\xa2": 556,
            "\xa3": 556,
            "\xa4": 556,
            "\xa5": 556,
            "\xa6": 260,
            "\xa7": 556,
            "\xa8": 333,
            "\xa9": 737,
            "\xaa": 370,
            "\xab": 556,
            "\xac": 584,
            "\xae": 737,
            "\xaf": 333,
            "\xb0": 400,
            "\xb1": 584,
            "\xb2": 333,
            "\xb3": 333,
            "\xb4": 333,
            "\xb5": 556,
            "\xb6": 537,
            "\xb7": 278,
            "\xb8": 333,
            "\xb9": 333,
            "\xba": 365,
            "\xbb": 556,
            "\xbc": 834,
            "\xbd": 834,
            "\xbe": 834,
            "\xbf": 611,
            "\xc0": 667,
            "\xc1": 667,
            "\xc2": 667,
            "\xc3": 667,
            "\xc4": 667,
            "\xc5": 667,
            "\xc6": 1000,
            "\xc7": 722,
            "\xc8": 667,
            "\xc9": 667,
            "\xca": 667,
            "\xcb": 667,
            "\xcc": 278,
            "\xcd": 278,
            "\xce": 278,
            "\xcf": 278,
            "\xd0": 722,
            "\xd1": 722,
            "\xd2": 778,
            "\xd3": 778,
            "\xd4": 778,
            "\xd5": 778,
            "\xd6": 778,
            "\xd7": 584,
            "\xd8": 778,
            "\xd9": 722,
            "\xda": 722,
            "\xdb": 722,
            "\xdc": 722,
            "\xdd": 667,
            "\xde": 667,
            "\xdf": 611,
            "\xe0": 556,
            "\xe1": 556,
            "\xe2": 556,
            "\xe3": 556,
            "\xe4": 556,
            "\xe5": 556,
            "\xe6": 889,
            "\xe7": 500,
            "\xe8": 556,
            "\xe9": 556,
            "\xea": 556,
            "\xeb": 556,
            "\xec": 278,
            "\xed": 278,
            "\xee": 278,
            "\xef": 278,
            "\xf0": 556,
            "\xf1": 556,
            "\xf2": 556,
            "\xf3": 556,
            "\xf4": 556,
            "\xf5": 556,
            "\xf6": 556,
            "\xf7": 584,
            "\xf8": 611,
            "\xf9": 556,
            "\xfa": 556,
            "\xfb": 556,
            "\xfc": 556,
            "\xfd": 500,
            "\xfe": 556,
            "\xff": 500,
            "\u0100": 667,
            "\u0101": 556,
            "\u0102": 667,
            "\u0103": 556,
            "\u0104": 667,
            "\u0105": 556,
            "\u0106": 722,
            "\u0107": 500,
            "\u010c": 722,
            "\u010d": 500,
            "\u010e": 722,
            "\u010f": 643,
            "\u0110": 722,
            "\u0111": 556,
            "\u0112": 667,
            "\u0113": 556,
            "\u0116": 667,
            "\u0117": 556,
            "\u0118": 667,
            "\u0119": 556,
            "\u011a": 667,
            "\u011b": 556,
            "\u011e": 778,
            "\u011f": 556,
            "\u0122": 778,
            "\u0123": 556,
            "\u012a": 278,
            "\u012b": 278,
            "\u012e": 278,
            "\u012f": 222,
            "\u0130": 278,
            "\u0131": 278,
            "\u0136": 667,
            "\u0137": 500,
            "\u0139": 556,
            "\u013a": 222,
            "\u013b": 556,
            "\u013c": 222,
            "\u013d": 556,
            "\u013e": 299,
            "\u0141": 556,
            "\u0142": 222,
            "\u0143": 722,
            "\u0144": 556,
            "\u0145": 722,
            "\u0146": 556,
            "\u0147": 722,
            "\u0148": 556,
            "\u014c": 778,
            "\u014d": 556,
            "\u0150": 778,
            "\u0151": 556,
            "\u0152": 1000,
            "\u0153": 944,
            "\u0154": 722,
            "\u0155": 333,
            "\u0156": 722,
            "\u0157": 333,
            "\u0158": 722,
            "\u0159": 333,
            "\u015a": 667,
            "\u015b": 500,
            "\u015e": 667,
            "\u015f": 500,
            "\u0160": 667,
            "\u0161": 500,
            "\u0162": 611,
            "\u0163": 278,
            "\u0164": 611,
            "\u0165": 317,
            "\u016a": 722,
            "\u016b": 556,
            "\u016e": 722,
            "\u016f": 556,
            "\u0170": 722,
            "\u0171": 556,
            "\u0172": 722,
            "\u0173": 556,
            "\u0178": 667,
            "\u0179": 611,
            "\u017a": 500,
            "\u017b": 611,
            "\u017c": 500,
            "\u017d": 611,
            "\u017e": 500,
            "\u0192": 556,
            "\u0218": 667,
            "\u0219": 500,
            "\u02c6": 333,
            "\u02c7": 333,
            "\u02d8": 333,
            "\u02d9": 333,
            "\u02da": 333,
            "\u02db": 333,
            "\u02dc": 333,
            "\u02dd": 333,
            "\u2013": 556,
            "\u2014": 1000,
            "\u2018": 222,
            "\u2019": 222,
            "\u201a": 222,
            "\u201c": 333,
            "\u201d": 333,
            "\u201e": 333,
            "\u2020": 556,
            "\u2021": 556,
            "\u2022": 350,
            "\u2026": 1000,
            "\u2030": 1000,
            "\u2039": 333,
            "\u203a": 333,
            "\u2044": 167,
            "\u2122": 1000,
            "\u2202": 476,
            "\u2206": 612,
            "\u2211": 600,
            "\u2212": 584,
            "\u221a": 453,
            "\u2260": 549,
            "\u2264": 549,
            "\u2265": 549,
            "\u25ca": 471,
            "\uf6c3": 250,
            "\ufb01": 500,
            "\ufb02": 500,
        },
    ),
    "Symbol": (
        {
            "FontName": "Symbol",
            "FontBBox": (-180.0, -293.0, 1090.0, 1010.0),
            "FontWeight": "Medium",
            "FontFamily": "Symbol",
            "Flags": 0,
            "ItalicAngle": 0.0,
        },
        {
            " ": 250,
            "!": 333,
            "#": 500,
            "%": 833,
            "&": 778,
            "(": 333,
            ")": 333,
            "+": 549,
            ",": 250,
            ".": 250,
            "/": 278,
            "0": 500,
            "1": 500,
            "2": 500,
            "3": 500,
            "4": 500,
            "5": 500,
            "6": 500,
            "7": 500,
            "8": 500,
            "9": 500,
            ":": 278,
            ";": 278,
            "<": 549,
            "=": 549,
            ">": 549,
            "?": 444,
            "[": 333,
            "]": 333,
            "_": 500,
            "{": 480,
            "|": 200,
            "}": 480,
            "\xac": 713,
            "\xb0": 400,
            "\xb1": 549,
            "\xb5": 576,
            "\xd7": 549,
            "\xf7": 549,
            "\u0192": 500,
            "\u0391": 722,
            "\u0392": 667,
            "\u0393": 603,
            "\u0395": 611,
            "\u0396": 611,
            "\u0397": 722,
            "\u0398": 741,
            "\u0399": 333,
            "\u039a": 722,
            "\u039b": 686,
            "\u039c": 889,
            "\u039d": 722,
            "\u039e": 645,
            "\u039f": 722,
            "\u03a0": 768,
            "\u03a1": 556,
            "\u03a3": 592,
            "\u03a4": 611,
            "\u03a5": 690,
            "\u03a6": 763,
            "\u03a7": 722,
            "\u03a8": 795,
            "\u03b1": 631,
            "\u03b2": 549,
            "\u03b3": 411,
            "\u03b4": 494,
            "\u03b5": 439,
            "\u03b6": 494,
            "\u03b7": 603,
            "\u03b8": 521,
            "\u03b9": 329,
            "\u03ba": 549,
            "\u03bb": 549,
            "\u03bd": 521,
            "\u03be": 493,
            "\u03bf": 549,
            "\u03c0": 549,
            "\u03c1": 549,
            "\u03c2": 439,
            "\u03c3": 603,
            "\u03c4": 439,
            "\u03c5": 576,
            "\u03c6": 521,
            "\u03c7": 549,
            "\u03c8": 686,
            "\u03c9": 686,
            "\u03d1": 631,
            "\u03d2": 620,
            "\u03d5": 603,
            "\u03d6": 713,
            "\u2022": 460,
            "\u2026": 1000,
            "\u2032": 247,
            "\u2033": 411,
            "\u2044": 167,
            "\u20ac": 750,
            "\u2111": 686,
            "\u2118": 987,
            "\u211c": 795,
            "\u2126": 768,
            "\u2135": 823,
            "\u2190": 987,
            "\u2191": 603,
            "\u2192": 987,
            "\u2193": 603,
            "\u2194": 1042,
            "\u21b5": 658,
            "\u21d0": 987,
            "\u21d1": 603,
            "\u21d2": 987,
            "\u21d3": 603,
            "\u21d4": 1042,
            "\u2200": 713,
            "\u2202": 494,
            "\u2203": 549,
            "\u2205": 823,
            "\u2206": 612,
            "\u2207": 713,
            "\u2208": 713,
            "\u2209": 713,
            "\u220b": 439,
            "\u220f": 823,
            "\u2211": 713,
            "\u2212": 549,
            "\u2217": 500,
            "\u221a": 549,
            "\u221d": 713,
            "\u221e": 713,
            "\u2220": 768,
            "\u2227": 603,
            "\u2228": 603,
            "\u2229": 768,
            "\u222a": 768,
            "\u222b": 274,
            "\u2234": 863,
            "\u223c": 549,
            "\u2245": 549,
            "\u2248": 549,
            "\u2260": 549,
            "\u2261": 549,
            "\u2264": 549,
            "\u2265": 549,
            "\u2282": 713,
            "\u2283": 713,
            "\u2284": 713,
            "\u2286": 713,
            "\u2287": 713,
            "\u2295": 768,
            "\u2297": 768,
            "\u22a5": 658,
            "\u22c5": 250,
            "\u2320": 686,
            "\u2321": 686,
            "\u2329": 329,
            "\u232a": 329,
            "\u25ca": 494,
            "\u2660": 753,
            "\u2663": 753,
            "\u2665": 753,
            "\u2666": 753,
            "\uf6d9": 790,
            "\uf6da": 790,
            "\uf6db": 890,
            "\uf8e5": 500,
            "\uf8e6": 603,
            "\uf8e7": 1000,
            "\uf8e8": 790,
            "\uf8e9": 790,
            "\uf8ea": 786,
            "\uf8eb": 384,
            "\uf8ec": 384,
            "\uf8ed": 384,
            "\uf8ee": 384,
            "\uf8ef": 384,
            "\uf8f0": 384,
            "\uf8f1": 494,
            "\uf8f2": 494,
            "\uf8f3": 494,
            "\uf8f4": 494,
            "\uf8f5": 686,
            "\uf8f6": 384,
            "\uf8f7": 384,
            "\uf8f8": 384,
            "\uf8f9": 384,
            "\uf8fa": 384,
            "\uf8fb": 384,
            "\uf8fc": 494,
            "\uf8fd": 494,
            "\uf8fe": 494,
            "\uf8ff": 790,
        },
    ),
    "Times-Bold": (
        {
            "FontName": "Times-Bold",
            "Descent": -217.0,
            "FontBBox": (-168.0, -218.0, 1000.0, 935.0),
            "FontWeight": "Bold",
            "CapHeight": 676.0,
            "FontFamily": "Times",
            "Flags": 0,
            "XHeight": 461.0,
            "ItalicAngle": 0.0,
            "Ascent": 683.0,
        },
        {
            " ": 250,
            "!": 333,
            '"': 555,
            "#": 500,
            "$": 500,
            "%": 1000,
            "&": 833,
            "'": 278,
            "(": 333,
            ")": 333,
            "*": 500,
            "+": 570,
            ",": 250,
            "-": 333,
            ".": 250,
            "/": 278,
            "0": 500,
            "1": 500,
            "2": 500,
            "3": 500,
            "4": 500,
            "5": 500,
            "6": 500,
            "7": 500,
            "8": 500,
            "9": 500,
            ":": 333,
            ";": 333,
            "<": 570,
            "=": 570,
            ">": 570,
            "?": 500,
            "@": 930,
            "A": 722,
            "B": 667,
            "C": 722,
            "D": 722,
            "E": 667,
            "F": 611,
            "G": 778,
            "H": 778,
            "I": 389,
            "J": 500,
            "K": 778,
            "L": 667,
            "M": 944,
            "N": 722,
            "O": 778,
            "P": 611,
            "Q": 778,
            "R": 722,
            "S": 556,
            "T": 667,
            "U": 722,
            "V": 722,
            "W": 1000,
            "X": 722,
            "Y": 722,
            "Z": 667,
            "[": 333,
            "\\": 278,
            "]": 333,
            "^": 581,
            "_": 500,
            "`": 333,
            "a": 500,
            "b": 556,
            "c": 444,
            "d": 556,
            "e": 444,
            "f": 333,
            "g": 500,
            "h": 556,
            "i": 278,
            "j": 333,
            "k": 556,
            "l": 278,
            "m": 833,
            "n": 556,
            "o": 500,
            "p": 556,
            "q": 556,
            "r": 444,
            "s": 389,
            "t": 333,
            "u": 556,
            "v": 500,
            "w": 722,
            "x": 500,
            "y": 500,
            "z": 444,
            "{": 394,
            "|": 220,
            "}": 394,
            "~": 520,
            "\xa1": 333,
            "\xa2": 500,
            "\xa3": 500,
            "\xa4": 500,
            "\xa5": 500,
            "\xa6": 220,
            "\xa7": 500,
            "\xa8": 333,
            "\xa9": 747,
            "\xaa": 300,
            "\xab": 500,
            "\xac": 570,
            "\xae": 747,
            "\xaf": 333,
            "\xb0": 400,
            "\xb1": 570,
            "\xb2": 300,
            "\xb3": 300,
            "\xb4": 333,
            "\xb5": 556,
            "\xb6": 540,
            "\xb7": 250,
            "\xb8": 333,
            "\xb9": 300,
            "\xba": 330,
            "\xbb": 500,
            "\xbc": 750,
            "\xbd": 750,
            "\xbe": 750,
            "\xbf": 500,
            "\xc0": 722,
            "\xc1": 722,
            "\xc2": 722,
            "\xc3": 722,
            "\xc4": 722,
            "\xc5": 722,
            "\xc6": 1000,
            "\xc7": 722,
            "\xc8": 667,
            "\xc9": 667,
            "\xca": 667,
            "\xcb": 667,
            "\xcc": 389,
            "\xcd": 389,
            "\xce": 389,
            "\xcf": 389,
            "\xd0": 722,
            "\xd1": 722,
            "\xd2": 778,
            "\xd3": 778,
            "\xd4": 778,
            "\xd5": 778,
            "\xd6": 778,
            "\xd7": 570,
            "\xd8": 778,
            "\xd9": 722,
            "\xda": 722,
            "\xdb": 722,
            "\xdc": 722,
            "\xdd": 722,
            "\xde": 611,
            "\xdf": 556,
            "\xe0": 500,
            "\xe1": 500,
            "\xe2": 500,
            "\xe3": 500,
            "\xe4": 500,
            "\xe5": 500,
            "\xe6": 722,
            "\xe7": 444,
            "\xe8": 444,
            "\xe9": 444,
            "\xea": 444,
            "\xeb": 444,
            "\xec": 278,
            "\xed": 278,
            "\xee": 278,
            "\xef": 278,
            "\xf0": 500,
            "\xf1": 556,
            "\xf2": 500,
            "\xf3": 500,
            "\xf4": 500,
            "\xf5": 500,
            "\xf6": 500,
            "\xf7": 570,
            "\xf8": 500,
            "\xf9": 556,
            "\xfa": 556,
            "\xfb": 556,
            "\xfc": 556,
            "\xfd": 500,
            "\xfe": 556,
            "\xff": 500,
            "\u0100": 722,
            "\u0101": 500,
            "\u0102": 722,
            "\u0103": 500,
            "\u0104": 722,
            "\u0105": 500,
            "\u0106": 722,
            "\u0107": 444,
            "\u010c": 722,
            "\u010d": 444,
            "\u010e": 722,
            "\u010f": 672,
            "\u0110": 722,
            "\u0111": 556,
            "\u0112": 667,
            "\u0113": 444,
            "\u0116": 667,
            "\u0117": 444,
            "\u0118": 667,
            "\u0119": 444,
            "\u011a": 667,
            "\u011b": 444,
            "\u011e": 778,
            "\u011f": 500,
            "\u0122": 778,
            "\u0123": 500,
            "\u012a": 389,
            "\u012b": 278,
            "\u012e": 389,
            "\u012f": 278,
            "\u0130": 389,
            "\u0131": 278,
            "\u0136": 778,
            "\u0137": 556,
            "\u0139": 667,
            "\u013a": 278,
            "\u013b": 667,
            "\u013c": 278,
            "\u013d": 667,
            "\u013e": 394,
            "\u0141": 667,
            "\u0142": 278,
            "\u0143": 722,
            "\u0144": 556,
            "\u0145": 722,
            "\u0146": 556,
            "\u0147": 722,
            "\u0148": 556,
            "\u014c": 778,
            "\u014d": 500,
            "\u0150": 778,
            "\u0151": 500,
            "\u0152": 1000,
            "\u0153": 722,
            "\u0154": 722,
            "\u0155": 444,
            "\u0156": 722,
            "\u0157": 444,
            "\u0158": 722,
            "\u0159": 444,
            "\u015a": 556,
            "\u015b": 389,
            "\u015e": 556,
            "\u015f": 389,
            "\u0160": 556,
            "\u0161": 389,
            "\u0162": 667,
            "\u0163": 333,
            "\u0164": 667,
            "\u0165": 416,
            "\u016a": 722,
            "\u016b": 556,
            "\u016e": 722,
            "\u016f": 556,
            "\u0170": 722,
            "\u0171": 556,
            "\u0172": 722,
            "\u0173": 556,
            "\u0178": 722,
            "\u0179": 667,
            "\u017a": 444,
            "\u017b": 667,
            "\u017c": 444,
            "\u017d": 667,
            "\u017e": 444,
            "\u0192": 500,
            "\u0218": 556,
            "\u0219": 389,
            "\u02c6": 333,
            "\u02c7": 333,
            "\u02d8": 333,
            "\u02d9": 333,
            "\u02da": 333,
            "\u02db": 333,
            "\u02dc": 333,
            "\u02dd": 333,
            "\u2013": 500,
            "\u2014": 1000,
            "\u2018": 333,
            "\u2019": 333,
            "\u201a": 333,
            "\u201c": 500,
            "\u201d": 500,
            "\u201e": 500,
            "\u2020": 500,
            "\u2021": 500,
            "\u2022": 350,
            "\u2026": 1000,
            "\u2030": 1000,
            "\u2039": 333,
            "\u203a": 333,
            "\u2044": 167,
            "\u2122": 1000,
            "\u2202": 494,
            "\u2206": 612,
            "\u2211": 600,
            "\u2212": 570,
            "\u221a": 549,
            "\u2260": 549,
            "\u2264": 549,
            "\u2265": 549,
            "\u25ca": 494,
            "\uf6c3": 250,
            "\ufb01": 556,
            "\ufb02": 556,
        },
    ),
    "Times-BoldItalic": (
        {
            "FontName": "Times-BoldItalic",
            "Descent": -217.0,
            "FontBBox": (-200.0, -218.0, 996.0, 921.0),
            "FontWeight": "Bold",
            "CapHeight": 669.0,
            "FontFamily": "Times",
            "Flags": 0,
            "XHeight": 462.0,
            "ItalicAngle": -15.0,
            "Ascent": 683.0,
        },
        {
            " ": 250,
            "!": 389,
            '"': 555,
            "#": 500,
            "$": 500,
            "%": 833,
            "&": 778,
            "'": 278,
            "(": 333,
            ")": 333,
            "*": 500,
            "+": 570,
            ",": 250,
            "-": 333,
            ".": 250,
            "/": 278,
            "0": 500,
            "1": 500,
            "2": 500,
            "3": 500,
            "4": 500,
            "5": 500,
            "6": 500,
            "7": 500,
            "8": 500,
            "9": 500,
            ":": 333,
            ";": 333,
            "<": 570,
            "=": 570,
            ">": 570,
            "?": 500,
            "@": 832,
            "A": 667,
            "B": 667,
            "C": 667,
            "D": 722,
            "E": 667,
            "F": 667,
            "G": 722,
            "H": 778,
            "I": 389,
            "J": 500,
            "K": 667,
            "L": 611,
            "M": 889,
            "N": 722,
            "O": 722,
            "P": 611,
            "Q": 722,
            "R": 667,
            "S": 556,
            "T": 611,
            "U": 722,
            "V": 667,
            "W": 889,
            "X": 667,
            "Y": 611,
            "Z": 611,
            "[": 333,
            "\\": 278,
            "]": 333,
            "^": 570,
            "_": 500,
            "`": 333,
            "a": 500,
            "b": 500,
            "c": 444,
            "d": 500,
            "e": 444,
            "f": 333,
            "g": 500,
            "h": 556,
            "i": 278,
            "j": 278,
            "k": 500,
            "l": 278,
            "m": 778,
            "n": 556,
            "o": 500,
            "p": 500,
            "q": 500,
            "r": 389,
            "s": 389,
            "t": 278,
            "u": 556,
            "v": 444,
            "w": 667,
            "x": 500,
            "y": 444,
            "z": 389,
            "{": 348,
            "|": 220,
            "}": 348,
            "~": 570,
            "\xa1": 389,
            "\xa2": 500,
            "\xa3": 500,
            "\xa4": 500,
            "\xa5": 500,
            "\xa6": 220,
            "\xa7": 500,
            "\xa8": 333,
            "\xa9": 747,
            "\xaa": 266,
            "\xab": 500,
            "\xac": 606,
            "\xae": 747,
            "\xaf": 333,
            "\xb0": 400,
            "\xb1": 570,
            "\xb2": 300,
            "\xb3": 300,
            "\xb4": 333,
            "\xb5": 576,
            "\xb6": 500,
            "\xb7": 250,
            "\xb8": 333,
            "\xb9": 300,
            "\xba": 300,
            "\xbb": 500,
            "\xbc": 750,
            "\xbd": 750,
            "\xbe": 750,
            "\xbf": 500,
            "\xc0": 667,
            "\xc1": 667,
            "\xc2": 667,
            "\xc3": 667,
            "\xc4": 667,
            "\xc5": 667,
            "\xc6": 944,
            "\xc7": 667,
            "\xc8": 667,
            "\xc9": 667,
            "\xca": 667,
            "\xcb": 667,
            "\xcc": 389,
            "\xcd": 389,
            "\xce": 389,
            "\xcf": 389,
            "\xd0": 722,
            "\xd1": 722,
            "\xd2": 722,
            "\xd3": 722,
            "\xd4": 722,
            "\xd5": 722,
            "\xd6": 722,
            "\xd7": 570,
            "\xd8": 722,
            "\xd9": 722,
            "\xda": 722,
            "\xdb": 722,
            "\xdc": 722,
            "\xdd": 611,
            "\xde": 611,
            "\xdf": 500,
            "\xe0": 500,
            "\xe1": 500,
            "\xe2": 500,
            "\xe3": 500,
            "\xe4": 500,
            "\xe5": 500,
            "\xe6": 722,
            "\xe7": 444,
            "\xe8": 444,
            "\xe9": 444,
            "\xea": 444,
            "\xeb": 444,
            "\xec": 278,
            "\xed": 278,
            "\xee": 278,
            "\xef": 278,
            "\xf0": 500,
            "\xf1": 556,
            "\xf2": 500,
            "\xf3": 500,
            "\xf4": 500,
            "\xf5": 500,
            "\xf6": 500,
            "\xf7": 570,
            "\xf8": 500,
            "\xf9": 556,
            "\xfa": 556,
            "\xfb": 556,
            "\xfc": 556,
            "\xfd": 444,
            "\xfe": 500,
            "\xff": 444,
            "\u0100": 667,
            "\u0101": 500,
            "\u0102": 667,
            "\u0103": 500,
            "\u0104": 667,
            "\u0105": 500,
            "\u0106": 667,
            "\u0107": 444,
            "\u010c": 667,
            "\u010d": 444,
            "\u010e": 722,
            "\u010f": 608,
            "\u0110": 722,
            "\u0111": 500,
            "\u0112": 667,
            "\u0113": 444,
            "\u0116": 667,
            "\u0117": 444,
            "\u0118": 667,
            "\u0119": 444,
            "\u011a": 667,
            "\u011b": 444,
            "\u011e": 722,
            "\u011f": 500,
            "\u0122": 722,
            "\u0123": 500,
            "\u012a": 389,
            "\u012b": 278,
            "\u012e": 389,
            "\u012f": 278,
            "\u0130": 389,
            "\u0131": 278,
            "\u0136": 667,
            "\u0137": 500,
            "\u0139": 611,
            "\u013a": 278,
            "\u013b": 611,
            "\u013c": 278,
            "\u013d": 611,
            "\u013e": 382,
            "\u0141": 611,
            "\u0142": 278,
            "\u0143": 722,
            "\u0144": 556,
            "\u0145": 722,
            "\u0146": 556,
            "\u0147": 722,
            "\u0148": 556,
            "\u014c": 722,
            "\u014d": 500,
            "\u0150": 722,
            "\u0151": 500,
            "\u0152": 944,
            "\u0153": 722,
            "\u0154": 667,
            "\u0155": 389,
            "\u0156": 667,
            "\u0157": 389,
            "\u0158": 667,
            "\u0159": 389,
            "\u015a": 556,
            "\u015b": 389,
            "\u015e": 556,
            "\u015f": 389,
            "\u0160": 556,
            "\u0161": 389,
            "\u0162": 611,
            "\u0163": 278,
            "\u0164": 611,
            "\u0165": 366,
            "\u016a": 722,
            "\u016b": 556,
            "\u016e": 722,
            "\u016f": 556,
            "\u0170": 722,
            "\u0171": 556,
            "\u0172": 722,
            "\u0173": 556,
            "\u0178": 611,
            "\u0179": 611,
            "\u017a": 389,
            "\u017b": 611,
            "\u017c": 389,
            "\u017d": 611,
            "\u017e": 389,
            "\u0192": 500,
            "\u0218": 556,
            "\u0219": 389,
            "\u02c6": 333,
            "\u02c7": 333,
            "\u02d8": 333,
            "\u02d9": 333,
            "\u02da": 333,
            "\u02db": 333,
            "\u02dc": 333,
            "\u02dd": 333,
            "\u2013": 500,
            "\u2014": 1000,
            "\u2018": 333,
            "\u2019": 333,
            "\u201a": 333,
            "\u201c": 500,
            "\u201d": 500,
            "\u201e": 500,
            "\u2020": 500,
            "\u2021": 500,
            "\u2022": 350,
            "\u2026": 1000,
            "\u2030": 1000,
            "\u2039": 333,
            "\u203a": 333,
            "\u2044": 167,
            "\u2122": 1000,
            "\u2202": 494,
            "\u2206": 612,
            "\u2211": 600,
            "\u2212": 606,
            "\u221a": 549,
            "\u2260": 549,
            "\u2264": 549,
            "\u2265": 549,
            "\u25ca": 494,
            "\uf6c3": 250,
            "\ufb01": 556,
            "\ufb02": 556,
        },
    ),
    "Times-Italic": (
        {
            "FontName": "Times-Italic",
            "Descent": -217.0,
            "FontBBox": (-169.0, -217.0, 1010.0, 883.0),
            "FontWeight": "Medium",
            "CapHeight": 653.0,
            "FontFamily": "Times",
            "Flags": 0,
            "XHeight": 441.0,
            "ItalicAngle": -15.5,
            "Ascent": 683.0,
        },
        {
            " ": 250,
            "!": 333,
            '"': 420,
            "#": 500,
            "$": 500,
            "%": 833,
            "&": 778,
            "'": 214,
            "(": 333,
            ")": 333,
            "*": 500,
            "+": 675,
            ",": 250,
            "-": 333,
            ".": 250,
            "/": 278,
            "0": 500,
            "1": 500,
            "2": 500,
            "3": 500,
            "4": 500,
            "5": 500,
            "6": 500,
            "7": 500,
            "8": 500,
            "9": 500,
            ":": 333,
            ";": 333,
            "<": 675,
            "=": 675,
            ">": 675,
            "?": 500,
            "@": 920,
            "A": 611,
            "B": 611,
            "C": 667,
            "D": 722,
            "E": 611,
            "F": 611,
            "G": 722,
            "H": 722,
            "I": 333,
            "J": 444,
            "K": 667,
            "L": 556,
            "M": 833,
            "N": 667,
            "O": 722,
            "P": 611,
            "Q": 722,
            "R": 611,
            "S": 500,
            "T": 556,
            "U": 722,
            "V": 611,
            "W": 833,
            "X": 611,
            "Y": 556,
            "Z": 556,
            "[": 389,
            "\\": 278,
            "]": 389,
            "^": 422,
            "_": 500,
            "`": 333,
            "a": 500,
            "b": 500,
            "c": 444,
            "d": 500,
            "e": 444,
            "f": 278,
            "g": 500,
            "h": 500,
            "i": 278,
            "j": 278,
            "k": 444,
            "l": 278,
            "m": 722,
            "n": 500,
            "o": 500,
            "p": 500,
            "q": 500,
            "r": 389,
            "s": 389,
            "t": 278,
            "u": 500,
            "v": 444,
            "w": 667,
            "x": 444,
            "y": 444,
            "z": 389,
            "{": 400,
            "|": 275,
            "}": 400,
            "~": 541,
            "\xa1": 389,
            "\xa2": 500,
            "\xa3": 500,
            "\xa4": 500,
            "\xa5": 500,
            "\xa6": 275,
            "\xa7": 500,
            "\xa8": 333,
            "\xa9": 760,
            "\xaa": 276,
            "\xab": 500,
            "\xac": 675,
            "\xae": 760,
            "\xaf": 333,
            "\xb0": 400,
            "\xb1": 675,
            "\xb2": 300,
            "\xb3": 300,
            "\xb4": 333,
            "\xb5": 500,
            "\xb6": 523,
            "\xb7": 250,
            "\xb8": 333,
            "\xb9": 300,
            "\xba": 310,
            "\xbb": 500,
            "\xbc": 750,
            "\xbd": 750,
            "\xbe": 750,
            "\xbf": 500,
            "\xc0": 611,
            "\xc1": 611,
            "\xc2": 611,
            "\xc3": 611,
            "\xc4": 611,
            "\xc5": 611,
            "\xc6": 889,
            "\xc7": 667,
            "\xc8": 611,
            "\xc9": 611,
            "\xca": 611,
            "\xcb": 611,
            "\xcc": 333,
            "\xcd": 333,
            "\xce": 333,
            "\xcf": 333,
            "\xd0": 722,
            "\xd1": 667,
            "\xd2": 722,
            "\xd3": 722,
            "\xd4": 722,
            "\xd5": 722,
            "\xd6": 722,
            "\xd7": 675,
            "\xd8": 722,
            "\xd9": 722,
            "\xda": 722,
            "\xdb": 722,
            "\xdc": 722,
            "\xdd": 556,
            "\xde": 611,
            "\xdf": 500,
            "\xe0": 500,
            "\xe1": 500,
            "\xe2": 500,
            "\xe3": 500,
            "\xe4": 500,
            "\xe5": 500,
            "\xe6": 667,
            "\xe7": 444,
            "\xe8": 444,
            "\xe9": 444,
            "\xea": 444,
            "\xeb": 444,
            "\xec": 278,
            "\xed": 278,
            "\xee": 278,
            "\xef": 278,
            "\xf0": 500,
            "\xf1": 500,
            "\xf2": 500,
            "\xf3": 500,
            "\xf4": 500,
            "\xf5": 500,
            "\xf6": 500,
            "\xf7": 675,
            "\xf8": 500,
            "\xf9": 500,
            "\xfa": 500,
            "\xfb": 500,
            "\xfc": 500,
            "\xfd": 444,
            "\xfe": 500,
            "\xff": 444,
            "\u0100": 611,
            "\u0101": 500,
            "\u0102": 611,
            "\u0103": 500,
            "\u0104": 611,
            "\u0105": 500,
            "\u0106": 667,
            "\u0107": 444,
            "\u010c": 667,
            "\u010d": 444,
            "\u010e": 722,
            "\u010f": 544,
            "\u0110": 722,
            "\u0111": 500,
            "\u0112": 611,
            "\u0113": 444,
            "\u0116": 611,
            "\u0117": 444,
            "\u0118": 611,
            "\u0119": 444,
            "\u011a": 611,
            "\u011b": 444,
            "\u011e": 722,
            "\u011f": 500,
            "\u0122": 722,
            "\u0123": 500,
            "\u012a": 333,
            "\u012b": 278,
            "\u012e": 333,
            "\u012f": 278,
            "\u0130": 333,
            "\u0131": 278,
            "\u0136": 667,
            "\u0137": 444,
            "\u0139": 556,
            "\u013a": 278,
            "\u013b": 556,
            "\u013c": 278,
            "\u013d": 611,
            "\u013e": 300,
            "\u0141": 556,
            "\u0142": 278,
            "\u0143": 667,
            "\u0144": 500,
            "\u0145": 667,
            "\u0146": 500,
            "\u0147": 667,
            "\u0148": 500,
            "\u014c": 722,
            "\u014d": 500,
            "\u0150": 722,
            "\u0151": 500,
            "\u0152": 944,
            "\u0153": 667,
            "\u0154": 611,
            "\u0155": 389,
            "\u0156": 611,
            "\u0157": 389,
            "\u0158": 611,
            "\u0159": 389,
            "\u015a": 500,
            "\u015b": 389,
            "\u015e": 500,
            "\u015f": 389,
            "\u0160": 500,
            "\u0161": 389,
            "\u0162": 556,
            "\u0163": 278,
            "\u0164": 556,
            "\u0165": 300,
            "\u016a": 722,
            "\u016b": 500,
            "\u016e": 722,
            "\u016f": 500,
            "\u0170": 722,
            "\u0171": 500,
            "\u0172": 722,
            "\u0173": 500,
            "\u0178": 556,
            "\u0179": 556,
            "\u017a": 389,
            "\u017b": 556,
            "\u017c": 389,
            "\u017d": 556,
            "\u017e": 389,
            "\u0192": 500,
            "\u0218": 500,
            "\u0219": 389,
            "\u02c6": 333,
            "\u02c7": 333,
            "\u02d8": 333,
            "\u02d9": 333,
            "\u02da": 333,
            "\u02db": 333,
            "\u02dc": 333,
            "\u02dd": 333,
            "\u2013": 500,
            "\u2014": 889,
            "\u2018": 333,
            "\u2019": 333,
            "\u201a": 333,
            "\u201c": 556,
            "\u201d": 556,
            "\u201e": 556,
            "\u2020": 500,
            "\u2021": 500,
            "\u2022": 350,
            "\u2026": 889,
            "\u2030": 1000,
            "\u2039": 333,
            "\u203a": 333,
            "\u2044": 167,
            "\u2122": 980,
            "\u2202": 476,
            "\u2206": 612,
            "\u2211": 600,
            "\u2212": 675,
            "\u221a": 453,
            "\u2260": 549,
            "\u2264": 549,
            "\u2265": 549,
            "\u25ca": 471,
            "\uf6c3": 250,
            "\ufb01": 500,
            "\ufb02": 500,
        },
    ),
    "Times-Roman": (
        {
            "FontName": "Times-Roman",
            "Descent": -217.0,
            "FontBBox": (-168.0, -218.0, 1000.0, 898.0),
            "FontWeight": "Roman",
            "CapHeight": 662.0,
            "FontFamily": "Times",
            "Flags": 0,
            "XHeight": 450.0,
            "ItalicAngle": 0.0,
            "Ascent": 683.0,
        },
        {
            " ": 250,
            "!": 333,
            '"': 408,
            "#": 500,
            "$": 500,
            "%": 833,
            "&": 778,
            "'": 180,
            "(": 333,
            ")": 333,
            "*": 500,
            "+": 564,
            ",": 250,
            "-": 333,
            ".": 250,
            "/": 278,
            "0": 500,
            "1": 500,
            "2": 500,
            "3": 500,
            "4": 500,
            "5": 500,
            "6": 500,
            "7": 500,
            "8": 500,
            "9": 500,
            ":": 278,
            ";": 278,
            "<": 564,
            "=": 564,
            ">": 564,
            "?": 444,
            "@": 921,
            "A": 722,
            "B": 667,
            "C": 667,
            "D": 722,
            "E": 611,
            "F": 556,
            "G": 722,
            "H": 722,
            "I": 333,
            "J": 389,
            "K": 722,
            "L": 611,
            "M": 889,
            "N": 722,
            "O": 722,
            "P": 556,
            "Q": 722,
            "R": 667,
            "S": 556,
            "T": 611,
            "U": 722,
            "V": 722,
            "W": 944,
            "X": 722,
            "Y": 722,
            "Z": 611,
            "[": 333,
            "\\": 278,
            "]": 333,
            "^": 469,
            "_": 500,
            "`": 333,
            "a": 444,
            "b": 500,
            "c": 444,
            "d": 500,
            "e": 444,
            "f": 333,
            "g": 500,
            "h": 500,
            "i": 278,
            "j": 278,
            "k": 500,
            "l": 278,
            "m": 778,
            "n": 500,
            "o": 500,
            "p": 500,
            "q": 500,
            "r": 333,
            "s": 389,
            "t": 278,
            "u": 500,
            "v": 500,
            "w": 722,
            "x": 500,
            "y": 500,
            "z": 444,
            "{": 480,
            "|": 200,
            "}": 480,
            "~": 541,
            "\xa1": 333,
            "\xa2": 500,
            "\xa3": 500,
            "\xa4": 500,
            "\xa5": 500,
            "\xa6": 200,
            "\xa7": 500,
            "\xa8": 333,
            "\xa9": 760,
            "\xaa": 276,
            "\xab": 500,
            "\xac": 564,
            "\xae": 760,
            "\xaf": 333,
            "\xb0": 400,
            "\xb1": 564,
            "\xb2": 300,
            "\xb3": 300,
            "\xb4": 333,
            "\xb5": 500,
            "\xb6": 453,
            "\xb7": 250,
            "\xb8": 333,
            "\xb9": 300,
            "\xba": 310,
            "\xbb": 500,
            "\xbc": 750,
            "\xbd": 750,
            "\xbe": 750,
            "\xbf": 444,
            "\xc0": 722,
            "\xc1": 722,
            "\xc2": 722,
            "\xc3": 722,
            "\xc4": 722,
            "\xc5": 722,
            "\xc6": 889,
            "\xc7": 667,
            "\xc8": 611,
            "\xc9": 611,
            "\xca": 611,
            "\xcb": 611,
            "\xcc": 333,
            "\xcd": 333,
            "\xce": 333,
            "\xcf": 333,
            "\xd0": 722,
            "\xd1": 722,
            "\xd2": 722,
            "\xd3": 722,
            "\xd4": 722,
            "\xd5": 722,
            "\xd6": 722,
            "\xd7": 564,
            "\xd8": 722,
            "\xd9": 722,
            "\xda": 722,
            "\xdb": 722,
            "\xdc": 722,
            "\xdd": 722,
            "\xde": 556,
            "\xdf": 500,
            "\xe0": 444,
            "\xe1": 444,
            "\xe2": 444,
            "\xe3": 444,
            "\xe4": 444,
            "\xe5": 444,
            "\xe6": 667,
            "\xe7": 444,
            "\xe8": 444,
            "\xe9": 444,
            "\xea": 444,
            "\xeb": 444,
            "\xec": 278,
            "\xed": 278,
            "\xee": 278,
            "\xef": 278,
            "\xf0": 500,
            "\xf1": 500,
            "\xf2": 500,
            "\xf3": 500,
            "\xf4": 500,
            "\xf5": 500,
            "\xf6": 500,
            "\xf7": 564,
            "\xf8": 500,
            "\xf9": 500,
            "\xfa": 500,
            "\xfb": 500,
            "\xfc": 500,
            "\xfd": 500,
            "\xfe": 500,
            "\xff": 500,
            "\u0100": 722,
            "\u0101": 444,
            "\u0102": 722,
            "\u0103": 444,
            "\u0104": 722,
            "\u0105": 444,
            "\u0106": 667,
            "\u0107": 444,
            "\u010c": 667,
            "\u010d": 444,
            "\u010e": 722,
            "\u010f": 588,
            "\u0110": 722,
            "\u0111": 500,
            "\u0112": 611,
            "\u0113": 444,
            "\u0116": 611,
            "\u0117": 444,
            "\u0118": 611,
            "\u0119": 444,
            "\u011a": 611,
            "\u011b": 444,
            "\u011e": 722,
            "\u011f": 500,
            "\u0122": 722,
            "\u0123": 500,
            "\u012a": 333,
            "\u012b": 278,
            "\u012e": 333,
            "\u012f": 278,
            "\u0130": 333,
            "\u0131": 278,
            "\u0136": 722,
            "\u0137": 500,
            "\u0139": 611,
            "\u013a": 278,
            "\u013b": 611,
            "\u013c": 278,
            "\u013d": 611,
            "\u013e": 344,
            "\u0141": 611,
            "\u0142": 278,
            "\u0143": 722,
            "\u0144": 500,
            "\u0145": 722,
            "\u0146": 500,
            "\u0147": 722,
            "\u0148": 500,
            "\u014c": 722,
            "\u014d": 500,
            "\u0150": 722,
            "\u0151": 500,
            "\u0152": 889,
            "\u0153": 722,
            "\u0154": 667,
            "\u0155": 333,
            "\u0156": 667,
            "\u0157": 333,
            "\u0158": 667,
            "\u0159": 333,
            "\u015a": 556,
            "\u015b": 389,
            "\u015e": 556,
            "\u015f": 389,
            "\u0160": 556,
            "\u0161": 389,
            "\u0162": 611,
            "\u0163": 278,
            "\u0164": 611,
            "\u0165": 326,
            "\u016a": 722,
            "\u016b": 500,
            "\u016e": 722,
            "\u016f": 500,
            "\u0170": 722,
            "\u0171": 500,
            "\u0172": 722,
            "\u0173": 500,
            "\u0178": 722,
            "\u0179": 611,
            "\u017a": 444,
            "\u017b": 611,
            "\u017c": 444,
            "\u017d": 611,
            "\u017e": 444,
            "\u0192": 500,
            "\u0218": 556,
            "\u0219": 389,
            "\u02c6": 333,
            "\u02c7": 333,
            "\u02d8": 333,
            "\u02d9": 333,
            "\u02da": 333,
            "\u02db": 333,
            "\u02dc": 333,
            "\u02dd": 333,
            "\u2013": 500,
            "\u2014": 1000,
            "\u2018": 333,
            "\u2019": 333,
            "\u201a": 333,
            "\u201c": 444,
            "\u201d": 444,
            "\u201e": 444,
            "\u2020": 500,
            "\u2021": 500,
            "\u2022": 350,
            "\u2026": 1000,
            "\u2030": 1000,
            "\u2039": 333,
            "\u203a": 333,
            "\u2044": 167,
            "\u2122": 980,
            "\u2202": 476,
            "\u2206": 612,
            "\u2211": 600,
            "\u2212": 564,
            "\u221a": 453,
            "\u2260": 549,
            "\u2264": 549,
            "\u2265": 549,
            "\u25ca": 471,
            "\uf6c3": 250,
            "\ufb01": 556,
            "\ufb02": 556,
        },
    ),
    "ZapfDingbats": (
        {
            "FontName": "ZapfDingbats",
            "FontBBox": (-1.0, -143.0, 981.0, 820.0),
            "FontWeight": "Medium",
            "FontFamily": "ITC",
            "Flags": 0,
            "ItalicAngle": 0.0,
        },
        {
            "\x01": 974,
            "\x02": 961,
            "\x03": 980,
            "\x04": 719,
            "\x05": 789,
            "\x06": 494,
            "\x07": 552,
            "\x08": 537,
            "\t": 577,
            "\n": 692,
            "\x0b": 960,
            "\x0c": 939,
            "\r": 549,
            "\x0e": 855,
            "\x0f": 911,
            "\x10": 933,
            "\x11": 945,
            "\x12": 974,
            "\x13": 755,
            "\x14": 846,
            "\x15": 762,
            "\x16": 761,
            "\x17": 571,
            "\x18": 677,
            "\x19": 763,
            "\x1a": 760,
            "\x1b": 759,
            "\x1c": 754,
            "\x1d": 786,
            "\x1e": 788,
            "\x1f": 788,
            " ": 790,
            "!": 793,
            '"': 794,
            "#": 816,
            "$": 823,
            "%": 789,
            "&": 841,
            "'": 823,
            "(": 833,
            ")": 816,
            "*": 831,
            "+": 923,
            ",": 744,
            "-": 723,
            ".": 749,
            "/": 790,
            "0": 792,
            "1": 695,
            "2": 776,
            "3": 768,
            "4": 792,
            "5": 759,
            "6": 707,
            "7": 708,
            "8": 682,
            "9": 701,
            ":": 826,
            ";": 815,
            "<": 789,
            "=": 789,
            ">": 707,
            "?": 687,
            "@": 696,
            "A": 689,
            "B": 786,
            "C": 787,
            "D": 713,
            "E": 791,
            "F": 785,
            "G": 791,
            "H": 873,
            "I": 761,
            "J": 762,
            "K": 759,
            "L": 892,
            "M": 892,
            "N": 788,
            "O": 784,
            "Q": 438,
            "R": 138,
            "S": 277,
            "T": 415,
            "U": 509,
            "V": 410,
            "W": 234,
            "X": 234,
            "Y": 390,
            "Z": 390,
            "[": 276,
            "\\": 276,
            "]": 317,
            "^": 317,
            "_": 334,
            "`": 334,
            "a": 392,
            "b": 392,
            "c": 668,
            "d": 668,
            "e": 732,
            "f": 544,
            "g": 544,
            "h": 910,
            "i": 911,
            "j": 667,
            "k": 760,
            "l": 760,
            "m": 626,
            "n": 694,
            "o": 595,
            "p": 776,
            "u": 690,
            "v": 791,
            "w": 790,
            "x": 788,
            "y": 788,
            "z": 788,
            "{": 788,
            "|": 788,
            "}": 788,
            "~": 788,
            "\x7f": 788,
            "\x80": 788,
            "\x81": 788,
            "\x82": 788,
            "\x83": 788,
            "\x84": 788,
            "\x85": 788,
            "\x86": 788,
            "\x87": 788,
            "\x88": 788,
            "\x89": 788,
            "\x8a": 788,
            "\x8b": 788,
            "\x8c": 788,
            "\x8d": 788,
            "\x8e": 788,
            "\x8f": 788,
            "\x90": 788,
            "\x91": 788,
            "\x92": 788,
            "\x93": 788,
            "\x94": 788,
            "\x95": 788,
            "\x96": 788,
            "\x97": 788,
            "\x98": 788,
            "\x99": 788,
            "\x9a": 788,
            "\x9b": 788,
            "\x9c": 788,
            "\x9d": 788,
            "\x9e": 788,
            "\x9f": 788,
            "\xa0": 894,
            "\xa1": 838,
            "\xa2": 924,
            "\xa3": 1016,
            "\xa4": 458,
            "\xa5": 924,
            "\xa6": 918,
            "\xa7": 927,
            "\xa8": 928,
            "\xa9": 928,
            "\xaa": 834,
            "\xab": 873,
            "\xac": 828,
            "\xad": 924,
            "\xae": 917,
            "\xaf": 930,
            "\xb0": 931,
            "\xb1": 463,
            "\xb2": 883,
            "\xb3": 836,
            "\xb4": 867,
            "\xb5": 696,
            "\xb6": 874,
            "\xb7": 760,
            "\xb8": 946,
            "\xb9": 865,
            "\xba": 967,
            "\xbb": 831,
            "\xbc": 873,
            "\xbd": 927,
            "\xbe": 970,
            "\xbf": 918,
            "\xc0": 748,
            "\xc1": 836,
            "\xc2": 771,
            "\xc3": 888,
            "\xc4": 748,
            "\xc5": 771,
            "\xc6": 888,
            "\xc7": 867,
            "\xc8": 696,
            "\xc9": 874,
            "\xca": 974,
            "\xcb": 762,
            "\xcc": 759,
            "\xcd": 509,
            "\xce": 410,
        },
    ),
}

# Aliases defined in implementation note 62 in Appecix H. related to section 5.5.1
# (Type 1 Fonts) in the PDF Reference.
FONT_METRICS["Arial"] = FONT_METRICS["Helvetica"]
FONT_METRICS["Arial,Italic"] = FONT_METRICS["Helvetica-Oblique"]
FONT_METRICS["Arial,Bold"] = FONT_METRICS["Helvetica-Bold"]
FONT_METRICS["Arial,BoldItalic"] = FONT_METRICS["Helvetica-BoldOblique"]
FONT_METRICS["CourierNew"] = FONT_METRICS["Courier"]
FONT_METRICS["CourierNew,Italic"] = FONT_METRICS["Courier-Oblique"]
FONT_METRICS["CourierNew,Bold"] = FONT_METRICS["Courier-Bold"]
FONT_METRICS["CourierNew,BoldItalic"] = FONT_METRICS["Courier-BoldOblique"]
FONT_METRICS["TimesNewRoman"] = FONT_METRICS["Times-Roman"]
FONT_METRICS["TimesNewRoman,Italic"] = FONT_METRICS["Times-Italic"]
FONT_METRICS["TimesNewRoman,Bold"] = FONT_METRICS["Times-Bold"]
FONT_METRICS["TimesNewRoman,BoldItalic"] = FONT_METRICS["Times-BoldItalic"]
