package jp.gr.java_conf.ykhr.csvutil;

import java.beans.IntrospectionException;
import java.beans.PropertyDescriptor;
import java.io.BufferedOutputStream;
import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.OutputStream;
import java.lang.reflect.InvocationTargetException;
import java.util.Collection;
import java.util.Map;
import java.util.TreeMap;

import jp.gr.java_conf.ykhr.csvutil.config.WriteConfig;
import jp.gr.java_conf.ykhr.csvutil.escaper.DefaultEscaper;
import jp.gr.java_conf.ykhr.csvutil.escaper.Escaper;
import jp.gr.java_conf.ykhr.csvutil.ex.CSVWriteException;


public class CSVWriter {

    private static final Escaper DEFAULT_ESCAPER = new DefaultEscaper();
    private static final Object[] EMPTY = new Object[0];
    
    private Escaper escaper;
    
    public CSVWriter() {
        this(DEFAULT_ESCAPER);
    }

    public CSVWriter(Escaper escaper) {
        this.escaper = escaper;
    }
    
    public <T>void write(File outputFile, Collection<T> elements, Class<T> clazz) 
    throws CSVWriteException {
        write(outputFile, elements, clazz, new WriteConfig());
    }
    
    public <T>void write(File outputFile, Collection<T> elements, Class<T> clazz, WriteConfig config) 
    throws CSVWriteException {
        
        OutputStream out = null;
        try {
            createOutputDir(outputFile);
            out = new BufferedOutputStream(new FileOutputStream(outputFile));
            write(out, elements, clazz, config);
        } catch (IOException e) {
            throw new CSVWriteException(e);
        } finally {
            if (out != null) {
                try {
                    out.close();
                } catch (IOException ignore) {
                }
            }
        }
    }
    
    public <T>void write(OutputStream out, Collection<?> elements, Class<T> clazz, WriteConfig config) 
    throws CSVWriteException {
        try {
            Map<CSVElement, PropertyDescriptor> csvToProperty
                = new TreeMap<CSVElement, PropertyDescriptor>(Utils.PRIORITY_COMP);
            csvToProperty.putAll(Utils.createCsvToPropertyInfo(clazz));
            
            String encoding = config.getEncoding();
            byte[] lineEnding = config.getLineEnding().getBytes(encoding);
            byte[] separator = config.getSeparator().getBytes(encoding);
            
            int fieldSize = csvToProperty.size();
            int wirteElementCount = 1;
            
            // output header
            if (config.isWriteHeader()) {
                for (CSVElement header : csvToProperty.keySet()) {
                    String writeHeader = toWriterString(header.name(), config);
                    out.write(writeHeader.getBytes(encoding));
                    
                    if (wirteElementCount++ < fieldSize) {
                        out.write(separator);
                    }
                }
                out.write(lineEnding);
            }
            
            // output element
            int writeRowCount = 0;
            for (Object element : elements) {
                
                wirteElementCount = 1;
                for (PropertyDescriptor desc : csvToProperty.values()) {
                    Object value = desc.getReadMethod().invoke(element, EMPTY);
                    String writeValue = toWriterString(value, config);
                    out.write(writeValue.getBytes(encoding));
                    
                    if (wirteElementCount++ < fieldSize) {
                        out.write(separator);
                    }
                }
                
                if (writeRowCount++ < fieldSize) {
                    out.write(lineEnding);
                }
            }
            out.flush();
            
        } catch (IntrospectionException e) {
            throw new CSVWriteException(e);
        } catch (IOException e) {
            throw new CSVWriteException(e);
        } catch (InvocationTargetException e) {
            throw new CSVWriteException(e);
        } catch (IllegalAccessException e) {
            throw new CSVWriteException(e);
        }  
    }
    
    protected String toWriterString(Object value, WriteConfig config) {
        String str = escaper.toStringValue(value);
        StringBuilder sb = new StringBuilder(str.length() + 10);
        
        boolean escaping = false;
        if (escaping = escaper.needEscape(str)) {
            sb.append(escaper.escape(str));
        } else {
            sb.append(str);
        }
        
        if (escaper.needQuote(str, escaping) 
                || config.isQuoteAlways()) {
            sb.insert(0, config.getQuoteString());
            sb.append(config.getQuoteString());
        }
        
        return sb.toString();
    }
    
    private void createOutputDir(File file) throws IOException {
        File parent = file.getCanonicalFile().getParentFile();
        if (!parent.exists()) {
            parent.mkdirs();
        }
    }

}
