//! @file		pf_timer.c
//! @brief		プラットフォーム(Timer)実装ファイル

// The MIT License (MIT)
// Copyright (c) 2023 @xm6_original
//
// Permission is hereby granted, free of charge, to any person obtaining a
// copy of this software and associated documentation files (the "Software"),
// to deal in the Software without restriction, including without limitation
// the rights to use, copy, modify, merge, publish, distribute, sublicense,
// and/or sell copies of the Software, and to permit persons to whom the
// Software is furnished to do so, subject to the following conditions:
//
// The above copyright notice and this permission notice shall be included in
// all copies or substantial portions of the Software.
//
// THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
// IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
// FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL
// THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
// LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
// FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER
// DEALINGS IN THE SOFTWARE.

#include "pf_types.h"
#include "nrf52833.h"
#include "nrf52833_bitfields.h"
#include "pf_interrupt.h"
#include "pf_timer.h"

//! @brief		Timerチャネル数
#define PF_TIMER_CHANNEL_MAX		((u4)5U)

//! @brief		コンペア最大数(チャネル0～チャネル4で共通)
#define PF_TIMER_COMPARE_MAX		((u4)4U)

//! @brief		Timer動作
typedef enum PF_TIMER_ACTION_Tag
{
	PF_TIMER_ACTION_ONESHOT,					//!< ワンショット動作
	PF_TIMER_ACTION_INTERVAL,					//!< インターバル動作
	PF_TIMER_ACTION_CAPTURE,					//!< キャプチャ動作
} PF_TIMER_ACTION;

//! @brief		Timer初期化情報構造体
typedef struct PF_TIMER_INIT_Tag
{
	u4				channel;					//!< チャネル(0～4)
	NRF_TIMER_Type*	dev;						//!< デバイス
	PF_INTERRUPT_PRI priority;					//!< 割り込み優先度
	PF_TIMER_ACTION	action;						//!< Timer動作
	u4				compares;					//!< コンペア数(1～4)
	u4				cc0;						//!< コンペア値(0)
	u4				cc1;						//!< コンペア値(1)
	u4				cc2;						//!< コンペア値(2)
	u4				cc3;						//!< コンペア値(3)
} PF_TIMER_INIT;

//! @brief		Timer初期化情報テーブル
const PF_TIMER_INIT pf_timer_table[PF_TIMER_ID_MAX] =
{
	// PF_TIMER_ID_DISPLAY
	{
		0,										// チャネル
		NRF_TIMER0,								// デバイス
		PF_INTERRUPT_PRI_DISPLAY,				// 割り込み優先度
		PF_TIMER_ACTION_INTERVAL,				// Timer動作
		2,										// コンペア数
		MICROSEC_TO_TIMERCC(800),				// コンペア値(0)
		MICROSEC_TO_TIMERCC(1000),				// コンペア値(1)
		0,										// コンペア値(2)
		0,										// コンペア値(3)
	},

	// PF_TIMER_ID_MUSIC
	{
		1,										// チャネル
		NRF_TIMER1,								// デバイス
		PF_INTERRUPT_PRI_MUSIC,					// 割り込み優先度
		PF_TIMER_ACTION_INTERVAL,				// Timer動作
		1,										// コンペア数
		MICROSEC_TO_TIMERCC(12500),				// コンペア値(0)
		0,										// コンペア値(1)
		0,										// コンペア値(2)
		0,										// コンペア値(3)
	},

	// PF_TIMER_ID_ULTRA
	{
		2,										// チャネル
		NRF_TIMER2,								// デバイス
		0,										// 割り込み優先度
		PF_TIMER_ACTION_CAPTURE,				// Timer動作
		0,										// コンペア数
		0,										// コンペア値(0)
		0,										// コンペア値(1)
		0,										// コンペア値(2)
		0,										// コンペア値(3)
	},
};

//! @brief		compares→CLEARビットテーブル
static const u4 pf_timer_clear_table[PF_TIMER_COMPARE_MAX + 1] =
{
	0,
	TIMER_SHORTS_COMPARE0_CLEAR_Enabled << TIMER_SHORTS_COMPARE0_CLEAR_Pos,
	TIMER_SHORTS_COMPARE1_CLEAR_Enabled << TIMER_SHORTS_COMPARE1_CLEAR_Pos,
	TIMER_SHORTS_COMPARE2_CLEAR_Enabled << TIMER_SHORTS_COMPARE2_CLEAR_Pos,
	TIMER_SHORTS_COMPARE3_CLEAR_Enabled << TIMER_SHORTS_COMPARE3_CLEAR_Pos,
};

//! @brief		compares→STOPビットテーブル
static const u4 pf_timer_stop_table[PF_TIMER_COMPARE_MAX + 1] =
{
	0,
	TIMER_SHORTS_COMPARE0_STOP_Enabled << TIMER_SHORTS_COMPARE0_STOP_Pos,
	TIMER_SHORTS_COMPARE1_STOP_Enabled << TIMER_SHORTS_COMPARE1_STOP_Pos,
	TIMER_SHORTS_COMPARE2_STOP_Enabled << TIMER_SHORTS_COMPARE2_STOP_Pos,
	TIMER_SHORTS_COMPARE3_STOP_Enabled << TIMER_SHORTS_COMPARE3_STOP_Pos,
};

//! @brief		Timer動作情報構造体
typedef struct PF_TIMER_INFO_Tag
{
	PF_TIMER_CALLBACK	callback;				//!< コールバック関数
	u4					cc[4];					//!< コンペア値
	BOOL				running;				//!< 動作中フラグ
} PF_TIMER_INFO;

//! @brief		Timer動作情報テーブル
static PF_TIMER_INFO pf_timer_info[PF_TIMER_ID_MAX];

//! @brief		INTENSETレジスタへの設定値を得る
//! @param		[in] id			TimerのID
//! @return		INTENSETレジスタへの設定値
static u4 pf_timer_get_intenset(PF_TIMER_ID id)
{
	u4 intenset;

	// オート変数初期化
	intenset = 0;

	// コンペア0あり?
	if (pf_timer_table[id].compares >= 1)
	{
		intenset |= TIMER_INTENSET_COMPARE0_Set << TIMER_INTENSET_COMPARE0_Pos;
	}

	// コンペア1あり?
	if (pf_timer_table[id].compares >= 2)
	{
		intenset |= TIMER_INTENSET_COMPARE1_Set << TIMER_INTENSET_COMPARE1_Pos;
	}

	// コンペア2あり?
	if (pf_timer_table[id].compares >= 3)
	{
		intenset |= TIMER_INTENSET_COMPARE2_Set << TIMER_INTENSET_COMPARE2_Pos;
	}

	// コンペア3あり?
	if (pf_timer_table[id].compares >= 4)
	{
		intenset |= TIMER_INTENSET_COMPARE3_Set << TIMER_INTENSET_COMPARE3_Pos;
	}

	return intenset;
}

//! @brief		Timer初期化(単一のID)
//! @param		[in] id			TimerのID
static void pf_timer_init_id(PF_TIMER_ID id)
{
	// タイマ停止
	pf_timer_table[id].dev->TASKS_STOP = TIMER_TASKS_STOP_TASKS_STOP_Trigger
					<< TIMER_TASKS_STOP_TASKS_STOP_Pos;

	// イベントのショートカット
	switch (pf_timer_table[id].action)
	{
	// ワンショット動作
	case PF_TIMER_ACTION_ONESHOT:
		// COMPARE[i]に達したらSTOP
		pf_timer_table[id].dev->SHORTS = pf_timer_stop_table[pf_timer_table[id].compares];
		break;

	// インターバル動作
	case PF_TIMER_ACTION_INTERVAL:
		// COMPARE[i]に達したらCLEAR
		pf_timer_table[id].dev->SHORTS = pf_timer_clear_table[pf_timer_table[id].compares];
		break;

	// その他(キャプチャ動作)
	default:
		pf_timer_table[id].dev->SHORTS = 0;
		break;
	}

	// 割り込み有効
	pf_timer_table[id].dev->INTENSET = pf_timer_get_intenset(id);

	// MODE(Timer固定)
	pf_timer_table[id].dev->MODE = TIMER_MODE_MODE_Timer << TIMER_MODE_MODE_Pos;

	// BITMODE(32bit幅固定)
	pf_timer_table[id].dev->BITMODE = TIMER_BITMODE_BITMODE_32Bit << TIMER_BITMODE_BITMODE_Pos;

	// PRESCALER(1/16固定)クロックソースはPCLK1MHzとなり1[us]ごとにカウントアップする
	pf_timer_table[id].dev->PRESCALER = 4U;

	// CC[0]設定
	if (pf_timer_table[id].compares >= 1)
	{
		pf_timer_table[id].dev->CC[0] = pf_timer_table[id].cc0;
	}

	// CC[1]設定
	if (pf_timer_table[id].compares >= 2)
	{
		pf_timer_table[id].dev->CC[1] = pf_timer_table[id].cc1;
	}

	// CC[2]設定
	if (pf_timer_table[id].compares >= 3)
	{
		pf_timer_table[id].dev->CC[2] = pf_timer_table[id].cc2;
	}

	// CC[3]設定
	if (pf_timer_table[id].compares >= 4)
	{
		pf_timer_table[id].dev->CC[3] = pf_timer_table[id].cc3;
	}

	// 動作情報テーブルを初期化
	pf_timer_info[id].callback = NULL;
	pf_timer_info[id].cc[0] = pf_timer_table[id].cc0;
	pf_timer_info[id].cc[1] = pf_timer_table[id].cc1;
	pf_timer_info[id].cc[2] = pf_timer_table[id].cc2;
	pf_timer_info[id].cc[3] = pf_timer_table[id].cc3;
	pf_timer_info[id].running = FALSE;

	// キャプチャモード以外は割り込みを使用する
	if (PF_TIMER_ACTION_CAPTURE != pf_timer_table[id].action)
	{
		// 割り込みセットアップ
		pf_interrupt_local_setup(pf_timer_table[id].priority);
	}
}

//! @brief		Timer初期化
//! @remarks	プラットフォーム初期化処理から呼び出すこと
void pf_timer_init(void)
{
	PF_TIMER_ID id;

	// オート変数初期化
	id = 0;

	// すべてのIDをループ
	for (id = 0; id < PF_TIMER_ID_MAX; id++)
	{
		// 1つのIDを初期化
		pf_timer_init_id(id);
	}
}

//! @brief		Timer開始
//! @param		[in] id			TimerのID
void pf_timer_start(PF_TIMER_ID id)
{
	// IDチェック
	if (id < PF_TIMER_ID_MAX)
	{
		// 開始に先立って、クリア
		pf_timer_table[id].dev->TASKS_CLEAR =
		TIMER_TASKS_CLEAR_TASKS_CLEAR_Trigger << TIMER_TASKS_CLEAR_TASKS_CLEAR_Pos;

		// 開始
		pf_timer_table[id].dev->TASKS_START =
		TIMER_TASKS_START_TASKS_START_Trigger << TIMER_TASKS_START_TASKS_START_Pos;

		// 動作中フラグセット(割り込み干渉を考慮する必要はない)
		pf_timer_info[id].running = TRUE;
	}
}

//! @brief		Timer停止
//! @param		[in] id			TimerのID
void pf_timer_stop(PF_TIMER_ID id)
{
	// IDチェック
	if (id < PF_TIMER_ID_MAX)
	{
		// 停止
		pf_timer_table[id].dev->TASKS_STOP = TIMER_TASKS_STOP_TASKS_STOP_Trigger
						<< TIMER_TASKS_STOP_TASKS_STOP_Pos;

		// 動作中フラグクリア(割り込み干渉を考慮する必要はない)
		pf_timer_info[id].running = FALSE;
	}
}

//! @brief		Timerコールバック関数設定
//! @param		[in] id			TimerのID
//! @param		[in] func		Timerコールバック関数へのポインタ
//! @attention	Timerコールバック関数は割り込みコンテキストで呼び出される
void pf_timer_callback(PF_TIMER_ID id, PF_TIMER_CALLBACK func)
{
	u4 enable;

	// オート変数初期化
	enable = 0;

	// IDチェック
	if (id < PF_TIMER_ID_MAX)
	{
		// 割り込み禁止
		enable = pf_interrupt_local_disable(pf_timer_table[id].priority);

		// コールバック関数設定
		pf_timer_info[id].callback = func;

		// 割り込み復元
		pf_interrupt_local_restore(pf_timer_table[id].priority, enable);
	}
}

//! @brief		Timerコンペアカウンタ設定
//! @param		[in] id			TimerのID
//! @param		[in] cc			コンペアカウンタ配列
void pf_timer_cc(PF_TIMER_ID id, u4 *cc)
{
	u4 enable;
	u4 loop;

	// オート変数初期化
	enable = 0;
	loop = 0;

	// パラメータチェック
	if ((id < PF_TIMER_ID_MAX) && (NULL != cc))
	{
		// 割り込み禁止
		enable = pf_interrupt_local_disable(pf_timer_table[id].priority);

		// CCループ
		for (loop = 0; loop < pf_timer_table[id].compares; loop++)
		{
			// 異なっている要素だけ設定する
			if (pf_timer_info[id].cc[loop] != cc[loop])
			{
				pf_timer_info[id].cc[loop] = cc[loop];
				pf_timer_table[id].dev->CC[loop] = pf_timer_info[id].cc[loop];
			}
		}

		// 割り込み復元
		pf_interrupt_local_restore(pf_timer_table[id].priority, enable);
	}
}

//! @brief		Timerキャプチャカウンタを取得
//! @param		[in] id			TimerのID
//! @return		キャプチャカウンタ
u4 pf_timer_get_capture(PF_TIMER_ID id)
{
	u4 capture;

	// オート変数初期化
	capture = 0;

	// パラメータチェック
	if (id < PF_TIMER_ID_MAX)
	{
		// キャプチャ値を取得
		capture = pf_timer_table[id].dev->CC[0];

		// 次回に備え、停止・クリア・CC[0]クリアを行う(割り込みは考慮しなくてよい)
		pf_timer_table[id].dev->TASKS_STOP = TIMER_TASKS_STOP_TASKS_STOP_Trigger
						<< TIMER_TASKS_STOP_TASKS_STOP_Pos;
		pf_timer_table[id].dev->TASKS_CLEAR = TIMER_TASKS_CLEAR_TASKS_CLEAR_Trigger
						<< TIMER_TASKS_CLEAR_TASKS_CLEAR_Pos;
		pf_timer_table[id].dev->CC[0] = 0;
	}

	return capture;
}

//! @brief		Timerスタートアドレス取得
//! @param		[in] id			TimerのID
//! @return		TASKS_STARTレジスタのアドレス
u4 pf_timer_get_tasks_start(PF_TIMER_ID id)
{
	u4 addr;

	// オート変数初期化
	addr = 0;

	// パラメータチェック
	if (id < PF_TIMER_ID_MAX)
	{
		addr = (u4)&(pf_timer_table[id].dev->TASKS_START);
	}

	return addr;
}

//! @brief		Timerキャプチャイベントアドレス取得
//! @param		[in] id			TimerのID
//! @return		TASKS_CAPTURE[0]レジスタのアドレス
u4 pf_timer_get_tasks_capture(PF_TIMER_ID id)
{
	u4 addr;

	// オート変数初期化
	addr = 0;

	// パラメータチェック
	if (id < PF_TIMER_ID_MAX)
	{
		addr = (u4)&(pf_timer_table[id].dev->TASKS_CAPTURE[0]);
	}

	return addr;
}

//! @brief		共通割り込みハンドラ
//! @param		[in] channel	Timerチャネル(0～4)
static void pf_timer_isr(u4 channel)
{
	PF_TIMER_ID id;
	u4 compare;
	u4 mask;
	u4 generated;
	u4 clear;

	// オート変数初期化
	id = 0;
	compare = 0;
	mask = TIMER_EVENTS_COMPARE_EVENTS_COMPARE_Msk;
	generated = TIMER_EVENTS_COMPARE_EVENTS_COMPARE_Generated
					<< TIMER_EVENTS_COMPARE_EVENTS_COMPARE_Pos;
	clear = TIMER_EVENTS_COMPARE_EVENTS_COMPARE_NotGenerated
					<< TIMER_EVENTS_COMPARE_EVENTS_COMPARE_Pos;

	// チャネル番号からIDを特定する
	for (id = 0; id < PF_TIMER_ID_MAX; id++)
	{
		if (channel == pf_timer_table[id].channel)
		{
			break;
		}
	}

	// IDが見つかった場合のみ処理
	if (id < PF_TIMER_ID_MAX)
	{
		// EVENTS_COMPARE[3]レジスタからコンペア情報を作成
		compare <<= 1;
		if (generated == (pf_timer_table[id].dev->EVENTS_COMPARE[3] & mask))
		{
			compare |= 1U;
			pf_timer_table[id].dev->EVENTS_COMPARE[3] = clear;
		}

		// EVENTS_COMPARE[2]レジスタからコンペア情報を作成
		compare <<= 1;
		if (generated == (pf_timer_table[id].dev->EVENTS_COMPARE[2] & mask))
		{
			compare |= 1U;
			pf_timer_table[id].dev->EVENTS_COMPARE[2] = clear;
		}

		// EVENTS_COMPARE[1]レジスタからコンペア情報を作成
		compare <<= 1;
		if (generated == (pf_timer_table[id].dev->EVENTS_COMPARE[1] & mask))
		{
			compare |= 1U;
			pf_timer_table[id].dev->EVENTS_COMPARE[1] = clear;
		}

		// EVENTS_COMPARE[0]レジスタからコンペア情報を作成
		compare <<= 1;
		if (generated == (pf_timer_table[id].dev->EVENTS_COMPARE[0] & mask))
		{
			compare |= 1U;
			pf_timer_table[id].dev->EVENTS_COMPARE[0] = clear;
		}

		// Timerコールバック関数が登録されていれば、呼び出す
		if (NULL != pf_timer_info[id].callback)
		{
			pf_timer_info[id].callback(compare);
		}
	}
}

//! @brief		Timer0割り込みハンドラ
//! @attention	データ競合(割り込み干渉)に注意する
void TIMER0_IRQHandler(void)
{
	pf_timer_isr(0);
}

//! @brief		Timer1割り込みハンドラ
//! @attention	データ競合(割り込み干渉)に注意する
void TIMER1_IRQHandler(void)
{
	pf_timer_isr(1);
}

//! @brief		Timer2割り込みハンドラ
//! @attention	データ競合(割り込み干渉)に注意する
void TIMER2_IRQHandler(void)
{
	pf_timer_isr(2);
}

//! @brief		Timer3割り込みハンドラ
//! @attention	データ競合(割り込み干渉)に注意する
void TIMER3_IRQHandler(void)
{
	pf_timer_isr(3);
}

//! @brief		Timer4割り込みハンドラ
//! @attention	データ競合(割り込み干渉)に注意する
void TIMER4_IRQHandler(void)
{
	pf_timer_isr(4);
}
