
/* --------------------------------------------- */
/*  H8-3069F USB Host function                   */
/*                (c) KAZ.Imamura                */
/* --------------------------------------------- */

#include "sl811.h"
#include "sc1602.h"

#define SL811_ADDRREG	(*(volatile unsigned char *)0x200000)
#define SL811_DATAREG	(*(volatile unsigned char *)0x200002)
#define SUB_PROC_RESULT	unsigned char

// -------------------------------------------
//  Proto type definitions
// -------------------------------------------
// Global
void sl811_1ms_handler(void);
void sl811_chip_irq(void);
int sl811_initialize(void);
int sl811_process(void);
int ui_function_usb(UI_COMMAND uicmd);
unsigned char sl811_status(unsigned char req);
unsigned char sl811_ctrl_receive(P_SETUP_FORMAT p_fmt);
void sl811_buffer_copy( unsigned char* p_buf, int size);
unsigned char sl811_bulk_out_request( unsigned char* p_buf, int size );
unsigned char sl811_bulk_in_request( unsigned char* p_buf, int size );

// Locals
static unsigned char sl811_read_reg(unsigned char reg);
static void sl811_write_reg(unsigned char reg, unsigned char data);
static void sl811_read_buf(unsigned char reg, unsigned char* buf, short size);
static void sl811_write_buf(unsigned char reg, unsigned char* buf, short size);
static SUB_PROC_RESULT sl811_write_setup(P_SETUP_FORMAT pSetupFormat);
static SUB_PROC_RESULT sl811_write(unsigned char ep, unsigned char* data, int size);
static SUB_PROC_RESULT sl811_read(unsigned char ep, unsigned char* data, int size);

static int ui_function_usb_debug(UI_COMMAND uicmd, int param);
static int ui_function_usb_nop(UI_COMMAND uicmd, int param);
static int ui_function_usb_desc_device(UI_COMMAND uicmd, int param);
static int ui_function_usb_desc_config(UI_COMMAND uicmd, int param);
static int ui_function_usb_desc_interface(UI_COMMAND uicmd, int param);
static int ui_function_usb_desc_endpoints(UI_COMMAND uicmd, int param);

// -------------------------------------------
//  Variables
// -------------------------------------------
// Locals
//unsigned char	getdesc[8];
//unsigned char	setaddr[8];

volatile int sl811_wait_timer;
volatile int sl811_disable_timer;

static unsigned char sl811_speed;
static unsigned char sl811_error;

static unsigned char sl811_proc;
static unsigned char ui_usb_proc;
static unsigned char write_setup_proc;
static unsigned char write_proc;
static unsigned char read_proc;

static unsigned char packet_status;
static unsigned char device_address;

static SETUP_FORMAT	setup_fmt;
static EP_INFO ep_info[MAX_EP];
static unsigned char bulk_out_ep, bulk_in_ep;

static DESCRIPTOR_DEVICE		descriptor_device;
static DESCRIPTOR_CONFIG		descriptor_config;
static DESCRIPTOR_INTERFACE		descriptor_interface;
static DESCRIPTOR_ENDPOINT		descriptor_endpoints[MAX_EP];

static unsigned char buf[256];
static unsigned char reg_debug[10];

//static DevDesc device_descriptor;

typedef struct st_debug_var_table{
	int	num;
	unsigned char	type;
	char			display[MAX_COLUMN];
	void*			p_variable;
	unsigned int	interval_time;
} DEBUG_VAR_TABLE;

typedef struct st_usb_app_table{
	int			num;
	int			param;
	int				(*pExecFunc)(UI_COMMAND uicmd, int param);
	char			display[MAX_COLUMN];
} USB_APP_TABLE;

static const USB_APP_TABLE usb_app_table[] = {					// interval time will be ignored.
	{	 0,		0,	ui_function_usb_debug,			"  Debug         "	},
	{	 1,		0,	ui_function_usb_desc_device,	"  DESC.Device   "	},
	{	 2,		0,	ui_function_usb_desc_config,	"  DESC.Config   "	},
	{	 3,		0,	ui_function_usb_desc_interface,	"  DESC.Interface"	},
	{	 4,		0,	ui_function_usb_desc_endpoints,	"  DESC.EP(#0)   "	},
	{	 5,		1,	ui_function_usb_desc_endpoints,	"  DESC.EP(#1)   "	},
	{	 6,		2,	ui_function_usb_desc_endpoints,	"  DESC.EP(#2)   "	},
	{	 7,		3,	ui_function_usb_desc_endpoints,	"  DESC.EP(#3)   "	},
	{	 8,		4,	ui_function_usb_desc_endpoints,	"  DESC.EP(#4)   "	},
	{	 9,		5,	ui_function_usb_desc_endpoints,	"  DESC.EP(#5)   "	},
	{	10,		6,	ui_function_usb_desc_endpoints,	"  DESC.EP(#6)   "	},
	{	11,		7,	ui_function_usb_desc_endpoints,	"  DESC.EP(#7)   "	},
	{	-1,		0,	ui_function_usb_nop,			"0123456789ABCDEF"	},
};

static const DEBUG_VAR_TABLE debug_var_table[] = {
	{	 0,		 1,	"Process number: ",	&sl811_proc,							 300		},
	{	 1,		 1,	"Detected speed: ",	&sl811_speed,							 300		},
	{	 2,		 1,	"Error code:     ",	&sl811_error,							 300		},
	{	 3,		 1,	"Packet status:  ",	&packet_status,							 300		},
	{	 4,		 1,	"SL811 address:  ",	&device_address,						 300		},
	{	 5,		 1,	"Setup proc num: ",	&write_setup_proc,						 300		},
	{	 6,		 1,	"Write proc num: ",	&write_proc,							 300		},
	{	 7,		 1,	"Read proc num:  ",	&read_proc,								 300		},
	{	 8,		 1,	"bulk_out_ep:    ",	&bulk_out_ep, 			 				 300		},
	{	 9,		 1,	"bulk_in_ep:     ",	&bulk_in_ep, 						 	 300		},
	{	10,		 1,	"ep[0].size:     ",	&ep_info[0].size,		 			 	 300		},
	{	11,		 1,	"ep[0].address:  ",	&ep_info[0].address,	 			 	 300		},
	{	12,		 2,	"ep[0].hostbsize:",	&ep_info[0].host_buf_size, 			 	 300		},
	{	13,		 1,	"ep[1].size:     ",	&ep_info[1].size,		 			 	 300		},
	{	14,		 1,	"ep[1].address:  ",	&ep_info[1].address,	 			 	 300		},
	{	15,		 2,	"ep[1].hostbsize:",	&ep_info[1].host_buf_size, 			 	 300		},
	{	16,		 1,	"SFmt.bReqType:  ",	&setup_fmt.bmRequestType, 			 	 300		},
	{	17,		 1,	"SFmt.bRequest:  ",	&setup_fmt.bRequest, 				 	 300		},
	{	18,		 2,	"SFmt.wValue:    ",	&setup_fmt.wValue, 						 300		},
	{	19,		 2,	"SFmt.wIndex:    ",	&setup_fmt.wIndex, 						 300		},
	{	20,		 2,	"SFmt.wLength:   ",	&setup_fmt.wLength, 					 300		},
	{	22,		 3,	"SL811 Reg1(W):  ",	&reg_debug,							 	 300		},
	{	23,		 3,	"SL811 Reg2(W):  ",	&reg_debug+8,						 	 300		},
	{	-1,		 1,	"0123456789ABCDEF", 0,										   0		},
};

const DEBUG_VAR_TABLE device_var_table[] = {
	{	 0,		 1,	"bLength:        ",	&descriptor_device.bLength, 			 300		},
	{	 1,		 1,	"bDescriptorType:",	&descriptor_device.bDescriptorType, 	 300		},
	{	 2,		 2,	"bcdUSB:         ",	&descriptor_device.bcdUSB,		 		 300		},
	{	 3,		 1,	"bDeviceClass:   ",	&descriptor_device.bDeviceClass,	 	 300		},
	{	 4,		 1,	"bDeviceSubClass:",	&descriptor_device.bDeviceSubClass, 	 300		},
	{	 5,		 1,	"bDeviceProtocol:",	&descriptor_device.bDeviceProtocol, 	 300		},
	{	 6,		 1,	"bMaxPacketSize0:",	&descriptor_device.bMaxPacketSize0, 	 300		},
	{	 7,		 2,	"idVendor:       ",	&descriptor_device.idVendor, 			 300		},
	{	 8,		 2,	"idProduct:      ",	&descriptor_device.idProduct, 			 300		},
	{	 9,		 2,	"bcdDevice:      ",	&descriptor_device.bcdDevice, 			 300		},
	{	10,		 1,	"iManufacturer:  ",	&descriptor_device.iManufacturer, 		 300		},
	{	11,		 1,	"iProduct:       ",	&descriptor_device.iProduct, 			 300		},
	{	12,		 1,	"iSerialNumber:  ",	&descriptor_device.iSerialNumber, 		 300		},
	{	13,		 1,	"bNumConfig:     ",	&descriptor_device.bNumConfigurations, 	 300		},
	{	-1,		 1,	"0123456789ABCDEF", 0,										   0		},
};

const DEBUG_VAR_TABLE config_var_table[] = {
	{	 0,		 1,	"bLength:        ",	&descriptor_config.bLength,				300		},
	{	 1,		 1,	"bDescriptorType:",	&descriptor_config.bDescriptorType,		300		},
	{	 2,		 2,	"wTotalLength:   ",	&descriptor_config.wTotalLength,		300		},
	{	 3,		 1,	"bNumInterface:  ",	&descriptor_config.bNumInterface,		300		},
	{	 4,		 1,	"bConfigValue:   ",	&descriptor_config.bConfigurationValue,	300		},
	{	 5,		 1,	"iConfiguration: ",	&descriptor_config.iConfiguration,		300		},
	{	 6,		 1,	"bmAttributes:   ",	&descriptor_config.bmAttributes,		300		},
	{	 7,		 1,	"bMaxPower:      ",	&descriptor_config.bMaxPower,			300		},
	{	-1,		 1,	"0123456789ABCDEF", 0,										  0		},
};

const DEBUG_VAR_TABLE interface_var_table[] = {
	{	 0,		 1,	"bLength:        ",	&descriptor_interface.bLength,				300		},
	{	 1,		 1,	"bDescriptorType:",	&descriptor_interface.bDescriptorType,		300		},
	{	 2,		 2,	"bInterfaceNum:  ",	&descriptor_interface.bInterfaceNumber,		300		},
	{	 3,		 1,	"bAltSetting:    ",	&descriptor_interface.bAlternateSetting,	300		},
	{	 4,		 1,	"bNumEndpoints:  ",	&descriptor_interface.bNumEndpoints,		300		},
	{	 5,		 1,	"bI/FClass:      ",	&descriptor_interface.bInterfaceClass,		300		},
	{	 6,		 1,	"bI/FSubClass:   ",	&descriptor_interface.bInterfaceSubClass,	300		},
	{	 7,		 1,	"bI/FProtocol:   ",	&descriptor_interface.bInterfaceProtocol,	300		},
	{	 8,		 1,	"iInterface:     ",	&descriptor_interface.iInterface,			300		},
	{	-1,		 1,	"0123456789ABCDEF", 0,											  0		},
};

const DEBUG_VAR_TABLE endpoints_var_table[] = {
	{	 0,		 1,	"bLength:        ",	&descriptor_endpoints[0].bLength,			300		},
	{	 1,		 1,	"bDescriptorType:",	&descriptor_endpoints[0].bDescriptorType,	300		},
	{	 2,		 1,	"bEndpointAddrs: ",	&descriptor_endpoints[0].bEndpointAddress,	300		},
	{	 3,		 1,	"bmAttributes:   ",	&descriptor_endpoints[0].bmAttributes,		300		},
	{	 4,		 2,	"wMaxPacketSize: ",	&descriptor_endpoints[0].wMaxPacketSize,	300		},
	{	 5,		 1,	"bInterval:      ",	&descriptor_endpoints[0].bInterval,			300		},
	{	-1,		 1,	"0123456789ABCDEF", 0,											  0		},
};

enum ui_usb_procedure {
	UI_USB_01_TOP_LEVEL,
	UI_USB_02_EXECUTING,
};

enum sl811_error_code {
	USB_ERR_NONE,				// 00
	USB_ERR_REG_TEST_FAIL,		// 01
	USB_ENUM_EP_FAIL1,			// 02
	USB_ENUM_EP_FAIL2,			// 03
	USB_ENUM_EP_FAIL3,			// 04
	USB_ENUM_EP_FAIL4,			// 05
	USB_ENUM_EP_FAIL5,			// 06
	USB_DETECTION_FAIL1,		// 07
	USB_DETECTION_FAIL2,		// 08
	USB_SETADDRESS_FAIL1,		// 09
	USB_SETADDRESS_FAIL2,		// 10
	USB_GET_DESC_DEV_FAIL1,		// 11
	USB_GET_DESC_DEV_FAIL2,		// 12
	USB_GET_DESC_CFG_FAIL1,		// 13
	USB_GET_DESC_CFG_FAIL2,		// 14
	USB_GET_DESC_IF_FAIL1,		// 15
	USB_GET_DESC_IF_FAIL2,		// 16
	USB_SET_CONFIG_FAIL1,		// 17
	USB_SET_CONFIG_FAIL2,		// 18
	USB_WRITE_SETUP_ERROR,		// 19
	USB_NO_DEVICE,				// 20
	USB_BULK_OUT_EP_ILL,		// 21
	USB_BULK_IN_EP_ILL,			// 22
	USB_PROCESS_ERROR = 100,
};


enum sl811_process_mode {
	USB_REG_TEST,				// 00
	USB_RESET_01,				// 01
	USB_RESET_02,				// 02
	USB_RESET_03,				// 03
	USB_RESET_OK_LOW,			// 04
	USB_RESET_OK_FULL,			// 05
	USB_DETECTION_01,			// 06
	USB_DETECTION_02,			// 07
	USB_DETECTION_03,			// 08
	USB_GET_DESC_DEVICE_01,		// 09
	USB_GET_DESC_DEVICE_02,		// 10
	USB_GET_DESC_DEVICE_03,		// 11
	USB_GET_DESC_CONFIG_01,		// 12
	USB_GET_DESC_CONFIG_02,		// 13
	USB_GET_DESC_CONFIG_03,		// 14
	USB_GET_DESC_STRING_01,		// 15
	USB_GET_DESC_STRING_02,		// 16
	USB_GET_DESC_STRING_03,		// 17
	USB_GET_DESC_ALL_01,		// 18
	USB_GET_DESC_ALL_02,		// 19
	USB_GET_DESC_ALL_03,		// 20
	USB_SETADDRESS_01,			// 21
	USB_SETADDRESS_02,			// 22
	USB_SETADDRESS_03,			// 23
	USB_SETADDRESS_04,			// 24
	USB_SET_CONFIGURATION_01,	// 25
	USB_SET_CONFIGURATION_02,	// 26
	USB_SET_CONFIGURATION_03,	// 27
	
	USB_IDLE = 0x80,			// 80		IDLE
	
	USB_CONTROL_SEND_01,		// 81
	USB_CONTROL_SEND_02,		// 82
	USB_CONTROL_SEND_03,		// 83
	USB_CONTROL_RECEIVE_01,		// 84
	USB_CONTROL_RECEIVE_02,		// 85
	USB_CONTROL_RECEIVE_03,		// 86
	USB_BULK_OUT_01,			// 87
	USB_BULK_OUT_02,			// 88
	USB_BULK_OUT_03,			// 89
	USB_BULK_OUT_04,			// 90
	USB_BULK_IN_01,				// 91
	USB_BULK_IN_02,				// 92
	USB_BULK_IN_03,				// 93
	USB_BULK_IN_04,				// 94
	USB_ERROR_END = 0xFF,
};

enum sl811_sub_proc_result_code {
	SUB_PROC_BUSY,
	SUB_PROC_DONE,
	SUB_PROC_ERROR,
};

// -------------------------------------------
//  Interrupt handlers
// -------------------------------------------
void sl811_1ms_handler(void) {
	if( sl811_wait_timer )    sl811_wait_timer--;
	if( sl811_disable_timer ) sl811_disable_timer--;
}

void sl811_chip_irq(void) {
	unsigned char reg;

//	reg = sl811_read_reg(SL811HS_STATUS);
//	sl811_write_reg(SL811HS_STATUS, reg);	// 荞݃NA
//	tprintf("SL811HS_STATUS     : %02lX\n", (int)reg);
//	reg = SL811_read_reg(SL811HS_USBA_HOSTSTATUS);
//	tprintf("SL811HS_HOSTSTATUS : %02lX\n", (int)reg);
//	reg = SL811_read_reg(SL811HS_INTENV);
//	tprintf("SL811HS_INTENV : %02lX\n", (int)reg);
}

// -------------------------------------------
//  Initialize
// -------------------------------------------
int sl811_initialize(void) {
	/* BUS  initialize */
	BSC.ASTCR.BYTE = 0xff;
	P8DDR = 0xee;	/* CS1-3 is enable */

	sl811_wait_timer = 0;
	sl811_proc = USB_REG_TEST;
	sl811_error = USB_ERR_NONE;
	
	// UI related
	ui_usb_proc = UI_USB_01_TOP_LEVEL;
	
//	getdesc[] = {0x80, 0x06, 0, 1, 0, 0, 64, 0};
//	setaddr[] = {0x00, 0x05, 2, 0, 0, 0, 0, 0};


}

// -------------------------------------------
//  Query for SL811 device class status
// -------------------------------------------
unsigned char sl811_status(unsigned char req) {
	if( req == SL811_REQ_RESET ) {
		if( sl811_proc < USB_IDLE ) {			// If initialize in progress
			return SL811_STS_RESET_IN_PROGRESS;	
		} else if( sl811_proc == USB_IDLE || sl811_proc == USB_ERROR_END) {	// Reset accepted
			sl811_proc = USB_REG_TEST;
			return SL811_STS_RESET_ACCEPTED;
		} else {								// Class is busy
			return SL811_STS_NOT_READY;
		}
	}
	
	// Status return
	if( sl811_proc < USB_IDLE )				return SL811_STS_RESET_IN_PROGRESS;	
	else if( sl811_proc == USB_IDLE ) 		return SL811_STS_READY;
	else if( sl811_proc == USB_ERROR_END ) 	return SL811_STS_ERROR_STOP;
	else 									return SL811_STS_NOT_READY;
}

// -------------------------------------------
//  Control Data Receive Request
// -------------------------------------------
unsigned char sl811_ctrl_receive(P_SETUP_FORMAT p_fmt) {
	if( sl811_proc != USB_IDLE ) return USB_REQ_BUSY;
	
	setup_fmt.bmRequestType 	= p_fmt->bmRequestType;
	setup_fmt.bRequest 			= p_fmt->bRequest;
	setup_fmt.wValue 			= p_fmt->wValue;
	setup_fmt.wIndex 			= p_fmt->wIndex;
	setup_fmt.wLength 			= p_fmt->wLength << 8;
	
	sl811_proc = USB_CONTROL_RECEIVE_01;
	return USB_REQ_ACCEPTED;
}

// -------------------------------------------
//  Buffer copy request
// -------------------------------------------
void sl811_buffer_copy( unsigned char* p_buf, int size) {
	int i;
	
	for(i=0; i<size; i++) *p_buf = buf[i];
}


// -------------------------------------------
//  Bulk out request
// -------------------------------------------
unsigned char sl811_bulk_out_request( unsigned char* p_buf, int size ) {
	if( sl811_proc != USB_IDLE )	return USB_REQ_BUSY;
	if( bulk_out_ep >= MAX_EP )		return USB_REQ_REFUSED;
	
	ep_info[bulk_out_ep].p_host_buf = p_buf;
	ep_info[bulk_out_ep].host_buf_size = size;
	ep_info[bulk_out_ep].toggle = 0;
	sl811_proc = USB_BULK_OUT_01;
	return USB_REQ_ACCEPTED;
}

// -------------------------------------------
//  Bulk in request
// -------------------------------------------
unsigned char sl811_bulk_in_request( unsigned char* p_buf, int size ) {
	if( sl811_proc != USB_IDLE )	return USB_REQ_BUSY;
	if( bulk_in_ep >= MAX_EP )		return USB_REQ_REFUSED;
	
	ep_info[bulk_in_ep].p_host_buf = p_buf;
	ep_info[bulk_in_ep].host_buf_size = size;
	ep_info[bulk_out_ep].toggle = 0;
	sl811_proc = USB_BULK_IN_01;
	return USB_REQ_ACCEPTED;
}


// -------------------------------------------
//  Main process
// -------------------------------------------
int sl811_process(void) {
	int	i, data, result;
	unsigned char *p;
	
	switch( sl811_proc ) {
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		// *   Register read / write test                                                *
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		case USB_REG_TEST:
			// Variable initialize
			write_setup_proc 	= 0x00;
			write_proc 			= 0x00;
			read_proc 			= 0x00;
			
			device_address = 0;				// initial value
			bulk_out_ep = 0xFF;
			bulk_in_ep  = 0xFF;
			ep_info[0].size = 8;			// Minimun size of ep
			for ( i=0; i<10; i++ ) reg_debug[i]=0;
			
			p=(unsigned char *) &descriptor_device;
			for ( i=0; i<sizeof(descriptor_device); i++ ) *(p++) = 0;
			p=(unsigned char *) &descriptor_config;
			for ( i=0; i<sizeof(descriptor_config); i++ ) *(p++) = 0;
			p=(unsigned char *) &descriptor_interface;
			for ( i=0; i<sizeof(descriptor_interface); i++ ) *(p++) = 0;
			p=(unsigned char *) &descriptor_endpoints;
			for ( i=0; i<sizeof(descriptor_endpoints); i++ ) *(p++) = 0;
//			p=(unsigned char *) &descriptor_endpoints[0];
//			for ( i=0; i<sizeof(descriptor_endpoints[0])*MAX_EP; i++ ) *(p++) = 0;

			// Register test
			result = 0;
			for(i = 16;i < 256; i++) {
				buf[i] = sl811_read_reg(i);
				sl811_write_reg(i, i);
			}
			for(i = 16;i < 256; i++) {
				data = sl811_read_reg(i);
				if(data != i) {
					result = -1;
				}
			}
			for(i = 16;i < 256;i++) sl811_write_reg(i, buf[i]);
			if( result == -1 ) {
				sl811_error = USB_ERR_REG_TEST_FAIL;
				sl811_proc = USB_ERROR_END;
			} else {
				sl811_proc = USB_RESET_01;
			}
			break;
			
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		// *   SL811HST Reset sequence                                                   *
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		case USB_RESET_01:
			sl811_write_reg(SL811HS_SOFCOUNTH, 0xae);
			sl811_write_reg(SL811HS_CONTROL1, 0x08);	// reset USB
			sl811_wait_timer = 20;
			sl811_proc = USB_RESET_02;
			break;
		
		case USB_RESET_02:
			if( !sl811_wait_timer ) {
				sl811_write_reg(SL811HS_CONTROL1, 0);	// remove SE0
				sl811_wait_timer = 100;
				sl811_proc = USB_RESET_03;
			}
			break;
			
		case USB_RESET_03:
			if( !sl811_wait_timer ) {
				sl811_write_reg(SL811HS_STATUS, 0xff);		// clear all interrupt bits
				data = sl811_read_reg(SL811HS_STATUS);
				if(data & 0x40){							// Check if device is removed
					sl811_speed = 0;	// None
					sl811_write_reg(SL811HS_INTENV,
							SL811HS_INTENV_BIT_USBA | 
							SL811HS_INTENV_BIT_SOFTIMER |
							SL811HS_INTENV_BIT_INSRMV);
					sl811_error = USB_NO_DEVICE;
					sl811_proc = USB_ERROR_END;
				} else {
					sl811_write_reg(SL811HS_USBB_HOSTBASEDLEN, 0);	//zero lenth
					sl811_write_reg(SL811HS_USBB_HOSTPID, 0x50);	//send SOF to EP0
					sl811_write_reg(SL811HS_USBB_HOSTDEVADDR, 0x01);	//address0
					sl811_write_reg(SL811HS_SOFCOUNTL, 0xe0);
					
					sl811_write_reg(SL811HS_CONTROL1, 0x8);
					sl811_wait_timer = 20;
					if(!(data & 0x80))	sl811_proc = USB_RESET_OK_LOW;
					else				sl811_proc = USB_RESET_OK_FULL;
				}
			}
			break;
			
		case USB_RESET_OK_LOW:
			if( !sl811_wait_timer ) {
				sl811_speed = USB_LOW;	// Low
				sl811_write_reg(SL811HS_SOFCOUNTH, 0xee);
				sl811_write_reg(SL811HS_CONTROL1, 0x21);
				sl811_write_reg(SL811HS_USBB_HOSTCTRL, 0x01);
				for(i=0;i<20;i++) sl811_write_reg(SL811HS_STATUS, 0xff);
				sl811_write_reg(SL811HS_INTENV, SL811HS_INTENV_BIT_USBA | 
				   SL811HS_INTENV_BIT_SOFTIMER|SL811HS_INTENV_BIT_INSRMV);
				sl811_proc = USB_DETECTION_01;
			}
			break;
			
		case USB_RESET_OK_FULL:
			if( !sl811_wait_timer ) {
				sl811_speed = USB_FULL;	// Full
				sl811_write_reg(SL811HS_SOFCOUNTH, 0xae);
				sl811_write_reg(SL811HS_CONTROL1, 0x01 );
				sl811_write_reg(SL811HS_USBB_HOSTCTRL, 0x01);
				sl811_write_reg(SL811HS_STATUS, 0xff);
				sl811_write_reg(SL811HS_INTENV, SL811HS_INTENV_BIT_USBA | 
				   SL811HS_INTENV_BIT_SOFTIMER|SL811HS_INTENV_BIT_INSRMV);
				sl811_proc = USB_DETECTION_01;
			}
			break;
			
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		// *   Target device detection sequence                                          *
		// *   To get max packet size firstly.                                           *
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		case USB_DETECTION_01:				// SETUP stage
			setup_fmt.bmRequestType 	= DIR_DEV2HOST | RECEIPIENT_DEVICE;
			setup_fmt.bRequest 			= REQ_TYPE_GET_DESCRIPTOR;
			setup_fmt.wValue 			= DESCRIPTOR_TYPE_DEVICE;
			setup_fmt.wIndex 			= 0;
			setup_fmt.wLength 			= (ep_info[0].size) << 8;
			
			ep_info[0].toggle = 0;
			switch( sl811_write_setup(&setup_fmt) ) {
				case SUB_PROC_DONE:
					sl811_proc = USB_DETECTION_02;
					break;
				case SUB_PROC_ERROR:
					sl811_error = USB_PROCESS_ERROR + sl811_proc;
					sl811_proc = USB_ERROR_END;
					break;
				case SUB_PROC_BUSY:
				default:
					break;
			}
			break;
		
		case USB_DETECTION_02:				// DATA(IN) Stage
			ep_info[0].toggle = 1;
			switch ( sl811_read(0, buf, ep_info[0].size) ) {
				case SUB_PROC_DONE:
					sl811_proc = USB_DETECTION_03;
					ep_info[0].size = buf[7];
					descriptor_device.bLength = buf[0];
					break;
				case SUB_PROC_ERROR:
					sl811_error = USB_PROCESS_ERROR + sl811_proc;
					sl811_proc = USB_ERROR_END;
					break;
				case SUB_PROC_BUSY:
				default:
					break;
			}
			break;
			
		case USB_DETECTION_03:				// STATUS stage
			ep_info[0].toggle = 1;
			switch ( sl811_write(0, buf, 0) ) {
				case SUB_PROC_DONE:
					sl811_proc = USB_GET_DESC_DEVICE_01;
					break;
				case SUB_PROC_ERROR:
					sl811_error = USB_PROCESS_ERROR + sl811_proc;
					sl811_proc = USB_ERROR_END;
					break;
				case SUB_PROC_BUSY:
				default:
					break;
			}
			break;
			
			
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		// *   Descriptor collection (DEVICE)                                            *
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		case USB_GET_DESC_DEVICE_01:				// SETUP stage
			setup_fmt.bmRequestType 	= DIR_DEV2HOST | RECEIPIENT_DEVICE;
			setup_fmt.bRequest 			= REQ_TYPE_GET_DESCRIPTOR;
			setup_fmt.wValue 			= DESCRIPTOR_TYPE_DEVICE;
			setup_fmt.wIndex 			= 0;
			setup_fmt.wLength 			= (descriptor_device.bLength) << 8;
			
			switch( sl811_write_setup(&setup_fmt) ) {
				case SUB_PROC_DONE:
					sl811_proc = USB_GET_DESC_DEVICE_02;
					break;
				case SUB_PROC_ERROR:
					sl811_error = USB_PROCESS_ERROR + sl811_proc;
					sl811_proc = USB_ERROR_END;
					break;
				case SUB_PROC_BUSY:
				default:
					break;
			}
			break;
		
		case USB_GET_DESC_DEVICE_02:				// DATA(IN) stage
			ep_info[0].toggle = 1;
			switch ( sl811_read(0, buf, descriptor_device.bLength) ) {
				case SUB_PROC_DONE:
					// Get descriptor
					descriptor_device.bLength				= buf[ 0];
					descriptor_device.bDescriptorType 		= buf[ 1];
					descriptor_device.bcdUSB		 		= buf[ 2] + (buf[ 3]<<8);
					descriptor_device.bDeviceClass	 		= buf[ 4];
					descriptor_device.bDeviceSubClass 		= buf[ 5];
					descriptor_device.bDeviceProtocol 		= buf[ 6];
					descriptor_device.bMaxPacketSize0 		= buf[ 7];
					descriptor_device.idVendor 				= buf[ 8] + (buf[ 9]<<8);;
					descriptor_device.idProduct		 		= buf[10] + (buf[11]<<8);;
					descriptor_device.bcdDevice 			= buf[12] + (buf[13]<<8);;
					descriptor_device.iManufacturer 		= buf[14];
					descriptor_device.iProduct 				= buf[15];
					descriptor_device.iSerialNumber 		= buf[16];
					descriptor_device.bNumConfigurations	= buf[17];
					
					sl811_proc = USB_GET_DESC_DEVICE_03;
					break;
				case SUB_PROC_ERROR:
					sl811_error = USB_PROCESS_ERROR + sl811_proc;
					sl811_proc = USB_ERROR_END;
					break;
				case SUB_PROC_BUSY:
				default:
					break;
			}
			break;
			
		case USB_GET_DESC_DEVICE_03:				// STATUS stage
			ep_info[0].toggle = 1;
			switch ( sl811_write(0, buf, 0) ) {
				case SUB_PROC_DONE:
					sl811_proc = USB_GET_DESC_CONFIG_01;
					break;
				case SUB_PROC_ERROR:
					sl811_error = USB_PROCESS_ERROR + sl811_proc;
					sl811_proc = USB_ERROR_END;
					break;
				case SUB_PROC_BUSY:
				default:
					break;
			}
			break;
			
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		// *   Descriptor collection (CONFIG)                                            *
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		case USB_GET_DESC_CONFIG_01:
			setup_fmt.bmRequestType 	= DIR_DEV2HOST | RECEIPIENT_DEVICE;
			setup_fmt.bRequest 			= REQ_TYPE_GET_DESCRIPTOR;
			setup_fmt.wValue 			= DESCRIPTOR_TYPE_CONFIG;
			setup_fmt.wIndex 			= 0;
			setup_fmt.wLength 			= sizeof(descriptor_config) << 8;
			
			switch( sl811_write_setup(&setup_fmt) ) {
				case SUB_PROC_DONE:
					sl811_proc = USB_GET_DESC_CONFIG_02;
					break;
				case SUB_PROC_ERROR:
					sl811_error = USB_PROCESS_ERROR + sl811_proc;
					sl811_proc = USB_ERROR_END;
					break;
				case SUB_PROC_BUSY:
				default:
					break;
			}
			break;
		
		case USB_GET_DESC_CONFIG_02:
			ep_info[0].toggle = 1;
			switch ( sl811_read(0, buf, sizeof(descriptor_config)) ) {
				case SUB_PROC_DONE:
					// Get descriptor
					descriptor_config.bLength					= buf[ 0];
					descriptor_config.bDescriptorType			= buf[ 1];
					descriptor_config.wTotalLength				= buf[ 2] + (buf[ 3]<<8);
					descriptor_config.bNumInterface				= buf[ 4];
					descriptor_config.bConfigurationValue		= buf[ 5];
					descriptor_config.iConfiguration			= buf[ 6];
					descriptor_config.bmAttributes				= buf[ 7];
					descriptor_config.bMaxPower					= buf[ 8];
						
					sl811_proc = USB_GET_DESC_CONFIG_03;
					break;
				case SUB_PROC_ERROR:
					sl811_error = USB_PROCESS_ERROR + sl811_proc;
					sl811_proc = USB_ERROR_END;
					break;
				case SUB_PROC_BUSY:
				default:
					break;
			}
			break;
			
		case USB_GET_DESC_CONFIG_03:				// STATUS stage
			ep_info[0].toggle = 1;
			switch ( sl811_write(0, buf, 0) ) {
				case SUB_PROC_DONE:
					sl811_proc = USB_GET_DESC_ALL_01;
//					sl811_proc = USB_SETADDRESS_01;
					break;
				case SUB_PROC_ERROR:
					sl811_error = USB_PROCESS_ERROR + sl811_proc;
					sl811_proc = USB_ERROR_END;
					break;
				case SUB_PROC_BUSY:
				default:
					break;
			}
			break;
			
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		// *   Descriptor collection (ALL)                                               *
		// *   TODO : to apply this sequence to multi interface device.                  *
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		case USB_GET_DESC_ALL_01:
			setup_fmt.bmRequestType 	= DIR_DEV2HOST | RECEIPIENT_DEVICE;
			setup_fmt.bRequest 			= REQ_TYPE_GET_DESCRIPTOR;
			setup_fmt.wValue 			= DESCRIPTOR_TYPE_CONFIG;
			setup_fmt.wIndex 			= 0;
			setup_fmt.wLength 			= (descriptor_config.wTotalLength) << 8;
			
			switch( sl811_write_setup(&setup_fmt) ) {
				case SUB_PROC_DONE:
					sl811_proc = USB_GET_DESC_ALL_02;
					break;
				case SUB_PROC_ERROR:
					sl811_error = USB_PROCESS_ERROR + sl811_proc;
					sl811_proc = USB_ERROR_END;
					break;
				case SUB_PROC_BUSY:
				default:
					break;
			}
			break;
		
		case USB_GET_DESC_ALL_02:
			ep_info[0].toggle = 1;
			switch ( sl811_read(0, buf, descriptor_config.wTotalLength ) ) {
				case SUB_PROC_DONE:
					// Get descriptor (Interface) only #0
					i = descriptor_config.bLength;
					descriptor_interface.bLength = buf[i];
					if(descriptor_interface.bLength >= 2) descriptor_interface.bDescriptorType		= buf[i + 1];
					if(descriptor_interface.bLength >= 3) descriptor_interface.bInterfaceNumber		= buf[i + 2];
					if(descriptor_interface.bLength >= 4) descriptor_interface.bAlternateSetting	= buf[i + 3];
					if(descriptor_interface.bLength >= 5) descriptor_interface.bNumEndpoints		= buf[i + 4];
					if(descriptor_interface.bLength >= 6) descriptor_interface.bInterfaceClass		= buf[i + 5];
					if(descriptor_interface.bLength >= 7) descriptor_interface.bInterfaceSubClass	= buf[i + 6];
					if(descriptor_interface.bLength >= 8) descriptor_interface.bInterfaceProtocol	= buf[i + 7];
					if(descriptor_interface.bLength >= 9) descriptor_interface.iInterface			= buf[i + 8];
					
					// Get descriptor (Endpoints)
					i = descriptor_config.bLength + descriptor_interface.bLength;
					for(data=0; data<descriptor_interface.bNumEndpoints; data++) {
						if( data < MAX_EP ) {
							descriptor_endpoints[data].bLength = buf[i];
							if(descriptor_endpoints[data].bLength >= 2) descriptor_endpoints[data].bDescriptorType	= buf[i + 1];
							if(descriptor_endpoints[data].bLength >= 3) descriptor_endpoints[data].bEndpointAddress	= buf[i + 2];
							if(descriptor_endpoints[data].bLength >= 4) descriptor_endpoints[data].bmAttributes		= buf[i + 3];
							if(descriptor_endpoints[data].bLength >= 6) descriptor_endpoints[data].wMaxPacketSize	= buf[i + 4] + (buf[i + 5]<<8);
							if(descriptor_endpoints[data].bLength >= 7) descriptor_endpoints[data].bInterval		= buf[i + 6];
							
							i += descriptor_endpoints[data].bLength;
							
							// Local end points information update
							ep_info[data].address = descriptor_endpoints[data].bEndpointAddress;
							ep_info[data].size    = descriptor_endpoints[data].wMaxPacketSize;
							
							if(  ( (descriptor_endpoints[data].bEndpointAddress & 0x80) == 0x80 )			// IN
							  && ( (descriptor_endpoints[data].bmAttributes     & 0x03) == 0x02 ) ) {		// BULK
							  bulk_in_ep = data;
							}
							if(  ( (descriptor_endpoints[data].bEndpointAddress & 0x80) == 0x00 )			// OUT
							  && ( (descriptor_endpoints[data].bmAttributes     & 0x03) == 0x02 ) ) {		// BULK
							  bulk_out_ep = data;
							}
						}
					}
					sl811_proc = USB_GET_DESC_ALL_03;
					break;
					
				case SUB_PROC_ERROR:
					sl811_error = USB_PROCESS_ERROR + sl811_proc;
					sl811_proc = USB_ERROR_END;
					break;
				case SUB_PROC_BUSY:
				default:
					break;
			}
			break;
			
		case USB_GET_DESC_ALL_03:				// STATUS stage
			ep_info[0].toggle = 1;
			switch ( sl811_write(0, buf, 0) ) {
				case SUB_PROC_DONE:
					sl811_proc = USB_SETADDRESS_01;
					break;
				case SUB_PROC_ERROR:
					sl811_error = USB_PROCESS_ERROR + sl811_proc;
					sl811_proc = USB_ERROR_END;
					break;
				case SUB_PROC_BUSY:
				default:
					break;
			}
			break;
			
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		// *   Set device address sequence                                               *
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		case USB_SETADDRESS_01:
//			device_address = DEFAULT_DEVICE_ADDRESS;
			setup_fmt.bmRequestType 	= DIR_HOST2DEV | RECEIPIENT_DEVICE;
			setup_fmt.bRequest 			= REQ_TYPE_SET_ADDRESS;
//			setup_fmt.wValue 			= ((unsigned short)device_address) << 8;
			setup_fmt.wValue 			= (DEFAULT_DEVICE_ADDRESS) << 8;
			setup_fmt.wIndex 			= 0;
			setup_fmt.wLength 			= 0;
						
			switch( sl811_write_setup(&setup_fmt) ) {
				case SUB_PROC_DONE:
					sl811_proc = USB_SETADDRESS_02;
					break;
				case SUB_PROC_ERROR:
					sl811_error = USB_PROCESS_ERROR + sl811_proc;
					sl811_proc = USB_ERROR_END;
					break;
				case SUB_PROC_BUSY:
				default:
					break;
			}
			break;
			
		case USB_SETADDRESS_02:
			ep_info[0].toggle = 1;
			switch ( sl811_read(0, buf, 0) ) {
				case SUB_PROC_DONE:
					device_address = DEFAULT_DEVICE_ADDRESS;
					sl811_proc = USB_SET_CONFIGURATION_01;
					break;
				case SUB_PROC_ERROR:
					sl811_error = USB_PROCESS_ERROR + sl811_proc;
					sl811_proc = USB_ERROR_END;
					break;
				case SUB_PROC_BUSY:
				default:
					break;
			}
			break;

		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		// *   Set device configuration sequence                                         *
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		case USB_SET_CONFIGURATION_01:
			setup_fmt.bmRequestType 	= DIR_HOST2DEV | RECEIPIENT_DEVICE;
			setup_fmt.bRequest 			= REQ_TYPE_SET_CONFIG;
			setup_fmt.wValue 			= (descriptor_config.bConfigurationValue) << 8;
			setup_fmt.wIndex 			= 0;
			setup_fmt.wLength 			= 0;
			ep_info[0].toggle = 1;
			
			switch( sl811_write_setup(&setup_fmt) ) {
				case SUB_PROC_DONE:
					sl811_proc = USB_SET_CONFIGURATION_02;
					break;
				case SUB_PROC_ERROR:
					sl811_error = USB_PROCESS_ERROR + sl811_proc;
					sl811_proc = USB_ERROR_END;
					break;
				case SUB_PROC_BUSY:
				default:
					break;
			}
			break;
			
		case USB_SET_CONFIGURATION_02:
			switch ( sl811_read(0, buf, 0) ) {
				case SUB_PROC_DONE:
					sl811_proc = USB_IDLE;
					break;
				case SUB_PROC_ERROR:
					sl811_error = USB_PROCESS_ERROR + sl811_proc;
					sl811_proc = USB_ERROR_END;
					break;
				case SUB_PROC_BUSY:
				default:
					break;
			}
			break;
		
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		// *   Idle                                                                      *
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		case USB_IDLE:
			break;
		
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		// *   Control data receive                                                      *
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		case USB_CONTROL_RECEIVE_01:				// SETUP stage
			switch( sl811_write_setup(&setup_fmt) ) {
				case SUB_PROC_DONE:
					sl811_proc = USB_CONTROL_RECEIVE_02;
					break;
				case SUB_PROC_ERROR:
					sl811_error = USB_PROCESS_ERROR + sl811_proc;
					sl811_proc = USB_ERROR_END;
					break;
				case SUB_PROC_BUSY:
				default:
					break;
			}
			break;
		
		case USB_CONTROL_RECEIVE_02:				// DATA(IN) stage
			ep_info[0].toggle = 1;
			switch ( sl811_read(0, buf, setup_fmt.wLength>>8) ) {
				case SUB_PROC_DONE:
					sl811_proc = USB_CONTROL_RECEIVE_03;
					break;
				case SUB_PROC_ERROR:
					sl811_error = USB_PROCESS_ERROR + sl811_proc;
					sl811_proc = USB_ERROR_END;
					break;
				case SUB_PROC_BUSY:
				default:
					break;
			}
			break;
			
		case USB_CONTROL_RECEIVE_03:				// STATUS stage
			ep_info[0].toggle = 1;
			switch ( sl811_write(0, buf, 0) ) {
				case SUB_PROC_DONE:
					sl811_proc = USB_IDLE;			// Back to idle
					break;
				case SUB_PROC_ERROR:
					sl811_error = USB_PROCESS_ERROR + sl811_proc;
					sl811_proc = USB_ERROR_END;
					break;
				case SUB_PROC_BUSY:
				default:
					break;
			}
			break;
			
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		// *   Bulk data out                                                             *
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		case USB_BULK_OUT_01:
			if( bulk_out_ep >= MAX_EP ) {
				sl811_error = USB_BULK_OUT_EP_ILL;
				sl811_proc = USB_ERROR_END;
				break;
			}
			ep_info[bulk_out_ep].toggle = 0;
			switch ( sl811_write(bulk_out_ep, ep_info[bulk_out_ep].p_host_buf, ep_info[bulk_out_ep].host_buf_size) ) {
				case SUB_PROC_DONE:
					sl811_proc = USB_IDLE;			// Back to idle
					break;
				case SUB_PROC_ERROR:
					sl811_error = USB_PROCESS_ERROR + sl811_proc;
					sl811_proc = USB_ERROR_END;
					break;
				case SUB_PROC_BUSY:
				default:
					break;
			}
			break;
			
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		// *   Bulk data in                                                              *
		// * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * * *
		case USB_BULK_IN_01:
			if( bulk_in_ep >= MAX_EP ) {
				sl811_error = USB_BULK_IN_EP_ILL;
				sl811_proc = USB_ERROR_END;
				break;
			}
			ep_info[bulk_in_ep].toggle = 0;
			switch ( sl811_read(bulk_out_ep, ep_info[bulk_in_ep].p_host_buf, ep_info[bulk_in_ep].host_buf_size) ) {
				case SUB_PROC_DONE:
					sl811_proc = USB_IDLE;			// Back to idle
					break;
				case SUB_PROC_ERROR:
					sl811_error = USB_PROCESS_ERROR + sl811_proc;
					sl811_proc = USB_ERROR_END;
					break;
				case SUB_PROC_BUSY:
				default:
					break;
			}
			break;

		}

}


// -------------------------------------------
//  [LOCAL] Register read
// -------------------------------------------
static unsigned char sl811_read_reg(unsigned char reg) {
	SL811_ADDRREG = (unsigned char)reg;
	return SL811_DATAREG;
}

// -------------------------------------------
//  [LOCAL] Register write
// -------------------------------------------
static void sl811_write_reg(unsigned char reg, unsigned char data) {
	SL811_ADDRREG = (unsigned char)reg;
	SL811_DATAREG = data;
	
	if(reg<10) reg_debug[reg] = data;
}

// -------------------------------------------
//  [LOCAL] Buffer read
// -------------------------------------------
static void sl811_read_buf(unsigned char reg, unsigned char* buf, short size) {
	if(size <= 0) return;
	SL811_ADDRREG = reg;
	while(size--) *buf++ = SL811_DATAREG;
}

// -------------------------------------------
//  [LOCAL] Buffer write
// -------------------------------------------
static void sl811_write_buf(unsigned char reg, unsigned char* buf, short size) {
	if(size <= 0) return;
	SL811_ADDRREG = reg;
	while(size--) {
		SL811_DATAREG = *buf;
		buf++;
	}
}

// -------------------------------------------
//  [LOCAL] SL811 SETUP packet send
// -------------------------------------------
static SUB_PROC_RESULT sl811_write_setup(P_SETUP_FORMAT pSetupFormat)
{
	SUB_PROC_RESULT ret;
	
	ret = SUB_PROC_BUSY;
	
	switch( write_setup_proc ) {
		case 0x00:
			sl811_write_buf(0x10, (unsigned char*) pSetupFormat, 8);
			sl811_write_reg(SL811HS_USBA_HOSTBASEADDR, 0x10);
			sl811_write_reg(SL811HS_USBA_HOSTBASEDLEN, 8);
			sl811_write_reg(SL811HS_USBA_HOSTDEVADDR, device_address);
			sl811_write_reg(SL811HS_USBA_HOSTPID, PID_SETUP);
			sl811_write_reg(SL811HS_USBA_HOSTCTRL, DATA0_WR);		// SETUP always uses a DATA0 PID. (Ref, 8.5.3 "Control Transfers")
			wait_100us(); wait_100us();								// 200us wait
			write_setup_proc++;
			break;
		case 0x01:
			if(sl811_read_reg(SL811HS_STATUS) & SL811HS_INTENV_BIT_USBA == 0) {
				break;
			} else {
				write_setup_proc++;
				break;
			}
			break;
		case 0x02:
			packet_status = sl811_read_reg(SL811HS_USBA_HOSTSTATUS);
			if( packet_status & SL811HS_HOSTSTATUS_BIT_ACK ) {
				ret = SUB_PROC_DONE;
				write_setup_proc = 0x00;
			} else {
				ret = SUB_PROC_ERROR;
			}
			break;
		default:
			ret = SUB_PROC_ERROR;
			break;
	}
	return ret;
}

// -------------------------------------------
//  [LOCAL] SL811 data send process
// TODO: Maintainance of recovery sequence
// -------------------------------------------
static SUB_PROC_RESULT sl811_write(unsigned char ep, unsigned char* data, int size)
{
	static int remained_size;
	static unsigned char packet_size;
	SUB_PROC_RESULT ret;
	
	ret = SUB_PROC_BUSY;
	
	switch( write_proc ) {
		case 0x00:
			remained_size = size;
			if(size < 0) {
				ret = SUB_PROC_ERROR;
				break;
			}
			write_proc++;
			// break;
			
		case 0x01:
			// Packet setting
			if( remained_size == 0 )						packet_size = 0;
			else if( ep_info[ep].size <= remained_size )	packet_size = ep_info[ep].size;
			else											packet_size = remained_size;
			
			sl811_write_buf(0x10, data+(size-remained_size), packet_size);
			sl811_write_reg(SL811HS_USBA_HOSTBASEADDR, 0x10);
			sl811_write_reg(SL811HS_USBA_HOSTBASEDLEN, packet_size);
			sl811_write_reg(SL811HS_USBA_HOSTDEVADDR, device_address);
			sl811_write_reg(SL811HS_USBA_HOSTPID, PID_OUT | ep);
			sl811_write_reg(SL811HS_USBA_HOSTCTRL, (ep_info[ep].toggle & 0x01 == 0) ? DATA0_WR : DATA1_WR);
			wait_100us(); wait_100us();			// 200us
			write_proc++;
//			break;
			
		case 0x02:
			if(sl811_read_reg(SL811HS_STATUS) & SL811HS_INTENV_BIT_USBA == 0) {
				break;
			} else {
				write_proc++;
				break;
			}
			
		case 0x03:
			packet_status = sl811_read_reg(SL811HS_USBA_HOSTSTATUS);
			if( packet_status & SL811HS_HOSTSTATUS_BIT_ACK ) {
				remained_size -= packet_size;
				ep_info[ep].toggle++;
				if( remained_size > 0 ) write_proc = 0x01;
				else {
					ret = SUB_PROC_DONE;
					write_proc = 0x00;
				}
			} else {
				ret = SUB_PROC_ERROR;
			}
			break;
		default:
			ret = SUB_PROC_ERROR;
			break;
	}
	return ret;
}

// -------------------------------------------
//  [LOCAL] SL811 data receive process
// TODO: Maintainance of recovery sequence
// -------------------------------------------
static SUB_PROC_RESULT sl811_read(unsigned char ep, unsigned char* data, int size)
{
	static int timeout_counter;
	static int remained_size;
	static unsigned char packet_size;
	SUB_PROC_RESULT ret;
	
	ret = SUB_PROC_BUSY;
	
	switch( read_proc ) {
		case 0x00:
			remained_size = size;
			if(size < 0) {
				ret = SUB_PROC_ERROR;
				break;
			}
			read_proc++;
			// break;
			
		case 0x01:
			// Packet setting
			if( remained_size == 0 )						packet_size = 0;
			else if( ep_info[ep].size <= remained_size )	packet_size = ep_info[ep].size;
			else											packet_size = remained_size;
			
			sl811_write_reg(SL811HS_USBA_HOSTBASEADDR, 0x10);
			sl811_write_reg(SL811HS_USBA_HOSTBASEDLEN, packet_size);
			sl811_write_reg(SL811HS_USBA_HOSTDEVADDR, device_address);
			sl811_write_reg(SL811HS_USBA_HOSTPID, PID_IN | ep);
			timeout_counter = 200;
			read_proc++;
			//break;
			
		case 0x02:
			// SETUP command (IN) transfer
			if( timeout_counter ) {
				sl811_write_reg(SL811HS_USBA_HOSTCTRL, (ep_info[ep].toggle & 0x01 == 0) ? DATA0_RD : DATA1_RD);
				wait_100us(); wait_100us();			// 200us
				read_proc = 0x03;
			} else {
				read_proc = 0x04;
			}
			break;
			
		case 0x03:
			// Wait for transfer complete
			if(sl811_read_reg(SL811HS_STATUS) & SL811HS_INTENV_BIT_USBA) {
				packet_status = sl811_read_reg(SL811HS_USBA_HOSTSTATUS);
				timeout_counter--;
				if(packet_status & SL811HS_HOSTSTATUS_BIT_NAK)	read_proc = 0x01;
				else											read_proc++;
			}
			break;
			
		case 0x04:
			// Copy to buffer
			if(packet_status & SL811HS_HOSTSTATUS_BIT_ACK) {
				sl811_read_buf(0x10, data+(size-remained_size), packet_size);
				
				remained_size -= packet_size;
				ep_info[ep].toggle++;
				if( remained_size > 0 ) read_proc = 0x01;
				else {
					ret = SUB_PROC_DONE;
					read_proc = 0x00;
				}
			} else {
				ret = SUB_PROC_ERROR;
			}
			break;
		default:
			ret = SUB_PROC_ERROR;
			break;
	}
	return ret;
}


// -------------------------------------------
//  UI Function - USB function
// -------------------------------------------
int ui_function_usb(UI_COMMAND uicmd) {
// ui function related
	static unsigned int current_index;
	static unsigned int cursol_position;
	int ret_val;
	UI_COMMAND ui_cmd_sub;

	ret_val = UI_RET_READY;
	
	switch( ui_usb_proc ) {
		case UI_USB_01_TOP_LEVEL:
			// Event check
			switch( uicmd.cmd ) {
				case UI_CMD_NOP:
				case UI_CMD_INTEVAL:
					break;
					
				case UI_CMD_STARTED:
					current_index = 0;
					cursol_position = 0;
					break;
					
				case UI_CMD_KEY_PRESS_BACK:
					if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
					ret_val = UI_RET_QUIT;
					break;
					
				case UI_CMD_KEY_PRESS_UP:
					if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
					if( cursol_position == 1 ) {
						cursol_position = 0;
						current_index--;
					} else {
						if( current_index != 0 ) current_index--;
					}
					break;
				case UI_CMD_KEY_PRESS_DOWN:
					if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
					if( cursol_position == 0 ) {
						cursol_position = 1;
						current_index++;
					} else {
						if( usb_app_table[current_index+1].num != -1 ) current_index++;
					}
					break;
				case UI_CMD_KEY_PRESS_OK:
					if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
					ui_cmd_sub.cmd = UI_CMD_STARTED;
					usb_app_table[current_index].pExecFunc(ui_cmd_sub, usb_app_table[current_index].param);	// Initialize
					ui_usb_proc = UI_USB_02_EXECUTING;
					break;
			}
			
			// Menu view
			if( cursol_position == 0) { // cursol in first line
				sc1602_set_buffer_cursol( 0, usb_app_table[current_index  ].display );
				sc1602_set_buffer       ( 1, usb_app_table[current_index+1].display );
			} else {
				sc1602_set_buffer       ( 0, usb_app_table[current_index-1].display );
				sc1602_set_buffer_cursol( 1, usb_app_table[current_index  ].display );
			}
			break;
		
		case UI_USB_02_EXECUTING:
			switch( usb_app_table[current_index].pExecFunc(uicmd, usb_app_table[current_index].param) ) {
				case UI_RET_QUIT:
					ui_usb_proc = UI_USB_01_TOP_LEVEL;
					break;
				default:
					break;
			}
			break;
		default:
			break;
	}
	return ret_val;
}

// -------------------------------------------
//  UI Sub Function - USB debug
// -------------------------------------------
int ui_function_usb_debug(UI_COMMAND uicmd, int param) {
	static unsigned char current_index;
	int ret_val;
	
	ret_val = UI_RET_READY;
	
	switch( uicmd.cmd ) {
	case UI_CMD_NOP:
	case UI_CMD_KEY_PRESS_OK:
		break;
		
	case UI_CMD_INTEVAL:
		// Title
		sc1602_set_buffer( 0, debug_var_table[current_index].display );
		switch(debug_var_table[current_index].type) {
			case 1:
				sc1602_set_buffer_variable1 ( 1, debug_var_table[current_index].p_variable );
				break;
			case 2:
				sc1602_set_buffer_variable2 ( 1, debug_var_table[current_index].p_variable );
				break;
			case 3:
				sc1602_set_buffer_dump (1, debug_var_table[current_index].p_variable );
				break;
		}
		break;
		
	case UI_CMD_STARTED:
		current_index = 0;
		break;
		
	case UI_CMD_KEY_PRESS_UP:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( current_index != 0 ) current_index--;
		break;
	case UI_CMD_KEY_PRESS_DOWN:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( debug_var_table[current_index+1].num !=  -1 ) current_index++;
		break;
	case UI_CMD_KEY_PRESS_BACK:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		ret_val = UI_RET_QUIT;
		break;
	}
	return ret_val;
}

// -------------------------------------------
//  UI Sub Function - USB Descriptor view (Device)
// -------------------------------------------
int ui_function_usb_desc_device(UI_COMMAND uicmd, int param) {
	static unsigned char current_index;
	int ret_val;
	
	ret_val = UI_RET_READY;
	
	switch( uicmd.cmd ) {
	case UI_CMD_NOP:
	case UI_CMD_KEY_PRESS_OK:
		break;
		
	case UI_CMD_INTEVAL:
		// Title
		sc1602_set_buffer( 0, device_var_table[current_index].display );
		switch(device_var_table[current_index].type) {
			case 1:
				sc1602_set_buffer_variable1 ( 1, device_var_table[current_index].p_variable );
				break;
			case 2:
				sc1602_set_buffer_variable2 ( 1, device_var_table[current_index].p_variable );
				break;
			case 3:
				sc1602_set_buffer_dump (1, device_var_table[current_index].p_variable );
				break;
		}
		break;
		
	case UI_CMD_STARTED:
		current_index = 0;
		break;
		
	case UI_CMD_KEY_PRESS_UP:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( current_index != 0 ) current_index--;
		break;
	case UI_CMD_KEY_PRESS_DOWN:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( device_var_table[current_index+1].num !=  -1 ) current_index++;
		break;
	case UI_CMD_KEY_PRESS_BACK:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		ret_val = UI_RET_QUIT;
		break;
	}
	return ret_val;
}

// -------------------------------------------
//  UI Sub Function - USB Descriptor view (Config)
// -------------------------------------------
int ui_function_usb_desc_config(UI_COMMAND uicmd, int param) {
	static unsigned char current_index;
	int ret_val;
	
	ret_val = UI_RET_READY;
	
	switch( uicmd.cmd ) {
	case UI_CMD_NOP:
	case UI_CMD_KEY_PRESS_OK:
		break;
		
	case UI_CMD_INTEVAL:
		// Title
		sc1602_set_buffer( 0, config_var_table[current_index].display );
		switch(config_var_table[current_index].type) {
			case 1:
				sc1602_set_buffer_variable1 ( 1, config_var_table[current_index].p_variable );
				break;
			case 2:
				sc1602_set_buffer_variable2 ( 1, config_var_table[current_index].p_variable );
				break;
			case 3:
				sc1602_set_buffer_dump (1, config_var_table[current_index].p_variable );
				break;
		}
		break;
		
	case UI_CMD_STARTED:
		current_index = 0;
		break;
		
	case UI_CMD_KEY_PRESS_UP:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( current_index != 0 ) current_index--;
		break;
	case UI_CMD_KEY_PRESS_DOWN:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( config_var_table[current_index+1].num !=  -1 ) current_index++;
		break;
	case UI_CMD_KEY_PRESS_BACK:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		ret_val = UI_RET_QUIT;
		break;
	}
	return ret_val;
}

// -------------------------------------------
//  UI Sub Function - USB Descriptor view (Interface - 0)
// -------------------------------------------
int ui_function_usb_desc_interface(UI_COMMAND uicmd, int param) {
	static unsigned char current_index;
	int ret_val;
	
	ret_val = UI_RET_READY;
	
	switch( uicmd.cmd ) {
	case UI_CMD_NOP:
	case UI_CMD_KEY_PRESS_OK:
		break;
		
	case UI_CMD_INTEVAL:
		// Title
		sc1602_set_buffer( 0, interface_var_table[current_index].display );
		switch(interface_var_table[current_index].type) {
			case 1:
				sc1602_set_buffer_variable1 ( 1, interface_var_table[current_index].p_variable );
				break;
			case 2:
				sc1602_set_buffer_variable2 ( 1, interface_var_table[current_index].p_variable );
				break;
			case 3:
				sc1602_set_buffer_dump (1, interface_var_table[current_index].p_variable );
				break;
		}
		break;
		
	case UI_CMD_STARTED:
		current_index = 0;
		break;
		
	case UI_CMD_KEY_PRESS_UP:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( current_index != 0 ) current_index--;
		break;
	case UI_CMD_KEY_PRESS_DOWN:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( interface_var_table[current_index+1].num !=  -1 ) current_index++;
		break;
	case UI_CMD_KEY_PRESS_BACK:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		ret_val = UI_RET_QUIT;
		break;
	}
	return ret_val;
}

// -------------------------------------------
//  UI Sub Function - USB Descriptor view (Endpoints)
// -------------------------------------------
int ui_function_usb_desc_endpoints(UI_COMMAND uicmd, int param) {
	static unsigned char current_index;
	int ret_val, offset;
	
	ret_val = UI_RET_READY;
	
	switch( uicmd.cmd ) {
	case UI_CMD_NOP:
	case UI_CMD_KEY_PRESS_OK:
		break;
		
	case UI_CMD_INTEVAL:
		// Title
		offset = param * sizeof(descriptor_endpoints[0]);
		sc1602_set_buffer( 0, endpoints_var_table[current_index].display);
		switch(endpoints_var_table[current_index].type) {
			case 1:
				sc1602_set_buffer_variable1 ( 1, endpoints_var_table[current_index].p_variable + offset );
				break;
			case 2:
				sc1602_set_buffer_variable2 ( 1, endpoints_var_table[current_index].p_variable + offset );
				break;
			case 3:
				sc1602_set_buffer_dump (1, endpoints_var_table[current_index].p_variable + offset );
				break;
		}
		break;
		
	case UI_CMD_STARTED:
		current_index = 0;
		break;
		
	case UI_CMD_KEY_PRESS_UP:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( current_index != 0 ) current_index--;
		break;
	case UI_CMD_KEY_PRESS_DOWN:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		if( endpoints_var_table[current_index+1].num !=  -1 ) current_index++;
		break;
	case UI_CMD_KEY_PRESS_BACK:
		if( uicmd.param == OFF_EDGE ) break; // Ignore off edge
		ret_val = UI_RET_QUIT;
		break;
	}
	return ret_val;
}
// -------------------------------------------
//  UI Sub Function - USB nop
// -------------------------------------------
int ui_function_usb_nop(UI_COMMAND uicmd, int param)
{
	return UI_RET_QUIT;
}
