
/* --------------------------------------------- */
/*  H8-3069F Switch control function             */
/*                (c) KAZ.Imamura                */
/* --------------------------------------------- */

#include "key.h"
#include "sc1602.h"
#include "ui.h"

#define _KEY0	P5DR.BIT.B0
#define _KEY1	P5DR.BIT.B1
#define _KEY2	P5DR.BIT.B2
#define _KEY3	P5DR.BIT.B3
#define CH_TIMEUP      10

typedef union {
	unsigned char BYTE;
	
	struct {
		unsigned char up:1;
		unsigned char down:1;
		unsigned char back:1;
		unsigned char ok:1;
		unsigned char reserve:4;
	} BIT;
} KEY_HW_INFO;

// -------------------------------------------
//  Proto type definitions
// -------------------------------------------
void key_1ms_handler(void);
int key_initialize(void);
int key_process(void);


// Locals
static void key_event_up  (unsigned char edge);
static void key_event_down(unsigned char edge);
static void key_event_back(unsigned char edge);
static void key_event_ok  (unsigned char edge);

// -------------------------------------------
//  Variables
// -------------------------------------------

// Locals
unsigned short key_timer;
unsigned short key_wait_timer;
unsigned short key_proc;
int ch_timer_up;				// Chattering wait timer for UP key
int ch_timer_down;				// Chattering wait timer for DOWN key
int ch_timer_back;				// Chattering wait timer for BACK key
int ch_timer_ok;				// Chattering wait timer for OK key

KEY_HW_INFO current_key_status;
KEY_HW_INFO previous_key_status;
KEY_HW_INFO status_key_status;

enum key_process_mode {
	KEY_00_INIT,
	KEY_01_STARTUP,
	KEY_02_IDLE,
	KEY_03_READ,
};

// -------------------------------------------
//  Interrupt handlers
// -------------------------------------------
void key_1ms_handler(void) {
	if( key_wait_timer )    key_wait_timer--;

	// 4ms interval, kick off key read process
	if( !( ++key_timer  & 0x03 ) ) {
		if( key_proc == KEY_02_IDLE ) key_proc = KEY_03_READ;
	}
}


// -------------------------------------------
//  Initialize
// -------------------------------------------
int key_initialize(void) {
	key_wait_timer = 0;
	key_proc = KEY_00_INIT;
	
	// Chattering timer initialize
	ch_timer_up = 0;
	ch_timer_down = 0;
	ch_timer_back = 0;
	ch_timer_ok = 0;
	
}

// -------------------------------------------
//  Main process
// -------------------------------------------
int key_process(void) {
	switch( key_proc ) {
	case KEY_00_INIT:
		// read conditions
		current_key_status.BIT.up    = _KEY0;
		current_key_status.BIT.down  = _KEY1;
		current_key_status.BIT.back  = _KEY2;
		current_key_status.BIT.ok    = _KEY3;
		
		status_key_status.BYTE = current_key_status.BYTE;
		previous_key_status.BYTE = current_key_status.BYTE;
		
		key_wait_timer = 500;       // Start up wait
		key_proc++;
		break;
		
	case KEY_01_STARTUP:
		if( !key_wait_timer ) key_proc++;
		break;
		
	case KEY_02_IDLE:
		break;
		
	case KEY_03_READ:
		// read conditions
		current_key_status.BIT.up    = _KEY0;
		current_key_status.BIT.down  = _KEY1;
		current_key_status.BIT.back  = _KEY2;
		current_key_status.BIT.ok    = _KEY3;
		
		// UP key
		if( current_key_status.BIT.up != previous_key_status.BIT.up ) {
			ch_timer_up = CH_TIMEUP;
		} else if ( ch_timer_up ) {
			if( !(--ch_timer_up) ) {
				status_key_status.BIT.up = !current_key_status.BIT.up;
				key_event_up(status_key_status.BIT.up);
			}
		}
		// DOWN key
		if( current_key_status.BIT.down != previous_key_status.BIT.down ) {
			ch_timer_down = CH_TIMEUP;
		} else if ( ch_timer_down ) {
			if( !(--ch_timer_down) ) {
				status_key_status.BIT.down = !current_key_status.BIT.down;
				key_event_down(status_key_status.BIT.down);
			}
		}
		// BACK_KEY
		if( current_key_status.BIT.back != previous_key_status.BIT.back ) {
			ch_timer_back = CH_TIMEUP;
		} else if ( ch_timer_back ) {
			if( !(--ch_timer_back) ) {
				status_key_status.BIT.back = !current_key_status.BIT.back;
				key_event_back(status_key_status.BIT.back);
			}
		}
		// OK key
		if( current_key_status.BIT.ok != previous_key_status.BIT.ok ) {
			ch_timer_ok = CH_TIMEUP;
		} else if ( ch_timer_ok ) {
			if( !(--ch_timer_ok) ) {
				status_key_status.BIT.ok = !current_key_status.BIT.ok;
				key_event_ok(status_key_status.BIT.ok);
			}
		}
		previous_key_status.BYTE = current_key_status.BYTE;
		key_proc = KEY_02_IDLE;
	}
	
	return 0;
}

// -------------------------------------------
//  Key events
// -------------------------------------------
static void key_event_up  (unsigned char edge) {
//	unsigned char da;
	UI_COMMAND uicmd;
	
	// UI Command set
	uicmd.cmd = UI_CMD_KEY_PRESS_UP;
	uicmd.param = edge;
	ui_cmdset(uicmd);
	
	if( edge == ON_EDGE ) {
//		da = DA.DADR0;
//		da += 0x04;
//		DA.DADR0 = da;
	} else {
	}
}

static void key_event_down(unsigned char edge) {
	unsigned char da;
	UI_COMMAND uicmd;
	
	// UI Command set
	uicmd.cmd = UI_CMD_KEY_PRESS_DOWN;
	uicmd.param = edge;
	ui_cmdset(uicmd);

	if( edge == ON_EDGE ) {
//		da = DA.DADR0;
//		da -= 0x04;
//		DA.DADR0 = da;
	} else {
	}
}

static void key_event_back(unsigned char edge) {
	UI_COMMAND uicmd;
	
	// UI Command set
	uicmd.cmd = UI_CMD_KEY_PRESS_BACK;
	uicmd.param = edge;
	ui_cmdset(uicmd);

	if( edge == ON_EDGE ) {
//		strcpy( sc1602_buffer[0], "BACK KEY ON     ");
//		sc1602_buffer_updates[0]=1;
	} else {
//		strcpy( sc1602_buffer[0], "BACK KEY OFF    ");
//		sc1602_buffer_updates[0]=1;
	}
}

static void key_event_ok  (unsigned char edge){
	UI_COMMAND uicmd;
	
	// UI Command set
	uicmd.cmd = UI_CMD_KEY_PRESS_OK;
	uicmd.param = edge;
	ui_cmdset(uicmd);
	
	if( edge == ON_EDGE ) {
//		strcpy( sc1602_buffer[1], "OK KEY ON       ");
//		sc1602_buffer_updates[1]=1;
	} else {
//		strcpy( sc1602_buffer[1], "OK KEY OFF      ");
//		sc1602_buffer_updates[1]=1;
	}
}

