/*
 * $Id: PropertyMessageResources.java 480549 2006-11-29 12:16:15Z niallp $
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.struts.util;

import java.io.IOException;
import java.io.InputStream;
import java.util.HashMap;
import java.util.Locale;
import java.util.Map;
import java.util.Properties;
import java.util.concurrent.ConcurrentHashMap;
import java.util.concurrent.ConcurrentMap;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;

/**
 * Concrete subclass of <code>MessageResources</code> that reads message keys
 * and corresponding strings from named property resources in a <b><i>similar</i></b> manner
 * (see <i>modes</i> below) that <code>java.util.PropertyResourceBundle</code> does.  The
 * <code>base</code> property defines the base property resource name, and
 * must be specified. <p> <strong>IMPLEMENTATION NOTE</strong> - This class
 * trades memory for speed by caching all messages located via generalizing
 * the Locale under the original locale as well. This results in specific
 * messages being stored in the message cache more than once, but improves
 * response time on subsequent requests for the same locale + key
 * combination.
 *
 * <h2>Operating Modes</h2>
 * This implementation can be configured to operate in one of three modes:
 * <ul>
 *    <li>1. <b>Default</b> - default, backwardly compatible, Struts behaviour (i.e. the way
 *    its always worked).</li>
 *    <li>2. <b>JSTL</b> - compatible with how JSTL finds messages
 *        (fix for <a href="http://issues.apache.org/struts/browse/STR-2925">STR-2925</a>)</li>
 *    <li>3. <b>Resource</b> - compatible with how Java's <code>PropertyResourceBundle</code>
 *        finds messages (fix for
 *        <a href="http://issues.apache.org/struts/browse/STR-2077">STR-2077</a>)</li>
 * </ul>
 *
 * <h3>1. Default Mode</h3>
 * <i>Default mode</i> is the way this implementation has always operated. It searches
 * for a message key for property resources in the following sequence:
 * <pre>
 *      base + "_" + localeLanguage + "_" + localeCountry + "_" + localeVariant
 *      base + "_" + localeLanguage + "_" + localeCountry
 *      base + "_" + localeLanguage
 *      base + "_" + default locale
 *      base
 * </pre>
 * <p>
 * This mode is the <i>default</i> and requires no additional configuration.
 *
 * <h3>2. JSTL Mode</h3>
 * <i>JSTL mode</i> is compatible with how JSTL operates and the default Locale
 * is not used when looking for a message key. <i>JSTL mode</i> searches for
 * a message key for property resources in the following sequence:
 * <pre>
 *      base + "_" + localeLanguage + "_" + localeCountry + "_" + localeVariant
 *      base + "_" + localeLanguage + "_" + localeCountry
 *      base + "_" + localeLanguage
 *      base
 * </pre>
 * <p>
 * Configure <code>PropertyMessageResources</code> to operate in this mode by
 * specifying a value of <code>JSTL</code> for the <code>mode</code>
 * key in your <code>struts-config.xml</code>:
 * <pre>
 *      &lt;message-resources parameter="mypackage.MyMessageResources"&gt;
 *          &lt;set-property key="mode" value="JSTL"/&gt;
 *      &lt;/message-resources&gt;
 * </pre>
 *
 * <h3>3. Resource Mode</h3>
 * <i>Resource mode</i> is compatible with how Java's <code>PropertyResourceBundle</code>
 * operates. <i>Resource mode</i> searches first through the specified Locale's language,
 * country and variant, then through the default Locale's language,
 * country and variant and finally using just the <code>base</code>:
 * <pre>
 *      base + "_" + localeLanguage + "_" + localeCountry + "_" + localeVariant
 *      base + "_" + localeLanguage + "_" + localeCountry
 *      base + "_" + localeLanguage
 *      base + "_" + defaultLanguage + "_" + defaultCountry + "_" + defaultVariant
 *      base + "_" + defaultLanguage + "_" + defaultCountry
 *      base + "_" + defaultLanguage
 *      base
 * </pre>
 * <p>
 * Configure <code>PropertyMessageResources</code> to operate in this mode by
 * specifying a value of <code>resource</code> for the <code>mode</code>
 * key in your <code>struts-config.xml</code>:
 * <pre>
 *      &lt;message-resources parameter="mypackage.MyMessageResources"&gt;
 *          &lt;set-property key="mode" value="resource"/&gt;
 *      &lt;/message-resources&gt;
 * </pre>
 *
 * @version $Rev: 480549 $ $Date: 2006-11-29 06:16:15 -0600 (Wed, 29 Nov 2006) $
 */
public class PropertyMessageResources extends MessageResources {

    /** serialVersionUID */
    private static final long serialVersionUID = 6110925527101249875L;

    /**
     * The <code>Log</code> instance for this class.
     */
    private static final Log LOG = LogFactory.getLog(PropertyMessageResources.class);

    // ------------------------------------------------------------- Properties

    /**
     * The set of locale keys for which we have already loaded messages, keyed
     * by the value calculated in <code>localeKey()</code>.
     */
    private final Map<String, String> locales = new HashMap<>();

    /**
     * The cache of messages we have accumulated over time, keyed by the value
     * calculated in <code>messageKey()</code>.
     */
    private final ConcurrentMap<String, String> messages = new ConcurrentHashMap<>();

    // ----------------------------------------------------------- Constructors

    /**
     * Construct a new PropertyMessageResources according to the specified
     * parameters.
     *
     * @param config  The configuration parameter for this MessageResources
     */
    public PropertyMessageResources(final String config) {
        super(config);
        LOG.trace("Initializing, config='" + config + "'");
    }

    /**
     * Construct a new PropertyMessageResources according to the specified
     * parameters.
     *
     * @param config     The configuration parameter for this
     *                   MessageResources
     * @param returnNull The returnNull property we should initialize with
     */
    public PropertyMessageResources(final String config, final boolean returnNull) {
        super(config, returnNull);
        LOG.trace("Initializing, config='" + config + "', returnNull=" + returnNull);
    }

    /**
     * Construct a new PropertyMessageResources according to the specified
     * parameters.
     *
     * @param config     The configuration parameter for this
     *                   MessageResources
     * @param returnNull The returnNull property we should initialize with
     * @param escape escape processing
     */
    public PropertyMessageResources(
            final String config, final boolean returnNull, final boolean escape) {
        super(config, returnNull, escape);
        LOG.trace("Initializing, config='" + config + "', returnNull=" + returnNull);
    }

    // --------------------------------------------------------- Public Methods

    /**
     * Returns a text message for the specified key, for the specified or default
     * Locale. A null string result will be returned by this method if no relevant
     * message resource is found for this key or Locale, if the
     * <code>returnNull</code> property is set.  Otherwise, an appropriate
     * error message will be returned. <p> This method must be implemented by
     * a concrete subclass.
     *
     * @param locale The requested message Locale, or <code>null</code> for
     *               the system default Locale
     * @param key    The message key to look up
     * @return text message for the specified key and locale
     */
    @Override
    public String getMessage(final Locale locale, final String key) {
        if (LOG.isDebugEnabled()) {
            LOG.debug("getMessage(" + locale + "," + key + ")");
        }

        // Initialize variables we will require
        String localeKey = localeKey(locale);
        String originalKey = messageKey(localeKey, key);

        // Search the specified Locale
        String message = findMessage(locale, key, originalKey);
        if (message != null) {
            return message;
        }

        // Find the message in the default properties file
        message = findMessage("", key, originalKey);
        if (message != null) {
            return message;
        }

        // Return an appropriate error indication
        if (super.getReturnNull()) {
            return null;
        }

        return "???" + messageKey(locale, key) + "???";
    }

    // ------------------------------------------------------ Protected Methods

    /**
     * Load the messages associated with the specified Locale key.  For this
     * implementation, the <code>config</code> property should contain a fully
     * qualified package and resource name, separated by periods, of a series
     * of property resources to be loaded from the class loader that created
     * this PropertyMessageResources instance.  This is exactly the same name
     * format you would use when utilizing the <code>java.util.PropertyResourceBundle</code>
     * class.
     *
     * @param localeKey Locale key for the messages to be retrieved
     */
    protected synchronized void loadLocale(final String localeKey) {
        if (LOG.isTraceEnabled()) {
            LOG.trace("loadLocale(" + localeKey + ")");
        }

        // Have we already attempted to load messages for this locale?
        if (this.locales.get(localeKey) != null) {
            return;
        }

        this.locales.put(localeKey, localeKey);

        // Set up to load the property resource for this locale key, if we can
        String name = super.getConfig().replace('.', '/');
        if (!localeKey.isEmpty()) {
            name = name + "_" + localeKey;
        }
        name += ".properties";

        // Load the specified property resource
        if (LOG.isTraceEnabled()) {
            LOG.trace("  Loading resource '" + name + "'");
        }

        ClassLoader classLoader = Thread.currentThread().getContextClassLoader();
        if (classLoader == null) {
            classLoader = this.getClass().getClassLoader();
        }

        Properties props = new Properties();
        try (InputStream is = classLoader.getResourceAsStream(name)) {
            if (is != null) {
                props.load(is);

                if (LOG.isTraceEnabled()) {
                    LOG.trace("  Loading resource completed");
                }
            } else {
                if (LOG.isWarnEnabled()) {
                    LOG.warn("  Resource " + name + " Not Found.");
                }
            }
        } catch (final IOException e) {
            LOG.error("loadLocale()", e);
        }

        for (final String key : props.stringPropertyNames()) {
            if (LOG.isTraceEnabled()) {
                LOG.trace("  Saving message key '" + messageKey(localeKey, key));
            }
            this.messages.putIfAbsent(messageKey(localeKey, key), props.getProperty(key));
        }
    }

    // -------------------------------------------------------- Private Methods

    /**
     * Returns a text message for the specified key, for the specified Locale.
     * <p>
     * A null string result will be returned by this method if no relevant
     * message resource is found. This method searches through the locale
     * <i>hierarchy</i> (i.e. variant --> languge --> country) for the message.
     *
     * @param locale The requested message Locale, or <code>null</code> for
     *  the system default Locale
     * @param key The message key to look up
     * @param originalKey The original message key to cache any found message under
     * @return text message for the specified key and locale
     */
    protected String findMessage(final Locale locale, final String key, final String originalKey) {

        // Initialize variables we will require
        String localeKey = super.localeKey(locale);

        String message = findMessage(localeKey, key, originalKey);

        // Loop from specific to general Locales looking for this message
        while (message == null) {
            // Strip trailing modifiers to try a more general locale key
            int underscore = localeKey.lastIndexOf('_');
            if (underscore < 0) {
                break;
            }
            localeKey = localeKey.substring(0, underscore);

            message = findMessage(localeKey, key, originalKey);
        }

        return message;

    }

    /**
     * Returns a text message for the specified key, for the specified Locale.
     * <p>
     * A null string result will be returned by this method if no relevant
     * message resource is found.
     *
     * @param localeKey The requested key of the Locale
     * @param key The message key to look up
     * @param originalKey The original message key to cache any found message under
     * @return text message for the specified key and locale
     */
    protected String findMessage(final String localeKey,
            final String key, final String originalKey) {

        // Load this Locale's messages if we have not done so yet
        loadLocale(localeKey);

        // Check if we have this key for the current locale key
        String messageKey = super.messageKey(localeKey, key);

        String message = this.messages.get(messageKey);
        if (message != null) {
            // Add if not found under the original key
            if (!messageKey.equals(originalKey)) {
                if (this.messages.putIfAbsent(originalKey, message) != null) {
                    message = this.messages.get(originalKey);
                }
            }
        }
        return message;
    }
}
