/*
 * $Id: MessageResourcesFactory.java 471754 2006-11-06 14:55:09Z husted $
 *
 * Licensed to the Apache Software Foundation (ASF) under one
 * or more contributor license agreements.  See the NOTICE file
 * distributed with this work for additional information
 * regarding copyright ownership.  The ASF licenses this file
 * to you under the Apache License, Version 2.0 (the
 * "License"); you may not use this file except in compliance
 * with the License.  You may obtain a copy of the License at
 *
 *  http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing,
 * software distributed under the License is distributed on an
 * "AS IS" BASIS, WITHOUT WARRANTIES OR CONDITIONS OF ANY
 * KIND, either express or implied.  See the License for the
 * specific language governing permissions and limitations
 * under the License.
 */

package org.apache.struts.util;

import org.apache.commons.logging.Log;
import org.apache.commons.logging.LogFactory;
import org.apache.struts.config.MessageResourcesConfig;

import java.io.Serializable;

/**
 * Factory for <code>MessageResources</code> instances.  The general usage
 * pattern for this class is:
 *
 * <ul>
 *
 * <li>Call <code>MessageResourcesFactory().createFactory()</code> to retrieve
 * a <code>MessageResourcesFactory</code> instance.</li> <li>Set properties as
 * required to configure this factory instance to create
 * <code>MessageResources</code> instances with desired characteristics.</li>
 *
 * <li>Call the <code>createResources()</code> method of the factory to
 * retrieve a newly instantiated <code>MessageResources</code> instance.</li>
 *
 * </ul>
 *
 * @version $Rev: 471754 $ $Date: 2005-08-29 23:57:50 -0400 (Mon, 29 Aug 2005)
 *          $
 */
public abstract class MessageResourcesFactory implements Serializable {

    // ------------------------------------------------------ Static Properties

    /** serialVersionUID */
    private static final long serialVersionUID = 7329616765280441078L;

    /**
     * Commons Logging instance.
     */
    private static final Log LOG = LogFactory.getLog(MessageResourcesFactory.class);

    /**
     * The fully qualified class name to be used for <code>MessageResourcesFactory</code>
     * instances.
     */
    private static final String DEFAULT_FACTORY_CLASS =
        "org.apache.struts.util.PropertyMessageResourcesFactory";

    /**
     * The default MessageResourcesFactory used to create MessageResources
     * instances.
     */
    private static final MessageResourcesFactory DEFAULT_FACTORY =
            MessageResourcesFactory.createFactory();

    // --------------------------------------------------------- Public Methods

    /**
     *
     * @param config Configuration parameter(s) for the requested bundle
     * @return MessageResources
     */
    public MessageResources createResources(final String config) {
        MessageResourcesConfig mrc = new MessageResourcesConfig();
        mrc.setParameter(config);
        return createResources(mrc);
    }

    /**
     * Create and return a newly instansiated <code>MessageResources</code>.
     * This method must be implemented by concrete subclasses.
     * @param mrc MessageResourcesConfig
     * @return MessageResources
     */
    public abstract MessageResources createResources(final MessageResourcesConfig mrc);

    // --------------------------------------------------------- Static Methods

    /**
     *
     * @return Default MessageResourcesFactory
     */
    public static MessageResourcesFactory createFactory() {
        return createFactory(DEFAULT_FACTORY_CLASS);
    }

    /**
     * Create and return a <code>MessageResourcesFactory</code> instance of
     * the appropriate class, which can be used to create customized
     * <code>MessageResources</code> instances.  If no such factory can be
     * created, return <code>null</code> instead.
     * @param factoryClass Factory Class Name
     * @return MessageResourcesFactory
     */
    public static MessageResourcesFactory createFactory(final String factoryClass) {
        try {
            Class<MessageResourcesFactory> clazz = ResponseUtils.applicationClass(factoryClass);
            MessageResourcesFactory factory = clazz.newInstance();
            return factory;

        } catch (final ReflectiveOperationException t) {
            LOG.error("MessageResourcesFactory.createFactory", t);
            return null;
        }
    }

    /**
     * Create and return an instance of <code>MessageResources</code> for the
     * created by the default <code>MessageResourcesFactory</code>.
     * @param config Configuration parameter for this message bundle.
     * @return MessageResources
     */
    public static MessageResources getMessageResources(final String config) {
        return DEFAULT_FACTORY.createResources(config);
    }
}
