/*
 * Licensed to the Apache Software Foundation (ASF) under one or more
 * contributor license agreements.  See the NOTICE file distributed with
 * this work for additional information regarding copyright ownership.
 * The ASF licenses this file to You under the Apache License, Version 2.0
 * (the "License"); you may not use this file except in compliance with
 * the License.  You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND, either express or implied.
 * See the License for the specific language governing permissions and
 * limitations under the License.
 */

package org.apache.commons.chain2.web.servlet;

import java.io.Serializable;
import java.util.ArrayList;
import java.util.Collection;
import java.util.Collections;
import java.util.HashSet;
import java.util.List;
import java.util.Map;
import java.util.Set;

import javax.servlet.http.HttpServletRequest;

import org.apache.commons.chain2.web.MapEntry;

/**
 * <p>Private implementation of <code>Map</code> for servlet request
 * attributes.</p>
 *
 * @version $Id$
 */
final class ServletRequestScopeMap implements Map<String, Object>, Serializable {

    /** serialVersionUID */
    private static final long serialVersionUID = -5371932525499773641L;

    private transient HttpServletRequest request = null;

    ServletRequestScopeMap(final HttpServletRequest req) {
        this.request = req;
    }

    @Override
    public void clear() {
        for (final String key : keySet()) {
            this.request.removeAttribute(key);
        }
    }

    @Override
    public boolean containsKey(final Object key) {
        return this.request.getAttribute(toKey(key)) != null;
    }

    @Override
    public boolean containsValue(final Object value) {
        if (value != null) {
            for (final String key : Collections.list(this.request.getAttributeNames())) {
                Object next = this.request.getAttribute(key);
                if (value.equals(next)) {
                    return true;
                }
            }
        }
        return false;
    }

    @Override
    public Set<Entry<String, Object>> entrySet() {
        Set<Entry<String, Object>> set = new HashSet<>();
        for (final String key : Collections.list(this.request.getAttributeNames())) {
            set.add(new MapEntry<>(key, this.request.getAttribute(key), true));
        }
        return set;
    }

    @Override
    public boolean equals(final Object o) {
        if (o != this) {
            if (!ServletRequestScopeMap.class.isInstance(o)) {
                return false;
            }
            ServletRequestScopeMap map = (ServletRequestScopeMap) o;
            return this.request.equals(map.request);
        }
        return true;
    }

    @Override
    public Object get(final Object key) {
        return this.request.getAttribute(toKey(key));
    }

    @Override
    public int hashCode() {
        return this.request.hashCode();
    }

    @Override
    public boolean isEmpty() {
        return size() < 1;
    }

    @Override
    public Set<String> keySet() {
        return new HashSet<>(Collections.list(this.request.getAttributeNames()));
    }

    @Override
    public Object put(final String key, final Object value) {
        if (value == null) {
            return remove(key);
        }
        Object previous = this.request.getAttribute(key);
        this.request.setAttribute(key, value);
        return previous;
    }

    @Override
    public void putAll(final Map<? extends String, ? extends Object> map) {
        for (final Entry<? extends String, ? extends Object> entry : map.entrySet()) {
            put(toKey(entry.getKey()), entry.getValue());
        }
    }

    @Override
    public Object remove(final Object key) {
        String skey = toKey(key);
        Object previous = this.request.getAttribute(skey);
        this.request.removeAttribute(skey);
        return previous;
    }

    @Override
    public int size() {
        return Collections.list(this.request.getAttributeNames()).size();
    }

    @Override
    public Collection<Object> values() {
        List<Object> list = new ArrayList<>();
        for (final String key : Collections.list(this.request.getAttributeNames())) {
            list.add(this.request.getAttribute(key));
        }
        return list;
    }

    private String toKey(final Object key) {
        if (key == null) {
            throw new IllegalArgumentException();
        }
        return key.toString();
    }

}
