package core.util;

import java.math.BigDecimal;

import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;

/**
 * TestNumberUtil
 *
 * @author Tadashi Nakayama
 */
public class NumberUtilTest {

	/**
	 * Numberオブジェクト化
	 *
	 * @param <T> Numberジェネリックス
	 * @param val 文字列
	 * @param cls Numberクラス
	 * @return Numberオブジェクト
	 */
	private <T extends Number> T toNumber(final String val, final Class<T> cls) {
		return NumberUtil.toNumber(val, cls);
	}

	/**
	 * 数値オブジェクトBigDecimal変換
	 */
	@Test
	@DisplayName("数値オブジェクトBigDecimal変換")
	public void toNumberTest1() {
		Assertions.assertEquals(new BigDecimal("123"), toNumber("123", BigDecimal.class));
	}

	/**
	 * 数値オブジェクトInteger変換
	 */
	@Test
	@DisplayName("数値オブジェクトInteger変換")
	public void toNumberTest2() {
		Assertions.assertEquals(Integer.valueOf("1234"), toNumber("1,234", Integer.class));
	}

	/**
	 * 数値オブジェクトInteger変換不可
	 */
	@Test
	@DisplayName("数値オブジェクトInteger変換不可")
	public void toNumberTest3() {
		Assertions.assertNull(toNumber("123AB", Integer.class));
	}

	/**
	 * 乗算
	 *
	 * @param vals BigDecimal配列
	 * @return 乗算結果
	 */
	private BigDecimal multiply(final BigDecimal... vals) {
		return NumberUtil.multiply(vals);
	}

	/**
	 * 数値オブジェクト掛算nullONE
	 */
	@Test
	@DisplayName("数値オブジェクト掛算nullONE")
	public void multiplyTest1() {
		Assertions.assertEquals(BigDecimal.ONE, multiply(null, BigDecimal.ONE));
	}

	/**
	 * 数値オブジェクト掛算ONEnull
	 */
	@Test
	@DisplayName("数値オブジェクト掛算ONEnull")
	public void multiplyTest2() {
		Assertions.assertEquals(BigDecimal.ONE, multiply(BigDecimal.ONE, null));
	}

	/**
	 * 数値オブジェクト掛算nullnull
	 */
	@Test
	@DisplayName("数値オブジェクト掛算nullnull")
	public void multiplyTest3() {
		Assertions.assertNull(multiply(null, null));
	}

	/**
	 * 数値オブジェクト掛算nullnullZERO
	 */
	@Test
	@DisplayName("数値オブジェクト掛算nullnullZERO")
	public void multiplyTest4() {
		Assertions.assertEquals(BigDecimal.ZERO, multiply(null, null, BigDecimal.ZERO));
	}

	/**
	 * 数値オブジェクト掛算onenullTEN
	 */
	@Test
	@DisplayName("数値オブジェクト掛算onenullTEN")
	public void multiplyTest5() {
		Assertions.assertEquals(BigDecimal.TEN, multiply(BigDecimal.ONE, null, BigDecimal.TEN));
	}

	/**
	 * 加算
	 *
	 * @param vals Integer配列
	 * @return 合計
	 */
	private Integer add(final Integer... vals) {
		return NumberUtil.add(vals);
	}

	/**
	 * Integer加算
	 */
	@Test
	@DisplayName("数値オブジェクト加算Integer")
	public void addIntegerTest1() {
		Assertions.assertEquals(Integer.valueOf(6), add(1, 2, 3));
	}

	/**
	 * 加算
	 *
	 * @param vals Long配列
	 * @return 合計
	 */
	private Long add(final Long... vals) {
		return NumberUtil.add(vals);
	}

	/**
	 * Long加算
	 */
	@Test
	@DisplayName("数値オブジェクト加算Long")
	public void addLongTest1() {
		Assertions.assertEquals(Long.valueOf(4), add(1L, 2L, 1L));
	}

	/**
	 * 加算
	 *
	 * @param vals BigDecimal配列
	 * @return 合計
	 */
	private BigDecimal add(final BigDecimal... vals) {
		return NumberUtil.add(vals);
	}

	/**
	 * BigDecimal加算
	 */
	@Test
	@DisplayName("数値オブジェクト加算BigDecimal")
	public void addLongBigDecimal1() {
		Assertions.assertEquals(new BigDecimal(11), add(BigDecimal.ONE, BigDecimal.TEN));
	}

	/**
	 * int化
	 *
	 * @param val 文字列
	 * @param def デフォルト値
	 * @return int
	 */
	private int toInt(final String val, final int def) {
		return NumberUtil.toInt(val, def);
	}

	/**
	 * int整数正常
	 */
	@Test
	@DisplayName("int整数正常")
	public void toIntTest1() {
		final var val = Integer.valueOf(1);
		Assertions.assertEquals(val, Integer.valueOf(toInt(val.toString(), 0)));
	}

	/**
	 * int小数正常
	 */
	@Test
	@DisplayName("int小数正常")
	public void toIntTest2() {
		final var val = "1.24";
		Assertions.assertEquals(1, toInt(val, 0));
	}
}
