package common.sql;

import java.math.BigDecimal;
import java.sql.Timestamp;
import java.util.ArrayList;
import java.util.Arrays;
import java.util.Calendar;
import java.util.Date;
import java.util.HashMap;
import java.util.List;
import java.util.Map;

import org.junit.jupiter.api.Assertions;
import org.junit.jupiter.api.DisplayName;
import org.junit.jupiter.api.Test;

/**
 * LineParsedNodeListTest
 *
 * @author Tadashi Nakayama
 */
public class LineParsedNodeListTest {

	/**
	 * テストターゲット取得
	 *
	 * @param arg 引数
	 * @return テストターゲット
	 */
	private LineParsedNodeList getTarget(final String arg) {
		return new LineParsedNodeList(arg);
	}

	/**
	 * 日付必須でパラメータあり
	 */
	@Test
	@DisplayName("日付必須でパラメータあり")
	public void buildTest1() {
		final var item = getTarget(
				"UPDATE_DATETIME = /*:DateTime*/TO_DATE('20091225','YYYYMMDD'),");

		final var map = new HashMap<String, Object>();
		map.put("DateTime", "20100118");
		final var list = new ArrayList<>();

		Assertions.assertEquals("UPDATE_DATETIME = ? , \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals(java.sql.Date.class, list.get(0).getClass());
		Assertions.assertEquals("2010-01-18", list.get(0).toString());
	}

	/**
	 * 数値デフォルトでパラメータあり
	 */
	@Test
	@DisplayName("数値デフォルトでパラメータあり")
	public void buildTest2() {
		final var item = getTarget(
				"OFFSET ((/*?Page*/1 - 1) * /*?Line*/20 ) LIMIT /*?Line*/20 \n");

		final var map = new HashMap<String, Object>();
		map.put("Page", "1");
		map.put("Line", "30");
		final var list = new ArrayList<>();

		Assertions.assertEquals("OFFSET ( ( ? - 1 ) * ? ) LIMIT ? \n", item.build(map, list));
		Assertions.assertEquals(3, list.size());
		Assertions.assertEquals(new BigDecimal(1), list.get(0));
		Assertions.assertEquals(new BigDecimal(30), list.get(1));
		Assertions.assertEquals(new BigDecimal(30), list.get(2));
	}

	/**
	 * IN句内複数中単一文字列デフォルトでパラメータなし
	 */
	@Test
	@DisplayName("IN句内複数中単一文字列デフォルトでパラメータなし")
	public void buildTest3() {
		final var item = getTarget(
				"AAA IN(/*?dtGyomu*/'200901', '20', '30')");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("AAA IN( ?, '20', '30' ) \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals("200901", list.get(0));
	}

	/**
	 * IN句内複数中単一数値デフォルトでパラメータなし
	 */
	@Test
	@DisplayName("IN句内複数中単一数値デフォルトでパラメータなし")
	public void buildTest4() {
		final var item = getTarget(
				"CASE WHEN SHUBETSU NOT IN (/*?Shubetsu*/1,2,3,4)");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("CASE WHEN SHUBETSU NOT IN ( ?,2,3,4 ) \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals(BigDecimal.ONE, list.get(0));
	}

	/**
	 * IN句内複数中単一数値可変でパラメータなし
	 */
	@Test
	@DisplayName("IN句内複数中単一数値可変でパラメータなし")
	public void buildTest5() {
		final var item = getTarget(
				"CASE WHEN SHUBETSU NOT IN (/*$Shubetsu*/1,2,3,4)");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("", item.build(map, list));
		Assertions.assertEquals(0, list.size());
	}

	/**
	 * OR連結IN句内一方数値可変でパラメータなし
	 */
	@Test
	@DisplayName("OR連結IN句内一方数値可変でパラメータなし")
	public void buildTest6() {
		final var item = getTarget(
				"AND (SHUBETSU IN (/*$Shubetsu*/1) OR SHUBETSU IN (2,3,4))");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("AND (SHUBETSU IN ( 2,3,4 )) \n", item.build(map, list));
		Assertions.assertEquals(0, list.size());
	}

	/**
	 * IN句内複数中単一数値可変で別行パラメータなし
	 */
	@Test
	@DisplayName("IN句内複数中単一数値可変で別行パラメータなし")
	public void buildTest7() {
		final var item = getTarget(
				"AND SHUBETSU IN (2,3,4 \n, /*$Shubetsu*/1\n)");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("AND SHUBETSU IN (2,3,4 \n) \n", item.build(map, list));
		Assertions.assertEquals(0, list.size());
	}

	/**
	 * IN句内複数中単一数値可変で別行パラメータあり
	 */
	@Test
	@DisplayName("IN句内複数中単一数値可変で別行パラメータあり")
	public void buildTest8() {
		final var item = getTarget(
				"AND SHUBETSU IN (2,3,4 \n, /*$Shubetsu*/1\n)");

		final var map = new HashMap<String, Object>();
		map.put("Shubetsu", 1);
		final var list = new ArrayList<>();

		Assertions.assertEquals("AND SHUBETSU IN (2,3,4 \n, ? \n) \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals(1, list.get(0));
	}

	/**
	 * IN句内全て文字列デフォルトでパラメータなし
	 */
	@Test
	@DisplayName("IN句内全て文字列デフォルトでパラメータなし")
	public void buildTest9() {
		final var item = getTarget(
			"CASE WHEN SHUBETSU NOT IN (/*?Shubetsu1*/'', "
			+ "/*?Shubetsu2*/'', /*?Shubetsu3*/'', /*?Shubetsu4*/'')");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals(
				"CASE WHEN SHUBETSU NOT IN ( ?, ?, ?, ? ) \n", item.build(map, list));
		Assertions.assertEquals(4, list.size());
		Assertions.assertEquals("", list.get(0));
		Assertions.assertEquals("", list.get(1));
		Assertions.assertEquals("", list.get(2));
		Assertions.assertEquals("", list.get(3));
	}

	/**
	 * IN句内全て文字列デフォルトでパラメータなしカンマ密着
	 */
	@Test
	@DisplayName("IN句内全て文字列デフォルトでパラメータなしカンマ密着")
	public void buildTest10() {
		final var item = getTarget(
				"CASE WHEN SHUBETSU NOT IN (/*?Shubetsu1*/'',/*?Shubetsu2*/'',"
				+ "/*?Shubetsu3*/'',/*?Shubetsu4*/'')");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("CASE WHEN SHUBETSU NOT IN ( ?,?,?,? ) \n", item.build(map, list));
		Assertions.assertEquals(4, list.size());
		Assertions.assertEquals("", list.get(0));
		Assertions.assertEquals("", list.get(1));
		Assertions.assertEquals("", list.get(2));
		Assertions.assertEquals("", list.get(3));
	}

	/**
	 * IN句内全て文字列可変でパラメータ一部あり
	 */
	@Test
	@DisplayName("IN句内全て文字列可変でパラメータ一部あり")
	public void buildTest11() {
		final var item = getTarget(
				"CASE WHEN SHUBETSU NOT IN (/*$Shubetsu1*/'',/*$Shubetsu2*/'',"
				+ "/*$Shubetsu3*/'',/*$Shubetsu4*/'')");

		final var map = new HashMap<String, Object>();
		map.put("Shubetsu1", "A");
		map.put("Shubetsu3", "B");
		map.put("Shubetsu4", "C");
		final var list = new ArrayList<>();

		Assertions.assertEquals("", item.build(map, list));
		Assertions.assertEquals(0, list.size());
	}

	/**
	 * IN句内全て文字列必須でパラメータ一部あり
	 */
	@Test
	@DisplayName("IN句内全て文字列必須でパラメータ一部あり")
	public void buildTest12() {
		final var item = getTarget(
				"CASE WHEN SHUBETSU NOT IN (/*:Shubetsu1*/'',/*:Shubetsu2*/'',"
				+ "/*:Shubetsu3*/'',/*:Shubetsu4*/'')");

		final var map = new HashMap<String, Object>();
		map.put("Shubetsu1", "A");
		map.put("Shubetsu3", "B");
		map.put("Shubetsu4", "C");
		final var list = new ArrayList<>();

		Assertions.assertEquals("CASE WHEN SHUBETSU NOT IN ( ?,?,?,? ) \n", item.build(map, list));
		Assertions.assertEquals(4, list.size());
		Assertions.assertEquals("A", list.get(0));
		Assertions.assertNull(list.get(1));
		Assertions.assertEquals("B", list.get(2));
		Assertions.assertEquals("C", list.get(3));
	}

	/**
	 * IN句内単一文字列可変でパラメータなし
	 */
	@Test
	@DisplayName("IN句内単一文字列可変でパラメータなし")
	public void buildTest13() {
		final var item = getTarget(
				"AND SHUBETSU IN (SELECT A.SB FROM SHUBE A "
				+ "WHERE A.S IN (/*$Shubetsu1*/'') AND A.VERSION > 0)");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("", item.build(map, list));
		Assertions.assertEquals(0, list.size());
	}

	/**
	 * IN句内単一文字列必須でパラメータなし
	 */
	@Test
	@DisplayName("IN句内単一文字列必須でパラメータなし")
	public void buildTest14() {
		final var item = getTarget(
				"AND SHUBETSU IN (SELECT A.SB FROM SHUBE A "
				+ "WHERE A.S IN (/*:Shubetsu1*/'') AND A.VERSION > 0)");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals(
			"AND SHUBETSU IN ( SELECT A.SB FROM SHUBE A WHERE A.S IN ( ? ) AND A.VERSION > 0 ) \n",
			item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertNull(list.get(0));
	}

	/**
	 * IN句内単一文字列可変で配列パラメータあり
	 */
	@Test
	@DisplayName("IN句内単一文字列可変で配列パラメータあり")
	public void buildTest15() {
		final var item = getTarget(
				"AND SHUBETSU IN (SELECT A.SB FROM SHUBE A "
				+ "WHERE A.S IN (/*$Shubetsu1*/'') AND A.VERSION > 0)");

		final var map = new HashMap<String, Object>();
		map.put("Shubetsu1", new String[]{"X", "Y", "Z"});
		final var list = new ArrayList<>();

		Assertions.assertEquals(
			"AND SHUBETSU IN ( SELECT A.SB FROM SHUBE A "
			+ "WHERE A.S IN ( ?,?,? ) AND A.VERSION > 0 ) \n",
			item.build(map, list));
		Assertions.assertEquals(3, list.size());
		Assertions.assertEquals("X", list.get(0));
		Assertions.assertEquals("Y", list.get(1));
		Assertions.assertEquals("Z", list.get(2));
	}

	/**
	 * IN句内単一文字列可変でリストパラメータあり
	 */
	@Test
	@DisplayName("IN句内単一文字列可変でリストパラメータあり")
	public void buildTest16() {
		final var item = getTarget(
				"AND SHUBETSU IN (SELECT A.SB FROM SHUBE A "
				+ "WHERE A.S IN (/*$Shubetsu1*/'') AND A.VERSION > 0)");

		final var map = new HashMap<String, Object>();
		map.put("Shubetsu1", new ArrayList<>(Arrays.asList("X", "Y", "Z")));
		final var list = new ArrayList<>();

		Assertions.assertEquals(
				"AND SHUBETSU IN ( SELECT A.SB FROM SHUBE A "
				+ "WHERE A.S IN ( ?,?,? ) AND A.VERSION > 0 ) \n",
				item.build(map, list));
		Assertions.assertEquals(3, list.size());
		Assertions.assertEquals("X", list.get(0));
		Assertions.assertEquals("Y", list.get(1));
		Assertions.assertEquals("Z", list.get(2));
	}

	/**
	 * 複数項目IN句内全て文字列デフォルトでパラメータなし
	 */
	@Test
	@DisplayName("複数項目IN句内全て文字列デフォルトでパラメータなし")
	public void buildTest17() {
		final var item = getTarget(
				"AND (SHUBETSU1, SHUBETSU2) NOT IN (/*?Shubetsu1*//*?Shubetsu2*/('A', 'B'))");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals(
				"AND ( SHUBETSU1, SHUBETSU2 ) NOT IN ( ( ?,? ) ) \n", item.build(map, list));
		Assertions.assertEquals(2, list.size());
		Assertions.assertEquals("A", list.get(0));
		Assertions.assertEquals("B", list.get(1));
	}

	/**
	 * 複数項目IN句内全て文字列デフォルトでパラメータあり
	 */
	@Test
	@DisplayName("複数項目IN句内全て文字列デフォルトでパラメータあり")
	public void buildTest18() {
		final LineParsedNodeList item = getTarget(
				"AND (SHUBETSU1, SHUBETSU2) IN (/*?Shubetsu1*//*?Shubetsu2*/('',''))");

		final var map = new HashMap<String, Object>();
		map.put("Shubetsu1", "A");
		map.put("Shubetsu2", "C");
		final var list = new ArrayList<>();

		Assertions.assertEquals(
				"AND ( SHUBETSU1, SHUBETSU2 ) IN ( ( ?,? ) ) \n", item.build(map, list));
		Assertions.assertEquals(2, list.size());
		Assertions.assertEquals("A", list.get(0));
		Assertions.assertEquals("C", list.get(1));
	}

	/**
	 * 複数項目IN句内単一文字列デフォルトでパラメータなし
	 */
	@Test
	@DisplayName("複数項目IN句内単一文字列デフォルトでパラメータなし")
	public void buildTest19() {
		final var item = getTarget(
				"AND (SHUBETSU1, SHUBETSU2) "
				+ "NOT IN (/*?Shubetsu1*//*?Shubetsu2*/('A', 'B'), ('C', 'D'))");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals(
				"AND ( SHUBETSU1, SHUBETSU2 ) NOT IN ( ( ?,? ) , ( 'C', 'D' ) ) \n",
				item.build(map, list));
		Assertions.assertEquals(2, list.size());
		Assertions.assertEquals("A", list.get(0));
		Assertions.assertEquals("B", list.get(1));
	}

	/**
	 * 複数項目IN句内全て文字列可変でパラメータなし
	 */
	@Test
	@DisplayName("複数項目IN句内全て文字列可変でパラメータなし")
	public void buildTest20() {
		final var item = getTarget(
				"AND (SHUBETSU1, SHUBETSU2) IN ( /*$Shubetsu1*//*$Shubetsu2*/('','') )");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("", item.build(map, list));
		Assertions.assertEquals(0, list.size());
	}

	/**
	 * 複数項目IN句内全て文字列可変で配列パラメータあり
	 */
	@Test
	@DisplayName("複数項目IN句内全て文字列可変で配列パラメータあり")
	public void buildTest21() {
		final var item = getTarget(
				"AND (SHUBETSU1, SHUBETSU2) IN (/*$Shubetsu1*//*$Shubetsu2*/('',''))");

		final var map = new HashMap<String, Object>();
		map.put("Shubetsu1", new String[]{"A", "B", "C"});
		map.put("Shubetsu2", new String[]{"D", "E", "F"});
		final var list = new ArrayList<>();

		Assertions.assertEquals(
				"AND ( SHUBETSU1, SHUBETSU2 ) IN ( ( ?,? ),( ?,? ),( ?,? ) ) \n",
				item.build(map, list));
		Assertions.assertEquals(6, list.size());
		Assertions.assertEquals("A", list.get(0));
		Assertions.assertEquals("D", list.get(1));
		Assertions.assertEquals("B", list.get(2));
		Assertions.assertEquals("E", list.get(3));
		Assertions.assertEquals("C", list.get(4));
		Assertions.assertEquals("F", list.get(5));
	}

	/**
	 * 複数項目IN句内全て文字列可変でリストパラメータあり
	 */
	@Test
	@DisplayName("複数項目IN句内全て文字列可変でリストパラメータあり")
	public void buildTest22() {
		final var item = getTarget(
				"AND (SHUBETSU1, SHUBETSU2) IN (/*$Shubetsu1*//*$Shubetsu2*/('',''))");

		final var map = new HashMap<String, Object>();
		map.put("Shubetsu1", new ArrayList<>(Arrays.asList("A", "B", "C")));
		map.put("Shubetsu2", new ArrayList<>(Arrays.asList("D", "E", "F")));
		final var list = new ArrayList<>();

		Assertions.assertEquals(
				"AND ( SHUBETSU1, SHUBETSU2 ) IN ( ( ?,? ),( ?,? ),( ?,? ) ) \n",
				item.build(map, list));
		Assertions.assertEquals(6, list.size());
		Assertions.assertEquals("A", list.get(0));
		Assertions.assertEquals("D", list.get(1));
		Assertions.assertEquals("B", list.get(2));
		Assertions.assertEquals("E", list.get(3));
		Assertions.assertEquals("C", list.get(4));
		Assertions.assertEquals("F", list.get(5));
	}

	/**
	 * 複数項目IN句内複数中単一文字列可変で別行パラメータなし
	 */
	@Test
	@DisplayName("複数項目IN句内複数中単一文字列可変で別行パラメータなし")
	public void buildTest23() {
		final var item = getTarget(
				"AND (SHUBETSU1, SHUBETSU2) IN "
				+ "(('1', '2')\n , /*$Shubetsu1*//*$Shubetsu2*/('','') \n)");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals(
				"AND (SHUBETSU1, SHUBETSU2) IN (('1', '2') \n) \n", item.build(map, list));
		Assertions.assertEquals(0, list.size());
	}

	/**
	 * 複数項目IN句内複数中単一文字列可変で別行配列パラメータあり
	 */
	@Test
	@DisplayName("複数項目IN句内複数中単一文字列可変で別行配列パラメータあり")
	public void buildTest24() {
		final var item = getTarget(
				"AND (SHUBETSU1, SHUBETSU2) "
				+ "IN (('1', '2')\n , /*$Shubetsu1*//*$Shubetsu2*/('','') \n)");

		final var map = new HashMap<String, Object>();
		map.put("Shubetsu1", new String[]{"A", "B", "C"});
		map.put("Shubetsu2", new String[]{"D", "E", "F"});
		final var list = new ArrayList<>();

		Assertions.assertEquals(
				"AND (SHUBETSU1, SHUBETSU2) IN (('1', '2') \n, ( ?,? ),( ?,? ),( ?,? ) \n) \n",
				item.build(map, list));
		Assertions.assertEquals(6, list.size());
		Assertions.assertEquals("A", list.get(0));
		Assertions.assertEquals("D", list.get(1));
		Assertions.assertEquals("B", list.get(2));
		Assertions.assertEquals("E", list.get(3));
		Assertions.assertEquals("C", list.get(4));
		Assertions.assertEquals("F", list.get(5));
	}

	/**
	 * 複数項目IN句内全て文字列必須でパラメータなし
	 */
	@Test
	@DisplayName("複数項目IN句内全て文字列必須でパラメータなし")
	public void buildTest25() {
		final var item = getTarget(
				"AND (SHUBETSU1, SHUBETSU2) NOT IN ( /*:Shubetsu1*//*:Shubetsu2*/('A', 'B'))");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals(
				"AND ( SHUBETSU1, SHUBETSU2 ) NOT IN ( ( ?,? ) ) \n", item.build(map, list));
		Assertions.assertEquals(2, list.size());
		Assertions.assertNull(list.get(0));
		Assertions.assertNull(list.get(1));
	}

	/**
	 * 複数項目IN句内全て文字列必須で配列パラメータあり
	 */
	@Test
	@DisplayName("複数項目IN句内全て文字列必須で配列パラメータあり")
	public void buildTest26() {
		final var item = getTarget(
				"AND (SHUBETSU1, SHUBETSU2) IN (/*:Shubetsu1*//*:Shubetsu2*/('',''))");

		final var map = new HashMap<String, Object>();
		map.put("Shubetsu1", new String[]{"A", "B"});
		map.put("Shubetsu2", new String[]{"C", "D"});
		final var list = new ArrayList<>();

		Assertions.assertEquals(
				"AND ( SHUBETSU1, SHUBETSU2 ) IN ( ( ?,? ),( ?,? ) ) \n", item.build(map, list));
		Assertions.assertEquals(4, list.size());
		Assertions.assertEquals("A", list.get(0));
		Assertions.assertEquals("C", list.get(1));
		Assertions.assertEquals("B", list.get(2));
		Assertions.assertEquals("D", list.get(3));
	}


	/**
	 * IN句全体に文字列デフォルトでパラメータなし
	 */
	@Test
	@DisplayName("IN句全体に文字列デフォルトでパラメータなし")
	public void buildTest27() {
		final var item = getTarget(
				"CASE WHEN SHUBETSU NOT IN /*?Shubetsu*/('1','2','3','4')");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("CASE WHEN SHUBETSU NOT IN ( ?,?,?,? ) \n", item.build(map, list));
		Assertions.assertEquals(4, list.size());
		Assertions.assertEquals("1", list.get(0));
		Assertions.assertEquals("2", list.get(1));
		Assertions.assertEquals("3", list.get(2));
		Assertions.assertEquals("4", list.get(3));
	}

	/**
	 * IN句全体に文字列デフォルトでパラメータあり
	 */
	@Test
	@DisplayName("IN句全体に文字列デフォルトでパラメータあり")
	public void buildTest28() {
		final var item = getTarget(
				"CASE WHEN SHUBETSU NOT IN /*?Shubetsu*/('1','2','3','4')");

		final var map = new HashMap<String, Object>();
		map.put("Shubetsu", "A");
		final var list = new ArrayList<>();

		Assertions.assertEquals("CASE WHEN SHUBETSU NOT IN ( ? ) \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals("A", list.get(0));
	}

	/**
	 * IN句全体に文字列デフォルトで配列パラメータあり
	 */
	@Test
	@DisplayName("IN句全体に文字列デフォルトで配列パラメータあり")
	public void buildTest29() {
		final var item = getTarget(
				"CASE WHEN SHUBETSU NOT IN /*?Shubetsu*/('1','2','3','4')");

		final var map = new HashMap<String, Object>();
		map.put("Shubetsu", new String[]{"A", "B"});
		final var list = new ArrayList<>();

		Assertions.assertEquals("CASE WHEN SHUBETSU NOT IN ( ?,? ) \n", item.build(map, list));
		Assertions.assertEquals(2, list.size());
		Assertions.assertEquals("A", list.get(0));
		Assertions.assertEquals("B", list.get(1));
	}

	/**
	 * IN句全体に数値デフォルトでパラメータなし
	 */
	@Test
	@DisplayName("IN句全体に数値デフォルトでパラメータなし")
	public void buildTest30() {
		final var item = getTarget(
				"CASE WHEN SHUBETSU NOT IN /*?Shubetsu*/(1,2,3,4)");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("CASE WHEN SHUBETSU NOT IN ( ?,?,?,? ) \n", item.build(map, list));
		Assertions.assertEquals(4, list.size());
		Assertions.assertEquals(BigDecimal.ONE, list.get(0));
		Assertions.assertEquals(new BigDecimal(2), list.get(1));
		Assertions.assertEquals(new BigDecimal(3), list.get(2));
		Assertions.assertEquals(new BigDecimal(4), list.get(3));
	}

	/**
	 * IN句全体に数値必須でパラメータなし
	 */
	@Test
	@DisplayName("IN句全体に数値必須でパラメータなし")
	public void buildTest31() {
		final var item = getTarget(
				"CASE WHEN SHUBETSU NOT IN /*:Shubetsu*/(1,2,3,4)");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("CASE WHEN SHUBETSU NOT IN ( ? ) \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertNull(list.get(0));
	}

	/**
	 * IN句全体に数値必須でパラメータあり
	 */
	@Test
	@DisplayName("IN句全体に数値必須でパラメータあり")
	public void buildTest32() {
		final var item = getTarget(
				"CASE WHEN SHUBETSU NOT IN /*:Shubetsu*/(1,2,3,4)");

		final var map = new HashMap<String, Object>();
		map.put("Shubetsu", 10);
		final var list = new ArrayList<>();

		Assertions.assertEquals("CASE WHEN SHUBETSU NOT IN ( ? ) \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals(new BigDecimal(10), list.get(0));
	}

	/**
	 * IN句全体に数値可変でパラメータなし
	 */
	@Test
	@DisplayName("IN句全体に数値可変でパラメータなし")
	public void buildTest33() {
		final var item = getTarget(
				"CASE WHEN SHUBETSU NOT IN /*$Shubetsu*/(1,2,3,4)");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("", item.build(map, list));
		Assertions.assertEquals(0, list.size());
	}

	/**
	 * IN句全体に数値可変でパラメータあり
	 */
	@Test
	@DisplayName("IN句全体に数値可変でパラメータあり")
	public void buildTest34() {
		final var item = getTarget(
				"CASE WHEN SHUBETSU NOT IN /*$Shubetsu*/(1,2,3,4)");

		final var map = new HashMap<String, Object>();
		map.put("Shubetsu", 10);
		final var list = new ArrayList<>();

		Assertions.assertEquals("CASE WHEN SHUBETSU NOT IN ( ? ) \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals(new BigDecimal(10), list.get(0));
	}

	/**
	 * IN句内文字列デフォルトでパラメータなし
	 */
	@Test
	@DisplayName("IN句内文字列デフォルトでパラメータなし")
	public void buildTest35() {
		final var item = getTarget("AAA IN(/*?dtGyomu*/'200901')");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("AAA IN( ? ) \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals("200901", list.get(0));
	}

	/**
	 * 括弧内複数中単一文字列必須でパラメータあり
	 */
	@Test
	@DisplayName("括弧内複数中単一文字列必須でパラメータあり")
	public void buildTest36() {
		final var item = getTarget(
				"COALESCE(/*:dtShuryo*/'0000', '0000') AS DT_TOROKU_YM, \n");

		final var map = new HashMap<String, Object>();
		map.put("dtShuryo", "9876");
		final var list = new ArrayList<>();

		Assertions.assertEquals("COALESCE( ?, '0000' ) AS DT_TOROKU_YM, \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals("9876", list.get(0));
	}

	/**
	 * 括弧内複数中途中文字列必須でパラメータあり
	 */
	@Test
	@DisplayName("括弧内複数中途中文字列必須でパラメータあり")
	public void buildTest37() {
		final var item = getTarget(
				"DECODE (DT_KOJO_YM, /*:dtKojo*/'200901', '1', '0') AS FG_TOU, \n");

		final var map = new HashMap<String, Object>();
		map.put("dtKojo", "201001");
		final var list = new ArrayList<>();

		Assertions.assertEquals(
				"DECODE ( DT_KOJO_YM, ?, '1', '0' ) AS FG_TOU, \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals("201001", list.get(0));
	}

	/**
	 * 括弧内複数中途中文字列可変でパラメータなし
	 */
	@Test
	@DisplayName("括弧内複数中途中文字列可変でパラメータなし")
	public void buildTest38() {
		final var item = getTarget(
				"DECODE (DT_KOJO_YM, /*$dtKojo*/'200901', '1', '0') AS FG_TOU, \n");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("", item.build(map, list));
		Assertions.assertEquals(0, list.size());
	}

	/**
	 * 括弧内複数中途中文字列可変でパラメータあり
	 */
	@Test
	@DisplayName("括弧内複数中途中文字列可変でパラメータあり")
	public void buildTest39() {
		final var item = getTarget(
				"DECODE (DT_KOJO_YM, /*$dtKojo*/'200901', '1', '0') AS FG_TOU, \n");

		final var map = new HashMap<String, Object>();
		map.put("dtKojo", "201001");
		final var list = new ArrayList<>();

		Assertions.assertEquals(
				"DECODE ( DT_KOJO_YM, ?, '1', '0' ) AS FG_TOU, \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals("201001", list.get(0));
	}

	/**
	 * 文字列可変でパラメータあり
	 */
	@Test
	@DisplayName("文字列可変でパラメータあり")
	public void buildTest40() {
		final var item = getTarget(
				"OR DT_KOJO_YM <= /*$dtKojo*/'200901' \n");

		final var map = new HashMap<String, Object>();
		map.put("dtKojo", "201001");
		final var list = new ArrayList<>();

		Assertions.assertEquals("OR DT_KOJO_YM <= ? \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals("201001", list.get(0));
	}

	/**
	 * 文字列可変でパラメータなし
	 */
	@Test
	@DisplayName("文字列可変でパラメータなし")
	public void buildTest41() {
		final var item = getTarget(
				"OR DT_KOJO_YM <= /*$dtKojo*/'200901' \n");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("", item.build(map, list));
		Assertions.assertEquals(0, list.size());
	}

	/**
	 * IN句内文字列可変で配列パラメータあり
	 */
	@Test
	@DisplayName("IN句内文字列可変で配列パラメータあり")
	public void buildTest42() {
		final var item = getTarget(
				"AND DT_KOJO_YM IN (/*$dtKojo*/'') \n");

		final var map = new HashMap<String, Object>();
		map.put("dtKojo", new String[]{"", "asdf"});
		final var list = new ArrayList<>();

		Assertions.assertEquals("AND DT_KOJO_YM IN ( ? ) \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals("asdf", list.get(0));
	}

	/**
	 * 文字列置換でパラメータあり
	 */
	@Test
	@DisplayName("文字列置換でパラメータあり")
	public void buildTest43() {
		final var item = getTarget(
				"ORDER BY /*@dtKojo*/AAA, BBB DESC, CCCC \n");

		final var map = new HashMap<String, Object>();
		map.put("dtKojo", "ZZZ DESC, YYYY, XXXX; \n");
		final var list = new ArrayList<>();

		Assertions.assertEquals("ORDER BY ZZZ DESC, YYYY, XXXX \n", item.build(map, list));
		Assertions.assertEquals(0, list.size());
	}

	/**
	 * 文字列行末置換でシングルクォートパラメータあり
	 */
	@Test
	@DisplayName("文字列行末置換でシングルクォートパラメータあり")
	public void buildTest44() {
		final var item = getTarget(
				"ORDER BY /*@dtKojo*/AAA, BBB DESC, CCCC \n");

		final var map = new HashMap<String, Object>();
		map.put("dtKojo", "ZZZ DESC, 'YYYY', XXXX \n");
		final var list = new ArrayList<>();

		Assertions.assertEquals("ORDER BY ZZZ DESC, YYYY, XXXX \n", item.build(map, list));
		Assertions.assertEquals(0, list.size());
	}

	/**
	 * 文字列行末置換でパラメータなし
	 */
	@Test
	@DisplayName("文字列行末置換でパラメータなし")
	public void buildTest45() {
		final var item = getTarget(
				"ORDER BY /*@dtKojo*/AAA, BBB DESC, CCCC \n");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("ORDER BY AAA, BBB DESC, CCCC \n", item.build(map, list));
		Assertions.assertEquals(0, list.size());
	}

	/**
	 * 文字列行末置換で配列パラメータあり
	 */
	@Test
	@DisplayName("文字列行末置換で配列パラメータあり")
	public void buildTest46() {
		final var item = getTarget(
				"ORDER BY /*@dtKojo*/AAA, BBB DESC, CCCC \n");

		final var map = new HashMap<String, Object>();
		map.put("dtKojo", new String[]{"ZZZ DESC", "'YYYY'", "XXXX"});
		final var list = new ArrayList<>();

		Assertions.assertEquals("ORDER BY ZZZ DESC, YYYY, XXXX \n", item.build(map, list));
		Assertions.assertEquals(0, list.size());
	}

	/**
	 * 文字列対象置換でパラメータあり
	 */
	@Test
	@DisplayName("文字列対象置換でパラメータあり")
	public void buildTest47() {
		final var item = getTarget(
				"FROM /*@YmTable/YYYYMM*/TR_MEISAI_YYYYMM WHERE AAA = 'ABC' \n");

		final var map = new HashMap<String, Object>();
		map.put("YmTable", "201202");
		final var list = new ArrayList<>();

		Assertions.assertEquals(
				"FROM TR_MEISAI_201202 WHERE AAA = 'ABC' \n", item.build(map, list));
		Assertions.assertEquals(0, list.size());
	}

	/**
	 * 文字列対象置換で配列パラメータあり
	 */
	@Test
	@DisplayName("文字列対象置換で配列パラメータあり")
	public void buildTest48() {
		final var item = getTarget(
				"FROM /*@YmTable1/YYYYMM*/TR_MEISAI_YYYYMM \n");

		final var map = new HashMap<String, Object>();
		map.put("YmTable1", new String[]{"201202", "'YYYY'", "XXXX"});
		final var list = new ArrayList<>();

		Assertions.assertEquals("FROM TR_MEISAI_201202, YYYY, XXXX \n", item.build(map, list));
		Assertions.assertEquals(0, list.size());
	}

	/**
	 * 文字列対象置換でパラメータなし
	 */
	@Test
	@DisplayName("文字列対象置換でパラメータなし")
	public void buildTest49() {
		final var item = getTarget(
				"FROM /*@YmTable/YYYYMM*/TR_MEISAI_YYYYMM \n");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("FROM TR_MEISAI_YYYYMM \n", item.build(map, list));
		Assertions.assertEquals(0, list.size());
	}

	/**
	 * 誤文字列対象置換でパラメータあり
	 */
	@Test
	@DisplayName("誤文字列対象置換でパラメータあり")
	public void buildTest50() {
		final var item = getTarget(
				"FROM /*@YmTable/YYYYMM*/TR_MEISAI_201201 \n");

		final var map = new HashMap<String, Object>();
		map.put("YmTable", "201202");
		final var list = new ArrayList<>();

		Assertions.assertEquals("FROM TR_MEISAI_201201 \n", item.build(map, list));
		Assertions.assertEquals(0, list.size());
	}

	/**
	 * 文字列対象文字列内置換でパラメータあり
	 */
	@Test
	@DisplayName("文字列対象文字列内置換でパラメータあり")
	public void buildTest51() {
		final var item = getTarget(
			"FROM TR_MEISAI_201201 WHERE ABC LIKE '%' || /*@YmTable/YYYYMM*/'YYYYMM' || '%' \n");

		final var map = new HashMap<String, Object>();
		map.put("YmTable", "201202");
		final var list = new ArrayList<>();

		Assertions.assertEquals(
				"FROM TR_MEISAI_201201 WHERE ABC LIKE '%' || '201202' || '%' \n",
				item.build(map, list));
		Assertions.assertEquals(0, list.size());
	}

	/**
	 * 数値可変でカンマパラメータあり
	 */
	@Test
	@DisplayName("数値可変でカンマパラメータあり")
	public void buildTest52() {
		final var item = getTarget("OR NO_KOJO <= /*$dtKojo*/200901 \n");

		final var map = new HashMap<String, Object>();
		map.put("dtKojo", "123,456,789");
		final var list = new ArrayList<>();

		Assertions.assertEquals("OR NO_KOJO <= ? \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals(new BigDecimal("123456789"), list.get(0));
	}

	/**
	 * 数値可変でカンマ小数点パラメータあり
	 */
	@Test
	@DisplayName("数値可変でカンマ小数点パラメータあり")
	public void buildTest53() {
		final var item = getTarget("OR NO_KOJO <= /*$dtKojo*/200901 \n");

		final var map = new HashMap<String, Object>();
		map.put("dtKojo", "123,456,789.001");
		final var list = new ArrayList<>();

		Assertions.assertEquals("OR NO_KOJO <= ? \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals(new BigDecimal("123456789.001"), list.get(0));
	}

	/**
	 * 数値可変でカンマ英字パラメータあり
	 */
	@Test
	@DisplayName("数値可変でカンマ英字パラメータあり")
	public void buildTest54() {
		final var item = getTarget("OR NO_KOJO <= /*$dtKojo*/200901 \n");

		final var map = new HashMap<String, Object>();
		map.put("dtKojo", "abcd123,50L");
		final var list = new ArrayList<>();

		Assertions.assertEquals("OR NO_KOJO <= ? \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertNull(list.get(0));
	}

	/**
	 * IN句内数値デフォルトでCSVパラメータあり
	 */
	@Test
	@DisplayName("IN句内数値デフォルトでCSVパラメータあり")
	public void buildTest55() {
		final var item = getTarget("AAA IN(/*?dtGyomu*/200)");

		final var map = new HashMap<String, Object>();
		map.put("dtGyomu", "300, 400");
		final var list = new ArrayList<>();

		Assertions.assertEquals("AAA IN( ?,? ) \n", item.build(map, list));
		Assertions.assertEquals(2, list.size());
		Assertions.assertEquals(new BigDecimal("300"), list.get(0));
		Assertions.assertEquals(new BigDecimal("400"), list.get(1));
	}

	/**
	 * LIKE句文字列可変でパラメータあり
	 */
	@Test
	@DisplayName("LIKE句文字列可変でパラメータあり")
	public void buildTest56() {
		final var item = getTarget(
				"AND DT_KOJO LIKE /*$dtKojo*/'%AAA%' \n");

		final var map = new HashMap<String, Object>();
		map.put("dtKojo", "ZZZZ");
		final var list = new ArrayList<>();

		Assertions.assertEquals("AND DT_KOJO LIKE ? \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals("%ZZZZ%", list.get(0));
	}

	/**
	 * LIKE句文字列可変でワイルドカードパラメータあり
	 */
	@Test
	@DisplayName("LIKE句文字列可変でワイルドカードパラメータあり")
	public void buildTest57() {
		final var item = getTarget(
				"AND DT_KOJO LIKE /*$dtKojo*/'%AAA%' \n");

		final var map = new HashMap<String, Object>();
		map.put("dtKojo", "ZZ%%ZZ");
		final var list = new ArrayList<>();

		Assertions.assertEquals("AND DT_KOJO LIKE ? \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals("%ZZ%%ZZ%", list.get(0));
	}

	/**
	 * 日付可変でパラメータなし
	 */
	@Test
	@DisplayName("日付可変でパラメータなし")
	public void buildTest58() {
		final var item = getTarget(
				"AND date_trunc('day', TRN.TS_TORIKOMI) = "
				+ "/*$DateTime*/TO_DATE('20100805','YYYYMMDD') \n");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("", item.build(map, list));
		Assertions.assertEquals(0, list.size());
	}

	/**
	 * 日付可変でパラメータあり
	 */
	@Test
	@DisplayName("日付可変でパラメータあり")
	public void buildTest59() {
		final var item = getTarget(
				"AND date_trunc('day', TRN.TS_TORIKOMI) = "
				+ "/*$DateTime*/TO_DATE('20100805','YYYYMMDD') \n");

		final var map = new HashMap<String, Object>();
		map.put("DateTime", new Timestamp(new Date().getTime()));
		final var list = new ArrayList<>();

		Assertions.assertEquals(
				"AND date_trunc( 'day', TRN.TS_TORIKOMI ) = ? \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
	}

	/**
	 * 日付デフォルトでパラメータなし
	 */
	@Test
	@DisplayName("日付デフォルトでパラメータなし")
	public void buildTest60() {
		final var item = getTarget(
				"AND TS_TORIKOMI = /*?DateTime*/TO_DATE('20100805','YYYYMMDD') \n");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();
		Assertions.assertEquals("AND TS_TORIKOMI = ? \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals(java.sql.Date.class, list.get(0).getClass());
		Assertions.assertEquals("2010-08-05", list.get(0).toString());
	}

	/**
	 * 日時可変でパラメータあり
	 */
	@Test
	@DisplayName("日時可変でパラメータあり")
	public void buildTest61() {
		final var item = getTarget(
				"AND TS_TOROKU <= "
				+ "/*$DtHenko*/TO_TIMESTAMP('99991231235959999','YYYYMMDDHH24MISSMS') \n");

		final var map = new HashMap<String, Object>();
		map.put("DtHenko", "20100805");
		final var list = new ArrayList<>();
		Assertions.assertEquals("AND TS_TOROKU <= ? \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals(java.sql.Timestamp.class, list.get(0).getClass());
		Assertions.assertEquals("2010-08-05 23:59:59.999", list.get(0).toString());
	}

	/**
	 * WHERE句内数値可変文字列可変でパラメータなし
	 */
	@Test
	@DisplayName("WHERE句内数値可変文字列可変でパラメータなし")
	public void buildTest62() {
		final var item = getTarget(
				"WHERE\n AAA = /*$dtGyomu*/200\n  AND BBB= /*$dtG*/'100'\nOFFSET 0");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("OFFSET 0 \n", item.build(map, list));
		Assertions.assertEquals(0, list.size());
	}

	/**
	 * WHERE句内数値可変でパラメータなし
	 */
	@Test
	@DisplayName("WHERE句内数値可変でパラメータなし")
	public void buildTest63() {
		final var item = getTarget(
				"WHERE\n AAA = /*$dtGyomu*/200\nLIMIT 0");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("LIMIT 0 \n", item.build(map, list));
		Assertions.assertEquals(0, list.size());
	}

	/**
	 * WHERE句内可変で一部パラメータあり
	 */
	@Test
	@DisplayName("WHERE句内可変で一部パラメータあり")
	public void buildTest64() {
		final var item = getTarget(
			"WHERE \n"
			+ "AAA = /*$dtGyomu*/200 \n"
			+ "AND DT_DATTAI_YM = /*$DtDattaiYm*/'aaa' \n"
			+ "AND (MN_PAY > /*$mnPay*/3 OR DT_DATTAI_YM >= /*$dtGyomu*/'200901' )");

		final var map = new HashMap<String, Object>();
		map.put("mnPay", 3);
		final var list = new ArrayList<>();

		Assertions.assertEquals("WHERE \n(MN_PAY > ?) \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals(3, list.get(0));
	}

	/**
	 * WHERE句内中間パラメータあり
	 */
	@Test
	@DisplayName("WHERE句内中間パラメータあり")
	public void buildTest65() {
		final var item = getTarget(
			"WHERE \n"
			+ "AAA = /*$dtGyomu*/200 \n"
			+ "AND DT_DATTAI_YM = /*$DtDattaiYm*/'aaa' \n"
			+ "AND (MN_PAY > /*$mnPay*/3 OR DT_DATTAI_YM >= /*$dtGyomu*/'200901' )");

		final var map = new HashMap<String, Object>();
		map.put("DtDattaiYm", "aaaaa");
		final var list = new ArrayList<>();

		Assertions.assertEquals("WHERE \nDT_DATTAI_YM = ? \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals("aaaaa", list.get(0));
	}

	/**
	 * AND_OR句文字列可変でパラメータあり
	 */
	@Test
	@DisplayName("AND_OR句文字列可変でパラメータあり")
	public void buildTest66() {
		final var item = getTarget(
			"AND  ((/*$StShoninSearch*/'0' = '0' AND TS_SHONIN IS NULL) "
			+ "OR (/*$StShoninSearch*/'0' = '1' AND TS_SHONIN IS NOT NULL))");

		final var map = new HashMap<String, Object>();
		map.put("StShoninSearch", "0");
		final var list = new ArrayList<>();

		Assertions.assertEquals(
				"AND ((? = '0' AND TS_SHONIN IS NULL) OR (? = '1' AND TS_SHONIN IS NOT NULL)) \n",
				item.build(map, list));
		Assertions.assertEquals(2, list.size());
	}

	/**
	 * AND_OR句文字列可変でパラメータなし
	 */
	@Test
	@DisplayName("AND_OR句文字列可変でパラメータなし")
	public void buildTest67() {
		final var item = getTarget(
			"AND  ((/*$StShoninSearch*/'0' = '0' AND TS_SHONIN IS NULL) "
			+ "OR (/*$StShoninSearch*/'0' = '1' AND TS_SHONIN IS NOT NULL))");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("", item.build(map, list));
		Assertions.assertEquals(0, list.size());
	}

	/**
	 * AND_OR句文字列可変でパラメータありOR後除去
	 */
	@Test
	@DisplayName("AND_OR句文字列可変でパラメータありOR後除去")
	public void buildTest68() {
		final var item = getTarget(
			"AND ((KB_URIAGE_KEIHI = '1' AND KB_URIAGE_KEIHI = /*$KbUriageSearch*/'111') "
			+ "OR (KB_URIAGE_KEIHI = '2' AND KB_URIAGE_KEIHI = /*$KbKeihiSearch*/'11'))");

		final var map = new HashMap<String, Object>();
		map.put("KbUriageSearch", "123");
		final var list = new ArrayList<>();

		Assertions.assertEquals(
				"AND ((KB_URIAGE_KEIHI = '1' AND KB_URIAGE_KEIHI = ?)) \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals("123", list.get(0));
	}

	/**
	 * AND_OR句文字列可変でパラメータありOR前除去
	 */
	@Test
	@DisplayName("AND_OR句文字列可変でパラメータありOR前除去")
	public void buildTest69() {
		final var item = getTarget(
			"AND ((KB_URIAGE_KEIHI = '1' AND KB_URIAGE_KEIHI = /*$KbUriageSearch*/'111') "
			+ "OR (KB_URIAGE_KEIHI = '2' AND KB_URIAGE_KEIHI = /*$KbKeihiSearch*/'11'))");

		final var map = new HashMap<String, Object>();
		map.put("KbKeihiSearch", "456");
		final var list = new ArrayList<>();

		Assertions.assertEquals(
				"AND ((KB_URIAGE_KEIHI = '2' AND KB_URIAGE_KEIHI = ?)) \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals("456", list.get(0));
	}

	/**
	 * AND_OR句複数文字列可変でパラメータなし
	 */
	@Test
	@DisplayName("AND_OR句複数文字列可変でパラメータなし")
	public void buildTest70() {
		final var item = getTarget(
			"AND ((CD_DR_KAMOKU = /*$CdKamokuSearch*/'CC77' "
			+ "AND CD_DR_HOJO = /*$CdHojoSearch*/'DD66') "
			+ "OR (CD_CR_KAMOKU = /*$CdKamokuSearch*/'CC77' "
			+ "AND CD_CR_HOJO = /*$CdHojoSearch*/'DD66'))");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("", item.build(map, list));
		Assertions.assertEquals(0, list.size());
	}

	/**
	 * AND_OR句複数文字列可変でパラメータありOR内AND後除去
	 */
	@Test
	@DisplayName("AND_OR句複数文字列可変でパラメータありOR内AND後除去")
	public void buildTest71() {
		final var item = getTarget(
			"AND ((CD_DR_KAMOKU = /*$CdKamokuSearch*/'CC77' "
			+ "AND CD_DR_HOJO = /*$CdHojoSearch*/'DD66') "
			+ "OR (CD_CR_KAMOKU = /*$CdKamokuSearch*/'CC77' "
			+ "AND CD_CR_HOJO = /*$CdHojoSearch*/'DD66'))");

		final var map = new HashMap<String, Object>();
		map.put("CdKamokuSearch", "123");
		final var list = new ArrayList<>();

		Assertions.assertEquals(
				"AND ((CD_DR_KAMOKU = ?) OR (CD_CR_KAMOKU = ?)) \n", item.build(map, list));
		Assertions.assertEquals(2, list.size());
		Assertions.assertEquals("123", list.get(0));
		Assertions.assertEquals("123", list.get(1));
	}

	/**
	 * AND_OR句複数文字列可変でパラメータありOR内AND前除去
	 */
	@Test
	@DisplayName("AND_OR句複数文字列可変でパラメータありOR内AND前除去")
	public void buildTest72() {
		final var item = getTarget(
			"AND ((CD_DR_KAMOKU = /*$CdKamokuSearch*/'CC77' "
			+ "AND CD_DR_HOJO = /*$CdHojoSearch*/'DD66') "
			+ "OR (CD_CR_KAMOKU = /*$CdKamokuSearch*/'CC77' "
			+ "AND CD_CR_HOJO = /*$CdHojoSearch*/'DD66'))");

		final var map = new HashMap<String, Object>();
		map.put("CdHojoSearch", "333");
		final var list = new ArrayList<>();

		Assertions.assertEquals(
				"AND ((CD_DR_HOJO = ?) OR (CD_CR_HOJO = ?)) \n", item.build(map, list));
		Assertions.assertEquals(2, list.size());
		Assertions.assertEquals("333", list.get(0));
		Assertions.assertEquals("333", list.get(1));
	}

	/**
	 * AND_OR句複数文字列可変でパラメータあり
	 */
	@Test
	@DisplayName("AND_OR句複数文字列可変でパラメータあり")
	public void buildTest73() {
		final var item = getTarget(
			"AND ((CD_DR_KAMOKU = /*$CdKamokuSearch*/'CC77' "
			+ "AND CD_DR_HOJO = /*$CdHojoSearch*/'DD66') "
			+ "OR (CD_CR_KAMOKU = /*$CdKamokuSearch*/'CC77' "
			+ "AND CD_CR_HOJO = /*$CdHojoSearch*/'DD66'))");

		final var map = new HashMap<String, Object>();
		map.put("CdKamokuSearch", "123");
		map.put("CdHojoSearch", "333");
		final var list = new ArrayList<>();

		Assertions.assertEquals(
				"AND ((CD_DR_KAMOKU = ? AND CD_DR_HOJO = ?) "
				+ "OR (CD_CR_KAMOKU = ? AND CD_CR_HOJO = ?)) \n",
				item.build(map, list));
		Assertions.assertEquals(4, list.size());
		Assertions.assertEquals("123", list.get(0));
		Assertions.assertEquals("333", list.get(1));
		Assertions.assertEquals("123", list.get(2));
		Assertions.assertEquals("333", list.get(3));
	}

	/**
	 * 文字列必須でパラメータあり前置
	 */
	@Test
	@DisplayName("文字列必須でパラメータあり前置")
	public void buildTest74() {
		final var item = getTarget(
			"AND (SELECT MIN(CD) FROM MS_CODE WHERE 'SHOHIZEI' = CD_SBT "
			+ "AND /*:DdShimebi*/'' <= CD AND 0 < VERSION) = CM.CODE");

		final var map = new HashMap<String, Object>();
		map.put("DdShimebi", "20101010");
		final var list = new ArrayList<>();

		Assertions.assertEquals(
				"AND ( SELECT MIN( CD ) FROM MS_CODE WHERE 'SHOHIZEI' = CD_SBT "
				+ "AND ? <= CD AND 0 < VERSION ) = CM.CODE \n",
				item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals("20101010", list.get(0));
	}

	/**
	 * 文字列必須でパラメータあり後置
	 */
	@Test
	@DisplayName("文字列必須でパラメータあり後置")
	public void buildTest75() {
		final var item = getTarget(
			"AND CM.CODE =(SELECT MIN(CD) FROM MS_CODE "
			+ "WHERE 'SHOHIZEI' = CD_SBT AND /*:DdShimebi*/'' <= CD AND 0 < VERSION)");

		final var map = new HashMap<String, Object>();
		map.put("DdShimebi", "20101010");
		final var list = new ArrayList<>();

		Assertions.assertEquals(
				"AND CM.CODE =( SELECT MIN( CD ) FROM MS_CODE "
				+ "WHERE 'SHOHIZEI' = CD_SBT AND ? <= CD AND 0 < VERSION ) \n",
				item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals("20101010", list.get(0));
	}

	/**
	 * 相対日付デフォルトでパラメータなし
	 */
	@Test
	@DisplayName("相対日付デフォルトでパラメータなし")
	public void buildTest76() {
		final var item = getTarget(
				"AND TS_TOROKU = /*?DtHenko*/TO_DATE('昨日','YYYYMMDD') \n");

		final var map = new HashMap<String, Object>();
		final var cl = Calendar.getInstance();
		cl.add(Calendar.DATE, -1);
		cl.set(Calendar.HOUR, 0);
		cl.set(Calendar.MINUTE, 0);
		cl.set(Calendar.SECOND, 0);

		final var list = new ArrayList<>();
		Assertions.assertEquals("AND TS_TOROKU = ? \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals(java.sql.Date.class, list.get(0).getClass());
		Assertions.assertEquals(
				new java.sql.Date(cl.getTimeInMillis()).toString(), list.get(0).toString());
	}

	/**
	 * 囲み内日付デフォルトでパラメータあり
	 */
	@Test
	@DisplayName("囲み内日付デフォルトでパラメータあり")
	public void buildTest77() {
		final var item = getTarget(
				"--<KbHenko\n AND TS_TOROKU = /*?DtHenko*/TO_DATE('20110404','YYYYMMDD') \n-->");

		final var map = new HashMap<String, Object>();
		map.put("KbHenko", "1");

		final var list = new ArrayList<>();
		Assertions.assertEquals("AND TS_TOROKU = ? \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals(java.sql.Date.class, list.get(0).getClass());
		Assertions.assertEquals("2011-04-04", list.get(0).toString());
	}

	/**
	 * 囲み内日付デフォルトでパラメータなし
	 */
	@Test
	@DisplayName("囲み内日付デフォルトでパラメータなし")
	public void buildTest78() {
		final var item = getTarget(
				"--<KbHenko\n AND TS_TOROKU = /*?DtHenko*/TO_DATE('20110404','YYYYMMDD') \n-->");

		final var map = new HashMap<String, Object>();

		final var list = new ArrayList<>();
		Assertions.assertEquals("", item.build(map, list));
		Assertions.assertEquals(0, list.size());
	}

	/**
	 * 囲み内日付デフォルトで比較正パラメータあり
	 */
	@Test
	@DisplayName("囲み内日付デフォルトで比較正パラメータあり")
	public void buildTest79() {
		final var item = getTarget(
				"--<KbHenko 10\n AND TS_TOROKU = /*?DtHenko*/TO_DATE('20110404','YYYYMMDD') \n-->");

		final var map = new HashMap<String, Object>();
		map.put("KbHenko", "10");

		final var list = new ArrayList<>();
		Assertions.assertEquals("AND TS_TOROKU = ? \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals(java.sql.Date.class, list.get(0).getClass());
		Assertions.assertEquals("2011-04-04", list.get(0).toString());
	}

	/**
	 * 囲み内日付デフォルトで比較誤パラメータあり
	 */
	@Test
	@DisplayName("囲み内日付デフォルトで比較誤パラメータあり")
	public void buildTest80() {
		final var item = getTarget(
				"--<KbHenko 10\n AND TS_TOROKU = /*?DtHenko*/TO_DATE('20110404','YYYYMMDD') \n-->");

		final var map = new HashMap<String, Object>();
		map.put("KbHenko", "1");

		final var list = new ArrayList<>();
		Assertions.assertEquals("", item.build(map, list));
		Assertions.assertEquals(0, list.size());
	}

	/**
	 * 否定囲み内日付デフォルトで比較パラメータあり
	 */
	@Test
	@DisplayName("否定囲み内日付デフォルトで比較パラメータあり")
	public void buildTest81() {
		final var item = getTarget(
			"--<!KbKomoku 1\n AND TS_TOROKU = /*?DtHenko*/TO_DATE('20110404','YYYYMMDD') \n-->");

		final var map = new HashMap<String, Object>();
		map.put("KbKomoku", "10");

		final var list = new ArrayList<>();
		Assertions.assertEquals("AND TS_TOROKU = ? \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals(java.sql.Date.class, list.get(0).getClass());
		Assertions.assertEquals("2011-04-04", list.get(0).toString());
	}

	/**
	 * 否定囲み内日付デフォルトで複数比較パラメータあり
	 */
	@Test
	@DisplayName("否定囲み内日付デフォルトで複数比較パラメータあり")
	public void buildTest82() {
		final var item = getTarget(
			"--<!KbKomoku 1 2\n AND TS_TOROKU = /*?DtHenko*/TO_DATE('20110404','YYYYMMDD') \n-->");

		final var map = new HashMap<String, Object>();
		map.put("KbKomoku", null);

		final var list = new ArrayList<>();
		Assertions.assertEquals("AND TS_TOROKU = ? \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals(java.sql.Date.class, list.get(0).getClass());
		Assertions.assertEquals("2011-04-04", list.get(0).toString());
	}

	/**
	 * AND_OR句内LIKE句複数文字列可変でパラメータなし
	 */
	@Test
	@DisplayName("AND_OR句内LIKE句複数文字列可変でパラメータなし")
	public void buildTest83() {
		final var item = getTarget(
			"AND ((KB.KOMOKU LIKE /*$KbKomoku1*/'%A%' "
			+ "AND KB.KOMOKU LIKE /*$KbKomoku2*/'%A%' AND KB.KOMOKU LIKE /*$KbKomoku3*/'%A%') "
			+ " OR (KK.KOMOKU LIKE /*$KbKomoku1*/'%B%' "
			+ "AND KK.KOMOKU LIKE /*$KbKomoku2*/'%B%' AND KK.KOMOKU LIKE /*$KbKomoku3*/'%C%'))");

		final var map = new HashMap<String, Object>();

		final var list = new ArrayList<>();
		Assertions.assertEquals("", item.build(map, list));
		Assertions.assertEquals(0, list.size());
	}

	/**
	 * AND_OR句内LIKE句複数文字列可変で１パラメータあり１
	 */
	@Test
	@DisplayName("AND_OR句内LIKE句複数文字列可変で１パラメータあり１")
	public void buildTest84() {
		final var item = getTarget(
			"AND ((KB.KOMOKU LIKE /*$KbKomoku1*/'%A%' "
			+ "AND KB.KOMOKU LIKE /*$KbKomoku2*/'%A%' AND KB.KOMOKU LIKE /*$KbKomoku3*/'%A%') "
			+ " OR (KK.KOMOKU LIKE /*$KbKomoku1*/'%B%' "
			+ "AND KK.KOMOKU LIKE /*$KbKomoku2*/'%B%' AND KK.KOMOKU LIKE /*$KbKomoku3*/'%C%'))");

		final var map = new HashMap<String, Object>();
		map.put("KbKomoku1", "KBUNKB");

		final var list = new ArrayList<>();
		Assertions.assertEquals(
				"AND ((KB.KOMOKU LIKE ?) OR (KK.KOMOKU LIKE ?)) \n", item.build(map, list));
		Assertions.assertEquals(2, list.size());
		Assertions.assertEquals("%KBUNKB%", list.get(0));
		Assertions.assertEquals("%KBUNKB%", list.get(1));
	}

	/**
	 * AND_OR句内LIKE句複数文字列可変で１パラメータあり２
	 */
	@Test
	@DisplayName("AND_OR句内LIKE句複数文字列可変で１パラメータあり２")
	public void buildTest85() {
		final var item = getTarget(
			"AND ((KB.KOMOKU LIKE /*$KbKomoku1*/'%A%' "
			+ "AND KB.KOMOKU LIKE /*$KbKomoku2*/'%A%' AND KB.KOMOKU LIKE /*$KbKomoku3*/'%A%') "
			+ " OR (KK.KOMOKU LIKE /*$KbKomoku1*/'%B%' "
			+ "AND KK.KOMOKU LIKE /*$KbKomoku2*/'%B%' AND KK.KOMOKU LIKE /*$KbKomoku3*/'%C%'))");

		final var map = new HashMap<String, Object>();
		map.put("KbKomoku2", "KBUNKBKBUNKB");

		final var list = new ArrayList<>();
		Assertions.assertEquals(
				"AND ((KB.KOMOKU LIKE ?) OR (KK.KOMOKU LIKE ?)) \n", item.build(map, list));
		Assertions.assertEquals(2, list.size());
		Assertions.assertEquals("%KBUNKBKBUNKB%", list.get(0));
		Assertions.assertEquals("%KBUNKBKBUNKB%", list.get(1));
	}

	/**
	 * AND_OR句内LIKE句複数文字列可変で１パラメータあり３
	 */
	@Test
	@DisplayName("AND_OR句内LIKE句複数文字列可変で１パラメータあり３")
	public void buildTest86() {
		final var item = getTarget(
			"AND ((KB.KOMOKU LIKE /*$KbKomoku1*/'%A%' "
			+ "AND KB.KOMOKU LIKE /*$KbKomoku2*/'%A%' AND KB.KOMOKU LIKE /*$KbKomoku3*/'%A%') "
			+ " OR (KK.KOMOKU LIKE /*$KbKomoku1*/'%B%' "
			+ "AND KK.KOMOKU LIKE /*$KbKomoku2*/'%B%' AND KK.KOMOKU LIKE /*$KbKomoku3*/'%C%'))");

		final var map = new HashMap<String, Object>();
		map.put("KbKomoku3", "KBUNKBKBUNKBKBUNKBKBUNKB");

		final var list = new ArrayList<>();
		Assertions.assertEquals(
				"AND ((KB.KOMOKU LIKE ?) OR (KK.KOMOKU LIKE ?)) \n", item.build(map, list));
		Assertions.assertEquals(2, list.size());
		Assertions.assertEquals("%KBUNKBKBUNKBKBUNKBKBUNKB%", list.get(0));
		Assertions.assertEquals("%KBUNKBKBUNKBKBUNKBKBUNKB%", list.get(1));
	}

	/**
	 * AND_OR句内LIKE句複数文字列可変で２パラメータあり１
	 */
	@Test
	@DisplayName("AND_OR句内LIKE句複数文字列可変で２パラメータあり１")
	public void buildTest87() {
		final var item = getTarget(
			"AND ((KB.KOMOKU LIKE /*$KbKomoku1*/'%A%' "
			+ "AND KB.KOMOKU LIKE /*$KbKomoku2*/'%A%' AND KB.KOMOKU LIKE /*$KbKomoku3*/'%A%') "
			+ " OR (KK.KOMOKU LIKE /*$KbKomoku1*/'%B%' "
			+ "AND KK.KOMOKU LIKE /*$KbKomoku2*/'%B%' AND KK.KOMOKU LIKE /*$KbKomoku3*/'%C%'))");

		final var map = new HashMap<String, Object>();
		map.put("KbKomoku1", "KBUNKB");
		map.put("KbKomoku2", "KBUNKBKBUNKB");

		final var list = new ArrayList<>();
		Assertions.assertEquals(
				"AND ((KB.KOMOKU LIKE ? AND KB.KOMOKU LIKE ?) "
				+ "OR (KK.KOMOKU LIKE ? AND KK.KOMOKU LIKE ?)) \n",
				item.build(map, list));
		Assertions.assertEquals(4, list.size());
		Assertions.assertEquals("%KBUNKB%", list.get(0));
		Assertions.assertEquals("%KBUNKBKBUNKB%", list.get(1));
		Assertions.assertEquals("%KBUNKB%", list.get(2));
		Assertions.assertEquals("%KBUNKBKBUNKB%", list.get(3));
	}

	/**
	 * AND_OR句内LIKE句複数文字列可変で２パラメータあり２
	 */
	@Test
	@DisplayName("AND_OR句内LIKE句複数文字列可変で２パラメータあり２")
	public void buildTest88() {
		final var item = getTarget(
			"AND ((KB.KOMOKU LIKE /*$KbKomoku1*/'%A%' "
			+ "AND KB.KOMOKU LIKE /*$KbKomoku2*/'%A%' AND KB.KOMOKU LIKE /*$KbKomoku3*/'%A%') "
			+ " OR (KK.KOMOKU LIKE /*$KbKomoku1*/'%B%' "
			+ "AND KK.KOMOKU LIKE /*$KbKomoku2*/'%B%' AND KK.KOMOKU LIKE /*$KbKomoku3*/'%C%'))");

		final var map = new HashMap<String, Object>();
		map.put("KbKomoku2", "KBUNKB");
		map.put("KbKomoku3", "KBUNKBKBUNKB");

		final var list = new ArrayList<>();
		Assertions.assertEquals(
				"AND ((KB.KOMOKU LIKE ? AND KB.KOMOKU LIKE ?) "
				+ "OR (KK.KOMOKU LIKE ? AND KK.KOMOKU LIKE ?)) \n",
				item.build(map, list));
		Assertions.assertEquals(4, list.size());
		Assertions.assertEquals("%KBUNKB%", list.get(0));
		Assertions.assertEquals("%KBUNKBKBUNKB%", list.get(1));
		Assertions.assertEquals("%KBUNKB%", list.get(2));
		Assertions.assertEquals("%KBUNKBKBUNKB%", list.get(3));
	}

	/**
	 * AND_OR句内LIKE句複数文字列可変で２パラメータあり３
	 */
	@Test
	@DisplayName("AND_OR句内LIKE句複数文字列可変で２パラメータあり３")
	public void buildTest89() {
		final LineParsedNodeList item = getTarget(
			"AND ((KB.KOMOKU LIKE /*$KbKomoku1*/'%A%' "
			+ "AND KB.KOMOKU LIKE /*$KbKomoku2*/'%A%' AND KB.KOMOKU LIKE /*$KbKomoku3*/'%A%') "
			+ " OR (KK.KOMOKU LIKE /*$KbKomoku1*/'%B%' "
			+ "AND KK.KOMOKU LIKE /*$KbKomoku2*/'%B%' AND KK.KOMOKU LIKE /*$KbKomoku3*/'%C%'))");

		final Map<String, Object> map = new HashMap<>();
		map.put("KbKomoku1", "KBUNKB");
		map.put("KbKomoku3", "KBUNKBKBUNKB");
		final List<Object> list = new ArrayList<>();

		Assertions.assertEquals(
				"AND ((KB.KOMOKU LIKE ? AND KB.KOMOKU LIKE ?) "
				+ "OR (KK.KOMOKU LIKE ? AND KK.KOMOKU LIKE ?)) \n",
				item.build(map, list));
		Assertions.assertEquals(4, list.size());
		Assertions.assertEquals("%KBUNKB%", list.get(0));
		Assertions.assertEquals("%KBUNKBKBUNKB%", list.get(1));
		Assertions.assertEquals("%KBUNKB%", list.get(2));
		Assertions.assertEquals("%KBUNKBKBUNKB%", list.get(3));
	}

	/**
	 * 複数文字列結合と数値デフォルトでパラメータなし
	 */
	@Test
	@DisplayName("複数文字列結合と数値デフォルトでパラメータなし")
	public void buildTest90() {
		final var item = getTarget(" 'XX''X:' || /*?NoPage*/1 \n");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("'XX''X:' || ? \n", item.build(map, list));
	}

	/**
	 * 文字列結合と数値デフォルトでパラメータあり
	 */
	@Test
	@DisplayName("文字列結合と数値デフォルトでパラメータあり")
	public void buildTest91() {
		final var item = getTarget(" 'XXX:' || /*?NoPage*/1 -- :?$\n");

		final var map = new HashMap<String, Object>();
		map.put("NoPage", 3);
		final var list = new ArrayList<>();

		Assertions.assertEquals("'XXX:' || ? \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals(3, list.get(0));
	}

	/**
	 * 文字列結合と数値デフォルトでパラメータなし
	 */
	@Test
	@DisplayName("文字列結合と数値デフォルトでパラメータなし")
	public void buildTest92() {
		final var item = getTarget(" 'XXX:' || /*?NoPage*/1 \n");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("'XXX:' || ? \n", item.build(map, list));
		Assertions.assertEquals(1, list.size());
		Assertions.assertEquals(new BigDecimal(1), list.get(0));
	}

	/**
	 * ラインコメント
	 */
	@Test
	@DisplayName("ラインコメント")
	public void buildTest93() {
		final var item = getTarget("-- 'XXX:' || /*?NoPage*/1 \n");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("", item.build(map, list));
	}

	/**
	 * 文字列内ラインコメント
	 */
	@Test
	@DisplayName("文字列内ラインコメント")
	public void buildTest94() {
		final var item = getTarget(" '--------------' AS CD_COMPANY \n");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("'--------------' AS CD_COMPANY \n", item.build(map, list));
	}

	/**
	 * 複数文字列内ラインコメント
	 */
	@Test
	@DisplayName("複数文字列内ラインコメント")
	public void buildTest95() {
		final var item = getTarget(
				" '-------''-------' AS CD_COMPANY \n");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("'-------''-------' AS CD_COMPANY \n", item.build(map, list));
	}

	/**
	 * コメント内ラインコメント
	 */
	@Test
	@DisplayName("コメント内ラインコメント")
	public void buildTest96() {
		final var item = getTarget(
				" ABCD /* '-------''-------' AS CD_COMPANY */ EFG\n");

		final var map = new HashMap<String, Object>();
		final var list = new ArrayList<>();

		Assertions.assertEquals("ABCD  EFG \n", item.build(map, list));
	}

	/**
	 * コメント内ラインコメントと文字列内必須コロン
	 */
	@Test
	@DisplayName("コメント内ラインコメントと文字列内必須コロン")
	public void buildTest97() {
		final var item = getTarget(
				" ABCD /* '-------''-------' AS CD_COMPANY */ 'XXX:' || /*?Code*/'1' \n");

		final var map = new HashMap<String, Object>();

		final var list = new ArrayList<>();
		Assertions.assertEquals("ABCD 'XXX:' || ? \n", item.build(map, list));
	}
}
