//
//  BSInlinePreviewer.m
//  BSInlinePreviewer
//
//  Created by Hori,Masaki on 08/08/14.
//  Copyright 2008 masakih. All rights reserved.
//

#import "BSInlinePreviewer.h"
#import "BSILinkInfomation.h"

#import "BSIPreferenceWindowController.h"
#import "BSIPReferenceViewController.h"


@interface BSInlinePreviewer(Private)
- (NSRange)linkRange;
- (NSImage *)downloadImageURL:(NSURL *)imageURL;
- (NSAttributedString *)attachmentAttributedStringWithImage:(NSImage *)image;
- (NSImage *)notFoundImage;
- (NSImage *)fitImage:(NSImage *)image toSize:(NSSize)targetSize;

- (void)showProgressPanel;
- (void)closeProgressPanel;
@end

@interface NSObject(BSInlinePreviewerDummy)
- (id)threadLayout;

- (float)messageHeadIndent;
@end

@implementation BSInlinePreviewer

NSString *const BSInlinePreviewerPreviewed = @"BSInlinePreviewerPreviewed";
const NSUInteger alreadyPreviewed = NSNotFound - 1;

static NSString *ThumbnailSizeKey = @"com.masakih.BSInlinePreviewer.thumbnailSize";

@synthesize totalDownloads, remainder;
@synthesize previewSize = _previewSize;

#pragma mark ## BSImagePreviewerProtocol
// Designated Initializer
- (id)initWithPreferences:(AppDefaults *)prefs
{
	self = [super init];
	if(self) {
		[self setPreferences:prefs];
		cache = [[NSCache alloc] init];
		[cache setName:@"BSInlinePreviewer"];
		[cache setCountLimit:100];
	}
	
	return self;
}
// Accessor
- (AppDefaults *)preferences
{
	return preference;
}
- (void)setPreferences:(AppDefaults *)aPreferences
{
	preference = aPreferences;
	NSSize size = NSSizeFromString([[aPreferences imagePreviewerPrefsDict] objectForKey:ThumbnailSizeKey]);
	if(NSEqualSizes(size, NSZeroSize)) {
		self.previewSize = NSMakeSize(100, 100);
	} else {
		self.previewSize = size;
	}
}

- (IBAction)showPreviewerPreferences:(id)sender
{
	if(!pref) {
		pref = [[BSIPreferenceWindowController alloc] init];
		pref.thumbnailSize = self.previewSize;
		[pref addObserver:self
			   forKeyPath:@"thumbnailSize"
				  options:0
				  context:pref];
	}
	[pref showWindow:nil];
}

- (NSView *)preferenceView
{
	
	if(!_prefViewCon) {
		_prefViewCon = [[BSIPReferenceViewController alloc] init];
		_prefViewCon.thumbnailSize = self.previewSize;
		[_prefViewCon addObserver:self
					   forKeyPath:@"thumbnailSize"
						  options:0
						  context:_prefViewCon];
	}
	return _prefViewCon.view;
}

- (void)observeValueForKeyPath:(NSString *)keyPath ofObject:(id)object change:(NSDictionary *)change context:(void *)context
{
	if(context == pref) {
		self.previewSize = pref.thumbnailSize;
		_prefViewCon.thumbnailSize = pref.thumbnailSize;
		[cache removeAllObjects];
		return;
	}
	if(context == _prefViewCon) {
		self.previewSize = _prefViewCon.thumbnailSize;
		pref.thumbnailSize = _prefViewCon.thumbnailSize;
		[cache removeAllObjects];
		return;
	}
	[super observeValueForKeyPath:keyPath ofObject:object change:change context:context];
}
// Action
- (BOOL)showImageWithURL:(NSURL *)imageURL
{
	return [self previewLink:imageURL];
}
- (BOOL)previewLink:(NSURL *)url
{
	// get document.
	if(![self prepareTarget]) {
		NSLog(@"%@ can not prepared.", NSStringFromClass([self class]));
		return NO;
	}
	
	// get insert position.
	NSRange linkRange = [self linkRange];
	if(linkRange.location == NSNotFound) {
		NSLog(@"con not found insert position.");
		return NO;
	}
	if(linkRange.location == alreadyPreviewed) return NO;
	unsigned insertIndex = linkRange.location;
	
	// download image.
	self.totalDownloads = self.remainder = 1;
	[self showProgressPanel];
	NSImage *image = [self downloadImageURL:url];
	self.remainder  = 0;
	[self closeProgressPanel];
	if(!image) { NSBeep(); return NO; }
	
	// insert image.
	NSTextStorage *ts = [self targetTextStorage];
	[ts beginEditing];
	{
		[ts addAttribute:BSInlinePreviewerPreviewed
				   value:[NSNumber numberWithBool:YES]
				   range:linkRange];
		
		id newInsertion = [self attachmentAttributedStringWithImage:image];
		[ts insertAttributedString:newInsertion atIndex:insertIndex];
	}
	[ts endEditing];
	
	return YES;
}

- (BOOL)previewLinks:(NSArray *)urls
{
	// get document.
	if(![self prepareTarget]) {
		NSLog(@"%@ can not prepared.", NSStringFromClass([self class]));
		return NO;
	}
	
	// get insert position.
	NSMutableArray *links = [NSMutableArray array];
	
	NSRange selectedRange = [threadView selectedRange];
	NSTextStorage *ts = [self targetTextStorage];
		
	NSUInteger selectionMax = NSMaxRange(selectedRange);
	id attr = nil;
	BOOL didChecked = NO;
	NSUInteger location = selectedRange.location;
	NSRange longest;
	NSRange range = selectedRange;
	do {
		attr = [ts attribute:NSLinkAttributeName
					 atIndex:location
	   longestEffectiveRange:&longest
					 inRange:range];
		didChecked = ([ts attribute:BSInlinePreviewerPreviewed
							atIndex:location
			  longestEffectiveRange:NULL
							inRange:range]) ? YES : NO;		
		if(!didChecked && attr
		   && ![attr hasPrefix:@"cmonar"]
		   && ![attr hasPrefix:@"cmbe"]
		   && ![attr hasPrefix:@"mailto"]) {
			BSILinkInfomation *link = [[[BSILinkInfomation alloc] init] autorelease];
			link.urlString = attr;
			link.range = longest;
			[links addObject:link];
		}
		location = range.location = NSMaxRange(longest);
		range.length -= longest.length;
	} while(location < selectionMax);
	
	// download images.
	self.totalDownloads = self.remainder = [links count];
	
	[self showProgressPanel];
	for(BSILinkInfomation *link in links) {
		NSURL *url = [NSURL URLWithString:link.urlString];
		if([self validateLink:url]) {
			link.image = [self downloadImageURL:url];
		}
		self.remainder--;
		[progressPanel displayIfNeeded];
	}
	[self closeProgressPanel];
	
	
	// insert images.
//	[ts beginEditing];
	NSUInteger linkOffset = 0;
	for(BSILinkInfomation *link in links) {
		if(!link.image) continue;
		NSRange linkRange = link.range;
		linkRange.location += linkOffset;
		[ts addAttribute:BSInlinePreviewerPreviewed
				   value:[NSNumber numberWithBool:YES]
				   range:linkRange];
		
		id newInsertion = [self attachmentAttributedStringWithImage:link.image];
		[ts insertAttributedString:newInsertion atIndex:linkRange.location];
		
		linkOffset += [newInsertion length];
		[[threadView window] displayIfNeeded];
	}
//	[ts endEditing];
	
	return YES;
}
	
- (BOOL)validateLink:(NSURL *)anURL
{
	NSArray *imageExtensions;
	NSString *extension;
	
	extension = [[[anURL path] pathExtension] lowercaseString];
	if(!extension) return NO;
	
	imageExtensions = [NSImage imageFileTypes];
	
	return [imageExtensions containsObject:extension];
}

#pragma mark ##  Accessor

- (void)setPreviewSize:(NSSize)previewSize
{
	_previewSize = previewSize;
	NSMutableDictionary *dict = [preference imagePreviewerPrefsDict];
	[dict setObject:NSStringFromSize(previewSize)
			 forKey:ThumbnailSizeKey];
}
- (void)loadProgressPanel
{
	if(progressPanel) return;
	
	NSBundle *bundle = [NSBundle bundleForClass:[self class]];
	[bundle loadNibFile:@"Panel"
	  externalNameTable:[NSDictionary dictionaryWithObject:self forKey:NSNibOwner]
			   withZone:[self zone]];
	
	[progress setUsesThreadedAnimation:YES];
}
- (void)showProgressPanel
{
	[self loadProgressPanel];
		
	NSWindow *mainWindow = [NSApp mainWindow];
	if(!mainWindow) {
		NSBeep();NSBeep();
		[NSThread sleepForTimeInterval:0.01];
		mainWindow = [NSApp mainWindow];
	}
	[NSApp beginSheet:progressPanel
	   modalForWindow:mainWindow
		modalDelegate:nil
	   didEndSelector:Nil
		  contextInfo:NULL];
	[progress startAnimation:self];
}
- (void)closeProgressPanel
{
	[progress stopAnimation:self];
	[progressPanel orderOut:self];
	[NSApp endSheet:progressPanel];
}

- (NSTimeInterval)timeoutInterval
{
	return 20.0;
}
- (id)previewAttributes
{
	if (!previewAttributes) {
		float					indent_;
		NSMutableParagraphStyle	*messageParagraphStyle_;
		
		indent_ = [[self preferences] messageHeadIndent];
		if(indent_ == 0) indent_ = 40;
		
		messageParagraphStyle_ = [[[NSMutableParagraphStyle defaultParagraphStyle] mutableCopy] autorelease];
		[messageParagraphStyle_ setFirstLineHeadIndent:indent_];
		[messageParagraphStyle_ setHeadIndent:indent_];
		
		previewAttributes = [NSDictionary dictionaryWithObjectsAndKeys:
							 messageParagraphStyle_, NSParagraphStyleAttributeName,
							 nil];
		[previewAttributes retain];
	}
	return previewAttributes;
}

- (void)setTargetThreadView:(id)view
{
	static Class theCMRThreadViewClass = NULL;
	if(!theCMRThreadViewClass) {
		theCMRThreadViewClass = NSClassFromString(@"CMRThreadView");
	}
	
	if(!view) {
		threadView = nil;
		return;
	}
	if(![view isKindOfClass:theCMRThreadViewClass]) return;
	
	threadView = view;
}
- (id)targetThreadView
{
	return threadView;
}
- (void)setTargetTextStorage:(NSTextStorage *)ts
{
	textStrage = ts;
}
- (NSTextStorage *)targetTextStorage
{
	return textStrage;
}
- (void)setActionIndex:(unsigned)p
{
	actionIndex = p;
}
- (unsigned)actionIndex
{
	return actionIndex;
}
- (NSImage *)notFoundImage
{
	NSBundle *myBundle = [NSBundle bundleForClass:[self class]];
	NSString *path = [myBundle pathForImageResource:@"notFound"];
	NSImage *image;
	
	if(!path) return nil;
	
	image = [[[NSImage alloc] initWithContentsOfFile:path] autorelease];
	
	return [self fitImage:image toSize:[self previewSize]];
}
#pragma mark ## Prepare
- (BOOL)prepareTarget
{
	[self setTargetThreadView:nil];
	[self setTargetTextStorage:nil];
	
	id window = [NSApp keyWindow];
	id firstResponder = [window firstResponder];
	
//	NSLog(@"first responder -> %@", firstResponder);
	// firstresponder is CMRThreadView.
	
	[self setTargetThreadView:firstResponder];
	if(![self targetThreadView]) return NO;
	
	
	id layout = [firstResponder threadLayout];
	NSTextStorage *ts = [layout textStorage];
	[self setTargetTextStorage:ts];
	
	
	NSEvent *event = [window currentEvent];
	NSPoint		mouseLocation_;
	unsigned	charIndex_;
	
	mouseLocation_ = [event locationInWindow];
	mouseLocation_ = [window convertBaseToScreen:mouseLocation_];
	charIndex_ = [firstResponder characterIndexForPoint:mouseLocation_];
	[self setActionIndex:charIndex_];
	
	return YES;
}

#pragma mark-

- (NSAttributedString *)attachmentAttributedStringWithImage:(NSImage *)image
{
	NSTextAttachment			*attachment_;
	NSMutableAttributedString	*attrs_ = nil;
	id<NSTextAttachmentCell>	cell_;
	Class aClass = NSClassFromString(@"CMXImageAttachmentCell");
	if(!aClass) return nil;
	
	// 画像リソースをNSTextAttachmentにする。
	attachment_ =  [[[NSTextAttachment alloc] init] autorelease];
	cell_ = [[[aClass alloc] initImageCell:image] autorelease];
	[attachment_ setAttachmentCell:cell_];
	
	attrs_ = (NSMutableAttributedString *)[NSMutableAttributedString attributedStringWithAttachment:attachment_];
	[attrs_ appendAttributedString:[[[NSAttributedString alloc] initWithString:@"\n"] autorelease]];
	[attrs_ addAttributes:[self previewAttributes]
					range:NSMakeRange(0, [attrs_ length])];
	
	return attrs_;
}

- (NSRange)linkRange
{
	unsigned	charIndex_ = [self actionIndex];
	NSTextStorage *ts = [self targetTextStorage];
	
	NSRange limit;
	NSRange longest;
	if(charIndex_ > 100) {
		limit = NSMakeRange(charIndex_ - 100, 200);
	} else {
		limit = NSMakeRange(0, 200);
	}
	unsigned over = NSMaxRange(limit) - [ts length];
	if(over > 0) {
		limit.length -= over;
	}
	
	id attr = nil;
	id checked = nil;
	@try {
		attr = [ts attribute:NSLinkAttributeName
					 atIndex:charIndex_
	   longestEffectiveRange:&longest
					 inRange:limit];
		//		NSLog(@"link attr -> %@", attr);
		//		NSLog(@"attr range -> %@", NSStringFromRange(longest));
		
		checked = [ts attribute:BSInlinePreviewerPreviewed
						atIndex:charIndex_
		  longestEffectiveRange:NULL
						inRange:limit];
	}
	@catch(NSException *ex) {
		if([[ex name] isEqualToString:NSRangeException]) {
			longest.location = NSNotFound;
		} else {
			@throw;
		}
	}
	
	if(!attr) {
		longest.location = NSNotFound;
	}
	if(checked) {
		longest.location = alreadyPreviewed;
	}
	
	return longest;
}

- (NSImage *)fitImage:(NSImage *)image toSize:(NSSize)targetSize
{
	if(!image) return nil;
	
	NSSize imageSize = [image size];
	
	float targetAspect = targetSize.width / targetSize.height;
	float imageAspect = imageSize.width / imageSize.height;
	float ratio;
	if(targetAspect > imageAspect) {
		ratio = targetSize.height / imageSize.height;
	} else {
		ratio = targetSize.width / imageSize.width;
	}
	imageSize.width *= ratio;
	imageSize.height *= ratio;
	
	[image setScalesWhenResized:YES];
	[image setSize:imageSize];
	
	return image;
}

- (id)cacheKeyForURL:(NSURL *)url
{
	return [url absoluteURL];
}
- (NSImage *)downloadImageURL:(NSURL *)imageURL
{
	NSImage *cachedImage = [cache objectForKey:[self cacheKeyForURL:imageURL]];
	if(cachedImage) return cachedImage;
	
	NSURLRequest *req;
	
	req = [NSURLRequest requestWithURL:imageURL
						   cachePolicy:NSURLRequestUseProtocolCachePolicy
					   timeoutInterval:[self timeoutInterval]];
	
	NSData *imageData;
	NSURLResponse *res = NULL;
	NSError *err = NULL;
	imageData = [NSURLConnection sendSynchronousRequest:req
									  returningResponse:&res
												  error:&err];
	if(err) {
		NSLog(@"Fail download. reason(%@)", [err localizedDescription]);
		goto notFound;
	}
	if(res) {
		if(![[res MIMEType] hasPrefix:@"image/"]) {
			NSLog(@"Fail download. reason(target type is %@)", [res MIMEType]);
			goto notFound;
		}
	}
	
	NSImage *image = [[[NSImage alloc] initWithData:imageData] autorelease];
	if(!image) {
		NSLog(@"Can not create image.");
		goto notFound;
	}
	
	cachedImage = [self fitImage:image toSize:[self previewSize]];
	[cache setObject:cachedImage forKey:[self cacheKeyForURL:imageURL]];
	return cachedImage;
notFound:
	[cache setObject:[self notFoundImage] forKey:[self cacheKeyForURL:imageURL]];
	return [self notFoundImage];
}

@end
