// export.js
/*
 * usage:
 * node export-single.js <target directory> <output directory>
 */

var path = require('path');
var fs = require('fs');

var exportParser = require('./export_parser');

var HTML_HEADER = 'header.txt';
var HTML_FOOTER = 'footer.txt';
var TEXT_HEADER = 'honmon_header.txt';
var IMAGE_DIR = 'figure';

var headerFullPath = path.join(path.dirname(module.filename), HTML_HEADER);
var footerFullPath = path.join(path.dirname(module.filename), HTML_FOOTER);
var textHeaderFullPath = path.join(path.dirname(module.filename), TEXT_HEADER);

var HTML_HEAD = fs.readFileSync(headerFullPath, {encoding: 'utf8'});
var HTML_FOOT = fs.readFileSync(footerFullPath, {encoding: 'utf8'});
var TEXT_HEAD = fs.readFileSync(textHeaderFullPath, {encoding: 'utf8'});

var usage = 'node export-single.js <target file> <output directory>';


function copyFiles(filelist, from, dest) {
  // create output directory
  if (!fs.existsSync(dest)) {
    fs.mkdirSync(dest);
  }

  filelist.forEach(function (target) {
    var fromPathname = path.join(from, target);
    var destPathname = path.join(dest, target);

    if (!fs.existsSync(fromPathname)) {
      console.log(fromPathname, 'not exists');
    } else {
      // copy file
      fs.createReadStream(fromPathname).
        pipe(fs.createWriteStream(destPathname));
    }
  });
}

function exportSingle(target, output, chapter, pageStart, pageCount) {
  chapter = chapter || 1;
  pageStart = pageStart || 0;
  pageCount = pageCount || 2;
  var pageEnd = pageStart + pageCount - 1;

  var textHeader = TEXT_HEAD;
  textHeader = textHeader.replace('<CHAPTER>', chapter)
    .replace('<PAGE_START>', pageStart)
    .replace('<PAGE_END>', pageEnd)
    .replace('<PAGE_COUNT>', pageCount);

  fs.readFile(target, 'utf8', function (err, data) {
    console.log(target);
    if (err) {
      console.log('Cannot read file: ' + target);
      return;
    }

    var result = exportParser.parseText(data, target);

    // create output directory
    if (!fs.existsSync(output)) {
      fs.mkdirSync(output);
    }

    // output honmon, caption
    fs.writeFileSync(path.join(output, 'honmon.txt'),
                     textHeader + result.bodies.join('\n'));

    if (result.captions.length != 0) {
      fs.writeFileSync(path.join(output, 'caption.txt'),
                       result.captions.join('\n'));
      // output html
      var out = fs.createWriteStream(path.join(output, 'figures.html'), {flags: 'w', encoding: 'utf8', mode: 0666});
      out.write(HTML_HEAD);
      out.write('\n');
      out.write(result.htmls.join('\n'));
      out.write('\n');
      out.write(HTML_FOOT);
      out.end();
    }

    // file copy
    var copyFrom = path.join(path.dirname(target), 'figure');
    if (result.figures.length != 0) {
      copyFiles(result.figures, copyFrom, path.join(output, 'figures'));
    }

  });
}

module.exports = exportSingle;

// main action
if (require.main == module) {
  if (process.argv.length < 4) {
    process.stdout.write(usage + '\n');
    process.exit(1);
  }
  var target = process.argv[2];
  var output = process.argv[3];
  exportSingle(target, output);
}
