use std::io::{Read, Write};

use egg::{*, rewrite as rw};

fn load_tree(handle :&mut Read) -> std::io::Result<RecExpr<SymbolLang>> {
    let mut input = String::new();
    handle.read_to_string(&mut input)?;
    input.parse().map_err(|e| std::io::Error::new(std::io::ErrorKind::InvalidData, e))
}

// Wadler 1989: https://www2.cs.sfu.ca/CourseCentral/831/burton/Notes/July14/free.pdf
// Elliott 2013: http://conal.net/blog/posts/optimizing-cccs

// "R when X is id" means that rule R is being specialized modulo the equality:
//     (comp id ?f) = ?f = (comp ?f id) = (comp (comp ?f id) id) = ...

// "free for X" means that the polymorphic type of X implies the underlying equality justifying
// the annotated rule, as in Wadler 1989.

// "dual from X" means that justification X formally dualizes to justify the following rule. For
// example, products and sums are dual.

// Rules marked "associative!" can cause explosions.

// Rules marked "improvement!" can improve IEEE 754 results in both precision and accuracy, so
// cannot be run backwards.

fn main() -> std::io::Result<()> {
    let rules :&[Rewrite<SymbolLang, ()>] = &[
        rw!("comp-id-left"; "(comp id ?x)" => "?x"),
        rw!("comp-id-right"; "(comp ?x id)" => "?x"),

        // associative!
        rw!("comp-assoc-left"; "(comp ?x (comp ?y ?z))" => "(comp (comp ?x ?y) ?z)"),
        rw!("comp-assoc-right"; "(comp (comp ?x ?y) ?z)" => "(comp ?x (comp ?y ?z))"),

        rw!("ignore-absorb-left"; "(comp ?x ignore)" => "ignore"),

        // free for fst
        rw!("fst-elim-pair"; "(comp (pair ?f ?g) fst)" => "?f"),
        // free for snd
        rw!("snd-elim-pair"; "(comp (pair ?f ?g) snd)" => "?g"),
        rw!("pair-fst-snd"; "(pair fst snd)" => "id"),
        rw!("pair-repack"; "(comp (pair ?f ?g) (pair (comp fst ?j) (comp snd ?k)))"
            => "(pair (comp ?f ?j) (comp ?g ?k))"),
        // pair-repack when ?j is id
        rw!("pair-repack-snd"; "(comp (pair ?f ?g) (pair fst (comp snd ?k)))"
            => "(pair ?f (comp ?g ?k))"),
        // pair-repack when ?k is id
        rw!("pair-repack-fst"; "(comp (pair ?f ?g) (pair (comp fst ?j) snd))"
            => "(pair (comp ?f ?j) ?g)"),
        // Elliott 2013
        rw!("pair-precompose"; "(pair (comp ?r ?f) (comp ?r ?g))" => "(comp ?r (pair ?f ?g))"),
        // pair-precompose when ?f is id
        rw!("pair-factor-fst"; "(pair ?r (comp ?r ?g))" => "(comp ?r (pair id ?g))"),
        // pair-precompose when ?g is id
        rw!("pair-factor-snd"; "(pair (comp ?r ?f) ?r)" => "(comp ?r (pair ?f id))"),
        // pair-precompose when ?f and ?g are id
        rw!("pair-factor-both"; "(pair ?r ?r)" => "(comp ?r (pair id id))"),

        // Turn braided products into symmetric products.
        rw!("pair-swap-invo"; "(comp (pair snd fst) (pair snd fst))" => "id"),

        // free for left
        rw!("left-elim-case"; "(comp left (case ?f ?g))" => "?f"),
        // free for right
        rw!("right-elim-case"; "(comp right (case ?f ?g))" => "?g"),
        rw!("case-left-right"; "(case left right)" => "id"),

        // dual of Elliott 2013
        rw!("case-postcompose"; "(case (comp ?f ?r) (comp ?g ?r))" => "(comp (case ?f ?g) ?r)"),
        rw!("case-factor"; "(case ?r ?r)" => "(comp (case id id) ?r)"),

        rw!("curry-uncurry-cancel"; "(uncurry (curry ?f))" => "?f"),
        rw!("uncurry-curry-cancel"; "(curry (uncurry ?f))" => "?f"),

        rw!("nat-elim-pr"; "(pr zero succ)" => "id"),
        rw!("nat-unroll-pr-zero"; "(comp zero (pr ?x ?f))" => "?x"),
        rw!("nat-unroll-pr-succ-post"; "(comp succ (pr ?x ?f))" => "(comp (pr ?x ?f) ?f)"),
        rw!("nat-unroll-pr-succ-pre"; "(comp succ (pr ?x ?f))" => "(pr (comp ?x ?f) ?f)"),

        rw!("list-elim-fold"; "(fold nil cons)" => "id"),
        rw!("list-unroll-fold-nil"; "(comp nil (fold ?x ?f))" => "?x"),
        rw!("list-unroll-fold-cons"; "(comp cons (fold ?x ?f))" => "(comp (pair fst (fold ?x ?f)) ?f)"),

        // Boolean negation
        rw!("bool-t-not"; "(comp t not)" => "f"),
        rw!("bool-f-not"; "(comp f not)" => "t"),
        rw!("bool-not-invo"; "(comp not not)" => "id"),

        // IEEE 754 addition
        rw!("f-add-comm"; "(comp (pair snd fst) f-add)" => "f-add"),
        rw!("f-add-unit-left"; "(comp (pair ?f f-zero) f-add)" => "?f"),
        rw!("f-add-unit-right"; "(comp (pair f-zero ?f) f-add)" => "?f"),
        rw!("f-add-negate-left"; "(comp (pair id f-negate) f-add)" => "f-zero"),
        rw!("f-add-negate-right"; "(comp (pair f-negate id) f-add)" => "f-zero"),

        // IEEE 754 multiplication
        rw!("f-mul-comm"; "(comp (pair snd fst) f-mul)" => "f-mul"),
        rw!("f-mul-unit-left"; "(comp (pair ?f f-one) f-mul)" => "?f"),
        rw!("f-mul-unit-right"; "(comp (pair f-one ?f) f-mul)" => "?f"),

        // improvement!
        rw!("f-negate-invo"; "(comp f-negate f-negate)" => "id"),
        rw!("f-recip-invo"; "(comp f-recip f-recip)" => "id"),

        // IEEE 754 even functions
        rw!("f-sign-negate-ana"; "(comp f-negate f-sign)" => "(comp f-sign not)"),
        rw!("f-sign-negate-kata"; "(comp f-sign not)" => "(comp f-negate f-sign)"),
        rw!("f-cos-even"; "(comp f-negate f-cos)" => "f-cos"),

        // IEEE 754 odd functions
        rw!("f-sin-odd-ana"; "(comp f-negate f-sin)" => "(comp f-sin f-negate)"),
        rw!("f-sin-odd-kata"; "(comp f-sin f-negate)" => "(comp f-negate f-sin)"),
    ];

    let tree = load_tree(&mut std::io::stdin())?;
    eprintln!("Input expression has cost {}", AstSize.cost_rec(&tree));

    let runner = Runner::default().with_expr(&tree).run(rules);
    eprintln!("E-graph finished running: {:?} after {} iterations", runner.stop_reason.unwrap(),
              runner.iterations.len());

    let mut extractor = Extractor::new(&runner.egraph, AstSize);
    let (best_cost, best_expr) = extractor.find_best(runner.roots[0]);
    eprintln!("Best expression has cost {}", best_cost);

    std::io::stdout().write_all(best_expr.pretty(78).as_bytes());
    std::io::stdout().write_all(b"\n")
}
