/*
 * Copyright 2004-2012 the Seasar Foundation and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package org.seasar.framework.mock.sql;

import java.io.InputStream;
import java.io.Reader;
import java.math.BigDecimal;
import java.net.URL;
import java.sql.Array;
import java.sql.Blob;
import java.sql.Clob;
import java.sql.Date;
import java.sql.ParameterMetaData;
import java.sql.PreparedStatement;
import java.sql.Ref;
import java.sql.ResultSet;
import java.sql.ResultSetMetaData;
import java.sql.SQLException;
import java.sql.Statement;
import java.sql.Time;
import java.sql.Timestamp;
import java.util.Calendar;

/**
 * {@link PreparedStatement}用のモッククラスです。
 * 
 * @author higa
 * 
 */
public class MockPreparedStatement extends MockStatement implements
        PreparedStatement {

    private String sql;

    private int autoGeneratedKeys = Statement.NO_GENERATED_KEYS;

    private int[] columnIndices;

    private String[] columnNames;

    /**
     * {@link MockPreparedStatement}を作成します。
     * 
     * @param connection
     *            コネクション
     * @param sql
     *            SQL
     */
    public MockPreparedStatement(MockConnection connection, String sql) {
        super(connection);
        setSql(sql);
    }

    /**
     * {@link MockPreparedStatement}を作成します。
     * 
     * @param connection
     *            コネクション
     * @param sql
     *            SQL
     * @param autoGeneratedKeys
     *            自動生成されたキーを返すかどうかのフラグ
     */
    public MockPreparedStatement(MockConnection connection, String sql,
            int autoGeneratedKeys) {
        this(connection, sql);
        setAutoGeneratedKeys(autoGeneratedKeys);
    }

    /**
     * {@link MockPreparedStatement}を作成します。
     * 
     * @param connection
     *            コネクション
     * @param sql
     *            SQL
     * @param columnIndices
     *            自動生成された値を返して欲しいカラムの位置の配列
     */
    public MockPreparedStatement(MockConnection connection, String sql,
            int[] columnIndices) {
        this(connection, sql);
        setColumnIndices(columnIndices);
    }

    /**
     * {@link MockPreparedStatement}を作成します。
     * 
     * @param connection
     *            コネクション
     * @param sql
     *            SQL
     * @param columnNames
     *            自動生成された値を返して欲しいカラムの名前の配列
     */
    public MockPreparedStatement(MockConnection connection, String sql,
            String[] columnNames) {
        this(connection, sql);
        setColumnNames(columnNames);
    }

    /**
     * {@link MockPreparedStatement}を作成します。
     * 
     * @param connection
     *            コネクション
     * @param sql
     *            SQl
     * @param resultSetType
     *            結果セットタイプ
     * @param resultSetConcurrency
     *            結果セット同時並行性
     */
    public MockPreparedStatement(MockConnection connection, String sql,
            int resultSetType, int resultSetConcurrency) {
        super(connection, resultSetType, resultSetConcurrency);
        setSql(sql);
    }

    /**
     * {@link MockPreparedStatement}を作成します。
     * 
     * @param connection
     *            コネクション
     * @param sql
     *            SQL
     * @param resultSetType
     *            結果セットタイプ
     * @param resultSetConcurrency
     *            結果セット同時並行性
     * @param resultSetHoldability
     *            結果セット保持力
     */
    public MockPreparedStatement(MockConnection connection, String sql,
            int resultSetType, int resultSetConcurrency,
            int resultSetHoldability) {
        super(connection, resultSetType, resultSetConcurrency,
                resultSetHoldability);
        setSql(sql);
    }

    public void addBatch() throws SQLException {
    }

    public void clearParameters() throws SQLException {
    }

    public boolean execute() throws SQLException {
        return false;
    }

    public ResultSet executeQuery() throws SQLException {
        MockResultSet rs = new MockResultSet();
        rs.setType(getResultSetType());
        rs.setFetchSize(getFetchSize());
        return rs;
    }

    public int executeUpdate() throws SQLException {
        return 0;
    }

    public ResultSetMetaData getMetaData() throws SQLException {
        return null;
    }

    public ParameterMetaData getParameterMetaData() throws SQLException {
        return null;
    }

    public void setArray(int i, Array x) throws SQLException {
    }

    public void setAsciiStream(int parameterIndex, InputStream x, int length)
            throws SQLException {
    }

    public void setBigDecimal(int parameterIndex, BigDecimal x)
            throws SQLException {
    }

    public void setBinaryStream(int parameterIndex, InputStream x, int length)
            throws SQLException {
    }

    public void setBlob(int i, Blob x) throws SQLException {
    }

    public void setBoolean(int parameterIndex, boolean x) throws SQLException {
    }

    public void setByte(int parameterIndex, byte x) throws SQLException {
    }

    public void setBytes(int parameterIndex, byte[] x) throws SQLException {
    }

    public void setCharacterStream(int parameterIndex, Reader reader, int length)
            throws SQLException {
    }

    public void setClob(int i, Clob x) throws SQLException {
    }

    public void setDate(int parameterIndex, Date x, Calendar cal)
            throws SQLException {
    }

    public void setDate(int parameterIndex, Date x) throws SQLException {
    }

    public void setDouble(int parameterIndex, double x) throws SQLException {
    }

    public void setFloat(int parameterIndex, float x) throws SQLException {
    }

    public void setInt(int parameterIndex, int x) throws SQLException {
    }

    public void setLong(int parameterIndex, long x) throws SQLException {
    }

    public void setNull(int paramIndex, int sqlType, String typeName)
            throws SQLException {
    }

    public void setNull(int parameterIndex, int sqlType) throws SQLException {
    }

    public void setObject(int parameterIndex, Object x, int targetSqlType,
            int scale) throws SQLException {
    }

    public void setObject(int parameterIndex, Object x, int targetSqlType)
            throws SQLException {
    }

    public void setObject(int parameterIndex, Object x) throws SQLException {
    }

    public void setRef(int i, Ref x) throws SQLException {
    }

    public void setShort(int parameterIndex, short x) throws SQLException {
    }

    public void setString(int parameterIndex, String x) throws SQLException {
    }

    public void setTime(int parameterIndex, Time x, Calendar cal)
            throws SQLException {
    }

    public void setTime(int parameterIndex, Time x) throws SQLException {
    }

    public void setTimestamp(int parameterIndex, Timestamp x, Calendar cal)
            throws SQLException {
    }

    public void setTimestamp(int parameterIndex, Timestamp x)
            throws SQLException {
    }

    public void setUnicodeStream(int parameterIndex, InputStream x, int length)
            throws SQLException {
    }

    public void setURL(int parameterIndex, URL x) throws SQLException {
    }

    /**
     * SQLを返します。
     * 
     * @return SQL
     */
    public String getSql() {
        return sql;
    }

    /**
     * SQLを設定します。
     * 
     * @param sql
     *            SQL
     */
    public void setSql(String sql) {
        this.sql = sql;
    }

    /**
     * 自動生成されたキーを返すかどうかのフラグを返します。
     * 
     * @return 自動生成されたキーを返すかどうかのフラグ
     */
    public int getAutoGeneratedKeys() {
        return autoGeneratedKeys;
    }

    /**
     * 自動生成されたキーを返すかどうかのフラグを設定します。
     * 
     * @param autoGeneratedKeys
     *            自動生成されたキーを返すかどうかのフラグ
     */
    public void setAutoGeneratedKeys(int autoGeneratedKeys) {
        this.autoGeneratedKeys = autoGeneratedKeys;
    }

    /**
     * 自動生成された値を返して欲しいカラムの位置の配列を返します。
     * 
     * @return 自動生成された値を返して欲しいカラムの位置の配列
     */
    public int[] getColumnIndices() {
        return columnIndices;
    }

    /**
     * 自動生成された値を返して欲しいカラムの位置の配列を設定します。
     * 
     * @param columnIndices
     *            自動生成された値を返して欲しいカラムの位置の配列
     */
    public void setColumnIndices(int[] columnIndices) {
        this.columnIndices = columnIndices;
    }

    /**
     * 自動生成された値を返して欲しいカラムの名前の配列を返します。
     * 
     * @return 自動生成された値を返して欲しいカラムの名前の配列
     */
    public String[] getColumnNames() {
        return columnNames;
    }

    /**
     * 自動生成された値を返して欲しいカラムの名前の配列を設定します。
     * 
     * @param columnNames
     *            自動生成された値を返して欲しいカラムの名前の配列
     */
    public void setColumnNames(String[] columnNames) {
        this.columnNames = columnNames;
    }
}