using System;
using nft.framework;

namespace nft.core.game
{
	public enum PlayerMode { FreeBuild, Government, LocalAuthority, Company, Other };
	public enum Ownership { None, Government, LocalAuthority, PlayerCompany, NonPlayerCompany, Private, Special };
	/// <summary>
	/// IGameMode ̊Tv̐łB
	/// </summary>
	public interface IGameMode : IHasNameAndID
	{
		PlayerMode PlayerMode { get; }
		bool DynamicLandEdit { get; }

		bool HasFareIncome { get; }
        bool HasTaxIncome { get; }
		bool CanOwnSubsidaries{ get; }
		bool CityDevelopment { get; }

		bool CheckBuildPermission(Ownership owner);
		bool CheckInvitePermission(Ownership owner); // Uv
        bool CheckEvictionPermission(Ownership owner); // ނ        
		bool CheckSellPermission(Ownership owner);
		bool CheckBuyPermission(Ownership owner);
		bool CheckRemovePermission(Ownership owner);
	}

    public abstract class BuiltInGameMode : IGameMode {
        /// <summary>
        /// FreeBuild mode is God mode. No financial restrictions, No development.
        /// </summary>
        public static readonly BuiltInGameMode FreeBuild = new FreeBuildMode();
        /// <summary>
        /// Government mode is 'SimCity' like national sovereign mode. 
        /// Can edit land, No commerciali actions.
        /// </summary>
        public static readonly BuiltInGameMode Government = new GovernmentMode();
        /// <summary>
        /// LocalAuthority mode is 'Traffic Tycoon' like city-officer mode. 
        /// No land edit, No commerciali actions and No development.
        /// </summary>
        public static readonly BuiltInGameMode LocalAuthority = new LocalAuthMode();
        /// <summary>
        /// Company mode is traditional 'Take the A-train' like management mode. 
        /// No land edit, enable commerciali actions and development.
        /// </summary>
        public static readonly BuiltInGameMode Company = new CompanyMode();

        internal BuiltInGameMode(PlayerMode pm, Ownership owner,
            bool landEdit, bool subsidaries, bool fareIncom, bool taxIncom, bool develop) {
            this.playerMode = pm;
            this.playerOwnership = owner;
            this.flgLandEdit = landEdit;
            this.flgFareIncome = fareIncom;
            this.flgTaxIncome = taxIncom;
            this.flgOwnSubsidaries = subsidaries;
            this.flgCityDevelop = develop;
        }

        protected PlayerMode playerMode;
        protected Ownership playerOwnership;
        protected bool flgLandEdit;
        protected bool flgFareIncome;
        protected bool flgTaxIncome;
        protected bool flgOwnSubsidaries;
        protected bool flgCityDevelop;

        #region IGameMode o

        public PlayerMode PlayerMode {
            get { return playerMode; }
        }

        public bool DynamicLandEdit {
            get { return flgLandEdit; }
        }

        public bool HasFareIncome {
            get { return flgFareIncome; }
        }

        public bool HasTaxIncome {
            get { return flgTaxIncome; }
        }

        public bool CanOwnSubsidaries {
            get { return flgFareIncome; }
        }

        public bool CityDevelopment {
            get { return flgCityDevelop; }
        }

        public virtual bool CheckBuildPermission(Ownership owner) {
            return owner == this.playerOwnership;
        }

        public virtual bool CheckInvitePermission(Ownership owner) {
            return false;
        }

        public virtual bool CheckEvictionPermission(Ownership owner) {
            return false;
        }

        public virtual bool CheckSellPermission(Ownership owner) {
            return owner == this.playerOwnership;
        }

        public virtual bool CheckBuyPermission(Ownership owner) {
            return false;
        }

        public virtual bool CheckRemovePermission(Ownership owner) {
            return owner == this.playerOwnership;
        }

        #endregion

        #region IHasNameAndID o

        public string ID {
            get { throw new NotImplementedException(); }
        }

        public string Name {
            get { throw new NotImplementedException(); }
        }

        #endregion
    }
    #region FreeBuildMode
    internal class FreeBuildMode : BuiltInGameMode {
        public FreeBuildMode() :
            base(PlayerMode.FreeBuild, Ownership.None, true, false, false, false, true) { }

        public override bool CheckBuildPermission(Ownership owner) {
            return true;
        }

        public override bool CheckInvitePermission(Ownership owner) {
            return true;
        }

        public override bool CheckEvictionPermission(Ownership owner) {
            return true;
        }

        public override bool CheckRemovePermission(Ownership owner) {
            return true;
        }

        public override bool CheckSellPermission(Ownership owner) {
            return false;
        }

        public override bool CheckBuyPermission(Ownership owner) {
            return false;
        }
    }
    #endregion
    #region GovernmentMode
    internal class GovernmentMode : BuiltInGameMode {
        public GovernmentMode() :
            base(PlayerMode.Government, Ownership.Government, true, false, false, true, true) { }

        public override bool CheckInvitePermission(Ownership owner) {
            return owner==Ownership.NonPlayerCompany||owner==Ownership.Private;
        }

        public override bool CheckEvictionPermission(Ownership owner) {
            return true;
        }

        public override bool CheckRemovePermission(Ownership owner) {
            return owner==Ownership.Government||owner==Ownership.LocalAuthority||owner==Ownership.None;
        }

        public override bool CheckSellPermission(Ownership owner) {
            return false;
        }
    }
    #endregion
    #region LocalAuthorityMode
    internal class LocalAuthMode : BuiltInGameMode {
        public LocalAuthMode() :
            base(PlayerMode.LocalAuthority, Ownership.LocalAuthority, false, false, true, true, false) { }

        public override bool CheckInvitePermission(Ownership owner) {
            return owner == Ownership.NonPlayerCompany || owner == Ownership.Private || owner == Ownership.Government;
        }

        public override bool CheckEvictionPermission(Ownership owner) {
            return owner!=Ownership.Government;
        }

        public override bool CheckRemovePermission(Ownership owner) {
            return owner == Ownership.LocalAuthority || owner == Ownership.None;
        }

        public override bool CheckSellPermission(Ownership owner) {
            return false;
        }
    }
    #endregion
    #region CompanyMode
    internal class CompanyMode : BuiltInGameMode {
        public CompanyMode() :
            base(PlayerMode.Company, Ownership.PlayerCompany, false, true, true, false, true) { }

        public override bool CheckInvitePermission(Ownership owner) {
            return owner == Ownership.LocalAuthority || owner == Ownership.Government;
        }

        public override bool CheckEvictionPermission(Ownership owner) {
            return owner == Ownership.Private || owner == Ownership.NonPlayerCompany;
        }

        public override bool CheckRemovePermission(Ownership owner) {
            return owner == Ownership.PlayerCompany || owner == Ownership.None;
        }

        public override bool CheckBuyPermission(Ownership owner) {
            return owner == Ownership.Private || owner == Ownership.NonPlayerCompany;
        }
    }
    #endregion
}
