﻿#pragma once

/*

Wu, Xiaolin

http://freespace.virgin.net/hugo.elias/graphics/x_wuline.htm
http://en.wikipedia.org/wiki/Xiaolin_Wu%27s_line_algorithm

http://www.cs.nps.navy.mil/people/faculty/capps/iap/class1/lines/lines.html
を参考にC++で実装。

8bit版

*/

#include <math.h>
#include <algorithm>
#include "buffer2d.h"
#include "arrayutil.h"

#include "ILineDrawer.h"

namespace gl
{

template <typename NumericT>
class LineDrawer8_Wu : public IBufferLineDrawer<NumericT, uint8_t>
{
protected:
	Buffer2D<uint8_t>*	pBuff_;		//!< 描画先
	
	//! Correction table depent on the slope
	static const size_t SLOPECORR_TABLE_SIZE = 512;
	NumericT slopeCorrTable_[SLOPECORR_TABLE_SIZE];	
	
	void initTable()
	{
		initSlopeCorrTable();
	}
	
	void initSlopeCorrTable()
	{
		for (size_t i=0; i<SLOPECORR_TABLE_SIZE; ++i) {
			double m = (i + 0.5) / SLOPECORR_TABLE_SIZE;
			slopeCorrTable_[i] = sqrt(m*m + 1) * 0.707106781; /* (m+1)^2 / sqrt(2) */
		}
	}
	
	inline
	void blend(
		uint8_t* p,
		NumericT v
		)
	{
		uint8_t old = *p;
		if (old != 0xff) {
			NumericT nv = NumericT(old) + v;
			*p = (nv >= NumericT(255.0)) ? 255 : halfAdjust(nv);
		}
	}
	
public:
	LineDrawer8_Wu()
		:
		pBuff_(0)
	{
		initTable();
	}
	
	void SetBuffer(Buffer2D<uint8_t>* pBuff)
	{
		pBuff_ = pBuff;
	}
	
	virtual
	void DrawVerticalLine(uint8_t col, NumericT x, NumericT y1, NumericT y2)
	{
	}
	
	void DrawLine(uint8_t intensity, NumericT x1, NumericT y1, NumericT x2, NumericT y2)
	{
		NumericT width = abs(x2 - x1);
		NumericT height = abs(y2 - y1);
		if (width + height == NumericT(0))
			return;
		// ピクセルの中央位置を仮に整数座標と認識している。
		NumericT palpha = NumericT(intensity);
		const int lineOffset = pBuff_->GetLineOffset();
		if (height < width) {
			if (x2 < x1) {
				std::swap(x1, x2);
				std::swap(y1, y2);
			}
			NumericT gradient = (y2 - y1) / width;
			NumericT slope = slopeCorrTable_[ ToInt( abs(gradient) * NumericT(SLOPECORR_TABLE_SIZE-1) ) ];
			palpha *= slope;
			
			unsigned char* ptr;
			NumericT yf;
			size_t prev_iyf;
			
			// Start Point
			{
				NumericT xgap = NumericT(1) - frac(x1 + NumericT(0.5));		// ピクセルの横占有率
				NumericT xend = halfAdjust(x1);								// 最近傍の実X座標を求める
				NumericT yend = y1 + gradient * (xend - x1);				// ピクセルの中央位置（仮想開始位置）までXを移動した場合のY座標を求める
				ptr = (unsigned char*) pBuff_->GetPixelPtr(xend, ToInt(yend));

				NumericT dist = frac(yend);									// 仮想ピクセル中央Y座標からの距離
				uint8_t* cptr = (uint8_t*)ptr;
				blend(cptr, palpha * xgap * (NumericT(1) - dist));
				cptr = (uint8_t*)(ptr + lineOffset);
				blend(cptr, palpha * xgap * dist);
				
				yf = yend;
				prev_iyf = ToInt(yf);
				yf += gradient;
				size_t cur_iyf = ToInt(yf);
				ptr += lineOffset * (cur_iyf - prev_iyf) + sizeof(uint8_t);
				prev_iyf = cur_iyf;
			}
			
			// Mid Points
			size_t loopCnt = halfAdjust(x2) - halfAdjust(x1);
			if (loopCnt == 0) {
				return;
			}
			while (--loopCnt) {
				NumericT dist = frac(yf);
				uint8_t* cptr = (uint8_t*) ptr;
				blend(cptr, palpha * (NumericT(1)-dist));
				cptr = (uint8_t*)(ptr + lineOffset);
				blend(cptr, palpha * dist);
				
				yf += gradient;
				size_t cur_iyf = ToInt(yf);
				ptr += lineOffset * (cur_iyf - prev_iyf) + sizeof(uint8_t);
				prev_iyf = cur_iyf;
			}
			
			// End Point
			{
				NumericT xgap = frac(x2 + NumericT(0.5));			// ピクセルの横占有率
				NumericT xend = halfAdjust(x2);						// 最近傍の実X座標を求める
				NumericT yend = y2 + gradient * (xend - x2);		// ピクセルの中央位置（仮想開始位置）までXを移動した場合のY座標を求める
				
				NumericT dist = frac(yf);							// 仮想ピクセル中央Y座標からの距離
				uint8_t* cptr = (uint8_t*)ptr;
				blend(cptr, palpha * xgap * (NumericT(1) - dist));
				cptr = (uint8_t*)(ptr + lineOffset);
				blend(cptr, palpha * xgap * dist);
			}
			
		}
		else {
			if (y2 < y1) {
				std::swap(x1, x2);
				std::swap(y1, y2);
			}
			NumericT gradient = (x2 - x1) / height;
			NumericT slope = slopeCorrTable_[ ToInt( abs(gradient) * NumericT(SLOPECORR_TABLE_SIZE-1) ) ];
			palpha *= slope;
			
			unsigned char* ptr;
			NumericT xf;
			size_t prev_ixf;
			
			// Start Point
			{
				NumericT ygap = NumericT(1) - frac(y1 + NumericT(0.5));		// ピクセルの縦占有率
				NumericT yend = halfAdjust(y1);								// 最近傍の実Y座標を求める
				NumericT xend = x1 + gradient * (yend - y1);				// ピクセルの中央位置（仮想開始位置）までYを移動した場合のX座標を求める
				ptr = (unsigned char*) pBuff_->GetPixelPtr(ToInt(xend), yend);
				
				NumericT dist = frac(xend);									// 仮想ピクセル中央X座標からの距離
				uint8_t* cptr = (uint8_t*)ptr;
				blend(cptr, palpha * ygap * (NumericT(1) - dist));
				++cptr;
				blend(cptr, palpha * ygap * dist);
				
				xf = xend;
				prev_ixf = ToInt(xf);
				xf += gradient;
				size_t cur_ixf = ToInt(xf);
				ptr += lineOffset + (cur_ixf - prev_ixf) * sizeof(uint8_t);
				prev_ixf = cur_ixf;
			}

			// Mid Points
			size_t loopCnt = halfAdjust(y2) - halfAdjust(y1);
			if (loopCnt == 0) {
				return;
			}
			while (--loopCnt) {
				NumericT dist = frac(xf);
				uint8_t* cptr = (uint8_t*) ptr;
				blend(cptr, palpha * (NumericT(1)-dist));
				++cptr;
				blend(cptr, palpha * dist);
				
				xf += gradient;
				size_t cur_ixf = ToInt(xf);
				ptr += lineOffset + (cur_ixf - prev_ixf) * sizeof(uint8_t);
				prev_ixf = cur_ixf;
			}

			// End Point
			{
				NumericT ygap = frac(y2 + NumericT(0.5));			// ピクセルの縦占有率
				NumericT yend = halfAdjust(y2);						// 最近傍の実Y座標を求める
				NumericT xend = x2 + gradient * (yend - y2);		// ピクセルの中央位置（仮想開始位置）までYを移動した場合のX座標を求める

				NumericT dist = frac(xf);							// 仮想ピクセル中央X座標からの距離
				uint8_t* cptr = (uint8_t*)ptr;
				blend(cptr, palpha * ygap * (NumericT(1) - dist));
				++cptr;
				blend(cptr, palpha * ygap * dist);
			}
		}
	}
	
};

} // namespace gl

