﻿Public Class Board

    Public Const SIZE_X As Integer = 8
    Public Const SIZE_Y As Integer = 8
    Public Const MAX_TURNS = 60



    Private _arrangement((SIZE_X + 2) * (SIZE_Y + 2) - 1) As DiscColor


    Private _turns As Integer = 0
    ''' <summary>
    ''' 何手目かを返す
    ''' </summary>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public Function GetTurns() As Integer
        Return _turns
    End Function

    Private _currentColor As DiscColor
    ''' <summary>
    ''' 現在の手番の色
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public ReadOnly Property CurrentColor As DiscColor
        Get
            Return _currentColor
        End Get
    End Property

    Protected _updateLogs As New List(Of ReverseLog)
    Public ReadOnly Property History As IList(Of Point)
        Get
            Dim points As New List(Of Point)
            For Each log As ReverseLog In _updateLogs
                points.Add(log.PutDisc)
            Next
            Return points
        End Get
    End Property

    Private _movableDir(MAX_TURNS, SIZE_X + 1, SIZE_Y + 1) As Direction
    Private _movablePos(MAX_TURNS) As List(Of Point)

    Public ReadOnly Property MovablePostions() As List(Of Point)
        Get
            Return _movablePos(_turns)
        End Get
    End Property


    ''' <summary>
    ''' ゲーム終了しているかどうか
    ''' </summary>
    ''' <value></value>
    ''' <returns></returns>
    ''' <remarks></remarks>
    Public ReadOnly Property IsGameEnd As Boolean
        Get
            If _turns = MAX_TURNS Then
                Return True
            End If

            If MovablePostions.Count <> 0 Then
                Return False
            End If

            Dim disc As New Disc(0, 0, GetReverseColor(CurrentColor))
            For x As Integer = 1 To SIZE_X
                disc.X = x
                For y As Integer = 1 To SIZE_Y
                    disc.Y = y
                    Dim dir As Direction = checkMobilityFast(disc)
                    If dir.CanSomewhere Then
                        Return False
                    End If
                Next
            Next
            Return True
        End Get
    End Property

    Public Function GetReverseColor(ByVal discColor As DiscColor) As DiscColor
        Return DirectCast(-discColor, DiscColor)
    End Function



    Public DiscsCount As New ColorStorage

    Public Function StoneCount(ByVal color As DiscColor) As Integer
        Return Me.DiscsCount(color)
    End Function

    Public ReadOnly Property Cells(ByVal x As Integer, ByVal y As Integer) As DiscColor
        Get
            Debug.Assert(1 <= x AndAlso x <= SIZE_X)
            Debug.Assert(1 <= y AndAlso y <= SIZE_Y)
            Return _arrangement(y * (SIZE_X + 2) + x)
        End Get
    End Property

    Private Property CellIncludeWall(ByVal x As Integer, ByVal y As Integer) As DiscColor
        Get
            Debug.Assert(0 <= x AndAlso x <= SIZE_X + 1)
            Debug.Assert(0 <= y AndAlso y <= SIZE_Y + 1)
            Return _arrangement(y * (SIZE_X + 2) + x)
        End Get
        Set(ByVal value As DiscColor)
            Debug.Assert(0 <= x AndAlso x <= SIZE_X + 1)
            Debug.Assert(0 <= y AndAlso y <= SIZE_Y + 1)
            _arrangement(y * (SIZE_X + 2) + x) = value
        End Set
    End Property

    Public WriteOnly Property InitDisc(ByVal x As Integer, ByVal y As Integer) As DiscColor
        Set(ByVal value As DiscColor)
            Debug.Assert(1 <= x AndAlso x <= SIZE_X)
            Debug.Assert(1 <= y AndAlso y <= SIZE_Y)
            Debug.Assert(value <> DiscColor.WALL)
            DiscsCount(Cells(x, y)) -= 1
            CellIncludeWall(x, y) = value
            DiscsCount(Cells(x, y)) += 1
            InitMovable()
        End Set
    End Property

    Public Shared Function ValidPosition(ByVal x As Integer, ByVal y As Integer) As Boolean
        Return (1 <= x AndAlso x <= SIZE_X) AndAlso
               (1 <= y AndAlso y <= SIZE_Y)
    End Function

    Public Sub New()
        Init()
    End Sub

    Public Sub Init()

        For Each cell In _arrangement
            cell = DiscColor.EMPTY
        Next

        For y As Integer = 0 To SIZE_Y + 1
            CellIncludeWall(0, y) = DiscColor.WALL
            CellIncludeWall(SIZE_X + 1, y) = DiscColor.WALL
        Next
        For x As Integer = 0 To SIZE_X + 1
            CellIncludeWall(x, 0) = DiscColor.WALL
            CellIncludeWall(x, SIZE_Y + 1) = DiscColor.WALL
        Next

        DiscsCount(DiscColor.EMPTY) = SIZE_X * SIZE_Y
        InitDisc(4, 4) = DiscColor.WHITE
        InitDisc(4, 5) = DiscColor.BLACK
        InitDisc(5, 4) = DiscColor.BLACK
        InitDisc(5, 5) = DiscColor.WHITE

        _currentColor = DiscColor.BLACK
        InitMovable()

        _turns = 0


        _updateLogs.Clear()
    End Sub

    Public Sub New(ByVal source As Board)
        Debug.Assert(source IsNot Nothing)
        For i As Integer = 0 To _arrangement.Length - 1
            Me._arrangement(i) = source._arrangement(i)
        Next
        Me._currentColor = source._currentColor
        Me._turns = source._turns

        Me.DiscsCount(DiscColor.BLACK) = source.DiscsCount(DiscColor.BLACK)
        Me.DiscsCount(DiscColor.WHITE) = source.DiscsCount(DiscColor.WHITE)
        Me.DiscsCount(DiscColor.EMPTY) = source.DiscsCount(DiscColor.EMPTY)

        For t As Integer = 0 To source._turns
            For x As Integer = 1 To SIZE_X
                For y As Integer = 1 To SIZE_Y
                    If source._turns <> MAX_TURNS Then
                        Debug.Assert(source._movableDir(t, x, y) IsNot Nothing)
                        Me._movableDir(t, x, y) = New Direction(source._movableDir(t, x, y))
                    End If
                Next
            Next
        Next

        For t As Integer = 0 To source._turns
            Me._movablePos(t) = New List(Of Point)
            For Each pos As Point In source._movablePos(t)
                Me._movablePos(t).Add(pos)
            Next
        Next

        For Each log As ReverseLog In source._updateLogs
            Me._updateLogs.Add(log)
        Next

    End Sub

    Public Function CanMoveAny() As Boolean
        Return 0 < MovablePostions.Count
    End Function

    Public Overridable Sub Move(ByVal point As Point)
        Debug.Assert(ValidPosition(point.X, point.Y))
        Debug.Assert(CanMove(point.X, point.Y))
        flipDiscs(point)
        _turns += 1
        _currentColor = GetReverseColor(_currentColor)
        InitMovable()
    End Sub

    Private Sub flipDiscs(ByVal point As Point)
        Dim opeDisc As New Disc(point.X, point.Y, CurrentColor)
        Dim dir As Direction = _movableDir(_turns, point.X, point.Y)
        If dir.LOWER_RIGHT Is Nothing Then
            checkMobility(opeDisc, dir)
        End If

        Dim updateLog As New ReverseLog

        CellIncludeWall(point.X, point.Y) = CurrentColor
        updateLog.PutDisc = opeDisc

        If dir.UPPER Then
            Dim x As Integer = point.X
            Dim y As Integer = point.Y - 1
            Do
                CellIncludeWall(x, y) = CurrentColor
                updateLog.AddPosition(New Point(x, y))
                y -= 1
            Loop While CellIncludeWall(x, y) <> CurrentColor
        End If

        If dir.LOWER Then
            Dim x As Integer = point.X
            Dim y As Integer = point.Y + 1
            Do
                CellIncludeWall(x, y) = CurrentColor
                updateLog.AddPosition(New Point(x, y))
                y += 1
            Loop While CellIncludeWall(x, y) <> CurrentColor
        End If

        If dir.LEFT Then
            Dim x As Integer = point.X - 1
            Dim y As Integer = point.Y
            Do
                CellIncludeWall(x, y) = CurrentColor
                updateLog.AddPosition(New Point(x, y))
                x -= 1
            Loop While CellIncludeWall(x, y) <> CurrentColor
        End If

        If dir.RIGHT Then
            Dim x As Integer = point.X + 1
            Dim y As Integer = point.Y
            Do
                CellIncludeWall(x, y) = CurrentColor
                updateLog.AddPosition(New Point(x, y))
                x += 1
            Loop While CellIncludeWall(x, y) <> CurrentColor
        End If

        If dir.UPPER_RIGHT Then
            Dim x As Integer = point.X + 1
            Dim y As Integer = point.Y - 1
            Do
                CellIncludeWall(x, y) = CurrentColor
                updateLog.AddPosition(New Point(x, y))
                x += 1
                y -= 1
            Loop While CellIncludeWall(x, y) <> CurrentColor
        End If

        If dir.UPPER_LEFT Then
            Dim x As Integer = point.X - 1
            Dim y As Integer = point.Y - 1
            Do
                CellIncludeWall(x, y) = CurrentColor
                updateLog.AddPosition(New Point(x, y))
                x -= 1
                y -= 1
            Loop While CellIncludeWall(x, y) <> CurrentColor
        End If

        If dir.LOWER_LEFT Then
            Dim x As Integer = point.X - 1
            Dim y As Integer = point.Y + 1
            Do
                CellIncludeWall(x, y) = CurrentColor
                updateLog.AddPosition(New Point(x, y))
                x -= 1
                y += 1
            Loop While CellIncludeWall(x, y) <> CurrentColor
        End If

        If dir.LOWER_RIGHT Then
            Dim x As Integer = point.X + 1
            Dim y As Integer = point.Y + 1
            Do
                CellIncludeWall(x, y) = CurrentColor
                updateLog.AddPosition(New Point(x, y))
                x += 1
                y += 1
            Loop While CellIncludeWall(x, y) <> CurrentColor
        End If

        Dim discdiff = updateLog.ReverseCount()
        DiscsCount(CurrentColor) += discdiff + 1
        DiscsCount(GetReverseColor(CurrentColor)) -= discdiff
        DiscsCount(DiscColor.EMPTY) -= 1
        _updateLogs.Add(updateLog)
    End Sub

    Public Sub Pass()
        Debug.Assert(Not CanMoveAny())
        Debug.Assert(Not IsGameEnd)
        _currentColor = GetReverseColor(_currentColor)
        _updateLogs.Add(New ReverseLog)
        InitMovable()
    End Sub

    Public Function CanUndo() As Boolean
        Return 0 < _turns
    End Function

    Public Overridable Sub Undo()
        Debug.Assert(0 < _turns)
        Dim reverseColor As DiscColor = _currentColor
        _currentColor = GetReverseColor(_currentColor)

        Dim log As ReverseLog = _updateLogs.Last

        If log.IsPass Then
            MovablePostions.Clear()
            For x As Integer = 1 To SIZE_X
                For y As Integer = 1 To SIZE_Y
                    Dim dir As Direction = _movableDir(_turns, x, y)
                    dir.UPPER = False
                    dir.UPPER_LEFT = False
                    dir.UPPER_RIGHT = False
                    dir.LEFT = False
                    dir.LOWER = False
                    dir.LOWER_LEFT = False
                    dir.LOWER_RIGHT = False
                    dir.RIGHT = False
                Next
            Next
        Else
            _turns -= 1
            Dim putDisc As Disc = log.PutDisc
            CellIncludeWall(putDisc.X, putDisc.Y) = DiscColor.EMPTY
            For Each p As Point In log.ReversePostions
                CellIncludeWall(p.X, p.Y) = reverseColor
            Next
            DiscsCount(_currentColor) -= log.ReversePostions.Count + 1
            DiscsCount(reverseColor) += log.ReverseCount
            DiscsCount(DiscColor.EMPTY) += 1
        End If

        _updateLogs.RemoveAt(_updateLogs.Count - 1)
    End Sub




    Private Sub InitMovable()

        _movablePos(_turns) = New List(Of Point)
        If _turns = MAX_TURNS Then
            Exit Sub
        End If
        For x As Integer = 1 To SIZE_X
            For y As Integer = 1 To SIZE_Y
                Dim disc As New Disc(x, y, CurrentColor)
                Dim dir As Direction = checkMobilityFast(disc)
                If dir.CanSomewhere Then
                    _movablePos(_turns).Add(disc)
                End If
                _movableDir(_turns, x, y) = dir
            Next
        Next
    End Sub

    Private Function checkMobilityFast(ByVal disc As Disc) As Direction
        Dim dir As New Direction
        Dim reverseColor As DiscColor = GetReverseColor(disc.Color)

        If CellIncludeWall(disc.X, disc.Y) <> DiscColor.EMPTY Then
            Return dir
        End If

        dir.UPPER = False
        If CellIncludeWall(disc.X, disc.Y - 1) = reverseColor Then
            Dim x As Integer = disc.X
            Dim y As Integer = disc.Y - 2
            While CellIncludeWall(x, y) = reverseColor
                y -= 1
            End While
            If CellIncludeWall(x, y) = disc.Color Then
                dir.UPPER = True
                dir.CanSomewhere = True
                Return dir
            End If
        End If

        dir.RIGHT = False
        If CellIncludeWall(disc.X + 1, disc.Y) = reverseColor Then
            Dim x As Integer = disc.X + 2
            Dim y As Integer = disc.Y
            While CellIncludeWall(x, y) = reverseColor
                x += 1
            End While
            If CellIncludeWall(x, y) = disc.Color Then
                dir.RIGHT = True
                dir.CanSomewhere = True
                Return dir
            End If
        End If

        dir.LOWER = False
        If CellIncludeWall(disc.X, disc.Y + 1) = reverseColor Then
            Dim x As Integer = disc.X
            Dim y As Integer = disc.Y + 2
            While CellIncludeWall(x, y) = reverseColor
                y += 1
            End While
            If CellIncludeWall(x, y) = disc.Color Then
                dir.LOWER = True
                dir.CanSomewhere = True
                Return dir
            End If
        End If

        dir.LEFT = False
        If CellIncludeWall(disc.X - 1, disc.Y) = reverseColor Then
            Dim x As Integer = disc.X - 2
            Dim y As Integer = disc.Y
            While CellIncludeWall(x, y) = reverseColor
                x -= 1
            End While
            If CellIncludeWall(x, y) = disc.Color Then
                dir.LEFT = True
                dir.CanSomewhere = True
                Return dir
            End If
        End If


        dir.UPPER_LEFT = False
        If CellIncludeWall(disc.X - 1, disc.Y - 1) = reverseColor Then
            Dim x As Integer = disc.X - 2
            Dim y As Integer = disc.Y - 2
            While CellIncludeWall(x, y) = reverseColor
                x -= 1
                y -= 1
            End While
            If CellIncludeWall(x, y) = disc.Color Then
                dir.UPPER_LEFT = True
                dir.CanSomewhere = True
                Return dir
            End If
        End If

        dir.UPPER_RIGHT = False
        If CellIncludeWall(disc.X + 1, disc.Y - 1) = reverseColor Then
            Dim x As Integer = disc.X + 2
            Dim y As Integer = disc.Y - 2
            While CellIncludeWall(x, y) = reverseColor
                x += 1
                y -= 1
            End While
            If CellIncludeWall(x, y) = disc.Color Then
                dir.UPPER_RIGHT = True
                dir.CanSomewhere = True
                Return dir
            End If
        End If

        dir.LOWER_LEFT = False
        If CellIncludeWall(disc.X - 1, disc.Y + 1) = reverseColor Then
            Dim x As Integer = disc.X - 2
            Dim y As Integer = disc.Y + 2
            While CellIncludeWall(x, y) = reverseColor
                x -= 1
                y += 1
            End While
            If CellIncludeWall(x, y) = disc.Color Then
                dir.LOWER_LEFT = True
                dir.CanSomewhere = True
                Return dir
            End If
        End If

        dir.LOWER_RIGHT = False
        If CellIncludeWall(disc.X + 1, disc.Y + 1) = reverseColor Then
            Dim x As Integer = disc.X + 2
            Dim y As Integer = disc.Y + 2
            While CellIncludeWall(x, y) = reverseColor
                x += 1
                y += 1
            End While
            If CellIncludeWall(x, y) = disc.Color Then
                dir.LOWER_RIGHT = True
                dir.CanSomewhere = True
                Return dir
            End If
        End If

        Return dir
    End Function

    Private Sub checkMobility(ByVal disc As Disc, ByVal dir As Direction)
        Debug.Assert(dir.CanSomewhere = True)
        Debug.Assert(dir.LOWER_RIGHT Is Nothing)

        Dim reverseColor As DiscColor = GetReverseColor(disc.Color)

        If dir.RIGHT Is Nothing Then
            dir.RIGHT = False
            If CellIncludeWall(disc.X + 1, disc.Y) = reverseColor Then
                Dim x As Integer = disc.X + 2
                Dim y As Integer = disc.Y
                While CellIncludeWall(x, y) = reverseColor
                    x += 1
                End While
                If CellIncludeWall(x, y) = disc.Color Then
                    dir.RIGHT = True
                End If
            End If
        End If

        If dir.LOWER Is Nothing Then
            dir.LOWER = False
            If CellIncludeWall(disc.X, disc.Y + 1) = reverseColor Then
                Dim x As Integer = disc.X
                Dim y As Integer = disc.Y + 2
                While CellIncludeWall(x, y) = reverseColor
                    y += 1
                End While
                If CellIncludeWall(x, y) = disc.Color Then
                    dir.LOWER = True
                End If
            End If
        End If

        If dir.LEFT Is Nothing Then
            If CellIncludeWall(disc.X - 1, disc.Y) = reverseColor Then
                Dim x As Integer = disc.X - 2
                Dim y As Integer = disc.Y
                While CellIncludeWall(x, y) = reverseColor
                    x -= 1
                End While
                If CellIncludeWall(x, y) = disc.Color Then
                    dir.LEFT = True
                End If
            End If
        End If

        If dir.UPPER_LEFT Is Nothing Then
            dir.UPPER_LEFT = False
            If CellIncludeWall(disc.X - 1, disc.Y - 1) = reverseColor Then
                Dim x As Integer = disc.X - 2
                Dim y As Integer = disc.Y - 2
                While CellIncludeWall(x, y) = reverseColor
                    x -= 1
                    y -= 1
                End While
                If CellIncludeWall(x, y) = disc.Color Then
                    dir.UPPER_LEFT = True
                End If
            End If
        End If

        If dir.UPPER_RIGHT Is Nothing Then
            dir.UPPER_RIGHT = False
            If CellIncludeWall(disc.X + 1, disc.Y - 1) = reverseColor Then
                Dim x As Integer = disc.X + 2
                Dim y As Integer = disc.Y - 2
                While CellIncludeWall(x, y) = reverseColor
                    x += 1
                    y -= 1
                End While
                If CellIncludeWall(x, y) = disc.Color Then
                    dir.UPPER_RIGHT = True
                End If
            End If
        End If

        If dir.LOWER_LEFT Is Nothing Then
            dir.LOWER_LEFT = False
            If CellIncludeWall(disc.X - 1, disc.Y + 1) = reverseColor Then
                Dim x As Integer = disc.X - 2
                Dim y As Integer = disc.Y + 2
                While CellIncludeWall(x, y) = reverseColor
                    x -= 1
                    y += 1
                End While
                If CellIncludeWall(x, y) = disc.Color Then
                    dir.LOWER_LEFT = True
                End If
            End If
        End If

        If dir.LOWER_RIGHT Is Nothing Then
            dir.LOWER_RIGHT = False
            If CellIncludeWall(disc.X + 1, disc.Y + 1) = reverseColor Then
                Dim x As Integer = disc.X + 2
                Dim y As Integer = disc.Y + 2
                While CellIncludeWall(x, y) = reverseColor
                    x += 1
                    y += 1
                End While
                If CellIncludeWall(x, y) = disc.Color Then
                    dir.LOWER_RIGHT = True
                End If
            End If
        End If
    End Sub

    Public Function CanMove(ByVal x As Integer, ByVal y As Integer) As Boolean
        Debug.Assert(ValidPosition(x, y))
        Return Me._movableDir(_turns, x, y).CanSomewhere
    End Function

End Class
