/*
 * Copyright (C) 2012 infodb.org. All rights reserved.
 * This program is made available under the terms of
 * the Common Public License v1.0
 */
package org.infodb.wax.core.db;

import java.io.IOException;
import java.io.Writer;
import java.util.ArrayList;
import java.util.List;
import java.util.Stack;
import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;

public class XmlWriter implements ContentHandler {
    private class XMLNS {
        public String prefix;
        public String uri;
        public XMLNS(String prefix, String uri) {
            this.prefix = prefix;
            this.uri = uri;
        }
    }
    private Writer writer;
    private boolean declaration;
    private Stack<List<XMLNS>> nsStack;
    private List<XMLNS> currentNS;
    
    public XmlWriter(Writer writer) {
        this.writer = writer;
        declaration = false;
        nsStack = new Stack<>();
        currentNS = null;
    }
    private void write(String value) throws SAXException {
        try {
            writer.write(value);
        }
        catch(IOException e) {
            throw new SAXException(e);
        }
    }
    private void write(char ch) throws SAXException {
        try {
            writer.write(ch);
        }
        catch(IOException e) {
            throw new SAXException(e);
        }
    }
    private void writeName(String qName) throws SAXException {
        write(qName);
    }
    private void writeAttributes(Attributes atts) throws SAXException {
        int len = atts.getLength();
        for(int i = 0; i < len; i++) {
            char ch[] = atts.getValue(i).toCharArray();
            write(' ');
            writeName(atts.getQName(i));
            write("=\"");
            writeEsc(ch, 0, ch.length, true);
            write('"');
        }
    }
    private void writeEsc(char ch[], int start, int length, boolean isAttr) throws SAXException {
        for(int i = start; i < start + length; i++) {
            switch(ch[i]) {
            case '&':
                write("&amp;");
                break;
            case '<':
                write("&lt;");
                break;
            case '>':
                write("&gt;");
                break;
            case '"':
                if(isAttr) {
                    write("&quot;");
                } else {
                    write('"');
                }
                break;
            default:
                write(ch[i]);
            }
        }
    }
    @Override
    public void setDocumentLocator(Locator locator) {
    }
    @Override
    public void startDocument() throws SAXException {
        if(declaration == false) {
            write("<?xml version=\"1.0\" encoding=\"UTF-8\"?>");
            declaration = true;
        }
    }
    @Override
    public void endDocument() throws SAXException {
        try {
            writer.flush();
        } catch (IOException e) {
            throw new SAXException(e);
        }
    }
    @Override
    public void startPrefixMapping(String prefix, String uri) throws SAXException {
        if(currentNS == null) {
            currentNS = new ArrayList<>();
            nsStack.push(currentNS);
        }
        currentNS.add(new XMLNS(prefix, uri));
    }
    @Override
    public void endPrefixMapping(String prefix) throws SAXException {
        // スタートした個数だけ呼ばれる計算。
        currentNS = nsStack.peek();
        currentNS.remove(0);
        if(currentNS.isEmpty()) {
            nsStack.pop();
        }
        currentNS = null;
    }
    @Override
    public void startElement(String uri, String localName, String qName, Attributes atts) throws SAXException {
        write('<');
        writeName(qName);
        if(currentNS != null) {
            for(XMLNS xmlns : currentNS) {
                char[] ch = xmlns.uri.toCharArray();
                write(' ');
                if(xmlns.prefix.equals("")) {
                    write("xmlns=\"");
                } else {
                    write("xmlns:");
                    write(xmlns.prefix);
                    write("=\"");
                }
                writeEsc(ch, 0, ch.length, true);
                write('"');
            }
            currentNS = null;
        }
        writeAttributes(atts);
        write('>');
    }
    @Override
    public void endElement(String uri, String localName, String qName) throws SAXException {
        write("</");
        writeName(qName);
        write('>');
    }
    @Override
    public void characters(char[] ch, int start, int length) throws SAXException {
        writeEsc(ch, start, length, false);
    }
    @Override
    public void ignorableWhitespace(char[] ch, int start, int length) throws SAXException {
        writeEsc(ch, start, length, false);
    }
    @Override
    public void processingInstruction(String target, String data) throws SAXException {
        write("<?");
        write(target);
        write(' ');
        write(data);
        write("?>");
    }
    @Override
    public void skippedEntity(String name) throws SAXException {
    }
}
