/*
 * Copyright (C) 2012 infodb.org. All rights reserved.
 * This program is made available under the terms of
 * the Common Public License v1.0
 */
package org.infodb.wax.core.db;

import java.io.IOException;
import java.io.Writer;
import java.util.Stack;
import javax.xml.parsers.DocumentBuilder;
import javax.xml.parsers.ParserConfigurationException;
import javax.xml.transform.TransformerConfigurationException;
import org.infodb.wax.core.utils.XMLHelper;
import org.w3c.dom.Document;
import org.w3c.dom.Element;
import org.w3c.dom.Node;
import org.w3c.dom.ProcessingInstruction;
import org.w3c.dom.Text;
import org.xml.sax.Attributes;
import org.xml.sax.ContentHandler;
import org.xml.sax.InputSource;
import org.xml.sax.Locator;
import org.xml.sax.SAXException;
import org.xml.sax.XMLReader;
import org.xml.sax.helpers.XMLReaderFactory;

/**
 * XMLの書き込みに使用するクラス。SAX解析と同時にDOMも生成する。
 */
public class XmlStore implements ContentHandler {
    private static final String FEATURE_NAMESPACES = "http://xml.org/sax/features/namespaces";
    private static final String FEATURE_NS_PREFIX = "http://xml.org/sax/features/namespace-prefixes";
    
    private InputSource xmlSource;
    private XmlWriter xmlWriter;
    
    private DocumentBuilder builder;
    private Document document;
    private Stack<Node> stack;
    
    public XmlStore() throws SAXException, TransformerConfigurationException, ParserConfigurationException {
        builder = XMLHelper.newDocumentBuilder();
        stack = new Stack<>();
    }
    public void setInputSource(InputSource xml) {
        xmlSource = xml;
    }
    protected void setWriter(Writer writer) {
        xmlWriter = new XmlWriter(writer);
    }
    protected void execute() throws SAXException {
        XMLReader reader = XMLReaderFactory.createXMLReader();
        reader.setContentHandler(this);
        try {
            reader.parse(xmlSource);
        } catch (IOException e) {
            throw new SAXException(e.getMessage(), e);
        }
    }
    public Document getDocument() {
        return document;
    }
    @Override
    public void setDocumentLocator(Locator locator) {
        xmlWriter.setDocumentLocator(locator);
    }
    @Override
    public void startDocument() throws SAXException {
        xmlWriter.startDocument();
    }
    @Override
    public void endDocument() throws SAXException {
        xmlWriter.endDocument();
    }
    @Override
    public void startPrefixMapping(String prefix, String uri) throws SAXException {
        xmlWriter.startPrefixMapping(prefix, uri);
    }
    @Override
    public void endPrefixMapping(String prefix) throws SAXException {
        xmlWriter.endPrefixMapping(prefix);
    }
    @Override
    public void startElement(String uri, String localName, String qName, Attributes atts) throws SAXException {
        xmlWriter.startElement(uri, localName, qName, atts);
        
        if(document == null) {
            document = builder.getDOMImplementation().createDocument(uri, qName, null);
            document.setXmlStandalone(true);
            Element element = document.getDocumentElement();
            for(int i = 0; i < atts.getLength(); i++) {
                element.setAttributeNS(atts.getURI(i), atts.getQName(i), atts.getValue(i));
            }
            stack.push(element);
        } else {
            Element element = document.createElementNS(uri, qName);
            for(int i = 0; i < atts.getLength(); i++) {
                element.setAttributeNS(atts.getURI(i), atts.getQName(i), atts.getValue(i));
            }
            stack.peek().appendChild(element);
            stack.push(element);
        }
    }
    @Override
    public void endElement(String uri, String localName, String qName) throws SAXException {
        xmlWriter.endElement(uri, localName, qName);

        stack.pop();
    }
    @Override
    public void characters(char[] ch, int start, int length) throws SAXException {
        xmlWriter.characters(ch, start, length);
        
        String value = new String(ch, start, length);
        Text text = document.createTextNode(value);
        stack.peek().appendChild(text);
    }
    @Override
    public void ignorableWhitespace(char[] ch, int start, int length) throws SAXException {
        xmlWriter.ignorableWhitespace(ch, start, length);
    }
    @Override
    public void processingInstruction(String target, String data) throws SAXException {
        xmlWriter.processingInstruction(target, data);
        
        ProcessingInstruction pi = document.createProcessingInstruction(target, data);
        stack.peek().appendChild(pi);
    }
    @Override
    public void skippedEntity(String name) throws SAXException {
        xmlWriter.skippedEntity(name);
    }
}
