/*
 * Copyright (C) 2012 infodb.org. All rights reserved.
 * This program is made available under the terms of
 * the Common Public License v1.0
 */
package org.infodb.wax.core.db;

import java.io.IOException;
import java.io.Reader;
import java.sql.Clob;
import java.sql.Connection;
import java.sql.DatabaseMetaData;
import java.sql.ResultSet;
import java.sql.SQLException;
import java.util.UUID;
import javax.inject.Inject;
import javax.sql.DataSource;
import org.xml.sax.InputSource;
import org.xml.sax.SAXException;

public class XmlStorage {
    private DataSource ds;
    private Definition def;
    private TableDef tableDef;
    
    private String querySQL;

    private String deleteSQL;
    private String insertSQL;
    private String copyJournalSQL;
    
    @Inject
    public XmlStorage(DataSource ds, Definition def, TableDef tableDef) {
        this.ds = ds;
        this.def = def;
        this.tableDef = tableDef;
        makeSQLs();
    }
    private void makeSQLs() {
        querySQL = makeQuerySQL();
        
        insertSQL = makeInsertSQL();
        deleteSQL = makeDeleteSQL();
        copyJournalSQL = makeCopyJournalSQL();
    }
    private String makeQuerySQL() {
        StringBuilder sb = new StringBuilder("SELECT x.content_type, ");
        sb.append(def.getSelectXmlCol("x.xml_data"));
        sb.append(" FROM ");
        sb.append(tableDef.getTableName());
        sb.append(" x");
        return sb.toString();
    }
    private String makeInsertSQL() {
        StringBuilder sb = new StringBuilder("INSERT INTO ");
        sb.append(tableDef.getTableName());
        sb.append(" VALUES (?, ?,");
        sb.append(def.getUpdateXmlCol());
        sb.append(", ");
        sb.append(def.getCurDateFunc());
        sb.append(", ?)");
        return sb.toString();
    }
    private String makeDeleteSQL() {
        StringBuilder sb = new StringBuilder("DELETE FROM ");
        sb.append(tableDef.getTableName());
        sb.append(" WHERE id = ?");
        return sb.toString();
    }
    private String makeCopyJournalSQL() {
        StringBuilder sb = new StringBuilder("INSERT INTO ");
        sb.append(tableDef.getTableName()).append("_journal");
        sb.append(" SELECT ?, id, content_type, xml_data, update_date, update_user FROM ");
        sb.append(tableDef.getTableName());
        sb.append(" WHERE id = ?");
        return sb.toString();
    }
    public boolean existTable() throws SQLException {
        try (Connection conn = ds.getConnection()) {
            DatabaseMetaData meta = conn.getMetaData();
            try (ResultSet rs = meta.getTables(null, null, tableDef.getTableName().toUpperCase(), null)) {
                return rs.next();
            }
        }
    }
    public void createTable() throws SQLException {
        String sessionID = UUID.randomUUID().toString();
        try (SQLDirect direct = new SQLDirect(def, ds.getConnection(), sessionID)) {
            SQLDDL ddl = new SQLDDL(def.getTypes());
            ddl.createTable(tableDef.getTableName());
            ddl.column("id", SQLDDL.VARCHAR, 256).notNull().primaryKey();
            ddl.column("content_type", SQLDDL.VARCHAR, 128).notNull();
            ddl.column("xml_data", SQLDDL.XML).nullable();
            ddl.column("update_date", SQLDDL.DATETIME).notNull();
            ddl.column("update_user", SQLDDL.VARCHAR, 64).notNull();
            ddl.end();
            direct.executeSQL(ddl.toString());

            ddl = new SQLDDL(def.getTypes());
            ddl.createTable(tableDef.getTableName() + "_journal");
            ddl.column("journal_id", SQLDDL.VARCHAR, 36).notNull();
            ddl.column("id", SQLDDL.VARCHAR, 256).notNull();
            ddl.column("content_type", SQLDDL.VARCHAR, 128).notNull();
            ddl.column("xml_data", SQLDDL.XML).notNull();
            ddl.column("update_date", SQLDDL.DATETIME).notNull();
            ddl.column("update_user", SQLDDL.VARCHAR, 64).notNull();
            ddl.end();
            direct.executeSQL(ddl.toString());

            if(tableDef.getIndexColumn() != null) {
                ddl = new SQLDDL(def.getTypes());
                ddl.createTable(tableDef.getTableName() + "_index");
                ddl.column("id", SQLDDL.VARCHAR, 256).notNull();
                for(ColumnPair pair : tableDef.getIndexColumn()) {
                    ddl.column(pair.getName(), SQLDDL.VARCHAR, pair.getLength()).isNull(pair.getNullable());
                }
                ddl.end();
                direct.executeSQL(ddl.toString());
            }
        }
    }
    public boolean query(String sessionID, XmlQuery xq) throws SQLException, SAXException, IOException {
        try (SQLQuery query = new SQLQuery(ds.getConnection(), sessionID)) {
            try (ResultSet rs = query.executeQuery(querySQL, xq.getWheres())) { // TODO:order byが必要かと思われる。
                if(rs.next()) {
                    String contentType = rs.getString(1);
                    xq.setContentType(contentType);
                    xq.begin();
                    do {
                        Clob clob = rs.getClob(2);
                        Reader reader = clob.getCharacterStream();
                        xq.setInputSouce(new InputSource(reader));
                        xq.execute();
                    } while(rs.next());
                    xq.end();
                    return true;
                }
            }
        }
        return false;
    }
    public void insert(String sessionID, String id, String contentType, XmlStore xs, String userName) throws SQLException {
        try (SQLDirect direct = new SQLDirect(def, ds.getConnection(), sessionID)) {
            direct.executeSQL(deleteSQL, id);
            direct.executeSQL(insertSQL, id, contentType, xs, userName);
            direct.executeSQL(copyJournalSQL, UUID.randomUUID().toString(), id);
            
            // TODO:インデックスへの保存
            direct.commit();
        }
    }
}
