/*
 * Copyright (C) 2012 infodb.org. All rights reserved.
 * This program is made available under the terms of
 * the Common Public License v1.0
 */
package org.infodb.wax.core;

import java.io.File;
import java.io.IOException;
import java.lang.reflect.InvocationTargetException;
import java.lang.reflect.Method;
import java.net.MalformedURLException;
import java.net.URL;
import java.net.URLClassLoader;
import java.sql.SQLException;
import java.util.ArrayList;
import java.util.HashMap;
import java.util.List;
import java.util.Map;
import java.util.ServiceLoader;
import org.infodb.wax.core.html.AbstractBody;
import org.infodb.wax.core.utils.DirectoryScanner;

public class PluginManager {
    private static class Item {
        public Plugin plugin;
        public Method method;
        public Item(Plugin plugin, Method method) {
            this.plugin = plugin;
            this.method = method;
        }
    }
    private List<Plugin> plugins;
    private Map<String, Item> actions;
    private Map<String, Item> views;
    
    public PluginManager() {
        plugins = new ArrayList<>();
        actions = new HashMap<>();
        views = new HashMap<>();
    }
    public void initLibrary(SystemEnv env) throws MalformedURLException, IOException, SQLException {
        final List<URL> list = new ArrayList<>();
        DirectoryScanner.search(new File(env.getCurrentDirectory(), "plugins"), true, new DirectoryScanner.Listener() {
            @Override
            public void found(File f) throws IOException {
                if(f.isFile() == true && f.getName().endsWith(".jar")) {
                    URL url = new URL("jar:file:" + f.getAbsolutePath() + "!/");
                    list.add(url);
                }
            }
        });
        URLClassLoader loader = new URLClassLoader(list.toArray(new URL[list.size()]));
        ServiceLoader<Plugin> pluginLoader = ServiceLoader.load(Plugin.class, loader);
        for(Plugin plugin : pluginLoader) {
            add(plugin);
        }
        startup(env);
    }
    private void add(Plugin plugin) {
        plugins.add(plugin);
        
        Method[] methods = plugin.getClass().getDeclaredMethods();
        for(Method m : methods) {
            Action a = m.getAnnotation(Action.class);
            if(a != null) {
                actions.put(a.value(), new Item(plugin, m));
            }
            View v = m.getAnnotation(View.class);
            if(v != null) {
                views.put(v.value(), new Item(plugin, m));
            }
        }
    }
    private void startup(SystemEnv env) throws IOException, SQLException {
        for(Plugin p : plugins) {
            p.startup(env);
        }
    }
    public void exit() {
        for(Plugin p : plugins) {
            p.exit();
        }
    }
    public boolean action(String name, ActionParams params) throws PluginException {
        try {
            Item item = actions.get(name);
            if(item == null) {
                return false;
            }
            item.method.invoke(item.plugin, params);
            return true;
        } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException ex) {
            throw new PluginException(ex);
        }
    }
    public void view(String name, WikiBridge bridge, AbstractBody body) throws PluginException, IOException {
        try {
            Item item = views.get(name);
            if(item == null) {
                body.text("Unknown plugin:" + name);
                return;
            }
            item.method.invoke(item.plugin, bridge, body);
        } catch (IllegalAccessException | IllegalArgumentException | InvocationTargetException ex) {
            throw new PluginException(ex);
        }
    }
}
