package net.takemix.testok.db;

import java.io.File;
import java.io.FileOutputStream;
import java.io.IOException;
import java.io.InputStream;
import java.io.InputStreamReader;
import java.io.OutputStream;
import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.EnumSet;
import java.util.HashMap;
import java.util.Locale;

import net.takemix.testok.PFConstants;
import net.takemix.testok.Question;

import android.content.ContentValues;
import android.content.Context;
import android.content.SharedPreferences;
import android.database.Cursor;
import android.database.DatabaseUtils;
import android.database.SQLException;
import android.database.sqlite.SQLiteDatabase;
import android.database.sqlite.SQLiteOpenHelper;
import android.util.Log;
import au.com.bytecode.opencsv.CSVReader;

//TODO IDがLongとIntegerでバラバラなので統一する

public class TestokDataBaseHelper extends SQLiteOpenHelper {
    public enum QuestionType {
        QT_LISTENING_PHOTOS,                /** 写真描写問題 */
        QT_LISTENING_QUESTION_RESPONSE,     /** 応答問題 */
        QT_LISTENING_CONVERSATIONS,         /** 会話問題 */
        QT_LISTENING_SHORT_TALKS,           /** 説明文問題 */
        QT_READING_INCOMPLETE_SENTENCES,    /** 短文穴埋め問題 */
        QT_READING_CLOZE_PASSAGES,          /** 長文穴埋め問題 */
        QT_READING_READING_COMPREHENSION,   /** 読解問題 */
        QT_EXTRA_FLASHCARD ,                /** 単語カード */
        QT_END;

        @Override
        public String toString() {
            switch(this) {
            case QT_LISTENING_PHOTOS:
                return "写真描写問題";
            case QT_LISTENING_QUESTION_RESPONSE:
                return "応答問題";
            case QT_LISTENING_CONVERSATIONS:
                return "会話問題";
            case QT_LISTENING_SHORT_TALKS:
                return "説明文問題";
            case QT_READING_INCOMPLETE_SENTENCES:
                return "短文穴埋め問題";
            case QT_READING_CLOZE_PASSAGES:
                return "長文穴埋";
            case QT_READING_READING_COMPREHENSION:
                return "読解問題";
            case QT_EXTRA_FLASHCARD:
                return "単語カード";
            default:
                break;
            }

            return "";
        }

        public Integer toValue() {
            switch(this) {
            case QT_LISTENING_PHOTOS:
                return 0x01;
            case QT_LISTENING_QUESTION_RESPONSE:
                return 0x02;
            case QT_LISTENING_CONVERSATIONS:
                return 0x03;
            case QT_LISTENING_SHORT_TALKS:
                return 0x04;
            case QT_READING_INCOMPLETE_SENTENCES:
                return 0x10;
            case QT_READING_CLOZE_PASSAGES:
                return 0x11;
            case QT_READING_READING_COMPREHENSION:
                return 0x12;
            case QT_EXTRA_FLASHCARD:
                return 0x20;
            case QT_END:
            default:
                break;
            }

            return 0xFF;
        }

        public Integer questions() {
            switch(this) {
            case QT_LISTENING_PHOTOS:
                return 10;
            case QT_LISTENING_QUESTION_RESPONSE:
                return 30;
            case QT_LISTENING_CONVERSATIONS:
                return 30;
            case QT_LISTENING_SHORT_TALKS:
                return 30;
            case QT_READING_INCOMPLETE_SENTENCES:
                return 40;
            case QT_READING_CLOZE_PASSAGES:
                return 12;
            case QT_READING_READING_COMPREHENSION:
                return 48;
            case QT_EXTRA_FLASHCARD:
                return 0;
            default:
                break;
            }
            return 0;
        }

        public Integer minutes() {
            switch(this) {
            case QT_LISTENING_PHOTOS:
                return 6;
            case QT_LISTENING_QUESTION_RESPONSE:
                return 13;
            case QT_LISTENING_CONVERSATIONS:
                return 13;
            case QT_LISTENING_SHORT_TALKS:
                return 13;
            case QT_READING_INCOMPLETE_SENTENCES:
                return 20;
            case QT_READING_CLOZE_PASSAGES:
                return 15;
            case QT_READING_READING_COMPREHENSION:
                return 40;
            case QT_EXTRA_FLASHCARD:
                return 0;
            default:
                break;
            }
            return 0;
        }

        static public QuestionType getElement(String name) {
            QuestionType elm;
            try {
                elm = QuestionType.valueOf(QuestionType.class, name);
            } catch (IllegalArgumentException iae) {
                return QT_END;
            } catch (NullPointerException npe) {
                return QT_END;
            }
            return elm;
        }

        static public QuestionType getElement(Integer value) {
            for (QuestionType elm:EnumSet.allOf(QuestionType.class)) {
                if (elm.toValue() == value) {
                    return elm;
                }
            }
            return QT_END;
        }
    }

    public enum AnswerType {
        AT_WORD_ANY,        // 指定
        AT_WORD_SYNONYM,    // 同意語
        AT_WORD_SAMECLASS,  // 同品詞
        AT_WORD_DIFFCLASS,  // 別品詞
        AT_SENTENCE_ANY,    // センテンス選択
        AT_END;

        public Integer toValue() {
            switch(this) {
            case AT_WORD_ANY:
                return 0x00;
            case AT_WORD_SYNONYM:
                return 0x01;
            case AT_WORD_SAMECLASS:
                return 0x02;
            case AT_WORD_DIFFCLASS:
                return 0x03;
            case AT_SENTENCE_ANY:
                return 0x10;
            case AT_END:
            default:
                break;
            }
            return 0xFF;
        }

        static public AnswerType getElement(String name) {
            AnswerType elm;
            try {
                elm = AnswerType.valueOf(AnswerType.class, name);
            } catch (IllegalArgumentException iae) {
                return AT_END;
            } catch (NullPointerException npe) {
                return AT_END;
            }
            return elm;
        }

        static public AnswerType getElement(Integer value) {
            for(AnswerType elm:EnumSet.allOf(AnswerType.class)) {
                if( elm.toValue() == value) {
                    return elm;
                }
            }
            return AT_END;
        }
    }

    public enum WordTypeRelation {
        WTR_SOMETHING,
        WTR_CLASS,
        WTR_CATEGORY,
        WTR_SYNONYM,
        WTR_WORDGROUP,
        WTR_WORDGROUP_MASTER,
        WTR_END;

        public Integer toValue() {
            switch(this) {
            case WTR_SOMETHING:
                return 0x00;
            case WTR_CLASS:
                return 0x01;
            case WTR_CATEGORY:
                return 0x02;
            case WTR_SYNONYM:
                return 0x03;
            case WTR_WORDGROUP:
                return 0x04;
            case WTR_WORDGROUP_MASTER:
                return 0x05;
            case WTR_END:
            default:
                break;
            }
            return 0xFF;
        }

        static public WordTypeRelation getElement(String name) {
            WordTypeRelation elm;
            try {
                elm = WordTypeRelation.valueOf(WordTypeRelation.class, name);
            } catch (IllegalArgumentException iae) {
                return WTR_END;
            } catch (NullPointerException npe) {
                return WTR_END;
            }
            return elm;
        }

        static public WordTypeRelation getElement(Integer value) {
            for(WordTypeRelation elm:EnumSet.allOf(WordTypeRelation.class)) {
                if( elm.toValue() == value) {
                    return elm;
                }
            }
            return WTR_END;
        }
    }

    public enum WordType {
        WT_CLASS_NOUN,          /** 名詞 */
        WT_CLASS_PRONOUN,       /** 代名詞 */
        WT_CLASS_ADJECTIVE,     /** 形容詞 */
        WT_CLASS_ADVERB,        /** 副詞 */
        WT_CLASS_VERB,          /** 動詞 */
        WT_CLASS_PREPOSITION,   /** 前置詞 */
        WT_CLASS_CONJUNCTION,   /** 接続詞 */
        WT_CLASS_INTERJECTION,  /** 間投詞 */
        WT_CLASS_IDIOM,         /** 熟語 */
        WT_CATEGORY_COUNTRY,            /** 国名 */
        WT_CATEGORY_COUNTRY_G8,         /** 国名(G8) */
        WT_CATEGORY_COUNTRY_G20,        /** 国名(G20) */
        WT_CATEGORY_COUNTRY_EU,         /** 国名(EU) */
        WT_CATEGORY_COUNTRY_ASEAN,      /** 国名(ASEAN) */
        WT_CATEGORY_CITY,               /** 都市名 */
        WT_CATEGORY_CITY_WORLD,         /** 都市名(世界都市) */
        WT_CATEGORY_CITY_USA_1,         /** 都市名(アメリカ１) */
        WT_CATEGORY_CITY_USA_2,         /** 都市名(アメリカ２) */
        WT_CATEGORY_CITY_USA_STATE,     /** 都市名(アメリカ州) */
        WT_CATEGORY_CITY_JPN_PREF,      /** 都市名(日本県) */
        WT_CATEGORY_PERSON,             /** 人名 */
        WT_CATEGORY_PERSON_ENG_LN_1,    /** 人名(英名：姓１) */
        WT_CATEGORY_PERSON_ENG_LN_2,    /** 人名(英名：姓２) */
        WT_CATEGORY_PERSON_ENG_LN_3,    /** 人名(英名：姓３) */
        WT_CATEGORY_PERSON_ENG_FN,      /** 人名(英名：名) */
        WT_CATEGORY_PERSON_ENG_W,       /** 人名(英名：女性) */
        WT_CATEGORY_PERSON_JPN_LN_1,    /** 人名(日名：性１) */
        WT_CATEGORY_PERSON_JPN_LN_2,    /** 人名(日名：性２) */
        WT_CATEGORY_GOVERNMENT,         /** 政府 */
        WT_CATEGORY_COMPANY,            /** 会社名 */
        WT_CATEGORY_LANG,               /** 言語 */
        WT_CATEGORY_MONTH,              /** 月 */
        WT_CATEGORY_DAYOFWEEK,          /** 曜日 */
        WT_SYNONYM_BASE,    /** TODO:類似語ベース*/
        WT_WORDGROUP_BASE,  /** TODO:単語グループベース */
        WT_END;

        @Override
        public String toString() {
            switch(this) {
            case WT_CLASS_NOUN:
                return "名詞";
            case WT_CLASS_PRONOUN:
                return "代名詞";
            case WT_CLASS_ADJECTIVE:
                return "形容詞";
            case WT_CLASS_ADVERB:
                return "副詞";
            case WT_CLASS_VERB:
                return "動詞";
            case WT_CLASS_PREPOSITION:
                return "前置詞";
            case WT_CLASS_CONJUNCTION:
                return "接続詞";
            case WT_CLASS_INTERJECTION:
                return "間投詞";
            case WT_CLASS_IDIOM:
                return "熟語";
            default:
                break;
            }

            return "";
        }

        public Integer toValue() {
            switch(this) {
            case WT_CLASS_NOUN:
                return 0x01;
            case WT_CLASS_PRONOUN:
                return 0x02;
            case WT_CLASS_ADJECTIVE:
                return 0x03;
            case WT_CLASS_ADVERB:
                return 0x04;
            case WT_CLASS_VERB:
                return 0x05;
            case WT_CLASS_PREPOSITION:
                return 0x06;
            case WT_CLASS_CONJUNCTION:
                return 0x07;
            case WT_CLASS_INTERJECTION:
                return 0x08;
            case WT_CLASS_IDIOM:
                return 0x09;
            case WT_CATEGORY_COUNTRY:
                return 0x20;
            case WT_CATEGORY_COUNTRY_G8:
                return 0x21;
            case WT_CATEGORY_COUNTRY_G20:
                return 0x22;
            case WT_CATEGORY_COUNTRY_EU:
                return 0x23;
            case WT_CATEGORY_COUNTRY_ASEAN:
                return 0x24;
            case WT_CATEGORY_CITY:
                return 0x30;
            case WT_CATEGORY_CITY_WORLD:
                return 0x31;
            case WT_CATEGORY_CITY_USA_1:
                return 0x32;
            case WT_CATEGORY_CITY_USA_2:
                return 0x33;
            case WT_CATEGORY_CITY_USA_STATE:
                return 0x34;
            case WT_CATEGORY_CITY_JPN_PREF:
                return 0x35;
            case WT_CATEGORY_PERSON:
                return 0x40;
            case WT_CATEGORY_PERSON_ENG_LN_1:
                return 0x41;
            case WT_CATEGORY_PERSON_ENG_LN_2:
                return 0x42;
            case WT_CATEGORY_PERSON_ENG_LN_3:
                return 0x43;
            case WT_CATEGORY_PERSON_ENG_FN:
                return 0x44;
            case WT_CATEGORY_PERSON_ENG_W:
                return 0x45;
            case WT_CATEGORY_PERSON_JPN_LN_1:
                return 0x46;
            case WT_CATEGORY_PERSON_JPN_LN_2:
                return 0x47;
            case WT_CATEGORY_GOVERNMENT:
                return 0x51;
            case WT_CATEGORY_COMPANY:
                return 0x52;
            case WT_CATEGORY_LANG:
                return 0x53;
            case WT_CATEGORY_MONTH:
                return 0x61;
            case WT_CATEGORY_DAYOFWEEK:
                return 0x62;
            case WT_SYNONYM_BASE:
                return 0x500;
            case WT_WORDGROUP_BASE:
                return 0x10000;
            case WT_END:
            default:
                break;
            }
            return 0xFFFFFFFF;
        }

        public String toSentenceTag() {
            switch(this) {
            case WT_CATEGORY_COUNTRY:            /** 国名 */
                return "OOO";
            case WT_CATEGORY_COUNTRY_G8:         /** 国名(G8) */
                return "AAA";
            case WT_CATEGORY_COUNTRY_G20:        /** 国名(G20) */
                return "BBB";
            case WT_CATEGORY_COUNTRY_EU:         /** 国名(EU) */
                return "PPP";
            case WT_CATEGORY_COUNTRY_ASEAN:      /** 国名(ASEAN) */
                return "QQQ";
            case WT_CATEGORY_CITY:               /** 都市名 */
                return "TTT";
            case WT_CATEGORY_CITY_WORLD:         /** 都市名(世界都市) */
                return "XXX";
            case WT_CATEGORY_CITY_USA_1:         /** 都市名(アメリカ１) */
                return "UUU";
            case WT_CATEGORY_CITY_USA_2:         /** 都市名(アメリカ２) */
                return "VVV";
            case WT_CATEGORY_CITY_USA_STATE:     /** 都市名(アメリカ州) */
                return "SSS";
            case WT_CATEGORY_CITY_JPN_PREF:      /** 都市名(日本県) */
                return "RRR";
            case WT_CATEGORY_PERSON:             /** 人名 */
                return "III";
            case WT_CATEGORY_PERSON_ENG_LN_1:    /** 人名(英名：姓１) */
                return "DDD";
            case WT_CATEGORY_PERSON_ENG_LN_2:    /** 人名(英名：姓２) */
                return "EEE";
            case WT_CATEGORY_PERSON_ENG_LN_3:    /** 人名(英名：姓３) */
                return "FFF";
            case WT_CATEGORY_PERSON_ENG_FN:      /** 人名(英名：名) */
                return "HHH";
            case WT_CATEGORY_PERSON_ENG_W:       /** 人名(英名：女性) */
                return "NNN";
            case WT_CATEGORY_PERSON_JPN_LN_1:    /** 人名(日名：性１) */
                return "JJJ";
            case WT_CATEGORY_PERSON_JPN_LN_2:    /** 人名(日名：性２) */
                return "KKK";
            case WT_CATEGORY_GOVERNMENT:         /** 政府 */
                return "GGG";
            case WT_CATEGORY_COMPANY:            /** 会社名 */
                return "CCC";
            case WT_CATEGORY_LANG:               /** 言語 */
                return "LLL";
            case WT_CATEGORY_MONTH:              /** 月 */
                return "MMM";
            case WT_CATEGORY_DAYOFWEEK:          /** 曜日 */
                return "WWW";
            default:
                break;
            }
            return "";
        }

        static public WordType getElement(String name) {
            WordType elm;
            try {
                elm = WordType.valueOf(WordType.class, name);
            } catch (IllegalArgumentException iae) {
                return WT_END;
            } catch (NullPointerException npe) {
                return WT_END;
            }
            return elm;
        }

        static public WordType getElement(Integer value) {
            for(WordType elm:EnumSet.allOf(WordType.class)) {
                if( elm.toValue() == value) {
                    return elm;
                }
            }
            return WT_END;
        }
    }

    public static final String HISTORY_MAPKEY_SEQUENCEID = "sequence_id";
    public static final String HISTORY_MAPKEY_QUESTIONS = "questions";

    private static final String TAG = "TestokDataBaseHelper";
    private static final String QUESTION_CSV_FILENAME = "question.csv";
    private static final String WORD_CLASS_CSV_FILENAME = "word_class.csv";
    private static final String WORD_CATEGORY_CSV_FILENAME = "word_category.csv";
    private static final String DATE_FORMAT = "yyyy.MM.dd";
    private static final String TIME_FORMAT = "yyyy.MM.dd HH:mm:ss";
    private static final Integer CSV_ANSWER_CHOICE_NUM = 4;

	private SQLiteDatabase db;
	private final Context context;
	private static final String dbName = "database";
    private static final String dbPath = "/data/data/net.takemix.testok/databases/";

    private Boolean dbLock = false;

    private Integer maxLevel;
    private Integer minLevel;
    private AnswerType answerType;
    private QuestionType questionType;
    private WordTypeRelation wordTypeRelation;
    private WordType wordType;

    public TestokDataBaseHelperInterface delegate;

    public Long questionCount;
    public Long sentenceCount;
    public Long wordCount;

	/**
	 * Constructor Takes and keeps a reference of the passed context in
	 * order to access to the application assets and resources.
	 *
	 * @param context
	 */
	public TestokDataBaseHelper(Context context) {
		super(context, dbName, null, 1);
		Log.i(TAG, "TestokDataBaseHelper - IN");

		this.context = context;

        Log.i(TAG, "TestokDataBaseHelper - OUT");
	}

    private synchronized void openDataBase() {
//        synchronized (this.db) {
            if (this.db == null) {
                String path = dbPath + dbName;
                while(this.dbLock) {
                    try {
                        Thread.sleep(1000);
                    } catch (InterruptedException e) {
                        e.printStackTrace();
                    }
                }
                try {
                    this.db = SQLiteDatabase.openDatabase(path, null, SQLiteDatabase.OPEN_READWRITE);
                } catch (SQLException e) {
                    e.printStackTrace();
                    Log.e(TAG, e.toString());
                }
                this.dbLock = true;
            }
//        }
    }

    private synchronized void closeDataBase() {

    }

        @Override
	public void close() {
//        synchronized (this.db) {
            super.close();
/*            if (this.dbLock != null) {
                this.db.close();
                this.db = null;
                this.dbLock = false;
            } */
  //      }
	}

	@Override
	public void onCreate(SQLiteDatabase db) {
        Log.i(TAG, "onCreate - IN");


        Log.i(TAG, "onCreate - OUT");
	}

	@Override
	public void onUpgrade(SQLiteDatabase db, int oldVersion, int newVersion) {
        Log.i(TAG, "onUpgrade - IN");

        Log.i(TAG, "onUpgrade - OUT");
	}

	public void setPreferences() {
        Log.i(TAG, "setPreferences - IN");
	    SharedPreferences pref = this.context.getSharedPreferences("testok_pref", Context.MODE_PRIVATE);

	    this.maxLevel = pref.getInt(PFConstants.getQuestionSettingMaxLevel(), 10);
        this.minLevel = pref.getInt(PFConstants.getQuestionSettingMinLevel(), 1);
	    this.questionType = QuestionType.getElement(pref.getString(PFConstants.getQuestionSettingType(), "QT_READING_INCOMPLETE_SENTENCES"));

        Log.i(TAG, "setPreferences - OUT [" + this.questionType.name() + "]");
	}

    public void copyDataBaseForce() throws IOException {
        Log.i(TAG, "copyDataBaseForce - IN");

        String fileName = dbPath + dbName;
        File file = new File(fileName);
        file.delete();

        this.copyDataBase();

        Log.i(TAG, "copyDataBaseForce - OUT");
    }

    public void copyDataBase() throws IOException {
        Log.i(TAG, "copyDataBase - IN");

        InputStream is = this.context.getAssets().open(dbName);

        String outFileName = dbPath + dbName;

        OutputStream os = new FileOutputStream(outFileName);

        byte[] buffer = new byte[1024];
        int length;
        while ((length = is.read(buffer)) > 0) {
            os.write(buffer, 0, length);
        }

        os.flush();
        os.close();
        is.close();

        Log.i(TAG, "copyDataBase - OUT");
    }

    private void makeQuestionType() {
        Log.i(TAG, "makeQuestionType - IN");

        this.openDataBase();

        for (int i = 0; i < QuestionType.values().length; i++) {
            QuestionType qt = QuestionType.values()[i];

            String sql = "insert into T_QUESTIONTYPE (question_type, alias_string, questions) ";
            sql += "values (" + qt.toValue() + ", \'" + qt.name() + "\', " + qt.questions() + ")";
            try {
                this.db.execSQL(sql);
            } catch (SQLException e) {
                Log.e(TAG, e.getLocalizedMessage());
            }
        }

        Log.i(TAG, "makeQuestionType - OUT");
    }

    private void makeAnswerType() {
        Log.i(TAG, "makeAnswerType - IN");

        this.openDataBase();

        for (int i = 0; i < AnswerType.values().length; i++) {
            AnswerType at = AnswerType.values()[i];

            String sql = "insert into T_ANSWERTYPE (answer_type, alias_string) ";
            sql += "values (" + at.toValue() + ", \'" + at.name() + "\')";
            try {
                this.db.execSQL(sql);
            } catch (SQLException e) {
                Log.e(TAG, e.getLocalizedMessage());
            }
        }

        Log.i(TAG, "makeAnswerType - OUT");
    }

    private void makeWordType() {
        Log.i(TAG, "makeWordType - IN");

        this.openDataBase();

        for (int i = 0; i < WordType.values().length; i++) {
            WordType wt = WordType.values()[i];

            String sql = "insert into T_WORDTYPE (word_type, alias_string) ";
            sql += "values (" + wt.toValue() + ", \'" + wt.name() + "\')";
            try {
                this.db.execSQL(sql);
            } catch (SQLException e) {
                Log.e(TAG, e.getLocalizedMessage());
            }
        }

        Log.i(TAG, "makeWordType - OUT");
    }

    private void makeWordRelationType() {
        Log.i(TAG, "makeWordRelationType - IN");

        this.openDataBase();

        for (int i = 0; i < WordTypeRelation.values().length; i++) {
            WordTypeRelation wr = WordTypeRelation.values()[i];

            String sql = "insert into T_RELATIONSHIP (relationship, alias_string) ";
            sql += "values (" + wr.toValue() + ", \'" + wr.name() + "\')";
            try {
                this.db.execSQL(sql);
            } catch (SQLException e) {
                Log.e(TAG, e.getLocalizedMessage());
            }
        }

        Log.i(TAG, "makeWordRelationType - OUT");
    }

    private void convertCsv2DataBase(String name) throws IOException {
        Log.i(TAG, "convertCsv2DataBase - IN [" + name + "]");

        InputStream is = this.context.getAssets().open(name);

        @SuppressWarnings("resource")
        CSVReader csv = new CSVReader(new InputStreamReader(is, "UTF-8"));
        @SuppressWarnings("unused")
        String[] columns = csv.readNext();

        String[] line;
        if (name.equals(QUESTION_CSV_FILENAME)) {
            DatabaseUtils.InsertHelper ihQuestion = new DatabaseUtils.InsertHelper(this.db, "T_QUESTION");
            DatabaseUtils.InsertHelper ihQuestionAnswerAssign = new DatabaseUtils.InsertHelper(this.db, "T_QUESTION_ANSWER_ASSIGN");
            DatabaseUtils.InsertHelper ihAnswer = new DatabaseUtils.InsertHelper(this.db, "T_ANSWER");
            DatabaseUtils.InsertHelper ihAnswerSentenceAssign = new DatabaseUtils.InsertHelper(this.db, "T_ANSWER_SENTENCE_ASSIGN");
            DatabaseUtils.InsertHelper ihAnswerWordAssign = new DatabaseUtils.InsertHelper(this.db, "T_ANSWER_WORD_ASSIGN");
            DatabaseUtils.InsertHelper ihSentence = new DatabaseUtils.InsertHelper(this.db, "T_SENTENCE");

            ContentValues values = new ContentValues();
            String qTypeIndex = "question_type";
            String qTextIndex = "text";
            String qHintIndex = "hint";
            String qLvIndex = "lv";

            String qaaQidIndex = "question_id";
            String qaaAidIndex = "answer_id";

            String aTypeIndex = "answer_type";
            String aGroupIndex = "answer_group";

            String asaAidIndex = "answer_id";
            String asaRightIndex = "right_id";
            String asaMisstake1Index = "misstake_id_1";
            String asaMisstake2Index = "misstake_id_2";
            String asaMisstake3Index = "misstake_id_3";

            String awaAidIndex = "answer_id";
            String awaRightIndex = "right_id";
            String awaMisstake1Index = "misstake_id_1";
            String awaMisstake2Index = "misstake_id_2";
            String awaMisstake3Index = "misstake_id_3";

            String sSentenceIndex = "sentence";

            long qid_aid = this.questionCount;
            while ((line = csv.readNext()) != null) {
                qid_aid++;
                Log.d(TAG, "qid_aid:" + qid_aid);

                QuestionType qt = QuestionType.getElement(line[0]);
                AnswerType at = AnswerType.getElement(line[8]);
                String subType = line[9];
                Log.d(TAG, "subType:" + subType);
                Log.d(TAG, "word:" + line[4]);

                String text = line[1];
                text = text.replace(line[4] + " ", "||" + line[4] + "|| ");
                for (WordType elm:EnumSet.allOf(WordType.class)) {
                    if (!elm.toSentenceTag().equals("")) {
                        text = text.replaceAll(elm.toSentenceTag(), "\\{\\{" + elm.name() + "\\}\\}");
                    }
                }

                values.clear();
                values.put(qTypeIndex, qt.toValue());
                values.put(qTextIndex, text);
                values.put(qHintIndex, line[2]);
                values.put(qLvIndex, Integer.valueOf(line[3]));
                ihQuestion.insert(values);

                values.clear();
                values.put(qaaQidIndex, qid_aid);
                values.put(qaaAidIndex, qid_aid);
                ihQuestionAnswerAssign.insert(values);

                values.clear();
                values.put(aTypeIndex, at.toValue());
                values.put(aGroupIndex, 0);
                ihAnswer.insert(values);

                if (at == AnswerType.AT_SENTENCE_ANY) {
                    /** Sentence question type */
                    for (int i = 0; i < CSV_ANSWER_CHOICE_NUM; i++) {
                        values.clear();
                        values.put(sSentenceIndex, line[4 + i]);
                        ihSentence.insert(values);
                    }

                    values.clear();
                    values.put(asaAidIndex, qid_aid);
                    values.put(asaRightIndex, ++this.sentenceCount);
                    values.put(asaMisstake1Index, ++this.sentenceCount);
                    values.put(asaMisstake2Index, ++this.sentenceCount);
                    values.put(asaMisstake3Index, ++this.sentenceCount);
                    ihAnswerSentenceAssign.insert(values);
                } else {
                    /** Word question type */
//                    Log.d(TAG, "word:" + line[4]);
//                    Log.d(TAG, "at:" + at.toString());
///*                    switch (at) {
//                    case AT_WORD_ANY:        // 指定
//                    {
//  */                       Long wid[] = new Long[CSV_ANSWER_CHOICE_NUM];
//                        for (int i = 0; i < CSV_ANSWER_CHOICE_NUM; i++) {
//                            wid[i] = this.getWordId(line[4 + i]);
//                            Log.d(TAG, "word[" + i + "]:" + line[4 + i]);
//                            Log.d(TAG, "wid[" + i + "]:" + wid[i]);
//                        }
//                        values.clear();
//                        values.put(awaAidIndex, qid_aid);
////                        values.put(awaRightIndex, wid[0]);
////                        values.put(awaMisstake1Index, wid[1]);
////                        values.put(awaMisstake2Index, wid[2]);
////                        values.put(awaMisstake3Index, wid[3]);
//                        ihAnswerWordAssign.insert(values);
//   /*                     break;
//                    }
//                    case AT_WORD_SYNONYM:    // 同意語
//                    case AT_WORD_SAMECLASS:  // 同品詞
//                    case AT_WORD_DIFFCLASS:  // 別品詞
//                    {
//                        Long wid = this.getWordId(line[4]);
//                        values.clear();
//                        values.put(awaAidIndex, qid_aid);
//                        values.put(awaRightIndex, wid);
//                        ihAnswerWordAssign.insert(values);
//                        break;
//                    }
//                    default:
//                        break;
//                    } */
//                    Log.d(TAG, "test1");
                }
                Log.d(TAG, "test2");
            }

            ihQuestion.close();
            ihQuestionAnswerAssign.close();
            ihAnswer.close();
            ihAnswerSentenceAssign.close();
            ihAnswerWordAssign.close();
            ihSentence.close();

            Log.d(TAG, "test3");
            this.questionCount = qid_aid;

        } else if (name.equals(WORD_CLASS_CSV_FILENAME) || name.equals(WORD_CATEGORY_CSV_FILENAME)) {
            DatabaseUtils.InsertHelper ihWord = new DatabaseUtils.InsertHelper(this.db, "T_WORD");
            DatabaseUtils.InsertHelper ihWordTypeRelation = new DatabaseUtils.InsertHelper(this.db, "T_WORDTYPE_RELATION");

            int wWordIndex = ihWord.getColumnIndex("word");
            int wTransIndex = ihWord.getColumnIndex("translate");

            int wrWidIndex = ihWordTypeRelation.getColumnIndex("word_id");
            int wrTypeIndex = ihWordTypeRelation.getColumnIndex("word_type");
            int wrRelationIndex = ihWordTypeRelation.getColumnIndex("relationship");

            WordTypeRelation wtr;
            if (name.equals(WORD_CLASS_CSV_FILENAME)) {
                wtr = WordTypeRelation.WTR_CLASS;
            } else {
                wtr = WordTypeRelation.WTR_CATEGORY;
            }

            long wid = this.wordCount;
            while ((line = csv.readNext()) != null) {
                wid++;

                ihWord.prepareForInsert();
                ihWord.bind(wWordIndex, line[0]);
                ihWord.bind(wTransIndex, line[1]);
                ihWord.execute();

                WordType wt = WordType.getElement(line[2]);

                ihWordTypeRelation.prepareForInsert();
                ihWordTypeRelation.bind(wrWidIndex, wid);
                ihWordTypeRelation.bind(wrTypeIndex, wt.toValue());
                ihWordTypeRelation.bind(wrRelationIndex, wtr.toValue());
                ihWordTypeRelation.execute();
            }
            this.wordCount = wid;

            ihWord.close();
            ihWordTypeRelation.close();
        }

        is.close();

        Log.i(TAG, "convertCsv2DataBase - OUT");
    }

    private String parseQuestionText(String text) {
        String destText = null;

        String[] strArray = text.split("\\|\\|");

        if (strArray.length == 3) {
            destText = strArray[0];
            destText += "_______";
            destText += strArray[2];
        } else {
            destText = text;
        }

        for (WordType elm:EnumSet.allOf(WordType.class)) {
            if (!elm.toSentenceTag().equals("")) {
                destText = destText.replaceAll("\\{\\{" + elm.name() + "\\}\\}", this.getWordByCategory(elm));
            }
        }

        return destText;
    }

    public void importDataBase() {
        Log.i(TAG, "importDataBase - IN");

        this.questionCount = 0L;
        this.wordCount = 0L;
        this.sentenceCount = 0L;

        Thread t = new Thread() {
            public void run() {
                makeQuestionType();
                makeAnswerType();
                makeWordType();
                makeWordRelationType();

                try {
//                    convertCsv2DataBase(WORD_CLASS_CSV_FILENAME);
//                    convertCsv2DataBase(WORD_CATEGORY_CSV_FILENAME);
                    convertCsv2DataBase(QUESTION_CSV_FILENAME);
                } catch (IOException ioe) {
                    Log.e(TAG, ioe.getLocalizedMessage());
                }

                delegate.importDataBaseComplete();
            }
        };

        t.start();

        Log.i(TAG, "importDataBase - OUT");
    }

    public Long getWordId(String word) {
        Long wid = 0L;

        this.openDataBase();

        String sql = "select word_id from T_WORD";
        sql += " where word = \'" + word + "\' limit 1";
        Cursor c = this.db.rawQuery(sql, null);
        if (c.getCount() <= 0) {
            Log.w(TAG, "getWordId - NOT FOUND; create new word");
            c.close();

            sql = "insert into T_WORD (word, translate) values (\'" + word + "\', \'\')";
            this.db.execSQL(sql);
            sql = "select ROWID from T_WORD where ROWID = last_insert_rowid()";
            c = this.db.rawQuery(sql, null);
            c.moveToFirst();
            wid = c.getLong(0);

            c.close();
            this.closeDataBase();
            return wid;
        }

        c.moveToFirst();
        wid = c.getLong(0);

        c.close();
        this.closeDataBase();

        return wid;
    }

    public Long getWordIdByClass(WordType wordType) {
        Long wid = 0L;

        this.openDataBase();

        String sql = "select word_id from V_WORD";
        sql += " where type_relationship = " + WordTypeRelation.WTR_CLASS.toValue() + " and type = " + wordType.toValue() + " order by RANDOM() limit 1";
        Cursor c = this.db.rawQuery(sql, null);
        if (c.getCount() <= 0) {
            Log.w(TAG, "getWordIdByClass - NOT FOUND");
            c.close();
            this.closeDataBase();
            return wid;
        }

        c.moveToFirst();
        wid = c.getLong(0);

        c.close();
        this.closeDataBase();

        return wid;
    }

    public String getWordByCategory(WordType wordType) {
        String val = "";

        this.openDataBase();

        String sql = "select word from V_WORD";
        sql += " where type_relationship = " + WordTypeRelation.WTR_CATEGORY.toValue() + " and type = " + wordType.toValue() + " order by RANDOM() limit 1";
        Cursor c = this.db.rawQuery(sql, null);
        if (c.getCount() <= 0) {
            c.close();
            this.closeDataBase();
            return val;
        }

        c.moveToFirst();
        val = c.getString(0);

        c.close();
        this.closeDataBase();

        return val;
    }

    public void insertDefaultUserProfile() {
        Log.i(TAG, "insertDefaultUserProfile - IN");

        this.openDataBase();

        String sql = "select * from T_USER_PROFILE";
        Cursor c = this.db.rawQuery(sql, null);
        if (c.getCount() == 0) {
            sql = "insert into T_USER_PROFILE (name, last_score, mock_score, study_time, exam_date) values (\'no name\', 0, 0, 0, \'\')";
            try {
                this.db.execSQL(sql);
            } catch (SQLException e) {
                Log.e(TAG, e.getLocalizedMessage());
            }
        } else {
            // nothing
        }

        c.close();
        this.closeDataBase();

        Log.i(TAG, "insertDefaultUserProfile - OUT");
    }

    public void insertUserProfile(String name, Integer score, Integer studyTime, Date examDate) {
        Log.i(TAG, "insertUserProfile - IN");

        this.openDataBase();

        String sql = "select * from T_USER_PROFILE";
        Cursor c = this.db.rawQuery(sql, null);
        if (c.getCount() == 0) {
            sql = "insert into T_USER_PROFILE (name, last_score, mock_score, study_time, exam_date) values (";
            sql += "\'" + name + "\', ";
            sql += "" + score + ", ";
            sql += "" + 0 + ", ";
            sql += "" + studyTime + ", ";
            sql += "\'" + date2dateString(examDate) + "\')";
            try {
                this.db.execSQL(sql);
            } catch (SQLException e) {
                Log.e(TAG, e.getLocalizedMessage());
            }
        } else {
            // nothing
        }

        c.close();
        this.closeDataBase();

        Log.i(TAG, "insertUserProfile - OUT");
    }

    public String userName() {
        Log.i(TAG, "userName - IN");

        String val;

        this.openDataBase();

        String sql = "select name from T_USER_PROFILE";
        Cursor c = this.db.rawQuery(sql, null);
        if (c.getCount() <= 0) {
            Log.w(TAG, "userName - NOT FOUND");
            c.close();
            this.closeDataBase();
            return "";
        }

        c.moveToFirst();
        val = c.getString(0);

        c.close();
        this.closeDataBase();

        Log.i(TAG, "userName - OUT [" + val + "]");

        return val;
    }

    public void setUserName(String val) {
        Log.i(TAG, "setUserName - IN [" + val + "]");

        this.openDataBase();

        String sql = "update T_USER_PROFILE set name = \'" + val + "\'";
        try {
            this.db.execSQL(sql);
        } catch (SQLException e) {
            Log.e(TAG, e.getLocalizedMessage());
        }

        this.closeDataBase();

        Log.i(TAG, "setUserName - OUT");
    }

    public Integer lastScore() {
        Log.i(TAG, "lastScore - IN");

        Integer val;

        this.openDataBase();

        String sql = "select last_score from T_USER_PROFILE";
        Cursor c = this.db.rawQuery(sql, null);
        if (c.getCount() <= 0) {
            Log.w(TAG, "lastScore - NOT FOUND");
            c.close();
            this.closeDataBase();
            return 0;
        }

        c.moveToFirst();
        val = c.getInt(0);

        c.close();
        this.closeDataBase();

        Log.i(TAG, "lastScore - OUT [" + val + "]");

        return val;
    }

    public void setLastScore(Integer val) {
        Log.i(TAG, "setLastScore - IN [" + val + "]");

        this.openDataBase();

        String sql = "update T_USER_PROFILE set last_score = \'" + val.toString() + "\'";
        try {
            this.db.execSQL(sql);
        } catch (SQLException e) {
            Log.e(TAG, e.getLocalizedMessage());
        }

        this.closeDataBase();

        Log.i(TAG, "setLastScore - OUT");
    }

    public Integer mockScore() {
        Log.i(TAG, "mockScore - IN");

        Integer val;

        this.openDataBase();

        String sql = "select mock_score from T_USER_PROFILE";
        Cursor c = this.db.rawQuery(sql, null);
        if (c.getCount() <= 0) {
            Log.w(TAG, "mockScore - NOT FOUND");
            c.close();
            this.closeDataBase();
            return 0;
        }

        c.moveToFirst();
        val = c.getInt(0);

        c.close();
        this.closeDataBase();

        Log.i(TAG, "mockScore - OUT [" + val + "]");

        return val;
    }

    public void setMockScore(Integer val) {
        Log.i(TAG, "setMockScore - IN [" + val + "]");

        this.openDataBase();

        String sql = "update T_USER_PROFILE set mock_score = \'" + val.toString() + "\'";
        try {
            this.db.execSQL(sql);
        } catch (SQLException e) {
            Log.e(TAG, e.getLocalizedMessage());
        }

        this.closeDataBase();

        Log.i(TAG, "setMockScore - OUT");
    }

    public Date examDate() {
        Log.i(TAG, "examDate - IN");

        String val;

        this.openDataBase();

        String sql = "select exam_date from T_USER_PROFILE";
        Cursor c = this.db.rawQuery(sql, null);
        if (c.getCount() <= 0) {
            Log.w(TAG, "examDate - NOT FOUND");
            c.close();
            this.closeDataBase();
            return new Date();
        }

        c.moveToFirst();
        val = c.getString(0);

        c.close();
        this.closeDataBase();

        Log.i(TAG, "examDate - OUT [" + val + "]");

        return dateString2date(val);
    }

    public void setExamDate(Date val) {
        Log.i(TAG, "setExamDate - IN [" + date2dateString(val) + "]");

        this.openDataBase();

        String sql = "update T_USER_PROFILE set exam_date = \'" + date2dateString(val) + "\'";
        try {
            this.db.execSQL(sql);
        } catch (SQLException e) {
            Log.e(TAG, e.getLocalizedMessage());
        }

        this.closeDataBase();

        Log.i(TAG, "setExamDate - OUT");
    }

    public Integer studyTime() {
        Log.i(TAG, "studyTime - IN");

        String val;

        this.openDataBase();

        String sql = "select study_time from T_USER_PROFILE";
        Cursor c = this.db.rawQuery(sql, null);
        if (c.getCount() <= 0) {
            Log.w(TAG, "studyTime - NOT FOUND");
            c.close();
            this.closeDataBase();
            return 0;
        }

        c.moveToFirst();
        val = c.getString(0);

        c.close();
        this.closeDataBase();

        Log.i(TAG, "studyTime - OUT [" + val + "]");

        return Integer.valueOf(val);
    }

    public void setStudyTime(Integer val) {
        Log.i(TAG, "setStudyTime - IN [" + val + "]");

        this.openDataBase();

        String sql = "update T_USER_PROFILE set study_time = \'" + val.toString() + "\'";
        try {
            this.db.execSQL(sql);
        } catch (SQLException e) {
            Log.e(TAG, e.getLocalizedMessage());
        }

        this.closeDataBase();

        Log.i(TAG, "setStudyTime - OUT");
    }

    public Question getQuestion() {
        Log.i(TAG, "getQuestion - IN");

        Question question = new Question();

        this.openDataBase();

        String sql = "select question_id, text, hint, lv, answer_id, answer_type, question_type from V_QUESTION";
        sql += " where question_type = " + this.questionType.toValue() + " and lv >= " + this.minLevel + " and lv <= " + this.maxLevel + " order by RANDOM() limit 1";
        Cursor c = this.db.rawQuery(sql, null);
        if (c.getCount() <= 0) {
            Log.w(TAG, "getQuestion - NOT FOUND");
            c.close();
            this.closeDataBase();
            return null;
        }

        c.moveToFirst();
        question.qid = c.getInt(0);
        question.questionSentence = this.parseQuestionText(c.getString(1));
        question.hint = c.getString(2);
        question.level = c.getInt(3);

        Integer answerId = c.getInt(4);
        AnswerType at = AnswerType.getElement(c.getInt(5));

        c.close();
        this.closeDataBase();

        this.getAnswerWithQuestion(question, answerId, at);

        Log.i(TAG, "getQuestion - OUT [" + question.qid + "]");

        return question;
    }

    private void getAnswerWithQuestion(Question question, Integer answerId, AnswerType at) {
        Log.i(TAG, "getAnswerWithQuestion - IN [" + answerId + "]");

        this.openDataBase();

        String getIdSql;
        String getTextSql;
        switch (at) {
        case AT_SENTENCE_ANY:
            getIdSql = "select right_id, misstake_id_1, misstake_id_2, misstake_id_3 from T_ANSWER_SENTENCE_ASSIGN where answer_id = " + answerId + "";
            getTextSql = "select sentence from T_SENTENCE where sentence_id = ";
            break;
        case AT_WORD_ANY:
        case AT_WORD_SYNONYM:
        case AT_WORD_DIFFCLASS:
            getIdSql = "select right_id, misstake_id_1, misstake_id_2, misstake_id_3 from T_ANSWER_WORD_ASSIGN where answer_id = " + answerId + "";
            getTextSql = "select sentence from T_WORD where sentence_id = ";
            break;
        default:
            return;
        }

        int[] choiceId = new int[4];
        Cursor c = this.db.rawQuery(getIdSql, null);
        c.moveToFirst();
        choiceId[0] = c.getInt(0);
        choiceId[1] = c.getInt(1);
        choiceId[2] = c.getInt(2);
        choiceId[3] = c.getInt(3);
        c.close();

        for (int i = 0; i < Question.MAX_ANSWER; i++) {
            String sql = getTextSql + choiceId[i];
            c = this.db.rawQuery(sql, null);
            if (c.getCount() <= 0) {
                Log.w(TAG, "getAnswer - NOT FOUND");
                question.answer[i] = "";
            } else {
                c.moveToFirst();
                question.answer[i] = c.getString(0);
            }
            c.close();
        }
        question.rightAnswerIndex = 0;
        question.random();

        this.closeDataBase();

        Log.i(TAG, "getAnswerWithQuestion - OUT [" + question.answer[question.rightAnswerIndex] + "]");
    }

    public void setMark(Integer questionId, Boolean mark) {
        Log.i(TAG, "setMark - IN [" + questionId + ", " + mark + "]");

        this.openDataBase();

        //QuestionTableのアップデートを考えるとMarkは別テーブルに保持したほうが良い？
        int mark_val = 0;
        if (mark) {
            mark_val = 1;
        }

        String sql = "update T_QUESTION set mark = " + mark_val + " where question_id = " + questionId;
        try {
            this.db.execSQL(sql);
        } catch (SQLException e) {
            Log.e(TAG, e.getLocalizedMessage());
        }

        this.closeDataBase();

        Log.i(TAG, "setMark - OUT");
    }

    public Boolean mark(Integer questionId) {
        Log.i(TAG, "mark - IN [" + questionId + "]");

        Boolean mark_val = false;
        int val;

        this.openDataBase();

        String sql = "select mark from T_QUESTION where question_id = " + questionId;
        Cursor c = this.db.rawQuery(sql, null);
        if (c.getCount() <= 0) {
            Log.w(TAG, "mark - NOT FOUND");
            c.close();
            this.closeDataBase();
            return mark_val;
        }

        c.moveToFirst();
        val = c.getInt(0);
        if (val != 0) {
            mark_val = true;
        }

        c.close();
        this.closeDataBase();

        Log.i(TAG, "mark - OUT [" + mark_val + "]");

        return mark_val;
    }

    public void setHistory(Long sequenceId, Question question) {
        Log.i(TAG, "setHistory - IN [" + sequenceId + ", " + question.qid + "]");

        this.openDataBase();

        String sql = "select * from T_HISTORY where sequence_id = " + sequenceId + " and question_id = " + question.qid;
        Cursor c = this.db.rawQuery(sql, null);

        String date = date2timeString(new Date());
        Integer match;
        if (question.selectedIndex == -1) {
            match = -1;
        } else {
            match = (question.rightAnswerIndex == question.selectedIndex) ? 1 : 0;
        }
        if (c.getCount() > 0) {
            Log.i(TAG, "setHistory - UPDATE");
            sql = "update T_HISTORY set selected_num = " + question.selectedIndex + ", match = " + match + ", date = '" + date + "', time = " + question.time;
            sql += " where sequence_id = " + sequenceId + " and question_id = " + question.qid;
        } else  {
            Log.i(TAG, "setHistory - INSERT");
            sql = "insert into T_HISTORY (sequence_id, question_id, date, answer_group, selected_num, right_num, question_text, choice_text_1, choice_text_2, choice_text_3, choice_text_4, match, time)";
            sql += " values (" + sequenceId + ", " + question.qid + ", '" + date + "', " + question.answerGroup + ", " + question.selectedIndex + ", " + question.rightAnswerIndex;
            String question_text = question.questionSentence.replace("'", "''");
            sql += ", '" + question_text + "'";
            for (int i = 0; i < Question.MAX_ANSWER; i++) {
                sql += ", '" + question.answer[i] + "'";
            }
            sql +=  ", " + match + ", " + question.time + ")";
        }
        try {
            this.db.execSQL(sql);
        } catch (SQLException e) {
            Log.e(TAG, e.getLocalizedMessage());
        }
        c.close();

        this.closeDataBase();

        Log.i(TAG, "setHistory - OUT");
    }

    public ArrayList<HashMap<String, Object>> getAllHistory() {
        Log.i(TAG, "getAllHistory - IN");

        ArrayList<HashMap<String, Object>> list = new ArrayList<HashMap<String, Object>>();
        ArrayList<Long> seqArray = new ArrayList<Long>();

        this.openDataBase();

        String sql = "select distinct sequence_id from T_HISTORY order by sequence_id desc";
        Cursor c = this.db.rawQuery(sql, null);
        if (c.getCount() <= 0) {
            Log.w(TAG, "getAllHistory - NOT FOUND");
            c.close();
            this.closeDataBase();
            return null;
        }

        c.moveToFirst();
        do {
            seqArray.add(c.getLong(0));
        } while(c.moveToNext());
        c.close();
        this.closeDataBase();

        HashMap<String, Object> map;

        for (Long seqId : seqArray) {
            map = new HashMap<String, Object>();
            map.put(HISTORY_MAPKEY_SEQUENCEID, seqId);
            map.put(HISTORY_MAPKEY_QUESTIONS, getHitory(seqId));
            list.add(map);
        }

        Log.i(TAG, "getAllHistory - OUT [" + list.size() + "]");

        return list;
    }

    public ArrayList<Question> getHitory(Long sequenceId) {
        Log.i(TAG, "getHitory - IN [" + sequenceId + "]");

        ArrayList<Question> questionArray = new ArrayList<Question>();

        this.openDataBase();

        String sql = "select question_id from T_HISTORY where sequence_id = " + sequenceId;
        Cursor c = this.db.rawQuery(sql, null);
        if (c.getCount() <= 0) {
            Log.w(TAG, "getHitory - NOT FOUND");
            c.close();
            this.closeDataBase();
            return questionArray;
        }

        c.moveToFirst();
        do {
            Question q = new Question();
            q.qid = c.getInt(0);
            questionArray.add(q);
        } while(c.moveToNext());

        c.close();
        this.closeDataBase();

        for (Question q : questionArray) {
            this.getHitory(sequenceId, q);
        }

        Log.i(TAG, "getHitory - OUT [" + questionArray.size() + "]");
        return questionArray;
    }

    public void getHitory(Long sequenceId, Question question) {
        Log.i(TAG, "getHitory - IN [" + sequenceId + ", " + question.qid + "]");

        this.openDataBase();

        String sql = "select question_text, date, answer_group, selected_num, right_num, choice_text_1, choice_text_2, choice_text_3, choice_text_4, time from T_HISTORY";
        sql += " where sequence_id = " + sequenceId + " and question_id = " + question.qid;
        Cursor c = this.db.rawQuery(sql, null);
        if (c.getCount() <= 0) {
            Log.w(TAG, "getHitory - NOT FOUND");
            c.close();
            this.closeDataBase();
            return;
        }

        int columnIdx = 0;
        c.moveToFirst();
        question.questionSentence = c.getString(columnIdx++);
        question.date = timeString2date(c.getString(columnIdx++));
        question.answerGroup = c.getInt(columnIdx++);
        question.selectedIndex = c.getInt(columnIdx++);
        question.rightAnswerIndex = c.getInt(columnIdx++);
        for (int i = 0; i < Question.MAX_ANSWER; i++) {
            question.answer[i] = c.getString(columnIdx++);
        }
        question.time = c.getLong(columnIdx++);

        c.close();
        this.closeDataBase();

        Log.i(TAG, "getHitory - OUT");
    }

    public Integer getQuestionCountFromHistory(Integer questionId) {
        Log.i(TAG, "getQuestionCountFromHistory - IN [" + questionId + "]");

        Integer questionCount = 0;

        this.openDataBase();

        String sql = "select question_id from T_HISTORY where question_id = " + questionId;
        Cursor c = this.db.rawQuery(sql, null);
        questionCount = c.getCount();

        c.close();
        this.closeDataBase();

        Log.i(TAG, "getQuestionCountFromHistory - OUT [" + questionCount + "]");

        return questionCount;
    }

    public Integer getMatchCountFromHistory(Integer questionId, Boolean match) {
        Log.i(TAG, "getMatchCountFromHistory - IN [" + questionId + ", " + match + "]");

        Integer matchCount = 0;

        this.openDataBase();

        int matchInt = match ? 1 : 0;
        String sql = "select match from T_HISTORY where question_id = " + questionId + " and match = " + matchInt;
        Cursor c = this.db.rawQuery(sql, null);
        matchCount = c.getCount();

        c.close();
        this.closeDataBase();

        Log.i(TAG, "getMatchCountFromHistory - OUT [" + matchCount + "]");

        return matchCount;
    }

    public static String date2dateString(Date date) {
        if (date == null) {
            return "";
        }
        SimpleDateFormat sdf = new SimpleDateFormat(DATE_FORMAT, Locale.JAPAN);
        return sdf.format(date);
    }

    public static Date dateString2date(String value) {
        if (value.equals("")) {
            return null;
        }
        SimpleDateFormat sdf = new SimpleDateFormat(DATE_FORMAT, Locale.JAPAN);
        try {
            return sdf.parse(value);
        } catch (ParseException e) {
            return null;
        }
    }

    public static String date2timeString(Date date) {
        if (date == null) {
            return "";
        }
        SimpleDateFormat sdf = new SimpleDateFormat(TIME_FORMAT, Locale.JAPAN);
        return sdf.format(date);
    }

    public static Date timeString2date(String value) {
        if (value.equals("")) {
            return null;
        }
        SimpleDateFormat sdf = new SimpleDateFormat(TIME_FORMAT, Locale.JAPAN);
        try {
            return sdf.parse(value);
        } catch (ParseException e) {
            return null;
        }
    }
}
