// ==============================================================
//                ORBITER MODULE: Quadcopter
//                  Part of the ORBITER SDK
//          Copyright (C) 2001-2019 Martin Schweiger
//                   All rights reserved
//
// QuadcopterLua.cpp
// Implementation of Quadcopter-specific Lua interface methods
// ==============================================================

#include "QuadcopterLua.h"
#include "Quadcopter.h"
#include "PropulsionSubsys.h"

int LuaInterface::InitInterpreter(void *context)
{
	lua_State *L = (lua_State*)context;

	// perform any initialisations here

	return 0;
}

int LuaInterface::InitInstance(void *context)
{
	lua_State *L = (lua_State*)context;

	// check if interpreter has DG table loaded already
	luaL_getmetatable(L, "VESSEL.QC");

	if (lua_isnil(L, -1)) { // register new functions
		lua_pop(L, 1);
		static const struct luaL_reg qcLib[] = {
			{ "set_directmode", setDirectMode},
			{ "set_autoheading", setAutoHeading},
			{ "set_heading", setHeading},
			{ "set_course", setCourse},
			{ "set_hspd", setHspd},
			{ "set_vspd", setVspd},
			{ "set_alt", setAlt},
			{ NULL, NULL }
		};

		// create metatable for vessel userdata
		luaL_newmetatable(L, "QC.vtable");

		// create a table for the overloaded methods
		luaL_openlib(L, "QC.method", qcLib, 0);

		// create metatable for accessing inherited methods from VESSEL
		luaL_newmetatable(L, "QC.base");
		lua_pushstring(L, "__index");
		luaL_getmetatable(L, "VESSEL.vtable");
		lua_settable(L, -3);

		// set DG.base as metatable for DG.method
		lua_setmetatable(L, -2);

		// point vessel userdata to DG.method
		lua_pushstring(L, "__index");
		lua_pushvalue(L, -2); // push DG.method
		lua_settable(L, -4);

		// pop DG.method from the stack
		lua_pop(L, 1);
	}

	lua_setmetatable(L, -2);

	return 0;
}

Quadcopter *LuaInterface::lua_toQC(lua_State *L, int idx)
{
	VESSEL **pv = (VESSEL**)lua_touserdata(L, idx);
	Quadcopter *qc = (Quadcopter*)*pv;
	return qc;
}

int LuaInterface::setDirectMode(lua_State *L)
{
	Quadcopter *qc = lua_toQC(L, 1);
	int action = lua_tointeger(L, 2);
	qc->ssysPropulsion()->setDirectMode(action);

	return 0;
}

int LuaInterface::setAutoHeading(lua_State *L)
{
	Quadcopter *qc = lua_toQC(L, 1);
	int action = lua_tointeger(L, 2);
	qc->ssysPropulsion()->setAutoHeading(action);

	return 0;
}

int LuaInterface::setHeading(lua_State *L)
{
	Quadcopter *qc = lua_toQC(L, 1);
	double hdg = lua_tonumber(L, 2) * RAD;
	qc->ssysPropulsion()->setHeadingCmd(hdg);

	return 0;
}

int LuaInterface::setCourse(lua_State *L)
{
	Quadcopter *qc = lua_toQC(L, 1);
	if (lua_isnil(L, 2)) {
		qc->ssysPropulsion()->unsetCourseCmd();
	}
	else {
		double crs = lua_tonumber(L, 2) * RAD;
		double spd = lua_tonumber(L, 3);
		qc->ssysPropulsion()->setCourseCmd(crs, spd);
	}
	return 0;
}

int LuaInterface::setHspd(lua_State *L)
{
	Quadcopter *qc = lua_toQC(L, 1);
	if (lua_isnil(L, 2)) {
		qc->ssysPropulsion()->unsetHspdCmd();
	}
	else {
		double hspd = lua_tonumber(L, 2);
		qc->ssysPropulsion()->setHspdCmd(hspd);
	}
	return 0;
}

int LuaInterface::setVspd(lua_State *L)
{
	Quadcopter *qc = lua_toQC(L, 1);
	if (lua_isnil(L, 2)) {
		qc->ssysPropulsion()->unsetVspdCmd();
	}
	else {
		double vspd = lua_tonumber(L, 2);
		qc->ssysPropulsion()->setVspdCmd(vspd);
	}
	return 0;
}

int LuaInterface::setAlt(lua_State *L)
{
	Quadcopter *qc = lua_toQC(L, 1);
	if (lua_isnil(L, 2)) {
		qc->ssysPropulsion()->unsetAltCmd();
	}
	else {
		double alt = lua_tonumber(L, 2);
		qc->ssysPropulsion()->setAltCmd(alt);
	}
	return 0;
}