# -*- coding: utf-8 -*-
#
#  Copyright (C) 2001, 2002 by Tamito KAJIYAMA
#  Copyright (C) 2002, 2003 by MATSUMURA Namihiko <nie@counterghost.net>
#  Copyright (C) 2002-2013 by Shyouzou Sugitani <shy@users.sourceforge.jp>
#  Copyright (C) 2003 by Shun-ichi TAHARA <jado@flowernet.gr.jp>
#
#  This program is free software; you can redistribute it and/or modify it
#  under the terms of the GNU General Public License (version 2) as
#  published by the Free Software Foundation.  It is distributed in the
#  hope that it will be useful, but WITHOUT ANY WARRANTY; without even the
#  implied warranty of MERCHANTABILITY or FITNESS FOR A PARTICULAR
#  PURPOSE.  See the GNU General Public License for more details.
#

import codecs
import logging
import os
import random
import re
import select
import socket
import sys
import time
import webbrowser
import urllib.parse
from collections import OrderedDict

from gi.repository import Gtk
from gi.repository import Gdk
from gi.repository import GLib
from gi.repository import Gio

try:
    from gi.repository import Gst
    Gst.init(None)
except:
    Gst = None

import ninix.surface
import ninix.balloon
import ninix.dll
import ninix.makoto
import ninix.pix
import ninix.script
import ninix.version
import ninix.update
from ninix.home import get_ninix_home, get_normalized_path
from ninix.metamagic import Meme


class ShellMeme(Meme):

    def __init__(self, key):
        Meme.__init__(self, key)
        self.request_parent = lambda *a: None # dummy

    def set_responsible(self, request_method):
        self.request_parent = request_method

    def create_menuitem(self, data):
        shell_name = data[0]
        subdir = data[1]
        base_path = self.request_parent('GET', 'get_prefix')
        thumbnail_path = os.path.join(base_path, b'shell',
                                      subdir, b'thumbnail.png')
        if not os.path.exists(thumbnail_path):
            thumbnail_path = None
        return self.request_parent(
            'GET', 'create_shell_menuitem', shell_name, self.key,
            thumbnail_path)

    def delete_by_myself(self):
        self.request_parent('NOTIFY', 'delete_shell', self.key)


class Sakura:

    BALLOON_LIFE   = 10  # [sec] (0: never closed automatically)
    SELECT_TIMEOUT = 15  # [sec]
    PAUSE_TIMEOUT  = 30  # [sec]
    SILENT_TIME    = 15  # [sec]
    # script modes
    BROWSE_MODE        = 1
    SELECT_MODE        = 2
    PAUSE_MODE         = 3
    WAIT_MODE          = 4 
    PAUSE_NOCLEAR_MODE = 5
    # script origins
    FROM_SSTP_CLIENT = 1
    FROM_GHOST       = 2
    # HTML entity definitions
    try:
        from html.entities import name2codepoint
    except:
        name2codepoint = None

    def __init__(self):
        self.request_parent = lambda *a: None # dummy
        self.sstp_handle = None
        self.sstp_entry_db = None
        self.sstp_request_handler = None
        # error = 'loose'(default) or 'strict'
        self.script_parser = ninix.script.Parser(error='loose')
        self.char = 2 # 'sakura' and 'kero'
        self.script_queue = []
        self.script_mode = self.BROWSE_MODE
        self.script_post_proc = []
        self.script_finally = []
        self.script_position = 0
        self.event_queue = []
        self.__current_script = ''
        self.__balloon_life = 0
        self.__surface_life = 0
        self.__boot = [0, 0]
        self.surface_mouse_motion = None ## FIXME
        self.time_critical_session = 0
        self.lock_repaint = 0
        self.passivemode = 0
        self.__running = 0
        self.anchor = None
        self.clock = (0, 0)
        self.synchronized_session = []
        self.force_quit = False
        ##
        self.old_otherghostname = None ## FIXME
        # create vanish dialog
        self.__vanish_dialog = VanishDialog()
        self.__vanish_dialog.set_responsible(self.handle_request)
        self.cantalk = 1
        self.__sender = 'ninix-aya'
        self.__charset = 'Shift_JIS'
        saori_lib = ninix.dll.Library('saori', sakura=self)
        self.__dll = ninix.dll.Library('shiori', saori_lib=saori_lib)
        self.__temp_mode = 0
        self.__observers = {}
        self.__listening = {
            'OnOtherGhostBooted': True,
            'OnOtherGhostClosed': True,
            'OnOtherGhostChanged': True,
            'OnOtherSurfaceChange': False,
            'OnOtherGhostVanished': True,
            'OnOtherGhostTalk': False,
            'OnOtherOverlap': True,
            'OnOtherOffscreen': True
            }
        self.balloon = ninix.balloon.Balloon()
        self.balloon.set_responsible(self.handle_request)
        self.surface = ninix.surface.Surface()
        self.surface.set_responsible(self.handle_request)
        self.keep_silence(False)
        self.updateman = ninix.update.NetworkUpdate()
        self.updateman.set_responsible(self.handle_request)
        if Gst is not None:
            self.audio_player = Gst.ElementFactory.make('playbin', 'player')
            fakesink = Gst.ElementFactory.make('fakesink', 'fakesink')
            self.audio_player.set_property('video-sink', fakesink)
            bus = self.audio_player.get_bus()
            bus.add_signal_watch()
            bus.connect('message', self.on_audio_message)
        else:
            self.audio_player = None
        self.audio_loop = False

    def set_responsible(self, request_method):
        self.request_parent = request_method

    def handle_request(self, event_type, event, *arglist, **argdict):
        assert event_type in ['GET', 'NOTIFY']
        handlers = {
            }
        handler = handlers.get(event, getattr(self, event, None))
        if handler is None:
            result = self.request_parent(
                event_type, event, *arglist, **argdict)
        else:
            result = handler(*arglist, **argdict)
        if event_type == 'GET':
            return result

    def attach_observer(self, observer):
        if observer not in self.__observers:
            self.__observers[observer] = 1

    def notify_observer(self, event, args=None):
        args = args or ()
        for observer in list(self.__observers.keys()):
            observer.observer_update(event, args)

    def detach_observer(self, observer):
        if observer in self.__observers:
            del self.__observers[observer]

    def delete_shell(self, key):
        assert key in self.shells
        del self.shells[key]

    def notify_installedshellname(self):
        installed = []
        for key in self.shells:
            installed.append(self.shells[key].baseinfo[0])
        self.notify_event('installedshellname', *installed)

    def get_shell_menu(self):
        current_key = self.get_current_shell()
        for key in self.shells:
            menuitem = self.shells[key].menuitem
            menuitem.set_sensitive(key != current_key) # not working
        return self.shell_menu

    def new(self, desc, shiori_dir, use_makoto, surface_set, prefix,
            shiori_dll, shiori_name): ## FIXME
        self.shiori = None
        self.desc = desc
        self.shiori_dir = shiori_dir
        self.use_makoto = use_makoto
        self.prefix = prefix
        self.shells = OrderedDict()
        shell_menuitems = OrderedDict()
        for key, value in surface_set.items():
            meme = ShellMeme(key)
            meme.set_responsible(self.handle_request)
            self.shells[key] = meme
            meme.baseinfo = value
            shell_menuitems[key] = meme.menuitem
        self.shell_menu = self.request_parent(
            'GET', 'create_shell_menu', shell_menuitems)
        self.shiori_dll = shiori_dll
        self.shiori_name = shiori_name
        name = (shiori_dll, shiori_name)
        self.shiori = self.__dll.request(name)
        char = 2
        while self.desc.get('char{0:d}.seriko.defaultsurface'.format(char)) is not None:
            char += 1
        if char > 2:
            self.char = char
        # XXX
        if self.desc.get('name') == 'BTH小っちゃいってことは便利だねっ':
            self.set_SSP_mode(1)
        else:
            self.set_SSP_mode(0)
        self.last_script = None
        self.status_icon = Gtk.StatusIcon()
        self.status_icon.set_title(self.get_name(default=''))
        self.status_icon.set_visible(False)

    def set_SSP_mode(self, flag): # XXX
        self.__sender = 'SSP' if flag else 'ninix-aya'

    def save_history(self):
        path = os.path.join(self.get_prefix(), b'HISTORY')
        try:
            with open(path, 'w') as f:
                f.write('time, {0}\n'.format(self.ghost_time))
                f.write('vanished_count, {0}\n'.format(self.vanished_count))
        except IOError as e:
            code, message = e.args
            logging.error('cannot write {0}'.format(path))

    def save_settings(self):
        path = os.path.join(self.get_prefix(), b'SETTINGS')
        try:
            with open(path, 'w') as f:
                if self.balloon_directory is not None:
                    f.write('balloon_directory, {0}\n'.format(
                            self.balloon_directory))
                if self.shell_directory is not None:
                    f.write('shell_directory, {0}\n'.format(
                            os.fsdecode(self.shell_directory)))
        except IOError as e:
            code, message = e.args
            logging.error('cannot write {0}'.format(path))

    def load_history(self):
        path = os.path.join(self.get_prefix(), b'HISTORY')
        if os.path.exists(path):
            ghost_time = 0
            ghost_vanished_count = 0
            try:
                with open(path, 'r') as f:
                    for line in f:
                        if ',' not in line:
                            continue
                        key, value = line.split(',', 1)
                        key = key.strip()
                        if key == 'time':
                            try:
                                ghost_time = int(value.strip())
                            except:
                                pass
                        elif key == 'vanished_count':
                            try:
                                ghost_vanished_count = int(value.strip())
                            except:
                                pass
            except IOError as e:
                code, message = e.args
                logging.error('cannot read {0}'.format(path))
            self.ghost_time = ghost_time
            self.vanished_count = ghost_vanished_count
        else:
            self.ghost_time = 0
            self.vanished_count = 0

    def load_settings(self):
        path = os.path.join(self.get_prefix(), b'SETTINGS')
        if os.path.exists(path):
            balloon_directory = None
            shell_directory = None
            try:
                with open(path, 'r') as f:
                    for line in f:
                        if ',' not in line:
                            continue
                        key, value = line.split(',', 1)
                        if key.strip() == 'balloon_directory':
                            balloon_directory = value.strip()
                        if key.strip() == 'shell_directory':
                            shell_directory = value.strip()
            except IOError as e:
                code, message = e.args
                logging.error('cannot read {0}'.format(path))
            self.balloon_directory = balloon_directory
            if shell_directory is not None:
                shell_directory = os.fsencode(shell_directory)
            self.shell_directory = shell_directory
        else:
            self.balloon_directory = None
            self.shell_directory = None

    def load_shiori(self):
        if self.shiori and self.shiori.load(self.shiori_dir):
            if getattr(self.shiori, 'show_description', None):
                self.shiori.show_description()
        else:
            logging.error('{0} cannot load SHIORI({1})'.format(
                    self.get_selfname(), self.shiori_name))
        self.__charset = 'Shift_JIS' # default
        self.get_event_response('OnInitialize', event_type='NOTIFY')
        self.get_event_response('basewareversion',
                                ninix.version.VERSION,
                                'ninix-aya',
                                ninix.version.NUMBER,
                                event_type='NOTIFY')

    def finalize(self):
        if self.script_finally: # XXX
            for proc in self.script_finally:
                proc(flag_break=False)
            self.script_finally = []
        if not self.__temp_mode:
            self.get_event_response('OnDestroy', event_type='NOTIFY')
            self.shiori.unload()
        self.stop()

    def enter_temp_mode(self):
        if not self.__temp_mode:
            self.__temp_mode = 2

    def leave_temp_mode(self):
        self.__temp_mode = 0

    def is_listening(self, key):
        return self.__listening.get(key)

    def on_audio_message(self, bus, message):
        if message is None: # XXX: workaround for Gst Version < 0.11
            if self.script_mode == self.WAIT_MODE:
                self.script_mode = self.BROWSE_MODE
            return
        t = message.type
        if t == Gst.MessageType.EOS:
            self.audio_player.set_state(Gst.State.NULL)
            if self.script_mode == self.WAIT_MODE:
                assert not self.audio_loop
                self.script_mode = self.BROWSE_MODE
            if self.audio_loop:
                self.audio_player.set_state(Gst.State.PLAYING)
        elif t == Gst.MessageType.ERROR:
            self.audio_player.set_state(Gst.State.NULL)
            err, debug = message.parse_error()
            logging.error('Error: {0}, {1}'.format(err, debug))
            self.audio_loop = False

    def set_surface(self, desc, alias, surface, name, surface_dir, tooltips, seriko_descript):
        default_sakura = self.desc.get('sakura.seriko.defaultsurface', '0')
        default_kero = self.desc.get('kero.seriko.defaultsurface', '10')
        self.surface.new(desc, alias, surface, name, surface_dir, tooltips, seriko_descript,
                         default_sakura, default_kero)
        for side in range(2, self.char):
            default = self.desc.get('char{0:d}.seriko.defaultsurface'.format(side))
            self.surface.add_window(side, default)
        icon = self.desc.get('icon', None)
        if icon is not None:
            icon_path = os.path.join(self.shiori_dir, os.fsencode(icon))
            if not os.path.exists(icon_path):
                icon_path = None
        else:
            icon_path = None
        self.surface.set_icon(icon_path)

    def set_balloon(self, desc, balloon):
        self.balloon.new(desc, balloon)
        for side in range(2, self.char):
            self.balloon.add_window(side)
        for side in range(self.char):
            balloon_win = self.balloon.get_window(side)
            surface_win = self.surface.get_window(side)
            balloon_win.set_transient_for(surface_win)

    def update_balloon_offset(self, side, x_delta, y_delta):
        if side >= self.char:
            return
        ox, oy = self.surface.window[side].balloon_offset # without scaling
        direction = self.balloon.window[side].direction
        sx, sy = self.get_surface_position(side)
        if direction == 0: # left
            nx = ox + x_delta
        else:
            w, h = self.surface.get_surface_size(side)
            nx = ox - x_delta
        ny = oy + y_delta
        self.surface.set_balloon_offset(side, (nx, ny))

    def enqueue_script(self, event, script, sender, handle,
                       host, show_sstp_marker, use_translator,
                       db=None, request_handler=None):
        if not self.script_queue and \
           not self.time_critical_session and not self.passivemode:
            if self.sstp_request_handler:
                self.sstp_request_handler.send_sstp_break()
                self.sstp_request_handler = None
            self.reset_script(1)
        self.script_queue.append((event, script, sender, handle, host,
                                  show_sstp_marker, use_translator,
                                  db, request_handler))

    reset_event = ['OnGhostChanging', 'OnShellChanging', 'OnVanishSelected']

    def check_event_queue(self):
        return bool(self.event_queue)

    def enqueue_event(self, event, *arglist, **argdict): ## FIXME
        for key in argdict:
            assert key in ['proc'] # trap typo, etc.
        if event in self.reset_event:
            self.reset_script(1)
        self.event_queue.append((event, arglist, argdict))

    EVENT_SCRIPTS = {
        'OnUpdateBegin': \
        ''.join((r'\t\h\s[0]',
                 _('Network Update has begun.'),
                 r'\e')),
        'OnUpdateComplete': \
        ''.join((r'\t\h\s[5]',
                 _('Network Update completed successfully.'),
                 r'\e')),
        'OnUpdateFailure': \
        ''.join((r'\t\h\s[4]',
                 _('Network Update failed.'),
                 r'\e')),
        }

    def handle_event(self): ## FIXME
        while self.event_queue:
            event, arglist, argdict = self.event_queue.pop(0)
            proc = argdict.get('proc', None)
            argdict = {'default': self.EVENT_SCRIPTS.get(event)}
            if self.notify_event(event, *arglist, **argdict):
                if proc is not None:
                    self.script_post_proc.append(proc)
                return 1
            elif proc is not None:
                proc()
                return 1
        return 0

    def is_running(self):
        return self.__running

    def is_paused(self):
        return self.script_mode in {self.PAUSE_MODE, self.PAUSE_NOCLEAR_MODE}

    def is_talking(self):
        return int(bool(self.processed_script or self.processed_text))

    def busy(self, check_updateman=True):
        return bool(self.time_critical_session or \
                        self.balloon.user_interaction or \
                        self.event_queue or \
                        self.passivemode or \
                        self.sstp_request_handler is not None or \
                        (check_updateman and self.updateman.is_active()))

    def get_silent_time(self):
        return self.silent_time

    def keep_silence(self, quiet):
        if quiet:
            self.silent_time = time.time()
        else:
            self.silent_time = 0
            self.reset_idle_time()

    def get_idle_time(self):
        now = time.time()
        idle = now - self.idle_start
        return idle

    def reset_idle_time(self):
        self.idle_start = time.time()

    def notify_preference_changed(self): ## FIXME
        self.balloon.reset_fonts() ## FIXME
        self.surface.reset_surface()
        self.notify_observer('set scale') ## FIXME
        self.balloon.reset_balloon()

    def get_surface_position(self, side):
        result = self.surface.get_position(side)
        return result if result is not None else (0, 0)

    def set_balloon_position(self, side, base_x, base_y):
        self.balloon.set_position(side, base_x, base_y)

    def set_balloon_direction(self, side, direction):
        if side >= self.char:
            return
        self.balloon.window[side].direction = direction

    def get_balloon_size(self, side):
        result = self.balloon.get_balloon_size(side)
        return result if result is not None else (0, 0)

    def get_balloon_windowposition(self, side):
        return self.balloon.get_balloon_windowposition(side)

    def get_balloon_position(self, side):
        result = self.balloon.get_position(side)
        return result if result is not None else (0, 0)

    def balloon_is_shown(self, side):
        return int(self.balloon and self.balloon.is_shown(side))

    def surface_is_shown(self, side):
        return int(self.surface and self.surface.is_shown(side))

    def is_URL(self, s):
        return s.startswith('http://') or \
               s.startswith('ftp://') or \
               s.startswith('file:/')

    def is_anchor(self, link_id):
        return int(len(link_id) == 2 and link_id[0] == 'anchor')

    def vanish(self):
        if self.busy():
            Gdk.beep() ## FIXME
            return
        self.notify_event('OnVanishSelecting')
        self.__vanish_dialog.show()

    def vanish_by_myself(self, next_ghost):
        self.vanished_count += 1
        self.ghost_time = 0
        self.request_parent('NOTIFY', 'vanish_sakura', self, next_ghost)

    def get_ifghost(self):
        return ''.join((self.get_selfname(), ',', self.get_keroname()))

    def ifghost(self, ifghost):
        names = self.get_ifghost()
        name = self.get_selfname()
        return bool(ifghost in [name, names])

    def get_name(self, default=_('Sakura&Unyuu')):
        return self.desc.get('name', default)

    def get_username(self):
        return self.getstring('username') or \
               self.surface.get_username() or \
               self.desc.get('user.defaultname', _('User'))

    def get_selfname(self, default=_('Sakura')):
        return self.surface.get_selfname() or \
               self.desc.get('sakura.name', default)

    def get_selfname2(self):
        return self.surface.get_selfname2() or \
               self.desc.get('sakura.name2', _('Sakura'))

    def get_keroname(self):
        return self.surface.get_keroname() or \
               self.desc.get('kero.name', _('Unyuu'))

    def get_friendname(self):
        return self.surface.get_friendname() or \
               self.desc.get('sakura.friend.name',
                             _('Tomoyo'))

    def getaistringrandom(self): # obsolete
        result = self.get_event_response('OnAITalk')
        return self.translate(result)

    def getdms(self):
        result = self.get_event_response('dms')
        return self.translate(result)

    def getword(self, word_type):
        result = self.get_event_response(word_type)
        return self.translate(result)

    def getstring(self, name):
        return self.get_event_response(name)

    def translate(self, s):
        if s is not None:
            if self.use_makoto:
                s = ninix.makoto.execute(s)
            else:
                r = self.get_event_response('OnTranslate', s, translate=0)
                if r:
                    s = r
        return s

    def get_value(self, response): # FIXME: check return code
        result = {}
        to = None
        for line in response.splitlines():
            line = str(line, self.__charset, 'ignore').strip()
            if not line:
                continue
            if ':' not in line:
                continue
            key, value = line.split(':', 1)
            key = key.strip()
            if key == 'Charset':
                charset = value.strip()
                if charset != self.__charset:
                    try:
                        codecs.lookup(charset)
                    except:
                        logging.warning(
                            'Unsupported charset {0}'.format(repr(charset)))
                    else:
                        self.__charset = charset
            result[key] = value
        for key, value in result.items():
            result[key] = value.strip()
        if 'Reference0' in result:
            to = result['Reference0']
        if 'Value' in result:
            return result['Value'], to
        else:
            return None, to

    def get_event_response_with_communication(self, event, *arglist, **argdict):
        if self.__temp_mode == 1:
            return ''
        for key in argdict:
            assert key in ['event_type', 'translate'] # trap typo, etc.
        ref = arglist
        event_type = argdict.get('event_type', 'GET')
        translate = argdict.get('translate', 1)
        header = ''.join(('{0} SHIORI/3.0\r\n'.format(event_type),
                          'Sender: {0}\r\n'.format(self.__sender),
                          'ID: {0}\r\n'.format(event),
                          'SecurityLevel: local\r\n',
                          'Charset: {0}\r\n'.format(self.__charset)))
        for i in range(len(ref)):
            value = ref[i]
            if value is not None:
                value = value if isinstance(value, str) \
                    else str(value)
                header = ''.join((header,
                                  'Reference', str(i), ': ',
                                  value, '\r\n'))
        header = ''.join((header, '\r\n'))
        header = header.encode(self.__charset, 'ignore')
        response = self.shiori.request(header)
        if event_type != 'NOTIFY' and self.cantalk:
            result, to = self.get_value(response)
            if translate:
                result = self.translate(result)
        else:
            result, to = '', None
        if result is None:
            result = ''
        if to and result:
            communication = to
        else:
            communication = None
        return result, communication

    def get_event_response(self, event, *arglist, **argdict):
        result, communication = self.get_event_response_with_communication(event, *arglist, **argdict)
        return result

    ###   CALLBACK   ###
    def notify_start(self, init, vanished, ghost_changed,
                     name, prev_name, prev_shell, path, last_script,
                     abend=None):
        if self.__temp_mode:
            default = None
        else:
            default = ninix.version.VERSION_INFO
        if init:
            if self.ghost_time == 0:
                if not self.notify_event('OnFirstBoot', self.vanished_count,
                                         None, None, None, None, None, None,
                                         self.surface.name):
                    if abend is not None:
                        self.notify_event('OnBoot', self.surface.name,
                                          None, None, None, None, None,
                                          'halt', abend, default=default)
                    else:
                        self.notify_event('OnBoot', self.surface.name,
                                          default=default)
            else:
                if abend is not None:
                    self.notify_event('OnBoot', self.surface.name,
                                      None, None, None, None, None,
                                      'halt', abend, default=default)
                else:
                    self.notify_event('OnBoot', self.surface.name,
                                      default=default)
            left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
            self.notify_event('OnDisplayChange',
                              Gdk.Visual.get_best_depth(),
                              scrn_w, scrn_h, event_type='NOTIFY')
        elif vanished:
            if self.ghost_time == 0:
                if self.notify_event('OnFirstBoot', self.vanished_count,
                                     None, None, None, None, None, None,
                                     self.surface.name):
                    return
            elif self.notify_event('OnVanished', name):
                return
            elif self.notify_event('OnGhostChanged', name, last_script,
                                   prev_name, None, None, None, None,
                                   pref_shell):
                return
            if abend is not None:
                self.notify_event('OnBoot', self.surface.name,
                                  None, None, None, None, None, None,
                                  'halt', abend, default=default)
            else:
                self.notify_event('OnBoot', self.surface.name, default=default)
        elif ghost_changed:
            if self.ghost_time == 0:
                if self.notify_event('OnFirstBoot', self.vanished_count,
                                     None, None, None, None, None, None,
                                     self.surface.name):
                    return
            elif self.notify_event('OnGhostChanged', name, last_script,
                                   prev_name, None, None, None, None,
                                   prev_shell):
                return
            if abend is not None:
                self.notify_event('OnBoot', self.surface.name,
                                  None, None, None, None, None,
                                  'halt', abend, default=default)
            else:
                self.notify_event('OnBoot', self.surface.name, default=default)
        else:
            pass ## FIXME

    def notify_vanish_selected(self):
        def proc(self=self):
            self.vanished_count += 1
            self.ghost_time = 0
            GLib.idle_add(
                lambda a: self.request_parent('NOTIFY', 'vanish_sakura', *a),
                (self, None))
        self.enqueue_event('OnVanishSelected', proc=proc)
        self.vanished = 1 ## FIXME

    def notify_vanish_canceled(self):
        self.notify_event('OnVanishCancel')

    def notify_iconified(self):
        self.cantalk = 0
        self.request_parent('NOTIFY', 'select_current_sakura')
        if not self.passivemode:
            self.reset_script(1)
            self.stand_by(1)
            self.notify_event('OnWindowStateMinimize')
        self.notify_observer('iconified')

    def notify_deiconified(self):
        if not self.cantalk:
            self.cantalk = 1
            self.request_parent('NOTIFY', 'select_current_sakura')
            if not self.passivemode:
                self.notify_event('OnWindowStateRestore')
        self.notify_observer('deiconified')

    def notify_link_selection(self, link_id, text, number):
        if self.script_origin == self.FROM_SSTP_CLIENT and \
           self.sstp_request_handler is not None:
            self.sstp_request_handler.send_answer(text)
            self.sstp_request_handler = None
        if self.is_anchor(link_id):
            self.notify_event('OnAnchorSelect', link_id[1])
        elif self.is_URL(link_id):
            webbrowser.open(link_id)
            self.reset_script(1)
            self.stand_by(0)
        elif self.sstp_entry_db:
            # leave the previous sstp message as it is
            self.start_script(self.sstp_entry_db.get(link_id, r'\e'))
            self.sstp_entry_db = None
        elif not self.notify_event('OnChoiceSelect', link_id, text, number):
            self.reset_script(1)
            self.stand_by(0)

    def notify_site_selection(self, args):
        title, url = args
        if self.is_URL(url):
            webbrowser.open(url)
        self.enqueue_event('OnRecommandedSiteChoice', title, url)

    def notify_surface_click(self, button, click, side, x, y):
        if button == 1 and click == 1:
            self.raise_all()
        if self.vanished:
            if side == 0 and button == 1:
                if self.sstp_request_handler:
                    self.sstp_request_handler.send_sstp_break()
                    self.sstp_request_handler = None
                self.reset_script(1)
                self.notify_event('OnVanishButtonHold', default=r'\e')
                self.vanished = 0
            return
        if self.updateman.is_active():
            if button == 1 and click == 2:
                self.updateman.interrupt()
            return
        if self.time_critical_session:
            return
        elif click == 1:
            if self.passivemode and \
               self.processed_script is not None:
                return
            part = self.surface.get_touched_region(side, x, y)
            if button in {1, 2, 3}:
                num_button = [0, 2, 1][button - 1]
                if not self.notify_event('OnMouseUp',
                                         x, y, 0, side, part, num_button,
                                         'mouse'): # FIXME
                    if button == 2:
                        if self.notify_event(
                            'OnMouseUpEx',
                            x, y, 0, side, part, 'middle',
                            'mouse'): # FIXME
                            return
                        if self.notify_event('OnMouseClickEx',
                                      x, y, 0, side, part, 'middle',
                                      'mouse'): # FIXME
                            return
                    self.notify_event('OnMouseClick',
                                      x, y, 0, side, part, num_button,
                                      'mouse') # FIXME
            elif button in {8, 9}:
                ex_button = {
                    2: 'middle',
                    8: 'xbutton1',
                    9: 'xbutton2'
                    }[button]
                if not self.notify_event('OnMouseUpEx',
                                         x, y, 0, side, part, ex_button,
                                         'mouse'): # FIXME
                    self.notify_event('OnMouseClickEx',
                                      x, y, 0, side, part, ex_button,
                                      'mouse') # FIXME
        elif self.passivemode:
            return
        elif button in {1, 3} and click == 2:
            if self.sstp_request_handler:
                self.sstp_request_handler.send_sstp_break()
                self.sstp_request_handler = None
            part = self.surface.get_touched_region(side, x, y)
            num_button = [0, 2, 1][button - 1]
            self.notify_event('OnMouseDoubleClick',
                              x, y, 0, side, part, num_button,
                              'mouse') # FIXME
        elif button in {2, 8, 9} and click == 2:
            part = self.surface.get_touched_region(side, x, y)
            ex_button = {
                2: 'middle',
                8: 'xbutton1',
                9: 'xbutton2'
                }[button]
            self.notify_event('OnMouseDoubleClickEx',
                              x, y, 0, side, part, ex_button,
                              'mouse') # FIXME

    def notify_balloon_click(self, button, click, side):
        if self.script_mode == self.PAUSE_MODE:
            self.script_mode = self.BROWSE_MODE
            self.balloon.clear_text_all()
            self.balloon.hide_all()
            self.script_side = 0
        elif self.script_mode == self.PAUSE_NOCLEAR_MODE:
            self.script_mode = self.BROWSE_MODE
        elif button == 1 and click == 1:
            self.raise_all()
        if self.vanished:
            return
        if self.updateman.is_active():
            if button == 1 and click == 2:
                self.updateman.interrupt()
            return
        if self.time_critical_session:
            self.time_critical_session = 0
            return
        elif self.passivemode:
            return
        elif button == 1 and click == 2:
            if self.sstp_request_handler:
                self.sstp_request_handler.send_sstp_break()
                self.sstp_request_handler = None
                self.reset_script(1)
                self.stand_by(0)
        elif button == 3 and click == 1:
            if self.sstp_request_handler:
                self.sstp_request_handler.send_sstp_break()
                self.sstp_request_handler = None
            if self.is_talking():
                self.notify_event('OnBalloonBreak',
                                  self.__current_script, side,
                                  self.script_position)
            else:
                self.notify_event('OnBalloonClose', self.__current_script)
            self.reset_script(1)
            self.stand_by(0)

    def notify_surface_mouse_motion(self, side, x, y, part):
        if self.surface_mouse_motion is not None:
            return
        if part:
            self.surface_mouse_motion = (side, x, y, part)
        else:
            self.surface_mouse_motion = None

    def notify_user_teach(self, word):
        if word is not None:
            script = self.translate(self.get_event_response('OnTeach', word))
            if script:
                self.start_script(script)
                self.balloon.hide_sstp_message()


    month_names = ['Jan', 'Feb', 'Mar', 'Apr', 'May', 'Jun',
                   'Jul', 'Aug', 'Sep', 'Oct', 'Nov', 'Dec']
    boot_event = ['OnBoot', 'OnFirstBoot', 'OnGhostChanged', 'OnShellChanged',
                  'OnUpdateComplete']
    reset_event = ['OnVanishSelecting', 'OnVanishCancel'] ## FIXME

    def notify_event(self, event, *arglist, **argdict):
        if self.time_critical_session and event.startswith('OnMouse'):
            return 0
        if event in self.reset_event:
            self.reset_script(1)
        for key in argdict:
            assert key in ['event_type', 'default'] # trap typo, etc.
        event_type = argdict.get('event_type', 'GET')
        default = argdict.get('default', None)
        argdict = {'event_type': event_type} ## FIXME
        script, communication = self.get_event_response_with_communication(event, *arglist, **argdict) or (default, None)
        if script or (not script and event != 'OnSecondChange'):
            t = time.localtime(time.time())
            m = self.month_names[t[1] - 1]
            logging.debug('\n[{0:02d}/{1}/{2:d}:{3:02d}:{4:02d}:{5:02d} {6:+05d}]'.format(
                    t[2], m, t[0], t[3], t[4], t[5], - time.timezone // 36))
            logging.debug('Event: {0}'.format(event))
            for n in range(len(arglist)):
                value = arglist[n]
                if value is not None:
                    value = value if isinstance(value, str) \
                        else str(value)
                    logging.debug(
                    'Reference{0:d}: {1}'.format(n, value))
        if event == 'OnCloseAll':
            self.force_quit = True
            if not script: # fallback
                script, communication = \
                    self.get_event_response_with_communication(
                    'OnClose', *arglist, **argdict) or (default, None)
            if script:
                self.start_script(script)
                self.balloon.hide_sstp_message()
            return 1
        if event == 'OnClose' and arglist[0] == 'shutdown': # XXX
            self.force_quit = True
        if not script: # an empty script is ignored
            if event in self.boot_event:
                self.surface_bootup()
            if event == 'OnMouseClick' and arglist[5] == 1:
                self.request_parent(
                    'NOTIFY', 'open_popup_menu', self, arglist[5], arglist[3])
            self.request_parent(
                'NOTIFY', 'notify_other', self.key,
                event, self.get_name(default=''),
                self.get_selfname(default=''),
                self.get_current_shell_name(),
                False, communication,
                None, False, script, arglist)
            return 0
        logging.debug('=> "{0}"'.format(script))
        if self.__temp_mode == 2:
            self.request_parent('NOTIFY', 'reset_sstp_flag')
            self.leave_temp_mode()
        if self.passivemode and \
           (event == 'OnSecondChange' or event == 'OnMinuteChange'):
            return 0
        self.start_script(script)
        self.balloon.hide_sstp_message()
        if event in self.boot_event:
            self.script_finally.append(self.surface_bootup)
        def proc(flag_break=False):
            self.request_parent(
                'NOTIFY', 'notify_other', self.key,
                event, self.get_name(default=''),
                self.get_selfname(default=''),
                self.get_current_shell_name(),
                flag_break, communication,
                None, False, script, arglist)
        self.script_finally.append(proc)
        return 1

    def get_prefix(self):
        return self.prefix

    def stick_window(self, flag):
        self.surface.window_stick(flag)

    def toggle_bind(self, args):
        self.surface.toggle_bind(args)

    def get_menu_pixmap(self):
        path_background, path_sidebar, path_foreground, \
            align_background, align_sidebar, align_foreground = \
            self.surface.get_menu_pixmap()
        top_dir = self.surface.prefix
        ghost_dir = os.path.join(self.get_prefix(), b'ghost', b'master')
        name = self.getstring('menu.background.bitmap.filename')
        if name:
            name = name.replace('\\', '/')
            path_background = os.path.join(top_dir, os.fsencode(name))
        if path_background is None:
            path_background = os.path.join(ghost_dir,
                                           os.fsencode('menu_background.png'))
        if not os.path.exists(path_background):
            path_background = None
        name = self.getstring('menu.sidebar.bitmap.filename')
        if name:
            name = name.replace('\\', '/')
            path_sidebar = os.path.join(top_dir, os.fsencode(name))
        if path_sidebar is None:
            path_sidebar = os.path.join(ghost_dir,
                                        os.fsencode('menu_sidebar.png'))
        if not os.path.exists(path_sidebar):
            path_sidebar = None
        name = self.getstring('menu.foreground.bitmap.filename')
        if name:
            name = name.replace('\\', '/')
            path_foreground = os.path.join(top_dir, os.fsencode(name))
        if path_foreground is None:
            path_foreground = os.path.join(ghost_dir,
                                           os.fsencode('menu_foreground.png'))
        if not os.path.exists(path_foreground):
            path_foreground = None
        align = self.getstring('menu.background.alignment')
        if align:
            align_background = align
        if align_background not in ['lefttop', 'righttop', 'centertop']:
            align_background = 'lefttop'
        align_background = align_background[:-3].encode('ascii') # XXX
        align = self.getstring('menu.sidebar.alignment')
        if align:
            align_sidebar = align
        if align_sidebar not in ['top', 'bottom']:
            align_sidebar = 'bottom'
        align_sidebar = align_sidebar.encode('ascii') # XXX
        align = self.getstring('menu.foreground.alignment')
        if align:
            align_foreground = align
        if align_foreground not in ['lefttop', 'righttop', 'centertop']:
            align_foreground = 'lefttop'
        align_foreground = align_foreground[:-3].encode('ascii') # XXX
        return path_background, path_sidebar, path_foreground, \
            align_background, align_sidebar, align_foreground

    def get_menu_fontcolor(self):
        background, foreground = self.surface.get_menu_fontcolor()
        color_r = self.getstring('menu.background.font.color.r')
        color_g = self.getstring('menu.background.font.color.g')
        color_b = self.getstring('menu.background.font.color.b')
        try:
            color_r = max(0, min(255, int(color_r)))
            color_g = max(0, min(255, int(color_g)))
            color_b = max(0, min(255, int(color_b)))
        except:
            pass
        else:
            background = (color_r, color_g, color_b)
        color_r = self.getstring('menu.foreground.font.color.r')
        color_g = self.getstring('menu.foreground.font.color.g')
        color_b = self.getstring('menu.foreground.font.color.b')
        try:
            color_r = max(0, min(255, int(color_r)))
            color_g = max(0, min(255, int(color_g)))
            color_b = max(0, min(255, int(color_b)))
        except:
            pass
        else:
            foreground = (color_r, color_g, color_b)
        return background, foreground

    def get_mayuna_menu(self):
        return self.surface.get_mayuna_menu()

    def get_current_balloon_directory(self):
        return self.balloon.get_balloon_directory()

    def get_current_shell(self):
        return self.shell_directory

    def get_current_shell_name(self):
        return self.shells[self.get_current_shell()].baseinfo[0]

    def get_default_shell(self):
        default = self.shell_directory or b'master'            
        if default not in self.shells:
            default = list(self.shells.keys())[0] # XXX
        return default

    def get_balloon_default_id(self):
        return self.desc.get('balloon.defaultsurface', '0')

    def select_shell(self, shell_key):
        assert self.shells and shell_key in self.shells
        self.shell_directory = shell_key # save user's choice
        surface_name, surface_dir, surface_desc, surface_alias, surface, surface_tooltips, seriko_descript = \
            self.shells[shell_key].baseinfo
        def proc(self=self, key=shell_key):
            logging.info('ghost {0} {1}'.format(self.key, key))
            self.set_surface(surface_desc, surface_alias, surface, surface_name,
                             surface_dir, surface_tooltips, seriko_descript)
            self.surface.reset_alignment()
            self.notify_event('OnShellChanged',
                              surface_name, surface_name, surface_dir)
        self.enqueue_event('OnShellChanging', surface_name, surface_dir,
                           proc=proc)

    def select_balloon(self, item, desc, balloon):
        self.balloon_directory = item # save user's choice
        if item == self.get_current_balloon_directory(): # no change
            return # need reloadning?
        assert item == os.fsdecode(balloon['balloon_dir'][0])
        path = os.path.join(get_ninix_home(), b'balloon', os.fsencode(item))
        self.balloon.hide_all()
        self.set_balloon(desc, balloon)
        self.balloon.set_balloon_default()
        self.position_balloons()
        name = desc.get('name', '')
        logging.info('balloon {0} {1}'.format(name, path))
        self.notify_event('OnBalloonChange', name, path)

    def surface_bootup(self, flag_break=False):
        for side in [0, 1]:
            if not self.__boot[side]:
                self.set_surface_default(side)
                self.surface.show(side)

    def get_uptime(self):
        uptime = int(time.time() - self.start_time) // 3600
        if uptime < 0:
            self.start_time = time.time()
            return 0
        return uptime

    def hide_all(self):
        self.surface.hide_all()
        self.balloon.hide_all()

    def position_balloons(self):
        self.surface.reset_balloon_position()

    def align_top(self, side):
        self.surface.set_alignment(side, 1)

    def align_bottom(self, side):
        self.surface.set_alignment(side, 0)

    def align_current(self):
        self.surface.set_alignment_current()

    def identify_window(self, win):
        return bool(self.surface.identify_window(win) or \
                        self.balloon.identify_window(win))

    def set_surface_default(self, side=None):
        self.surface.set_surface_default(side)

    def get_surface_scale(self):
        return self.request_parent('GET', 'get_preference', 'surface_scale')

    def get_surface_size(self, side):
        result = self.surface.get_surface_size(side)
        return result if result is not None else (0, 0)

    def set_surface_position(self, side, x, y):
        self.surface.set_position(side, x, y)

    def set_surface_id(self, side, id):
        self.surface.set_surface(side, id)

    def get_surface_id(self, side):
        return self.surface.get_surface(side)

    def surface_is_shown(self, side):
        return bool(self.surface and self.surface.is_shown(side))

    def get_kinoko_position(self, baseposition):
        side = 0
        x, y = self.get_surface_position(side)
        w, h = self.get_surface_size(side)
        if baseposition == 1:
            rect = self.surface.get_collision_area(side, 'face')
            if rect is not None:
                x1, y1, x2, y2 = rect
                return x + (x2 - x1) // 2, y + (y2 - y1) // 2
            else:
                return x + w // 2, y + h // 4
        elif baseposition == 2:
            rect = self.surface.get_collision_area(side, 'bust')
            if rect is not None:
                x1, y1, x2, y2 = rect
                return x + (x2 - x1) // 2, y + (y2 - y1) // 2
            else:
                return x + w // 2, y + h // 2
        elif baseposition == 3:
            centerx, centery = self.surface.get_center(side)
            if centerx is None:
                centerx = w // 2
            if centery is None:
                centery = h // 2
            return x + centerx, y + centery
        else: # baseposition == 0 or baseposition not in [1, 2, 3]: # AKF
            centerx, centery = self.surface.get_kinoko_center(side)
            if centerx is None or centery is None:
                rect = self.surface.get_collision_area(side, 'head')
                if rect is not None:
                    x1, y1, x2, y2 = rect
                    return x + (x2 - x1) // 2, y + (y2 - y1) // 2
                else:
                    return x + w // 2, y + h // 8
            return x + centerx, y + centery

    def raise_surface(self, side):
        self.surface.raise_(side)

    def lower_surface(self, side):
        self.surface.lower(side)

    def raise_all(self):
        self.surface.raise_all()
        self.balloon.raise_all()

    def lower_all(self):
        self.surface.lower_all()
        self.balloon.lower_all()

    ###   STARTER   ###
    def stand_by(self, reset_surface):
        self.balloon.hide_all()
        self.balloon.hide_sstp_message()
        default_sakura = self.desc.get('sakura.seriko.defaultsurface', '0')
        default_kero = self.desc.get('kero.seriko.defaultsurface', '10')
        if reset_surface:
            self.set_surface_default()
            self.balloon.set_balloon_default()
        elif self.get_surface_id(0) != default_sakura or \
             self.get_surface_id(1) != default_kero:
            self.__surface_life = random.randint(20, 30)
            ##logging.debug('surface_life = {0:d}'.format(self.__surface_life))

    def start(self, key, init, temp, vanished, ghost_changed,
              prev_self_name, prev_name, prev_shell, last_script, abend):
        if self.is_running():
            if temp:
                self.enter_temp_mode()
            else:
                if self.__temp_mode == 1:
                    self.__temp_mode = 2
                    self.load_shiori()
                    self.notify_start(
                        init, vanished, ghost_changed,
                        prev_self_name, prev_name, prev_shell,
                        '', last_script, abend)
            return
        self.ghost_time = 0
        self.vanished_count = 0
        self.__running = 1
        self.__temp_mode = temp
        self.key = key
        self.force_quit = False
        logging.info('ghost {0}'.format(key))
        self.load_settings()
        shell_key = self.get_default_shell()
        self.shell_directory = shell_key # XXX
        assert self.shells and shell_key in self.shells
        surface_name, surface_dir, surface_desc, surface_alias, surface, surface_tooltips, seriko_descript = \
            self.shells[shell_key].baseinfo
        name = prev_self_name if ghost_changed else surface_name
        self.set_surface(surface_desc, surface_alias, surface, surface_name,
                         surface_dir, surface_tooltips, seriko_descript)
        balloon = None
        if not self.request_parent('GET', 'get_preference', 'ignore_default'): ## FIXME: change prefs key
            balloon_path = self.desc.get('deault.balloon.path', '')
            balloon_name = self.desc.get('balloon', '')
            if balloon_path:
                balloon = self.request_parent(
                    'GET', 'find_balloon_by_subdir', balloon_path)
            if balloon is None and balloon_name:
                balloon = self.request_parent(
                    'GET', 'find_balloon_by_name', balloon_name)
        if balloon is None:
            if self.balloon_directory is not None:
                balloon = self.balloon_directory
            else:
                balloon = self.request_parent(
                    'GET', 'get_preference', 'default_balloon')
        desc, balloon = self.request_parent(
            'GET', 'get_balloon_description', balloon)
        self.set_balloon(desc, balloon)
        if not temp:
            self.load_shiori()
        self.restart()
        self.start_time = time.time()
        self.notify_start(
            init, vanished, ghost_changed,
            name, prev_name, prev_shell, surface_dir, last_script, abend)
        GLib.timeout_add(10, self.do_idle_tasks) # 10[ms]

    def restart(self):
        self.load_history()
        self.vanished = 0
        self.__boot = [0, 0]
        self.old_otherghostname = None ## FIXME
        self.reset_script(1)
        self.surface.reset_alignment()
        self.stand_by(1)
        self.surface.reset_position()
        self.reset_idle_time()
        self.__running = 1
        self.force_quit = False

    def stop(self):
        if not self.__running:
            return
        self.notify_observer('finalize')
        self.__running = 0
        self.save_settings()
        self.save_history()
        self.request_parent('NOTIFY', 'rebuild_ghostdb', self, None)
        self.hide_all()
        self.surface.finalize()
        self.balloon.finalize()
        if self.audio_player is not None:
            self.audio_player.set_state(Gst.State.NULL)
        self.audio_loop = False

    def process_script(self):
        now = time.time()
        idle = self.get_idle_time()
        minute, second = time.localtime(now)[4:6]
        if self.clock[0] != second: ## FIXME
            if not self.__temp_mode:
                self.ghost_time += 1
            self.request_parent(
                'NOTIFY', 'rebuild_ghostdb',
                self,
                self.get_selfname(),
                self.get_surface_id(0),
                self.get_surface_id(1))
            otherghostname = self.request_parent(
                'GET', 'get_otherghostname', self.get_selfname())
            if otherghostname != self.old_otherghostname:
                args = []
                args.extend(otherghostname)
                args.insert(0, 'otherghostname')
                args = tuple(args)
                keyword = {'event_type': 'NOTIFY'}
                self.notify_event(*args, **keyword)
            self.old_otherghostname = otherghostname
        if not self.__running:
            pass
        elif self.script_mode in {self.PAUSE_MODE, self.PAUSE_NOCLEAR_MODE}:
            ##if idle > self.PAUSE_TIMEOUT:
            ##    self.script_mode = self.BROWSE_MODE
            pass
        elif self.script_mode == self.WAIT_MODE:
            pass
        elif self.processed_script or self.processed_text:
            self.interpret_script()
        elif self.script_post_proc:
            for proc in self.script_post_proc:
                proc()
            self.script_post_proc = []
        elif self.script_finally:
            for proc in self.script_finally:
                proc()
            self.script_finally = []
        elif self.script_mode == self.SELECT_MODE:
            if self.passivemode:
                pass
            elif idle > self.SELECT_TIMEOUT:
                self.script_mode = self.BROWSE_MODE
                if self.sstp_request_handler:
                    self.sstp_request_handler.send_timeout()
                    self.sstp_request_handler = None
                if not self.notify_event('OnChoiceTimeout'):
                    self.stand_by(0)
        elif self.sstp_handle is not None:
            self.close_sstp_handle()
        elif self.balloon.user_interaction:
            pass
        elif idle > self.__balloon_life > 0 and not self.passivemode:
            self.__balloon_life = 0
            for side in range(self.char):
                if self.balloon_is_shown(side):
                    self.notify_event('OnBalloonTimeout',
                                      self.__current_script)
                    break
            self.stand_by(0)
            if self.request_parent('GET', 'get_preference', 'sink_after_talk'):
                self.surface.lower_all()
        elif self.event_queue and self.handle_event():
            pass
        elif self.script_queue and not self.passivemode:
            if self.get_silent_time() > 0:
                self.keep_silence(True) # extend silent time
            event, script, sender, self.sstp_handle, \
                    host, show_sstp_marker, use_translator, \
                    self.sstp_entry_db, self.sstp_request_handler = \
                    self.script_queue.pop(0)
            if self.cantalk:
                if show_sstp_marker:
                    self.balloon.show_sstp_message(sender, host)
                else:
                    self.balloon.hide_sstp_message()
                # XXX: how about the use_translator flag?
                self.start_script(script, self.FROM_SSTP_CLIENT)
                def proc(flag_break=False):
                    self.request_parent(
                        'NOTIFY', 'notify_other', self.key,
                        event, self.get_name(default=''),
                        self.get_selfname(default=''),
                        self.get_current_shell_name(),
                        flag_break, None,
                        (sender, host), not use_translator, script, ())
                self.script_finally.append(proc)
        elif self.get_silent_time() > 0:
            if now - self.get_silent_time() > self.SILENT_TIME:
                self.keep_silence(False)
        elif self.clock[0] != second and \
             self.notify_event('OnSecondChange', self.get_uptime(),
                               self.surface.get_mikire(),
                               self.surface.get_kasanari(),
                               not self.passivemode and self.cantalk):
            pass
        elif self.clock[1] != minute and \
             self.notify_event('OnMinuteChange', self.get_uptime(),
                               self.surface.get_mikire(),
                               self.surface.get_kasanari(),
                               not self.passivemode and self.cantalk):
            pass
        elif self.surface_mouse_motion is not None:
            side, x, y, part = self.surface_mouse_motion
            self.notify_event('OnMouseMove', x, y, '', side, part)
            self.surface_mouse_motion = None
        elif idle > self.__surface_life > 0 and not self.passivemode:
            self.__surface_life = 0
            self.notify_event('OnSurfaceRestore',
                              self.get_surface_id(0),
                              self.get_surface_id(1))
        self.clock = (second, minute)

    reload_event = None

    def do_idle_tasks(self):
        if not self.__running:
            return False
        if self.force_quit and not self.busy() and \
           not (self.processed_script or self.processed_text):
            self.quit()
        self.request_parent('NOTIFY', 'update_working', self.get_name())
        if self.__temp_mode:
            self.process_script()
            if not self.busy() and \
               not self.script_queue and \
               not (self.processed_script or \
                    self.processed_text):
                if self.__temp_mode == 1:
                    time.sleep(1.4)
                    self.finalize()
                    self.request_parent('NOTIFY', 'close_ghost', self)
                    self.request_parent('NOTIFY', 'reset_sstp_flag')
                    return False
                else:
                    self.request_parent('NOTIFY', 'reset_sstp_flag')
                    self.leave_temp_mode()
                    return True
            else:
                return True
        if self.reload_event and not self.busy() and \
           not (self.processed_script or self.processed_text):
            self.hide_all()
            logging.info('reloading....')
            self.shiori.unload()
            self.updateman.clean_up() # Don't call before unloading SHIORI
            self.request_parent(
                'NOTIFY', 'stop_sakura', self,
                lambda a: self.request_parent(
                    'NOTIFY', 'reload_current_sakura', a),
                (self))
            self.load_settings()
            self.restart() ## FIXME
            logging.info('done.')
            self.enqueue_event(*self.reload_event)
            self.reload_event = None
        # continue network update (enqueue events)
        if self.updateman.is_active():
            self.updateman.run()
            while 1:
                event = self.updateman.get_event()
                if not event:
                    break
                if event[0] == 'OnUpdateComplete' and event[1] == 'changed':
                    self.reload_event = event
                else:
                    self.enqueue_event(*event)
        self.process_script()
        return True

    def quit(self):
        self.request_parent('NOTIFY', 'stop_sakura', self)

    ###   SCRIPT PLAYER   ###
    def start_script(self, script, origin=None):
        if not script:
            return
        self.last_script = script
        self.script_origin = origin or self.FROM_GHOST
        self.reset_script(1)
        self.__current_script = script
        if not script.rstrip().endswith(r'\e'):
            script = ''.join((script, r'\e'))
        self.processed_script = []
        self.script_position = 0
        while 1:
            try:
                self.processed_script.extend(self.script_parser.parse(script))
            except ninix.script.ParserError as e:
                logging.error('-' * 50)
                logging.error('{0}'.format(e)) # 'UTF-8'
                done, script = e
                self.processed_script.extend(done)
            else:
                break
        self.script_mode = self.BROWSE_MODE
        self.script_wait = None
        self.script_side = 0
        self.time_critical_session = 0
        self.quick_session = 0
        self.set_synchronized_session(reset=1)
        self.balloon.hide_all()
        node = self.processed_script[0]
        if node[0] == ninix.script.SCRIPT_TAG and node[1] == r'\C':
            self.processed_script.pop(0)
            self.script_position = node[-1]
        else:
            self.balloon.clear_text_all()
        self.balloon.set_balloon_default()
        self.current_time = time.localtime(time.time())
        self.reset_idle_time()
        if self.request_parent('GET', 'get_preference', 'raise_before_talk'):
            self.raise_all()

    def __yen_e(self, args):
        surface_id = self.get_surface_id(self.script_side)
        self.surface.invoke_yen_e(self.script_side, surface_id)
        self.reset_script()
        self.__balloon_life = self.BALLOON_LIFE
    
    def __yen_0(self, args):
        ##self.balloon.show(0)
        self.script_side = 0

    def __yen_1(self, args):
        ##self.balloon.show(1)
        self.script_side = 1

    def __yen_p(self, args):
        try:
            chr_id = int(args[0])
        except:
            return
        if chr_id >= 0:
            self.script_side = chr_id

    def __yen_4(self, args):
        if self.script_side == 0:
            sw, sh = self.get_surface_size(1)
            sx, sy = self.get_surface_position(1)
        elif self.script_side == 1:
            sw, sh = self.get_surface_size(0)
            sx, sy = self.get_surface_position(0)
        else:
            return
        w, h = self.get_surface_size(self.script_side)
        x, y = self.get_surface_position(self.script_side)
        left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
        if sx + sw // 2 > left + scrn_w // 2:
            new_x = min(x - scrn_w // 20, sx - scrn_w // 20)
        else:
            new_x = max(x + scrn_w // 20, sx + scrn_w // 20)
        step = -10 if x > new_x else 10
        for current_x in range(x, new_x, step):
            self.set_surface_position(self.script_side, current_x, y)
        self.set_surface_position(self.script_side, new_x, y)

    def __yen_5(self, args):
        if self.script_side == 0:
            sw, sh = self.get_surface_size(1)
            sx, sy = self.get_surface_position(1)
        elif self.script_side == 1:
            sw, sh = self.get_surface_size(0)
            sx, sy = self.get_surface_position(0)
        else:
            return
        w, h = self.get_surface_size(self.script_side)
        x, y = self.get_surface_position(self.script_side)
        left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
        if x < sx + sw // 2 < x + w or sx < x + w // 2 < sx + sw:
            return
        if sx + sw // 2 > x + w // 2:
            new_x = sx - w // 2 + 1
        else:
            new_x = sx + sw - w // 2 - 1
        new_x = max(new_x, left)
        new_x = min(new_x, left + scrn_w - w)
        step = -10 if x > new_x else 10
        for current_x in range(x, new_x, step):
            self.set_surface_position(self.script_side, current_x, y)
        self.set_surface_position(self.script_side, new_x, y)

    def __yen_s(self, args):
        surface_id = args[0]
        if surface_id == '-1':
            self.surface.hide(self.script_side)
        else:
            self.set_surface_id(self.script_side, surface_id)
            self.surface.show(self.script_side)
        if self.script_side in [0, 1] and not self.__boot[self.script_side]:
            self.__boot[self.script_side] = 1

    def __yen_b(self, args):
        if args[0] == '-1':
            self.balloon.hide(self.script_side)
        else:
            try:
                balloon_id = int(args[0]) // 2
            except ValueError:
                balloon_id = 0
            else:
                self.balloon.set_balloon(self.script_side, balloon_id)

    def __yen__b(self, args):
        try:
            filename, x, y = self.expand_meta(args[0]).split(',')
        except:
            filename, param = self.expand_meta(args[0]).split(',')
            assert param == 'inline'
            x, y = 0, 0 ## FIXME
        filename = get_normalized_path(filename)
        path = os.path.join(self.get_prefix(), b'ghost/master', filename)
        if os.path.isfile(path):
            self.balloon.append_image(self.script_side, path, x, y)
        else:
            path = b''.join((path, b'.png'))
            if os.path.isfile(path):
                self.balloon.append_image(self.script_side, path, x, y)

    def __yen_n(self, args):
        if args and self.expand_meta(args[0]) == 'half':
            self.balloon.append_text(self.script_side, '\n[half]')
        else:
            self.balloon.append_text(self.script_side, '\n')

    def __yen_c(self, args):
        self.balloon.clear_text(self.script_side)

    def __set_weight(self, value, unit):
        try:
            amount = int(value) * unit - 0.01
        except ValueError:
            amount = 0
        if amount > 0:
            self.script_wait = time.time() + amount

    def __yen_w(self, args):
        script_speed = self.request_parent(
            'GET', 'get_preference', 'script_speed')
        if not self.quick_session and script_speed >= 0:
            self.__set_weight(args[0], 0.05) # 50[ms]

    def __yen__w(self, args):
        script_speed = self.request_parent(
            'GET', 'get_preference', 'script_speed')
        if not self.quick_session and script_speed >= 0:
            self.__set_weight(args[0], 0.001) # 1[ms]

    def __yen_t(self, args):
        self.time_critical_session = not self.time_critical_session

    def __yen__q(self, args):
        self.quick_session = not self.quick_session

    def __yen__s(self, args):
        self.set_synchronized_session([int(arg) for arg in args])

    def __yen__e(self, args):
        self.balloon.hide(self.script_side)
        self.balloon.clear_text(self.script_side)

    def __yen_q(self, args):
        newline_required = 0
        if len(args) == 3: # traditional syntax
            num, link_id, text = args
            newline_required = 1
        else: # new syntax
            text, link_id = args
        text = self.expand_meta(text)
        self.balloon.append_link(self.script_side, link_id, text,
                                 newline_required)
        self.script_mode = self.SELECT_MODE

    def __yen_URL(self, args):
        text = self.expand_meta(args[0])
        if len(args) == 1:
            link = text
        else:
            link = '#cancel'
        self.balloon.append_link(self.script_side, link, text)
        for i in range(1, len(args), 2):
            link = self.expand_meta(args[i])
            text = self.expand_meta(args[i + 1])
            self.balloon.append_link(self.script_side, link, text)
        self.script_mode = self.SELECT_MODE

    def __yen__a(self, args):
        if self.anchor:
            anchor_id = self.anchor[0]
            text = self.anchor[1]
            self.balloon.append_link_out(self.script_side, anchor_id, text)
            self.anchor = None
        else:
            anchor_id = args[0]
            self.anchor = [('anchor', anchor_id), '']
            self.balloon.append_link_in(self.script_side, self.anchor[0])

    def __yen_x(self, args):
        if self.script_mode == self.BROWSE_MODE:
            if len(args) > 0 and self.expand_meta(args[0]) == 'noclear':
                self.script_mode = self.PAUSE_NOCLEAR_MODE
            else:
                self.script_mode = self.PAUSE_MODE

    def __yen_a(self, args):
        self.start_script(self.getaistringrandom())

    def __yen_i(self, args):
        try:
            actor_id = int(args[0])
        except ValueError:
            pass
        else:
            self.surface.invoke(self.script_side, actor_id)

    def __yen_j(self, args):
        jump_id = args[0]
        if self.is_URL(jump_id):
            webbrowser.open(jump_id)
        elif self.sstp_entry_db:
            self.start_script(self.sstp_entry_db.get(jump_id, r'\e'))

    def __yen_minus(self, args):
        self.quit()

    def __yen_plus(self, args):
        self.request_parent('NOTIFY', 'select_ghost', self, 1)

    def __yen__plus(self, args):
        self.request_parent('NOTIFY', 'select_ghost', self, 0)

    def __yen_m(self, args):
        self.write_sstp_handle(self.expand_meta(args[0]))

    def __yen_and(self, args):
        if self.name2codepoint is not None:
            try:
                text = chr(self.name2codepoint.get(args[0]))
            except:
                text = None
        else:
            text = None
        if text is None:
            text = '?'
        self.balloon.append_text(self.script_side, text)

    def __yen__m(self, args):
        try:
            num = int(args[0], 16)
        except ValueError:
            num = 0
        if 0x20 <= num <= 0x7e:
            text = chr(num)
        else:
            text = '?'
        self.balloon.append_text(self.script_side, text)

    def __yen__u(self, args):
        if re.match('0x[a-fA-F0-9]{4}', args[0]):
            text = eval(''.join(('"\\u', args[0][2:], '"')))
            self.balloon.append_text(self.script_side, text)
        else:
            self.balloon.append_text(self.script_side, '?')

    def __yen__v(self, args):
        if self.audio_player is None:
            return
        filename = self.expand_meta(args[0])
        filename = get_normalized_path(filename)
        path = os.path.join(self.get_prefix(), b'ghost/master', filename)
        if os.path.isfile(path):
            self.audio_player.set_state(Gst.State.NULL)
            self.audio_player.set_property(
                'uri', 'file://' + urllib.parse.quote(path))
            self.audio_loop = False
            self.audio_player.set_state(Gst.State.PLAYING)

    def __yen_8(self, args):
        if self.audio_player is None:
            return
        filename = self.expand_meta(args[0])
        filename = get_normalized_path(filename)
        basename, ext = os.path.splitext(filename)
        ext = ext.lower()
        if ext != b'.wav':
            return
        path = os.path.join(self.get_prefix(), b'ghost/master', filename)
        if os.path.isfile(path):
            self.audio_player.set_state(Gst.State.NULL)
            self.audio_player.set_property(
                'uri', 'file://' + urllib.parse.quote(path))
            self.audio_loop = False
            self.audio_player.set_state(Gst.State.PLAYING)

    def __yen__V(self, args):
        if self.audio_loop:
            return # nothing to do
        if self.audio_player.get_state(timeout=Gst.SECOND)[1] == Gst.State.PLAYING:
            self.script_mode = self.WAIT_MODE

    def __yen_exclamation(self, args): ## FIXME
        if not args:
            return
        argc = len(args)
        args = [self.expand_meta(s) for s in args]
        if args[0] == 'raise' and argc >= 2:
            self.notify_event(*args[1:10])
        elif args[0:2] == ['open', 'readme']:
            ReadmeDialog().show(self.get_name(), self.get_prefix())
        elif args[0:2] == ['open', 'browser'] and argc > 2:
            webbrowser.open(args[2])
        elif args[0:2] == ['open', 'communicatebox']:
            if not self.passivemode:
                self.balloon.open_communicatebox()
        elif args[0:2] == ['open', 'teachbox']:
            if not self.passivemode:
                self.balloon.open_teachbox()
        elif args[0:2] == ['open', 'inputbox'] and argc > 2:
            if not self.passivemode:
                if argc > 4:
                    self.balloon.open_inputbox(args[2], args[3], args[4])
                elif argc == 4:
                    self.balloon.open_inputbox(args[2], args[3])
                else:
                    self.balloon.open_inputbox(args[2])
        elif args[0:2] == ['open', 'passwordinputbox'] and argc > 2:
            if not self.passivemode:
                if argc > 4:
                    self.balloon.open_passwordinputbox(args[2], args[3], args[4])
                elif argc == 4:
                    self.balloon.open_passwordinputbox(args[2], args[3])
                else:
                    self.balloon.open_passwordinputbox(args[2])
        elif args[0:2] == ['open', 'configurationdialog']:
            self.request_parent('NOTIFY', 'edit_preferences')
        elif args[0:2] == ['close', 'inputbox'] and argc > 2:
            self.balloon.close_inputbox(args[2])
        elif args[0:2] == ['change', 'balloon'] and argc > 2:
            key = self.request_parent('GET', 'find_balloon_by_name', args[2])
            if key is not None:
                desc, balloon = self.request_parent(
                    'GET', 'get_balloon_description', key)
                self.select_balloon(key, desc, balloon)
        elif args[0:2] == ['change', 'shell'] and argc > 2:
            for key in self.shells:
                shell_name = self.shells[key].baseinfo[0]
                if shell_name == args[2]:
                    self.select_shell(key)
                    break
        elif args[0:2] == ['change', 'ghost'] and argc > 2:
            if args[2] == 'random':
                self.request_parent('NOTIFY', 'select_ghost', self, 0, 0)
            else:
                self.request_parent(
                    'NOTIFY', 'select_ghost_by_name', self, args[2], 0)
        elif args[0:2] == ['call', 'ghost'] and argc > 2:
            key = self.request_parent('GET', 'find_ghost_by_name', args[2])
            if key is not None:
                self.request_parent('NOTIFY', 'start_sakura_cb', key, self)
        elif args[0:1] == ['updatebymyself']:
            if not self.busy(check_updateman=False):
                self.__update()
        elif args[0:1] == ['vanishbymyself']:
            self.vanished = 1 ## FIXME
            if argc > 1:
                next_ghost = args[1]
            else:
                next_ghost = None
            self.vanish_by_myself(next_ghost)
        elif args[1:2] == ['repaint']:
            if args[0:1] == ['lock']:
                self.lock_repaint = 1
            elif args[0:1] == ['unlock']:
                self.lock_repaint = 0
        elif args[1:2] == ['passivemode']:
            if args[0:1] == ['enter']:
                self.passivemode = 1
            elif args[0:1] == ['leave']:
                self.passivemode = 0
        elif args[1:2] == ['collisionmode']:
            if args[0:1] == ['enter']:
                if args[2:3] == ['rect']:
                    self.request_parent(
                        'NOTIFY', 'set_collisionmode', True, rect=True)
                else:
                    self.request_parent(
                        'NOTIFY', 'set_collisionmode', True)
            elif args[0:1] == ['leave']:
                self.request_parent(
                    'NOTIFY', 'set_collisionmode', False)
        elif args[0:2] == ['set', 'alignmentondesktop'] and argc > 2:
            if args[2] == 'bottom':
                if self.synchronized_session:
                    for chr_id in self.synchronized_session:
                        self.align_bottom(chr_id)
                else:
                    self.align_bottom(self.script_side)
            elif args[2] == 'top':
                if self.synchronized_session:
                    for chr_id in self.synchronized_session:
                        self.align_top(chr_id)
                else:
                    self.align_top(self.script_side)
            elif args[2] == 'free':
                if self.synchronized_session:
                    for chr_id in self.synchronized_session:
                        self.surface.set_alignment(chr_id, 2)
                else:
                    self.surface.set_alignment(self.script_side, 2)
            elif args[2] == 'default':
                self.surface.reset_alignment()
        elif args[0:2] == ['set', 'autoscroll'] and argc > 2:
            if args[2] == 'disable':
                self.balloon.set_autoscroll(False)
            elif args[2] == 'enable':
                self.balloon.set_autoscroll(True)
            else:
                pass ## FIXME
        elif args[0:2] == ['set', 'windowstate'] and argc > 2:
            if args[2] == 'minimize':
                self.surface.window_iconify(True)
            ##elif args[2] == '!minimize':
            ##    self.surface.window_iconify(False)
            elif args[2] == 'stayontop':
                self.surface.window_stayontop(True)
            elif args[2] == '!stayontop':
                self.surface.window_stayontop(False)
        elif args[0:2] == ['set', 'trayicon'] and argc > 2:
            path = os.path.join(self.get_prefix(), os.fsencode(args[2]))
            if os.path.exists(path):
                self.status_icon.set_from_file(os.fsdecode(path)) # XXX
            if argc > 3:
                text = args[3]
                if text:
                    self.status_icon.set_has_tooltip(True)
                    self.status_icon.set_tooltip_text(text)
                else:
                    self.status_icon.set_has_tooltip(False)
            else:
                self.status_icon.set_has_tooltip(False)
            self.status_icon.set_visible(True)
        elif args[0:2] == ['set', 'wallpaper'] and argc > 2:
            path = os.path.join(self.get_prefix(), os.fsencode(args[2]))
            opt = None
            if argc > 3:
                # possible picture_options value:
                # "none", "wallpaper", "centered", "scaled", "stretched"
                options = {
                    'center': 'centered',
                    'tile': 'wallpaper',
                    'stretch': 'stretched'
                    }
                if args[3] not in options:
                    opt = None
                else:
                    opt = options[args[3]]
            if opt is None:
                opt = 'centered' # default
            if os.path.exists(path):
                if os.name == 'posix':
                    # for GNOME3
                    gsettings = Gio.Settings.new(
                        'org.gnome.desktop.background')
                    gsettings.set_string('picture-uri',
                                         ''.join(('file://', os.fsdecode(path))))
                    gsettings.set_string('picture-options', opt)
                else:
                    pass # not implemented yet
        elif args[0:2] == ['set', 'otherghosttalk'] and argc > 2:
            if args[2] == 'true':
                self.__listening['OnOtherGhostTalk'] = True
            elif args[2] == 'false':
                self.__listening['OnOtherGhostTalk'] = False
            else:
                pass ## FIXME
        elif args[0:2] == ['set', 'othersurfacechange'] and argc > 2:
            if args[2] == 'true':
                self.__listening['OnOtherSurfaceChange'] = True
            elif args[2] == 'false':
                self.__listening['OnOtherSurfaceChange'] = False
            else:
                pass ## FIXME
        elif args[0:2] == ['set', 'balloonoffset'] and argc > 3:
            try:
                x = int(args[2])
                y = int(args[3])
            except:
                pass
            else:
                self.surface.set_balloon_offset(self.script_side, (x, y))
        elif args[0] == 'sound' and argc > 1:
            command = args[1]
            if self.audio_player is None:
                return
            if command == 'stop':
                self.audio_player.set_state(Gst.State.NULL)
                self.audio_loop = False
            elif command == 'play' and argc > 2:
                filename = args[2]
                filename = get_normalized_path(filename)
                path = os.path.join(self.get_prefix(),
                                    b'ghost/master', filename)
                if os.path.isfile(path):
                    self.audio_player.set_state(Gst.State.NULL)
                    self.audio_player.set_property(
                        'uri', 'file://' + urllib.parse.quote(path))
                    self.audio_loop = False
                    self.audio_player.set_state(Gst.State.PLAYING)
            elif command == 'cdplay' and argc > 2:
                self.audio_player.set_state(Gst.State.NULL)
                try:
                    track = int(args[2])
                except:
                    return
                self.audio_player.set_property(
                    'uri', 'cdda://{}'.format(track))
                self.audio_loop = False
                self.audio_player.set_state(Gst.State.PLAYING)
            elif command == 'loop' and argc > 2:
                filename = args[2]
                filename = get_normalized_path(filename)
                path = os.path.join(self.get_prefix(),
                                    b'ghost/master', filename)
                if os.path.isfile(path):
                    self.audio_player.set_state(Gst.State.NULL)
                    self.audio_player.set_property(
                        'uri', 'file://' + urllib.parse.quote(path))
                    self.audio_loop = True
                    self.audio_player.set_state(Gst.State.PLAYING)
            elif command == 'wait':
                if self.audio_loop:
                    return # nothing to do
                if self.audio_player.get_state(timeout=Gst.SECOND)[1] == Gst.State.PLAYING:
                    self.script_mode = self.WAIT_MODE
            elif command == 'pause':
                if self.audio_player.get_state(timeout=Gst.SECOND)[1] == Gst.State.PLAYING:
                    self.audio_player.set_state(Gst.State.PAUSED)
            elif command == 'resume':
                if self.audio_player.get_state(timeout=Gst.SECOND)[1] == Gst.State.PAUSED:
                    self.audio_player.set_state(Gst.State.PLAYING)
            else:
                pass ## FIXME
        elif args[0] == '*':
            self.balloon.append_sstp_marker(self.script_side)
        elif args[0] == 'quicksession' and argc > 1:
            if args[1] == 'true':
                self.quick_session = 1
            elif args[1] == 'false':
                self.quick_session = 0
            else:
                pass ## FIXME
        elif args[0] == 'bind' and argc > 2:
            category = args[1]
            name = args[2]
            if argc < 4:
                flag = 'toggle'
            else:
                flag = args[3]
            bind = self.surface.window[self.script_side].bind # XXX
            for key in bind:
                group = bind[key][0].split(',')
                if category != group[0]:
                    continue
                if name and name != group[1]:
                    continue
                if flag in ['true', '1']:
                    if bind[key][1]:
                        continue
                elif flag in ['false', '0']:
                    if not bind[key][1]:
                        continue
                else: # 'toggle'
                    pass
                self.surface.toggle_bind((self.script_side, key))
        else:
            pass ## FIXME

    def __yen___c(self, args):
        self.balloon.open_communicatebox()

    def __yen___t(self, args): 
        self.balloon.open_teachbox()

    def __yen_v(self, args):
        self.raise_surface(self.script_side)

    def __yen_f(self, args):
        if len(args) != 2: ## FIXME
            return
        tag = None
        if args[0] == 'sup':
            tag = '<sup>' if args[1] == 'true' else '</sup>'
        elif args[0] == 'sub':
            tag = '<sub>' if args[1] == 'true' else '</sub>'
        elif args[0] == 'strike':
            tag = '<s>' if args[1] in ['true', '1', 1] else '</s>'
        elif args[0] == 'underline':
            tag = '<u>' if args[1] in ['true', '1', 1] else '</u>'
        else:
            pass ## FIXME
        if tag is not None:
            self.balloon.append_meta(self.script_side, tag)

    __script_tag = {
        r'\e': __yen_e,
        r'\y': __yen_e,
        r'\z': __yen_e,
        r'\0': __yen_0,
        r'\h': __yen_0,
        r'\1': __yen_1,
        r'\u': __yen_1,
        r'\p': __yen_p,
        r'\4': __yen_4,
        r'\5': __yen_5,
        r'\s': __yen_s,
        r'\b': __yen_b,
        r'\_b': __yen__b,
        r'\n': __yen_n,
        r'\c': __yen_c,
        r'\w': __yen_w,
        r'\_w': __yen__w,
        r'\t': __yen_t,
        r'\_q': __yen__q,
        r'\_s': __yen__s,
        r'\_e': __yen__e,
        r'\q': __yen_q,
        r'\URL': __yen_URL,
        r'\_a': __yen__a,
        r'\x': __yen_x,
        r'\a': __yen_a, # Obsolete: only for old SHIORI
        r'\i': __yen_i,
        r'\j': __yen_j,
        r'\-': __yen_minus,
        r'\+': __yen_plus,
        r'\_+': __yen__plus,
        r'\m': __yen_m,
        r'\&': __yen_and,
        r'\_m': __yen__m,
        r'\_u': __yen__u,
        r'\_v': __yen__v,
        r'\8': __yen_8,
        r'\_V': __yen__V,
        r'\!': __yen_exclamation,
        r'\__c': __yen___c,
        r'\__t': __yen___t, 
        r'\v': __yen_v,
        r'\f': __yen_f,
        r'\C': lambda *a: None, # dummy
        }

    def interpret_script(self):
        if self.script_wait is not None:
            if time.time() < self.script_wait:
                return
            self.script_wait = None
        if self.processed_text:
            self.balloon.show(self.script_side)
            self.balloon.append_text(self.script_side, self.processed_text[0])
            self.processed_text = self.processed_text[1:]
            surface_id = self.get_surface_id(self.script_side)
            count = self.balloon.get_text_count(self.script_side)
            if self.surface.invoke_talk(self.script_side, surface_id, count):
                self.balloon.reset_text_count(self.script_side)
            script_speed = self.request_parent(
                'GET', 'get_preference', 'script_speed')
            if script_speed > 0:
                self.script_wait = time.time() + script_speed * 0.02
            return
        node = self.processed_script.pop(0)
        self.script_position = node[-1]
        if node[0] == ninix.script.SCRIPT_TAG:
            name, args = node[1], node[2:-1]
            if name in self.__script_tag:
                self.__script_tag[name](self, args)
            else:
                pass ## FIMXE
        elif node[0] == ninix.script.SCRIPT_TEXT:
            text = self.expand_meta(node[1])
            if self.anchor:
                self.anchor[1] = ''.join((self.anchor[1], text))
            script_speed = self.request_parent(
                'GET', 'get_preference', 'script_speed')
            if not self.quick_session and script_speed >= 0:
                self.processed_text = text
            else:
                self.balloon.append_text(self.script_side, text)

    def reset_script(self, reset_all=0):
        if reset_all:
            self.script_mode = self.BROWSE_MODE
            if self.script_finally:
                for proc in self.script_finally:
                    proc(flag_break=True)
                self.script_finally = []
            self.script_post_proc = []
            self.__current_script = ''
        self.processed_script = None
        self.processed_text = ''
        self.script_position = 0
        self.time_critical_session = 0
        self.quick_session = 0
        self.set_synchronized_session(reset=1)
        self.balloon.set_autoscroll(True)
        self.reset_idle_time()

    def set_synchronized_session(self, list=[], reset=0):
        if reset:
            self.synchronized_session = []
        elif not list:
            if self.synchronized_session:
                self.synchronized_session = []
            else:
                self.synchronized_session = [0, 1]
        else:
            self.synchronized_session = list
        self.balloon.synchronize(self.synchronized_session)

    def expand_meta(self, text_node):
        buf = []
        for chunk in text_node:
            if chunk[0] == ninix.script.TEXT_STRING:
                buf.append(chunk[1])
            elif chunk[1] == '%month':
                buf.append(str(self.current_time[1]))
            elif chunk[1] == '%day':
                buf.append(str(self.current_time[2]))
            elif chunk[1] == '%hour':
                buf.append(str(self.current_time[3]))
            elif chunk[1] == '%minute':
                buf.append(str(self.current_time[4]))
            elif chunk[1] == '%second':
                buf.append(str(self.current_time[5]))
            elif chunk[1] in ['%username', '%c']:
                buf.append(self.get_username())
            elif chunk[1] == '%selfname':
                buf.append(self.get_selfname())
            elif chunk[1] == '%selfname2':
                buf.append(self.get_selfname2())
            elif chunk[1] == '%keroname':
                buf.append(self.get_keroname())
            elif chunk[1] == '%friendname':
                buf.append(self.get_friendname())
            elif chunk[1] == '%screenwidth':
                left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
                buf.append(str(scrn_w))
            elif chunk[1] == '%screenheight':
                left, top, scrn_w, scrn_h = ninix.pix.get_workarea()
                buf.append(str(scrn_h))
            elif chunk[1] == '%et':
                buf.append('{0:d}万年'.format(self.current_time[7]))
            elif chunk[1] == '%wronghour':
                wrongtime = time.time() + random.choice((-2, -1, 1, 2)) * 3600
                buf.append(str(time.localtime(wrongtime)[3]))
            elif chunk[1] == '%exh':
                buf.append(str(self.get_uptime()))
            elif chunk[1] in ['%ms', '%mz', '%ml', '%mc', '%mh', \
                              '%mt', '%me', '%mp', '%m?']:
                buf.append(
                    self.getword(''.join(('\\', chunk[1][1:]))))
            elif chunk[1] == '%dms':
                buf.append(self.getdms())
            else: # %c, %songname
                buf.append(chunk[1])
        return ''.join(buf)

    ###   SEND SSTP/1.3   ###
    def _send_sstp_handle(self, data):
        r, w, e = select.select([], [self.sstp_handle], [], 0)
        if not w:
            return
        try:
            self.sstp_handle.send(''.join((data, '\n')))
        except socket.error:
            pass

    def write_sstp_handle(self, data):
        if self.sstp_handle is None:
            return
        self._send_sstp_handle(''.join(('+', data)))
        ##logging.debug('write_sstp_handle({0})'.format(repr(data)))

    def close_sstp_handle(self):
        if self.sstp_handle is None:
            return
        self._send_sstp_handle('-')
        ##logging.debug('close_sstp_handle()')
        try:
            self.sstp_handle.close()
        except socket.error:
            pass
        self.sstp_handle = None

    def close(self, reason='user'):
        if self.busy():
            if reason == 'user':
                Gdk.beep() ## FIXME
                return
            else: # shutdown
                if self.updateman.is_active():
                    self.updateman.interrupt()
        self.reset_script(1)
        self.enqueue_event('OnClose', reason)

    def about(self):
        if self.busy():
            Gdk.beep() ## FIXME
            return
        self.start_script(ninix.version.VERSION_INFO)
        self.balloon.hide_sstp_message()

    def __update(self):
        if self.updateman.is_active():
            return
        homeurl = self.getstring('homeurl')
        if not homeurl:
            self.start_script(
                ''.join((r'\t\h\s[0]',
                         _("I'm afraid I don't have Network Update yet."),
                         r'\e')))
            self.balloon.hide_sstp_message()
            return
        ghostdir = self.get_prefix()
        logging.info('homeurl = {0}'.format(homeurl))
        logging.info('ghostdir = {0}'.format(ghostdir))
        self.updateman.start(homeurl, ghostdir)

    def network_update(self):
        if self.busy():
            Gdk.beep() ## FIXME
            return
        self.__update()


class VanishDialog:

    def __init__(self):
        self.request_parent = lambda *a: None # dummy
        self.dialog = Gtk.Dialog()
        self.dialog.connect('delete_event', lambda *a: True) # XXX
        self.dialog.set_title('Vanish')
        self.dialog.set_modal(True)
        self.dialog.set_resizable(False)
        self.dialog.set_position(Gtk.WindowPosition.CENTER)
        self.label = Gtk.Label(label=_('Vanish'))
        content_area = self.dialog.get_content_area()
        content_area.add(self.label)
        self.label.show()
        self.dialog.add_button(Gtk.STOCK_YES, Gtk.ResponseType.YES)
        self.dialog.add_button(Gtk.STOCK_NO, Gtk.ResponseType.NO)
        self.dialog.connect('response', self.response)

    def set_responsible(self, request_method):
        self.request_parent = request_method

    def set_message(self, message):
        self.label.set_text(message)

    def show(self):
        self.dialog.show()

    def ok(self):
        self.dialog.hide()
        self.request_parent('NOTIFY', 'notify_vanish_selected')
        return True

    def cancel(self):
        self.dialog.hide()
        self.request_parent('NOTIFY', 'notify_vanish_canceled')
        return True

    def response(self, widget, response):
        func = {Gtk.ResponseType.YES: self.ok,
                Gtk.ResponseType.NO: self.cancel,
                Gtk.ResponseType.DELETE_EVENT: self.cancel,
                }
        func[response]()
        return True


class ReadmeDialog:

    def __init__(self):
        self.request_parent = lambda *a: None # dummy
        self.dialog = Gtk.Dialog()
        self.dialog.connect('delete_event', lambda *a: True) # XXX
        self.dialog.set_title('Readme.txt')
        self.dialog.set_modal(False)
        self.dialog.set_resizable(False)
        self.dialog.set_position(Gtk.WindowPosition.CENTER)
        self.label = Gtk.Label()
        self.label.show()
        self.textview = Gtk.TextView()
        self.textview.set_editable(False)
        self.textview.set_cursor_visible(False)
        self.textview.show()
        scroll = Gtk.ScrolledWindow(None, None)
        scroll.set_policy(Gtk.PolicyType.AUTOMATIC, Gtk.PolicyType.AUTOMATIC)
        scroll.add(self.textview)
        scroll.show()
        vbox = Gtk.Box(orientation=Gtk.Orientation.VERTICAL)
        vbox.set_size_request(720, 460)
        vbox.show()
        vbox.pack_start(self.label, False, True, 0)
        vbox.pack_start(scroll, True, True, 0)
        content_area = self.dialog.get_content_area()
        content_area.add(vbox)
        self.dialog.add_button(Gtk.STOCK_CLOSE, Gtk.ResponseType.CLOSE)
        self.dialog.connect('response', self.response)

    def set_responsible(self, request_method):
        self.request_parent = request_method

    def show(self, name, base_path):
        self.label.set_text(name)
        path = os.path.join(base_path, b'readme.txt')
        if os.path.exists(path):
            with open(path, encoding='CP932') as f: # XXX
                text = f.read()
                self.textview.get_buffer().set_text(text)
                self.dialog.show()

    def response(self, widget, response):
        func = {Gtk.ResponseType.CLOSE: widget.hide,
                Gtk.ResponseType.DELETE_EVENT: widget.hide,
                }
        func[response]()
        return True

def test():
    pass

if __name__ == '__main__':
    test()
