/**
 * ======================================================================
 * LS-Y201 device driver (Version 0.0.1)
 * Reference documents:
 * LinkSprite JPEG Color Camera Serial UART Interface January 2010
 * ======================================================================
 * Copyright (c) 2010 Shinichiro Nakamura (CuBeatSystems)
 *
 * Permission is hereby granted, free of charge, to any person obtaining
 * a copy of this software and associated documentation files
 * (the "Software"), to deal in the Software without restriction,
 * including without limitation the rights to use, copy, modify, merge,
 * publish, distribute, sublicense, and/or sell copies of the Software,
 * and to permit persons to whom the Software is furnished to do so,
 * subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF
 * MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT.
 * IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY
 * CLAIM, DAMAGES OR OTHER LIABILITY, WHETHER IN AN ACTION OF CONTRACT,
 * TORT OR OTHERWISE, ARISING FROM, OUT OF OR IN CONNECTION WITH THE
 * SOFTWARE OR THE USE OR OTHER DEALINGS IN THE SOFTWARE.
 * ======================================================================
 */

#ifndef LS_Y201_H
#define LS_Y201_H

/**
 * Error code.
 */
typedef enum {
    NoError = 0,
    UnexpectedReply,
    SendError,
    RecvError,
    InvalidArguments
} ErrorCode;

/**
 * Image size.
 */
typedef enum {
    ImageSize160x120,   /**< 160x120. */
    ImageSize320x280,   /**< 320x280. */
    ImageSize640x480    /**< 640x480. */
} ImageSize;

/**
 * @brief LS-Y201 handler.
 */
typedef struct {
    int (*write)(unsigned char *buf, int len);
    int (*read)(unsigned char *buf, int len);
    char version[128];
} ls_y201_t;

/**
 * @brief Setup the handler.
 * @param P Handler.
 * @param WRITE Write function.
 * @param READ Read function.
 */
#define LS_Y201_SETUP(P, WRITE, READ)  do { (P)->write = WRITE; (P)->read = READ; } while (0)

#define LS_Y201_VERSION(P)  (&((P)->version[0]))

/**
 * Reset module.
 *
 * @return Error code.
 */
ErrorCode ls_y201_reset(ls_y201_t *p);

/**
 * Set image size.
 *
 * @param is Image size.
 * @return Error code.
 */
ErrorCode ls_y201_set_image_size(ls_y201_t *p, ImageSize is);

/**
 * Take picture.
 *
 * @return Error code.
 */
ErrorCode ls_y201_take_picture(ls_y201_t *p);

/**
 * Read jpeg file size.
 *
 * @param fileSize File size.
 * @return Error code.
 */
ErrorCode ls_y201_read_jpeg_file_size(ls_y201_t *p, int *fileSize);

/**
 * Read jpeg file content.
 *
 * @param func A pointer to a call back function.
 * @return Error code.
 */
ErrorCode ls_y201_read_jpeg_file_content(
        ls_y201_t *p,
        int (*func)(
            const int done,
            const int total,
            const unsigned char *buf,
            const int siz));

/**
 * Stop taking pictures.
 *
 * @return Error code.
 */
ErrorCode ls_y201_stop_taking_pictures(ls_y201_t *p);

#endif

