#include "defines.h"
#include "kozos.h"
#include "intr.h"
#include "interrupt.h"
#include "syscall.h"
#include "memory.h"
#include "lib.h"

#define THREAD_NUM 12
#define PRIORITY_NUM 16
#define THREAD_NAME_SIZE 15

/* åɡƥ */
typedef struct _kz_context {
  uint32 sp; /* åݥ */
} kz_context;

/* ȥ롦֥å(TCB) */
typedef struct _kz_thread {
  struct _kz_thread *next;
  char name[THREAD_NAME_SIZE + 1]; /* å̾ */
  int priority;   /* ͥ */
  char *stack;    /* å */
  uint32 flags;   /* Ƽե饰 */
#define KZ_THREAD_FLAG_READY (1 << 0)

  struct { /* åɤΥȡå(thread_init())Ϥѥ᡼ */
    kz_func_t func; /* åɤΥᥤؿ */
    int argc;       /* åɤΥᥤؿϤ argc */
    char **argv;    /* åɤΥᥤؿϤ argv */
  } init;

  struct { /* ƥࡦѥХåե */
    kz_syscall_type_t type;
    kz_syscall_param_t *param;
  } syscall;

  kz_context context; /* ƥȾ */
} kz_thread;

/* åХåե */
typedef struct _kz_msgbuf {
  struct _kz_msgbuf *next;
  kz_thread *sender; /* åå */
  struct { /* åΥѥ᡼¸ΰ */
    int size;
    char *p;
  } param;
} kz_msgbuf;

/* åܥå */
typedef struct _kz_msgbox {
  kz_thread *receiver; /* Ԥ֤Υå */
  kz_msgbuf *head;
  kz_msgbuf *tail;

  /*
   * H816ӥåCPUʤΤǡ32ӥåФƤξ軻̿᤬̵ä
   * ¤ΤΥγˤʤäƤʤȡ¤ΤΥǥå
   * ׻Ǿ軻Ȥơ___mulsi3̵פʤɤΥ󥯡顼ˤʤ礬
   * 롥(γʤХեȱ黻ѤΤϽФʤ)
   * кȤơγˤʤ褦˥ߡФĴ롥
   * ¾¤ΤƱͤΥ顼ФˤϡƱͤн򤹤뤳ȡ
   */
  long dummy[1];
} kz_msgbox;

/* åɤΥǥ塼 */
static struct {
  kz_thread *head;
  kz_thread *tail;
} readyque[PRIORITY_NUM];

static kz_thread *current; /* ȡå */
static kz_thread threads[THREAD_NUM]; /* ȥ롦֥å */
static kz_handler_t handlers[SOFTVEC_TYPE_NUM]; /* ߥϥɥ */
static kz_msgbox msgboxes[MSGBOX_ID_NUM]; /* åܥå */

void dispatch(kz_context *context);

/* ȡåɤǥ塼ȴФ */
static int getcurrent(void)
{
  if (current == NULL) {
    return -1;
  }
  if (!(current->flags & KZ_THREAD_FLAG_READY)) {
    /* Ǥ̵̵ */
    return 1;
  }

  /* ȡåɤɬƬˤϤʤΤǡƬȴФ */
  readyque[current->priority].head = current->next;
  if (readyque[current->priority].head == NULL) {
    readyque[current->priority].tail = NULL;
  }
  current->flags &= ~KZ_THREAD_FLAG_READY;
  current->next = NULL;

  return 0;
}

/* ȡåɤǥ塼˷Ҥ */
static int putcurrent(void)
{
  if (current == NULL) {
    return -1;
  }
  if (current->flags & KZ_THREAD_FLAG_READY) {
    /* Ǥ̵ͭ */
    return 1;
  }

  /* ǥ塼³ */
  if (readyque[current->priority].tail) {
    readyque[current->priority].tail->next = current;
  } else {
    readyque[current->priority].head = current;
  }
  readyque[current->priority].tail = current;
  current->flags |= KZ_THREAD_FLAG_READY;

  return 0;
}

static void thread_end(void)
{
  kz_exit();
}

/* åɤΥȡå */
static void thread_init(kz_thread *thp)
{
  /* åɤΥᥤؿƤӽФ */
  thp->init.func(thp->init.argc, thp->init.argv);
  thread_end();
}

/* ƥࡦν(kz_run():åɤεư) */
static kz_thread_id_t thread_run(kz_func_t func, char *name, int priority,
				 int stacksize, int argc, char *argv[])
{
  int i;
  kz_thread *thp;
  uint32 *sp;
  extern char userstack; /* 󥫡ץȤ륹åΰ */
  static char *thread_stack = &userstack;

  /* Ƥ륿ȥ롦֥å򸡺 */
  for (i = 0; i < THREAD_NUM; i++) {
    thp = &threads[i];
    if (!thp->init.func) /* Ĥä */
      break;
  }
  if (i == THREAD_NUM) /* Ĥʤä */
    return -1;

  memset(thp, 0, sizeof(*thp));

  /* ȥ롦֥å(TCB) */
  strcpy(thp->name, name);
  thp->next     = NULL;
  thp->priority = priority;
  thp->flags    = 0;

  thp->init.func = func;
  thp->init.argc = argc;
  thp->init.argv = argv;

  /* åΰ */
  memset(thread_stack, 0, stacksize);
  thread_stack += stacksize;

  thp->stack = thread_stack; /* å */

  /* åν */
  sp = (uint32 *)thp->stack;
  *(--sp) = (uint32)thread_end;

  /*
   * ץࡦ󥿤ꤹ롥
   * åɤͥ٤ξˤϡ߶ػߥåɤȤ롥
   */
  *(--sp) = (uint32)thread_init | ((uint32)(priority ? 0 : 0xc0) << 24);

  *(--sp) = 0; /* ER6 */
  *(--sp) = 0; /* ER5 */
  *(--sp) = 0; /* ER4 */
  *(--sp) = 0; /* ER3 */
  *(--sp) = 0; /* ER2 */
  *(--sp) = 0; /* ER1 */

  /* åɤΥȡå(thread_init())Ϥ */
  *(--sp) = (uint32)thp;  /* ER0 */

  /* åɤΥƥȤ */
  thp->context.sp = (uint32)sp;

  /* ƥࡦƤӽФåɤǥ塼᤹ */
  putcurrent();

  /* åɤ򡤥ǥ塼³ */
  current = thp;
  putcurrent();

  return (kz_thread_id_t)current;
}

/* ƥࡦν(kz_exit():åɤνλ) */
static int thread_exit(void)
{
  /*
   * ʤ饹åƺѤǤ褦ˤ٤ά
   * Τᡤåɤˤõ褦ʤȤϸǤǤʤ
   */
  puts(current->name);
  puts(" EXIT.\n");
  memset(current, 0, sizeof(*current));
  return 0;
}

/* ƥࡦν(kz_wait():åɤμ¹Ը) */
static int thread_wait(void)
{
  putcurrent();
  return 0;
}

/* ƥࡦν(kz_sleep():åɤΥ꡼) */
static int thread_sleep(void)
{
  return 0;
}

/* ƥࡦν(kz_wakeup():åɤΥå) */
static int thread_wakeup(kz_thread_id_t id)
{
  /* åפƤӽФåɤǥ塼᤹ */
  putcurrent();

  /* ꤵ줿åɤǥ塼³ƥåפ */
  current = (kz_thread *)id;
  putcurrent();

  return 0;
}

/* ƥࡦν(kz_getid():åID) */
static kz_thread_id_t thread_getid(void)
{
  putcurrent();
  return (kz_thread_id_t)current;
}

/* ƥࡦν(kz_chpri():åɤͥѹ) */
static int thread_chpri(int priority)
{
  int old = current->priority;
  if (priority >= 0)
    current->priority = priority; /* ͥѹ */
  putcurrent(); /* ͥ٤Υǥ塼˷Ҥľ */
  return old;
}

/* ƥࡦν(kz_kmalloc():ưŪ) */
static void *thread_kmalloc(int size)
{
  putcurrent();
  return kzmem_alloc(size);
}

/* ƥࡦν(kz_kfree():) */
static int thread_kmfree(char *p)
{
  kzmem_free(p);
  putcurrent();
  return 0;
}

/* å */
static void sendmsg(kz_msgbox *mboxp, kz_thread *thp, int size, char *p)
{
  kz_msgbuf *mp;

  /* åХåեκ */
  mp = (kz_msgbuf *)kzmem_alloc(sizeof(*mp));
  if (mp == NULL)
    kz_sysdown();
  mp->next       = NULL;
  mp->sender     = thp;
  mp->param.size = size;
  mp->param.p    = p;

  /* åܥå˥å³ */
  if (mboxp->tail) {
    mboxp->tail->next = mp;
  } else {
    mboxp->head = mp;
  }
  mboxp->tail = mp;
}

/* åμ */
static void recvmsg(kz_msgbox *mboxp)
{
  kz_msgbuf *mp;
  kz_syscall_param_t *p;

  /* åܥåƬˤåȴФ */
  mp = mboxp->head;
  mboxp->head = mp->next;
  if (mboxp->head == NULL)
    mboxp->tail = NULL;
  mp->next = NULL;

  /* å륹åɤ֤ͤꤹ */
  p = mboxp->receiver->syscall.param;
  p->un.recv.ret = (kz_thread_id_t)mp->sender;
  if (p->un.recv.sizep)
    *(p->un.recv.sizep) = mp->param.size;
  if (p->un.recv.pp)
    *(p->un.recv.pp) = mp->param.p;

  /* ԤåɤϤʤʤäΤǡNULL᤹ */
  mboxp->receiver = NULL;

  /* åХåեβ */
  kzmem_free(mp);
}

/* ƥࡦν(kz_send():å) */
static int thread_send(kz_msgbox_id_t id, int size, char *p)
{
  kz_msgbox *mboxp = &msgboxes[id];

  putcurrent();
  sendmsg(mboxp, current, size, p); /* å */

  /* Ԥåɤ¸ߤƤˤϼԤ */
  if (mboxp->receiver) {
    current = mboxp->receiver; /* Ԥå */
    recvmsg(mboxp); /* åμ */
    putcurrent(); /* ˤưǽˤʤäΤǡ֥å */
  }

  return size;
}

/* ƥࡦν(kz_recv():å) */
static kz_thread_id_t thread_recv(kz_msgbox_id_t id, int *sizep, char **pp)
{
  kz_msgbox *mboxp = &msgboxes[id];

  if (mboxp->receiver) /* ¾ΥåɤǤ˼ԤƤ */
    kz_sysdown();

  mboxp->receiver = current; /* Ԥåɤ */

  if (mboxp->head == NULL) {
    /*
     * åܥå˥å̵Τǡåɤ
     * ꡼פ롥(ƥࡦ뤬֥å)
     */
    return -1;
  }

  recvmsg(mboxp); /* åμ */
  putcurrent(); /* åǤΤǡǥ֤ˤ */

  return current->syscall.param->un.recv.ret;
}

/* ƥࡦν(kz_setintr():ߥϥɥϿ) */
static int thread_setintr(softvec_type_t type, kz_handler_t handler)
{
  static void thread_intr(softvec_type_t type, unsigned long sp);

  /*
   * ߤդ뤿ˡեȥߥ٥
   * OSγ߽ȤʤؿϿ롥
   */
  softvec_setintr(type, thread_intr);

  handlers[type] = handler; /* OS¦ƤӽФߥϥɥϿ */
  putcurrent();

  return 0;
}

static void call_functions(kz_syscall_type_t type, kz_syscall_param_t *p)
{
  /* ƥࡦμ¹current񤭴Τ */
  switch (type) {
  case KZ_SYSCALL_TYPE_RUN: /* kz_run() */
    p->un.run.ret = thread_run(p->un.run.func, p->un.run.name,
			       p->un.run.priority, p->un.run.stacksize,
			       p->un.run.argc, p->un.run.argv);
    break;
  case KZ_SYSCALL_TYPE_EXIT: /* kz_exit() */
    /* TCBõΤǡͤ񤭹ǤϤʤ */
    thread_exit();
    break;
  case KZ_SYSCALL_TYPE_WAIT: /* kz_wait() */
    p->un.wait.ret = thread_wait();
    break;
  case KZ_SYSCALL_TYPE_SLEEP: /* kz_sleep() */
    p->un.sleep.ret = thread_sleep();
    break;
  case KZ_SYSCALL_TYPE_WAKEUP: /* kz_wakeup() */
    p->un.wakeup.ret = thread_wakeup(p->un.wakeup.id);
    break;
  case KZ_SYSCALL_TYPE_GETID: /* kz_getid() */
    p->un.getid.ret = thread_getid();
    break;
  case KZ_SYSCALL_TYPE_CHPRI: /* kz_chpri() */
    p->un.chpri.ret = thread_chpri(p->un.chpri.priority);
    break;
  case KZ_SYSCALL_TYPE_KMALLOC: /* kz_kmalloc() */
    p->un.kmalloc.ret = thread_kmalloc(p->un.kmalloc.size);
    break;
  case KZ_SYSCALL_TYPE_KMFREE: /* kz_kmfree() */
    p->un.kmfree.ret = thread_kmfree(p->un.kmfree.p);
    break;
  case KZ_SYSCALL_TYPE_SEND: /* kz_send() */
    p->un.send.ret = thread_send(p->un.send.id,
				 p->un.send.size, p->un.send.p);
    break;
  case KZ_SYSCALL_TYPE_RECV: /* kz_recv() */
    p->un.recv.ret = thread_recv(p->un.recv.id,
				 p->un.recv.sizep, p->un.recv.pp);
    break;
  case KZ_SYSCALL_TYPE_SETINTR: /* kz_setintr() */
    p->un.setintr.ret = thread_setintr(p->un.setintr.type,
				       p->un.setintr.handler);
    break;
  default:
    break;
  }
}

/* ƥࡦν */
static void syscall_proc(kz_syscall_type_t type, kz_syscall_param_t *p)
{
  /*
   * ƥࡦƤӽФåɤǥ塼
   * ֤ǽؿƤӽФΤ᥷ƥࡦ
   * ƤӽФåɤ򤽤Τޤư³ˤϡ
   * ؿ putcurrent() Ԥɬפ롥
   */
  getcurrent();
  call_functions(type, p);
}

/* ӥν */
static void srvcall_proc(kz_syscall_type_t type, kz_syscall_param_t *p)
{
  /*
   * ƥࡦȥӥνؿǡ
   * ƥࡦμ¹ԤåID뤿 current 
   * ȤƤʬ(Ȥ thread_send() ʤ)
   * current ĤäƤȸư뤿 NULL ꤹ롥
   * ӥ thread_intrvec() γߥϥɥƤӽФ
   * ĹǸƤФƤϤǤʤΤǡƤӽФ thread_intrvec() 
   * 塼󥰽Ԥ졤current Ϻꤵ롥
   */
  current = NULL;
  call_functions(type, p);
}

/* åɤΥ塼 */
static void schedule(void)
{
  int i;

  /*
   * ̤ͥι⤤(ͥ٤οͤξ)˥ǥ塼򸫤ơ
   * ưǽʥåɤ򸡺롥
   */
  for (i = 0; i < PRIORITY_NUM; i++) {
    if (readyque[i].head) /* Ĥä */
      break;
  }
  if (i == PRIORITY_NUM) /* Ĥʤä */
    kz_sysdown();

  current = readyque[i].head; /* ȡåɤꤹ */
}

static void syscall_intr(void)
{
  syscall_proc(current->syscall.type, current->syscall.param);
}

static void softerr_intr(void)
{
  puts(current->name);
  puts(" DOWN.\n");
  getcurrent(); /* ǥ塼鳰 */
  thread_exit(); /* åɽλ */
}

/* ߽ؿ */
static void thread_intr(softvec_type_t type, unsigned long sp)
{
  /* ȡåɤΥƥȤ¸ */
  current->context.sp = sp;

  /*
   * ߤȤν¹Ԥ롥
   * SOFTVEC_TYPE_SYSCALL, SOFTVEC_TYPE_SOFTERR ξ
   * syscall_intr(), softerr_intr() ϥɥϿƤΤǡ
   * 餬¹Ԥ롥
   * ʳξϡkz_setintr()ˤäƥ桼Ͽ줿ϥɥ餬
   * ¹Ԥ롥
   */
  if (handlers[type])
    handlers[type]();

  schedule(); /* åɤΥ塼 */

  /*
   * åɤΥǥѥå
   * (dispatch()ؿΤstartup.sˤꡤ֥ǵҤƤ)
   */
  dispatch(&current->context);
  /* ˤ֤äƤʤ */
}

void kz_start(kz_func_t func, char *name, int priority, int stacksize,
	      int argc, char *argv[])
{
  kzmem_init(); /* ưŪν */

  /*
   * ʹߤǸƤӽФåɴϢΥ饤֥ؿ current 
   * Ƥ礬Τǡcurrent  NULL ˽Ƥ
   */
  current = NULL;

  memset(readyque, 0, sizeof(readyque));
  memset(threads,  0, sizeof(threads));
  memset(handlers, 0, sizeof(handlers));
  memset(msgboxes, 0, sizeof(msgboxes));

  /* ߥϥɥϿ */
  thread_setintr(SOFTVEC_TYPE_SYSCALL, syscall_intr); /* ƥࡦ */
  thread_setintr(SOFTVEC_TYPE_SOFTERR, softerr_intr); /* װȯ */

  /* ƥࡦȯԲĤʤΤľܴؿƤӽФƥåɺ */
  current = (kz_thread *)thread_run(func, name, priority, stacksize,
				    argc, argv);

  /* ǽΥåɤư */
  dispatch(&current->context);

  /* ˤ֤äƤʤ */
}

void kz_sysdown(void)
{
  puts("system error!\n");
  while (1)
    ;
}

/* ƥࡦƤӽФѥ饤֥ؿ */
void kz_syscall(kz_syscall_type_t type, kz_syscall_param_t *param)
{
  current->syscall.type  = type;
  current->syscall.param = param;
  asm volatile ("trapa #0"); /* ȥå׳ȯ */
}

/* ӥƤӽФѥ饤֥ؿ */
void kz_srvcall(kz_syscall_type_t type, kz_syscall_param_t *param)
{
  srvcall_proc(type, param);
}
