#include "driver_lcd.h"
#include "sg12232c.h"
#include "kozos.h"
#include "lib.h"

#define LCDDRV_CMD_LCD_CLEAR        'c' /* LCD$B$N>C5n(B */
#define LCDDRV_CMD_LCD_DRAW_BOX     'b' /* BOX$BIA2h%3%^%s%I(B */
#define LCDDRV_CMD_LCD_DRAW_LOGO    'l' /* LOGO$BIA2h%3%^%s%I(B */
#define LCDDRV_CMD_LCD_DRAW_TEXT    't' /* TEXT$BIA2h%3%^%s%I(B */

void lcddrv_clear(void)
{
  char *p;
  p = kz_kmalloc(1);
  p[0] = LCDDRV_CMD_LCD_CLEAR;
  kz_send(MSGBOX_ID_LCDDRIVE, 1, p);
}

void lcddrv_draw_box(int x1, int y1, int x2, int y2, int on)
{
  char *p;
  p = kz_kmalloc(6);
  p[0] = LCDDRV_CMD_LCD_DRAW_BOX;
  p[1] = x1;
  p[2] = y1;
  p[3] = x2;
  p[4] = y2;
  p[5] = !on;
  kz_send(MSGBOX_ID_LCDDRIVE, 6, p);
}

void lcddrv_draw_logo(int x, int y, int size)
{
  char *p;
  p = kz_kmalloc(4);
  p[0] = LCDDRV_CMD_LCD_DRAW_LOGO;
  p[1] = x;
  p[2] = y;
  p[3] = size;
  kz_send(MSGBOX_ID_LCDDRIVE, 4, p);
}

void lcddrv_draw_text(int x, int y, char *str)
{
  char *p;
  int len;
  len = strlen(str);
  p = kz_kmalloc(3 + len + 1);
  p[0] = LCDDRV_CMD_LCD_DRAW_TEXT;
  p[1] = x;
  p[2] = y;
  memcpy(&p[3], str, len);
  p[3 + len] = '\0';
  kz_send(MSGBOX_ID_LCDDRIVE, 3 + len + 1, p);
}

static void draw_logo(const int ofsx, const int ofsy, const int size)
{
  static const uint8 logo_40x32[] = {
      0x00,0x00,0x00,0x00,0x00,
      0x00,0x00,0x00,0x00,0x00,
      0x00,0x1f,0xfc,0x00,0x00,
      0x00,0x1f,0xff,0xc0,0x00,
      0x00,0x1f,0xff,0xf0,0x00,
      0x00,0x0f,0x7f,0xfc,0x00,
      0x00,0x0f,0x7f,0xff,0x00,
      0x00,0x0f,0x7f,0xff,0xc0,
      0x00,0x0f,0x7f,0xff,0xe0,
      0x00,0x07,0x3f,0xff,0xf0,
      0x00,0x07,0x9f,0xff,0xfc,
      0x00,0x07,0xc1,0xff,0xfe,
      0x3f,0xff,0xff,0xff,0xff,
      0x3f,0xff,0xff,0xff,0xff,
      0x1f,0xff,0xff,0xff,0xff,
      0x1f,0xff,0xff,0xff,0xff,
      0x1f,0xff,0xfc,0xff,0xff,
      0x1f,0xff,0xf0,0x3f,0xff,
      0x1f,0xff,0xf0,0x07,0xff,
      0x1f,0xff,0xe0,0x41,0xff,
      0x0f,0xff,0xe0,0x21,0xff,
      0x0f,0xff,0xe2,0x13,0xff,
      0x07,0xff,0xf1,0x07,0xff,
      0x07,0xff,0xf8,0x8f,0xff,
      0x03,0xff,0xf8,0x1f,0xff,
      0x03,0xff,0xfc,0x3f,0xff,
      0x01,0xff,0xfc,0x7f,0xff,
      0x00,0xff,0xfe,0xff,0xff,
      0x00,0xff,0xff,0xff,0xff,
      0x00,0x7f,0xff,0xff,0xff,
      0x00,0x3f,0xff,0xff,0xff,
      0x00,0x1f,0xff,0xff,0xff,
  };
  static const uint8 logo_30x24[] = {
      0x00,0x00,0x00,0x00,
      0x00,0xfe,0x00,0x00,
      0x00,0xff,0xe0,0x00,
      0x00,0x7f,0xf8,0x00,
      0x00,0x77,0xff,0x00,
      0x00,0x77,0xff,0x80,
      0x00,0x77,0xff,0xe0,
      0x00,0x73,0xff,0xf0,
      0x00,0x38,0x7f,0xf8,
      0x3f,0xff,0xff,0xfc,
      0x3f,0xff,0xff,0xfc,
      0x3f,0xff,0xff,0xfc,
      0x3f,0xff,0x1f,0xfc,
      0x3f,0xfe,0x07,0xfc,
      0x1f,0xfe,0x23,0xfc,
      0x1f,0xfc,0x11,0xfc,
      0x1f,0xfe,0x8b,0xfc,
      0x0f,0xfe,0x47,0xfc,
      0x0f,0xff,0x2f,0xfc,
      0x07,0xff,0x1f,0xfc,
      0x03,0xff,0xbf,0xfc,
      0x03,0xff,0xff,0xfc,
      0x01,0xff,0xff,0xfc,
      0x00,0xff,0xff,0xfc,
  };
  static const uint8 logo_20x16[] = {
      0x00,0x00,0x00,
      0x03,0xf0,0x00,
      0x03,0xfe,0x00,
      0x03,0x7f,0x00,
      0x03,0x7f,0xc0,
      0x03,0x8f,0xe0,
      0x7f,0xff,0xf0,
      0x7f,0xff,0xf0,
      0x7f,0xc7,0xf0,
      0x3f,0x89,0xf0,
      0x3f,0xa5,0xf0,
      0x3f,0x93,0xf0,
      0x1f,0xc7,0xf0,
      0x0f,0xef,0xf0,
      0x0f,0xff,0xf0,
      0x07,0xff,0xf0,
  };
  int x, y;
  uint8 c = 0;
  switch (size) {
    case 0:
      for (y = 0; y < 16; y++) {
        for (x = 0; x < 20; x++) {
          if ((x % 8) == 0) {
            c = logo_20x16[(y * 3) + (x / 8)];
          }
          int pix = (c & (1 << (7 - (x % 8)))) ? 1 : 0;
          sg12232c_set_pixel(ofsx + x, ofsy + y, pix);
        }
      }
      break;
    case 1:
      for (y = 0; y < 24; y++) {
        for (x = 0; x < 30; x++) {
          if ((x % 8) == 0) {
            c = logo_30x24[(y * 4) + (x / 8)];
          }
          int pix = (c & (1 << (7 - (x % 8)))) ? 1 : 0;
          sg12232c_set_pixel(ofsx + x, ofsy + y, pix);
        }
      }
      break;
    case 2:
      for (y = 0; y < 32; y++) {
        for (x = 0; x < 40; x++) {
          if ((x % 8) == 0) {
            c = logo_40x32[(y * 5) + (x / 8)];
          }
          int pix = (c & (1 << (7 - (x % 8)))) ? 1 : 0;
          sg12232c_set_pixel(ofsx + x, ofsy + y, pix);
        }
      }
      break;
  }
}

/* $B%9%l%C%I$+$i$NMW5a$r=hM}$9$k(B */
static int lcddrv_command(char *p)
{
  int cmd = p[0];
  switch (cmd) {
  case LCDDRV_CMD_LCD_CLEAR:
    sg12232c_clear();
    break;
  case LCDDRV_CMD_LCD_DRAW_BOX:
    sg12232c_draw_box(p[1], p[2], p[3], p[4], p[5]);
    break;
  case LCDDRV_CMD_LCD_DRAW_LOGO:
    draw_logo(p[1], p[2], p[3]);
    break;
  case LCDDRV_CMD_LCD_DRAW_TEXT:
    sg12232c_draw_string(p[1], p[2], &p[3], 0);
    break;
  default:
    break;
  }
  kz_kmfree(p);

  return 0;
}

int driver_lcd(int argc, char *argv[])
{
  int size;
  char *p;

  sg12232c_init();

  while (1) {
    kz_recv(MSGBOX_ID_LCDDRIVE, &size, &p);
    lcddrv_command(p);
  }

  return 0;
}
