/**
 * @file statement.c
 * @author Shinichiro Nakamura
 */

/*
 * ===============================================================
 * "Natural Tiny Basic (NT-Basic)"
 * "A tiny BASIC interpreter"
 * ---------------------------------------------------------------
 * Core statement module
 * ===============================================================
 * Copyright (c) 2012 Shinichiro Nakamura
 *
 * Permission is hereby granted, free of charge, to any person
 * obtaining a copy of this software and associated documentation
 * files (the "Software"), to deal in the Software without
 * restriction, including without limitation the rights to use,
 * copy, modify, merge, publish, distribute, sublicense, and/or
 * sell copies of the Software, and to permit persons to whom the
 * Software is furnished to do so, subject to the following
 * conditions:
 *
 * The above copyright notice and this permission notice shall be
 * included in all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND,
 * EXPRESS OR IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES
 * OF MERCHANTABILITY, FITNESS FOR A PARTICULAR PURPOSE AND
 * NONINFRINGEMENT. IN NO EVENT SHALL THE AUTHORS OR COPYRIGHT
 * HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER LIABILITY,
 * WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING
 * FROM, OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR
 * OTHER DEALINGS IN THE SOFTWARE.
 * ===============================================================
 */

#include "ntlibc.h"
#include "statement.h"
#include "expression.h"
#include "error.h"
#include "core.h"
#include "hal.h"

/* Push function for the FOR stack. */
static void for_stack_push(core_t *core, for_stack_t i)
{
  if (core->ftos > FOR_MAX_DEPTH) {
    error_message(core, TooManyNestedFORLoops);
  }

  core->fstack[core->ftos] = i;
  core->ftos++;
}

static for_stack_t for_stack_pop(core_t *core)
{
  core->ftos--;
  if (core->ftos < 0) {
    error_message(core, NEXTWithoutFOR);
  }
  return(core->fstack[core->ftos]);
}

/* GOSUB stack push function. */
static void gosub_push(core_t *core, char *s)
{
  core->gtos++;

  if (core->gtos == GOSUB_MAX_DEPTH) {
    error_message(core, TooManyNestedGOSUBs);
    return;
  }

  core->gstack[core->gtos] = s;
}

/* GOSUB stack pop function. */
static char *gosub_pop(core_t *core)
{
  if (core->gtos == 0) {
    error_message(core, RETURNWithoutGOSUB);
    return 0;
  }

  return (core->gstack[core->gtos--]);
}

static int print_integer(int number)
{
    const int base = 10;
    int work = number;
    int bnum = 1;
    int i;
    int length = 10;
    int active = 0;

    if (number == 0) {
      ntlibc_putc('0');
      return 0;
    }

    for (i = 0; i < length - 1; i++) {
        bnum = bnum * base;
    }

    for (i = 0; i < length; i++) {
        int r = work / bnum;
        if (r != 0) {
          if (active == 0) {
            active = 1;
          }
        }
        if (active) {
          ntlibc_putc("0123456789"[r % 10]);
        }
        work = work - (r * bnum);
        bnum = bnum / base;
    }

    return 0;
}

/**
 * @brief Execute a simple version of the BASIC PRINT statement
 *
 * @param core core object.
 */
void statement_print(core_t *core)
{
  int answer;
  int len = 0, spaces;
  char last_delim = 0;

  do {
    core_get_token(core); /* get next list item */
    if ((core->token.statement == EOL) || (core->token.statement == FINISHED)) {
      break;
    }
    if (core->token.type == QUOTE) {
      /*
       * is string
       */
      ntlibc_puts(core->token.text);
      len += ntlibc_strlen(core->token.text);
      core_get_token(core);
    } else {
      /*
       * is expression
       */
      core_putback(core);
      expression_entry(core, &answer);
      core_get_token(core);
      len += print_integer(answer);
    }
    last_delim = *core->token.text;

    if (*core->token.text == ';') {
      /*
       * compute number of spaces to move to next tab
       */
      spaces = 8 - (len % 8);
      len += spaces; /* add in the tabbing position */
      while (spaces) {
        ntlibc_puts(" ");
        spaces--;
      }
    } else if (*core->token.text == ',') {
      /*
       * do nothing
       */
    } else if ((core->token.statement != EOL) && (core->token.statement != FINISHED)) {
      error_message(core, SyntaxError);
    }
  } while ((*core->token.text == ';') || (*core->token.text == ','));

  /*
   * Check the current statement.
   */
  if ((core->token.statement == EOL) || (core->token.statement == FINISHED)) {
    if ((last_delim != ';') && (last_delim != ',')) {
      ntlibc_puts("\n");
    }
  } else {
    error_message(core, SyntaxError); /* error is not , or ; */
  }
}

/**
 * @brief Execute a GOTO statement.
 */
void statement_goto(core_t *core)
{
  char *loc;

  core_get_token(core); /* get label to go to */
  /* find the location of the label */
  loc = core_find_label(core, core->token.text);
  if (loc == '\0') {
    error_message(core, UndefinedLabel); /* label not defined */
  } else {
    core->location.pointer = loc;  /* start core running at that loc */
  }
}

/* Execute an IF statement. */
void statement_if(core_t *core)
{
  int x , y, cond;
  char op;

  expression_entry(core, &x); /* get left expression */

  core_get_token(core); /* get the operator */
  if (!ntlibc_strchr("=<>", *core->token.text)) {
    error_message(core, SyntaxError); /* not a legal operator */
    return;
  }
  op = *core->token.text;

  expression_entry(core, &y); /* get right expression */

  /* determine the outcome */
  cond = 0;
  switch (op) {
    case '<':
      if (x < y) {
        cond = 1;
      }
      break;
    case '>':
      if (x > y) {
        cond = 1;
      }
      break;
    case '=':
      if (x == y) {
        cond = 1;
      }
      break;
  }
  if (cond) { /* is true so process target of IF */
    core_get_token(core);
    if (core->token.statement != THEN) {
      error_message(core, THENExpected);
      return;
    }/* else core execution starts on next line */
  } else {
    core_find_eol(core); /* find start of next line */
  }
}

/* Execute a FOR loop. */
void statement_for(core_t *core)
{
  for_stack_t i;
  int value;

  core_get_token(core); /* read the control variable */
  if (!ntlibc_isalpha(*core->token.text)) {
    error_message(core, NotAVariable);
    return;
  }

  i.var = ntlibc_toupper(*core->token.text) - 'A'; /* save its index */

  core_get_token(core); /* read the equals sign */
  if (*core->token.text != '=') {
    error_message(core, EqualsSignExpected);
    return;
  }

  expression_entry(core, &value); /* get initial value */

  core->variables[i.var] = value;

  core_get_token(core);
  if (core->token.statement != TO) {
    error_message(core, TOExpected); /* read and discard the TO */
  }

  expression_entry(core, &i.target); /* get target value */

  /* if loop can execute at least once, push info on stack */
  if (value >= core->variables[i.var]) {
    i.loc = core->location.pointer;
    for_stack_push(core, i);
  } else {
    /* otherwise, skip loop code altogether */
    while (core->token.statement != NEXT) {
      core_get_token(core);
    }
  }
}

/* Execute a NEXT statement. */
void statement_next(core_t *core)
{
  for_stack_t i;

  i = for_stack_pop(core); /* read the loop info */

  core->variables[i.var]++; /* increment control variable */
  if (core->variables[i.var] > i.target) {
    return;  /* all done */
  }
  for_stack_push(core, i);  /* otherwise, restore the info */
  core->location.pointer = i.loc;  /* loop */
}

/* Execute a simple form of the BASIC INPUT command */
void statement_input(core_t *core)
{
  char buf[80];
  char var;
  int i;

  core_get_token(core); /* see if prompt string is present */
  if (core->token.type == QUOTE) {
    ntlibc_puts(core->token.text); /* if so, print it and check for comma */
    core_get_token(core);
    if (*core->token.text != ',') {
      error_message(core, UnbalancedParentheses);
    }
    core_get_token(core);
  } else {
    ntlibc_puts("? "); /* otherwise, prompt with / */
  }
  var = ntlibc_toupper(*core->token.text) - 'A'; /* get the input var */

  i = ntlibc_atoi(ntlibc_gets(buf, sizeof(buf)));    /* read input */

  core->variables[(int)var] = i; /* store it */
}

/* Execute a GOSUB command. */
void statement_gosub(core_t *core)
{
  char *loc;

  core_get_token(core);
  /* find the label to call */
  loc = core_find_label(core, core->token.text);
  if (loc == '\0') {
    error_message(core, UndefinedLabel); /* label not defined */
  } else {
    gosub_push(core, core->location.pointer); /* save place to return to */
    core->location.pointer = loc;  /* start core running at that loc */
  }
}

/* Return from GOSUB. */
void statement_return(core_t *core)
{
  core->location.pointer = gosub_pop(core);
}

void statement_rem(core_t *core)
{
  core_find_eol(core);   /* find start of next line */
}

void statement_read(core_t *core)
{
  unsigned int port, value;
  int var;
  core_get_token(core);
  if (core->token.type != NUMBER) {
    var = ntlibc_toupper(*core->token.text) - 'A';
    port = core->variables[var];
  } else {
    port = ntlibc_atoi(core->token.text);
  }
  core_get_token(core);
  if (*core->token.text != ',') {
    error_message(core, SyntaxError);
  }
  core_get_token(core);
  var = ntlibc_toupper(*core->token.text) - 'A'; /* get the input var */
  hal_read(port, &value);
  core->variables[var] = value;
  core_find_eol(core); /* find start of next line */
}

void statement_write(core_t *core)
{
  int port, value;
  int var;
  core_get_token(core);
  if (core->token.type != NUMBER) {
    var = ntlibc_toupper(*core->token.text) - 'A';
    port = core->variables[var];
  } else {
    port = ntlibc_atoi(core->token.text);
  }
  core_get_token(core);
  if (*core->token.text != ',') {
    error_message(core, SyntaxError);
  }
  core_get_token(core);
  var = ntlibc_toupper(*core->token.text) - 'A'; /* get the input var */
  value = core->variables[var];
  hal_write(port, value);
  core_find_eol(core); /* find start of next line */
}

