﻿/*
 * Copyright (C) 2013 FooProject
 * * This program is free software; you can redistribute it and/or modify it under the terms of the GNU General Public License as published by
 * the Free Software Foundation; either version 3 of the License, or (at your option) any later version.

 * This program is distributed in the hope that it will be useful, but WITHOUT ANY WARRANTY; without even the implied warranty of 
 * MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE. See the GNU General Public License for more details.

You should have received a copy of the GNU General Public License along with this program. If not, see <http://www.gnu.org/licenses/>.
 */

//#define TEST_ASYNC

using System;
using System.IO;
using System.Collections.Generic;
using System.Text;
using System.Text.RegularExpressions;
using System.Threading;
using System.Threading.Tasks;
using System.Linq;

namespace FooEditEngine
{
    /// <summary>
    /// 進行状況を表す列挙体
    /// </summary>
    public enum ProgressState
    {
        /// <summary>
        /// 操作が開始したことを表す
        /// </summary>
        Start,
        /// <summary>
        /// 操作が終了したことを表す
        /// </summary>
        Complete,
    }
    /// <summary>
    /// 進行状況を表すためのイベントデータ
    /// </summary>
    public sealed class ProgressEventArgs : EventArgs
    {
        /// <summary>
        /// 進行状況
        /// </summary>
        public ProgressState state;
        /// <summary>
        /// コンストラクター
        /// </summary>
        /// <param name="state">ProgressStateオブジェクト</param>
        public ProgressEventArgs(ProgressState state)
        {
            this.state = state;
        }
    }

    /// <summary>
    /// 進行状況を通知するためのデリゲート
    /// </summary>
    /// <param name="sender">送信元クラス</param>
    /// <param name="e">イベントデータ</param>
    public delegate void ProgressEventHandler(object sender, ProgressEventArgs e);

    /// <summary>
    /// 更新タイプを表す列挙体
    /// </summary>
    public enum UpdateType
    {
        /// <summary>
        /// ドキュメントが置き換えられたことを表す
        /// </summary>
        Replace,
        /// <summary>
        /// ドキュメント全体が削除されたことを表す
        /// </summary>
        Clear,
    }

    /// <summary>
    /// 更新タイプを通知するためのイベントデータ
    /// </summary>
    public sealed class DocumentUpdateEventArgs : EventArgs
    {
        /// <summary>
        /// 更新タイプ
        /// </summary>
        public UpdateType type;
        /// <summary>
        /// 開始位置
        /// </summary>
        public int startIndex;
        /// <summary>
        /// 削除された長さ
        /// </summary>
        public int removeLength;
        /// <summary>
        /// 追加された長さ
        /// </summary>
        public int insertLength;
        /// <summary>
        /// コンストラクター
        /// </summary>
        /// <param name="type">更新タイプ</param>
        /// <param name="startIndex">開始インデックス</param>
        /// <param name="removeLength">削除された長さ</param>
        /// <param name="insertLength">追加された長さ</param>
        public DocumentUpdateEventArgs(UpdateType type, int startIndex, int removeLength, int insertLength)
        {
            this.type = type;
            this.startIndex = startIndex;
            this.removeLength = removeLength;
            this.insertLength = insertLength;
        }
    }

    /// <summary>
    /// ドキュメントに更新があったことを伝えるためのデリゲート
    /// </summary>
    /// <param name="sender">送信元クラス</param>
    /// <param name="e">イベントデータ</param>
    public delegate void DocumentUpdateEventHandler(object sender, DocumentUpdateEventArgs e);

    /// <summary>
    /// ドキュメントの管理を行う
    /// </summary>
    /// <remarks>この型のすべてのメソッド・プロパティはスレッドセーフです</remarks>
    public sealed class Document : IEnumerable<char>, IRandomEnumrator<char>
    {
        const int MaxSemaphoreCount = 1;
        Regex regex;
        Match match;
        StringBuffer buffer;
        LineToIndexTable _LayoutLines;
        bool _EnableFireUpdateEvent = true;
        SemaphoreSlim Semaphore = new SemaphoreSlim(MaxSemaphoreCount);

        /// <summary>
        /// コンストラクター
        /// </summary>
        internal Document()
            : this(null)
        {
        }

        internal Document(Document doc)
        {
            if (doc == null)
                this.buffer = new StringBuffer();
            else
                this.buffer = new StringBuffer(doc.buffer);
            this.buffer.Update += new DocumentUpdateEventHandler(buffer_Update);
            this.UpdateCalledAlways += (s, e) => { };
            this.Update += new DocumentUpdateEventHandler((s, e) => { });
            this.ChangeFireUpdateEvent += new EventHandler((s, e) => { });
            this.Markers = new MarkerCollection(this);
            this.UndoManager = new UndoManager();
            this._LayoutLines = new LineToIndexTable(this);
            this._LayoutLines.SpilitString = LayoutLines_SpilitStringByChar;
            this._LayoutLines.Clear();
        }

        /// <summary>
        /// レイアウト行を表す
        /// </summary>
        public LineToIndexTable LayoutLines
        {
            get
            {
                return this._LayoutLines;
            }
        }

        IList<LineToIndexTableData> LayoutLines_SpilitStringByChar(object sender, SpilitStringEventArgs e)
        {
            return this.CreateLineList(e.index, e.length);
        }

        /// <summary>
        /// レイアウト行を返す
        /// </summary>
        /// <param name="index">開始インデックス</param>
        /// <param name="length">長さ</param>
        /// <param name="lineLimitLength">１行当たりの最大文字数。-1で無制限</param>
        /// <returns>レイアウト行リスト</returns>
        internal IList<LineToIndexTableData> CreateLineList(int index, int length, int lineLimitLength = -1)
        {
            int startIndex = index;
            int endIndex = index + length - 1;
            List<LineToIndexTableData> output = new List<LineToIndexTableData>();

            foreach (Tuple<int, int> range in this.ForEachLines(startIndex, endIndex, lineLimitLength))
            {
                int lineHeadIndex = range.Item1;
                int lineLength = range.Item2;
                char c = this.buffer[lineHeadIndex + lineLength - 1];
                bool hasNewLine = c == Document.NewLine;
                output.Add(this.LayoutLines.CreateLineToIndexTableData(lineHeadIndex, lineLength, hasNewLine, null));
            }

            if (output.Count > 0)
                output.Last().LineEnd = true;

            return output;
        }

        internal void FireUpdate(DocumentUpdateEventArgs e)
        {
            this.buffer_Update(this.buffer, e);
        }

        /// <summary>
        /// ドキュメントが更新された時に呼ばれるイベント
        /// </summary>
        public event DocumentUpdateEventHandler Update;

        /// <summary>
        /// ドキュメントが更新された時に呼びされるイベント
        /// </summary>
        /// <remarks>
        /// FireUpdateEventの値に関わらず常に呼びされます
        /// </remarks>
        internal event DocumentUpdateEventHandler UpdateCalledAlways;

        /// <summary>
        /// FireUpdateEventの値が変わったときに呼び出されるイベント
        /// </summary>
        public event EventHandler ChangeFireUpdateEvent;

        /// <summary>
        /// 改行コードの内部表現
        /// </summary>
        public const char NewLine = '\n';

        /// <summary>
        /// EOFの内部表現
        /// </summary>
        public const char EndOfFile = '\u001a';

        /// <summary>
        /// ロック中なら真を返し、そうでないなら偽を返す
        /// </summary>
        public bool IsLocked
        {
            get
            {
                return this.Semaphore.CurrentCount == 0;
            }
        }

        /// <summary>
        /// アンドゥ管理クラスを表す
        /// </summary>
        public UndoManager UndoManager
        {
            get;
            private set;
        }

        /// <summary>
        /// 文字列の長さ
        /// </summary>
        public int Length
        {
            get
            {
                return this.buffer.Length;
            }
        }

        /// <summary>
        /// 変更のたびにUpdateイベントを発生させるかどうか
        /// </summary>
        public bool FireUpdateEvent
        {
            get
            {
                return this._EnableFireUpdateEvent;
            }
            set
            {
                this._EnableFireUpdateEvent = value;
                this.ChangeFireUpdateEvent(this, null);
            }
        }

        /// <summary>
        /// インデクサー
        /// </summary>
        /// <param name="i">インデックス（自然数でなければならない）</param>
        /// <returns>Char型</returns>
        public char this[int i]
        {
            get
            {
                return this.buffer[i];
            }
        }

        /// <summary>
        /// マーカーコレクション
        /// </summary>
        public MarkerCollection Markers
        {
            get;
            private set;
        }

        internal StringBuffer StringBuffer
        {
            get
            {
                return this.buffer;
            }
        }

        /// <summary>
        /// DocumentReaderを作成します
        /// </summary>
        /// <returns>DocumentReaderオブジェクト</returns>
        public DocumentReader CreateReader()
        {
            return new DocumentReader(this.buffer);
        }

        /// <summary>
        /// ロックを解除します
        /// </summary>
        public void UnLock()
        {
            this.Semaphore.Release();
        }

        /// <summary>
        /// ロックします
        /// </summary>
        public void Lock()
        {
            this.Semaphore.Wait();
        }

        /// <summary>
        /// ロックします
        /// </summary>
        /// <returns>Taskオブジェクト</returns>
        public Task LockAsync()
        {
            return this.Semaphore.WaitAsync();
        }

        /// <summary>
        /// マーカーを設定する
        /// </summary>
        /// <param name="id">マーカーID</param>
        /// <param name="m">設定したいマーカー</param>
        public void SetMarker(int id,Marker m)
        {
            if (m.start < 0 || m.start + m.length > this.Length)
                throw new ArgumentOutOfRangeException("startもしくはendが指定できる範囲を超えています");

            this.Markers.Add(id,m);
        }

        /// <summary>
        /// マーカーを削除する
        /// </summary>
        /// <param name="id">マーカーID</param>
        /// <param name="start">開始インデックス</param>
        /// <param name="length">削除する長さ</param>
        public void RemoveMarker(int id,int start, int length)
        {
            if (start < 0 || start + length > this.Length)
                throw new ArgumentOutOfRangeException("startもしくはendが指定できる範囲を超えています");

            this.Markers.RemoveAll(id,start, length);
        }

        /// <summary>
        /// マーカーを削除する
        /// </summary>
        /// <param name="id">マーカーID</param>
        /// <param name="type">削除したいマーカーのタイプ</param>
        public void RemoveMarker(int id, HilightType type)
        {
            this.Markers.RemoveAll(id,type);
        }

        /// <summary>
        /// インデックスに対応するマーカーを得る
        /// </summary>
        /// <param name="id">マーカーID</param>
        /// <param name="index">インデックス</param>
        /// <returns>Marker構造体の列挙子</returns>
        public IEnumerable<Marker> GetMarkers(int id, int index)
        {
            if (index < 0 || index > this.Length)
                throw new ArgumentOutOfRangeException("indexが範囲を超えています");
            return this.Markers.Get(id,index);
        }

        /// <summary>
        /// 部分文字列を取得する
        /// </summary>
        /// <param name="index">開始インデックス</param>
        /// <param name="length">長さ</param>
        /// <returns>Stringオブジェクト</returns>
        public string ToString(int index, int length)
        {
            return this.buffer.ToString(index, length);
        }

        /// <summary>
        /// インデックスを開始位置とする文字列を返す
        /// </summary>
        /// <param name="index">開始インデックス</param>
        /// <returns>Stringオブジェクト</returns>
        public string ToString(int index)
        {
            return this.ToString(index, this.buffer.Length - index);
        }

        /// <summary>
        /// 行を取得する
        /// </summary>
        /// <param name="startIndex">開始インデックス</param>
        /// <param name="endIndex">終了インデックス</param>
        /// <param name="maxCharCount">最大長</param>
        /// <returns>行イテレーターが返される</returns>
        public IEnumerable<string> GetLines(int startIndex, int endIndex, int maxCharCount = -1)
        {
            return this.buffer.GetLines(startIndex, endIndex, maxCharCount);
        }

        internal IEnumerable<Tuple<int, int>> ForEachLines(int startIndex, int endIndex, int maxCharCount = -1)
        {
            return this.buffer.ForEachLines(startIndex, endIndex, maxCharCount);
        }


        /// <summary>
        /// 文字列を追加する
        /// </summary>
        /// <param name="s">追加したい文字列</param>
        /// <remarks>非同期操作中はこのメソッドを実行することはできません</remarks>
        public void Append(string s)
        {
            this.Replace(this.buffer.Length, 0, s);
        }

        /// <summary>
        /// 文字列を挿入する
        /// </summary>
        /// <param name="index">開始インデックス</param>
        /// <param name="s">追加したい文字列</param>
        /// <remarks>読み出し操作中はこのメソッドを実行することはできません</remarks>
        public void Insert(int index, string s)
        {
            this.Replace(index, 0, s);
        }

        /// <summary>
        /// 文字列を削除する
        /// </summary>
        /// <param name="index">開始インデックス</param>
        /// <param name="length">長さ</param>
        /// <remarks>読み出し操作中はこのメソッドを実行することはできません</remarks>
        public void Remove(int index, int length)
        {
            this.Replace(index, length, "");
        }

        /// <summary>
        /// ドキュメントを置き換える
        /// </summary>
        /// <param name="index">開始インデックス</param>
        /// <param name="length">長さ</param>
        /// <param name="s">文字列</param>
        /// <remarks>読み出し操作中はこのメソッドを実行することはできません</remarks>
        public void Replace(int index, int length, string s)
        {
            if (index < 0 || index > this.buffer.Length || index + length > this.buffer.Length || length < 0)
                throw new ArgumentOutOfRangeException();
            if (length == 0 && (s == string.Empty || s == null))
                return;

            foreach(int id in this.Markers.IDs)
                this.RemoveMarker(id,index, length);

            ReplaceCommand cmd = new ReplaceCommand(this.buffer, index, length, s);
            this.UndoManager.push(cmd);
            cmd.redo();
        }

        /// <summary>
        /// 物理行をすべて削除する
        /// </summary>
        /// <remarks>Dirtyフラグも同時にクリアーされます</remarks>
        /// <remarks>非同期操作中はこのメソッドを実行することはできません</remarks>
        public void Clear()
        {
            this.buffer.Clear();
        }

        /// <summary>
        /// ストリームからドキュメントを非同期的に構築します
        /// </summary>
        /// <param name="fs">IStreamReaderオブジェクト</param>
        /// <param name="tokenSource">キャンセルトークン</param>
        /// <returns>Taskオブジェクト</returns>
        /// <remarks>
        /// 読み取り操作は別スレッドで行われます。
        /// また、非同期操作中はこのメソッドを実行することはできません。
        /// </remarks>
        internal async Task LoadAsync(IStreamReader fs, CancellationTokenSource tokenSource = null)
        {
            if (fs.IsEnd())
                return;

            try
            {
                await this.LockAsync().ConfigureAwait(false);
                this.Clear();
                this.FireUpdateEvent = false;
                this.UndoManager.BeginLock();
                string str;
                for (int i = 0; (str = await fs.ReadLineAsync().ConfigureAwait(false)) != null; i++)
                {
                    int index = this.Length;
                    if (index < 0)
                        index = 0;

                    this.Replace(index, 0, str + Document.NewLine);

                    if (tokenSource != null)
                        tokenSource.Token.ThrowIfCancellationRequested();
#if TEST_ASYNC
                    System.Threading.Thread.Sleep(10);
#endif
                }
            }
            finally
            {
                this.FireUpdateEvent = true;
                this.UndoManager.EndLock();
                this.UnLock();
            }
        }

        /// <summary>
        /// ストリームに非同期モードで保存します
        /// </summary>
        /// <param name="fs">IStreamWriterオブジェクト</param>
        /// <param name="tokenSource">キャンセルトークン</param>
        /// <returns>Taskオブジェクト</returns>
        /// <remarks>非同期操作中はこのメソッドを実行することはできません</remarks>
        internal async Task SaveAsync(IStreamWriter fs, CancellationTokenSource tokenSource = null)
        {
            try
            {
                await this.LockAsync().ConfigureAwait(false);
                StringBuilder line = new StringBuilder();
                for (int i = 0; i < this.Length; i++)
                {
                    char c = this[i];
                    line.Append(c);
                    if (c == Document.NewLine || i == this.Length - 1)
                    {
                        string str = line.ToString();
                        str = str.Replace(Document.NewLine.ToString(), fs.NewLine);
                        await fs.WriteAsync(str).ConfigureAwait(false);
                        line.Clear();
                        if (tokenSource != null)
                            tokenSource.Token.ThrowIfCancellationRequested();
#if TEST_ASYNC
                    System.Threading.Thread.Sleep(10);
#endif
                    }
                }
            }
            finally
            {
                this.UnLock();
            }
        }

        /// <summary>
        /// Find()およびReplaceAll()で使用するパラメーターをセットします
        /// </summary>
        /// <param name="pattern">検索したい文字列</param>
        /// <param name="UseRegex">正規表現を使用するなら真</param>
        /// <param name="opt">RegexOptions列挙体</param>
        public void SetFindParam(string pattern, bool UseRegex, RegexOptions opt)
        {
            this.match = null;
            if (UseRegex)
                this.regex = new Regex(pattern, opt);
            else
                this.regex = new Regex(Regex.Escape(pattern), opt);
        }

        /// <summary>
        /// 現在の検索パラメーターでWatchDogを生成する
        /// </summary>
        /// <param name="type">ハイライトタイプ</param>
        /// <param name="color">色</param>
        /// <returns>WatchDogオブジェクト</returns>
        public RegexMarkerPattern CreateWatchDogByFindParam(HilightType type,Color color)
        {
            if (this.regex == null)
                throw new InvalidOperationException("SetFindParam()を呼び出してください");
            return new RegexMarkerPattern(this.regex,type,color);
        }

        /// <summary>
        /// 指定した文字列を検索します
        /// </summary>
        /// <returns>見つかった場合はSearchResult列挙子を返却します</returns>
        /// <remarks>見つかったパターン以外を置き換えた場合、正常に動作しないことがあります</remarks>
        public IEnumerator<SearchResult> Find()
        {
            return this.Find(0, this.Length);
        }

        /// <summary>
        /// 指定した文字列を検索します
        /// </summary>
        /// <returns>見つかった場合はSearchResult列挙子を返却します</returns>
        /// <param name="start">開始インデックス</param>
        /// <param name="length">検索する長さ</param>
        /// <remarks>見つかったパターン以外を置き換えた場合、正常に動作しないことがあります</remarks>
        public IEnumerator<SearchResult> Find(int start, int length)
        {
            if (this.regex == null)
                throw new InvalidOperationException();
            if (start < 0 || start >= this.Length)
                throw new ArgumentOutOfRangeException();

            int end = start + length - 1;

            if(end > this.Length - 1)
                throw new ArgumentOutOfRangeException();

            StringBuilder line = new StringBuilder();
            int oldLength = this.Length;
            for (int i = start; i <= end; i++)
            {
                char c = this[i];
                line.Append(c);
                if (c == Document.NewLine || i == end)
                {
                    this.match = this.regex.Match(line.ToString());
                    while (this.match.Success)
                    {
                        int startIndex = i - line.Length + 1 + this.match.Index;
                        int endIndex = startIndex + this.match.Length - 1;

                        yield return new SearchResult(this.match, startIndex, endIndex);

                        if (this.Length != oldLength)   //長さが変わった場合は置き換え後のパターンの終点＋１まで戻る
                        {
                            int delta = this.Length - oldLength;
                            i = endIndex + delta;
                            end = end + delta;
                            oldLength = this.Length;
                            break;
                        }

                        this.match = this.match.NextMatch();
                    }
                    line.Clear();
                }
            }
        }

        /// <summary>
        /// 任意のパターンですべて置き換えます
        /// </summary>
        /// <param name="replacePattern">置き換え後のパターン</param>
        /// <param name="groupReplace">グループ置き換えを行うなら真。そうでないなら偽</param>
        public void ReplaceAll(string replacePattern,bool groupReplace)
        {
            if (this.regex == null)
                throw new InvalidOperationException();
            ReplaceAllCommand cmd = new ReplaceAllCommand(this.buffer, this.regex, replacePattern, groupReplace);
            this.UndoManager.push(cmd);
            cmd.redo();
        }

        /// <summary>
        /// 任意のパターンで置き換える
        /// </summary>
        /// <param name="target">対象となる文字列</param>
        /// <param name="pattern">置き換え後の文字列</param>
        /// <param name="ci">大文字も文字を区別しないなら真。そうでないなら偽</param>
        /// <remarks>
        /// 検索時に大文字小文字を区別します。また、このメソッドでは正規表現を使用することはできません
        /// </remarks>
        public void ReplaceAll2(string target, string pattern,bool ci = false)
        {
            FastReplaceAllCommand cmd = new FastReplaceAllCommand(this.buffer, target, pattern,ci);
            this.UndoManager.push(cmd);
            cmd.redo();
        }

        #region IEnumerable<char> メンバー

        /// <summary>
        /// 列挙子を返します
        /// </summary>
        /// <returns>IEnumeratorオブジェクトを返す</returns>
        public IEnumerator<char> GetEnumerator()
        {
            return this.buffer.GetEnumerator();
        }

        #endregion

        #region IEnumerable メンバー

        System.Collections.IEnumerator System.Collections.IEnumerable.GetEnumerator()
        {
            throw new NotImplementedException();
        }

        #endregion

        void buffer_Update(object sender, DocumentUpdateEventArgs e)
        {
            switch (e.type)
            {
                case UpdateType.Replace:
                    this._LayoutLines.UpdateAsReplace(e.startIndex, e.removeLength, e.insertLength);
                    break;
                case UpdateType.Clear:
                    this._LayoutLines.Clear();
                    break;
            }
            this.UpdateCalledAlways(this, e);
            if(this.FireUpdateEvent)
                this.Update(this, e);
        }
    }

    public interface IStreamReader
    {
        /// <summary>
        /// ストリームが空かどうかを返す
        /// </summary>
        bool IsEnd();

        /// <summary>
        /// ストリームから行を読み取った物を返す。LoadAsyncを呼び出す場合は必ず実装してください
        /// </summary>
        Task<string> ReadLineAsync();
    }

    public interface IStreamWriter
    {
        /// <summary>
        /// ストリームに書き込む。SaveAsyncを呼び出す場合は必ず実装してください
        /// </summary>
        Task WriteAsync(string str);

        /// <summary>
        /// 書き込む際に使用する改行コード
        /// </summary>
        string NewLine
        {
            get;
            set;
        }
    }

    /// <summary>
    /// 検索結果を表す
    /// </summary>
    public class SearchResult
    {
        private Match Match;

        /// <summary>
        /// 一致した場所の開始位置を表す
        /// </summary>
        public int Start;

        /// <summary>
        /// 一致した場所の終了位置を表す
        /// </summary>
        public int End;

        /// <summary>
        /// 見つかった文字列を返す
        /// </summary>
        public string Value
        {
            get { return this.Match.Value; }
        }

        /// <summary>
        /// 指定したパターンを置き換えて返す
        /// </summary>
        /// <param name="replacement">置き換える文字列</param>
        /// <returns>置き換え後の文字列</returns>
        public string Result(string replacement)
        {
            return this.Match.Result(replacement);
        }

        /// <summary>
        /// コンストラクター
        /// </summary>
        /// <param name="m">Matchオブジェクト</param>
        /// <param name="start">開始インデックス</param>
        /// <param name="end">終了インデックス</param>
        public SearchResult(Match m, int start,int end)
        {
            this.Match = m;
            this.Start = start;
            this.End = end;
        }
    }

    /// <summary>
    /// ドキュメントリーダー
    /// </summary>
    public class DocumentReader : TextReader
    {
        StringBuffer document;      
        int currentIndex;

        /// <summary>
        /// コンストラクター
        /// </summary>
        /// <param name="doc"></param>
        internal DocumentReader(StringBuffer doc)
        {
            if (doc == null)
                throw new ArgumentNullException();
            this.document = doc;
        }

        /// <summary>
        /// 文字を取得する
        /// </summary>
        /// <returns>文字。取得できない場合は-1</returns>
        public override int Peek()
        {
            if (this.document == null)
                throw new InvalidOperationException();
            if (this.currentIndex >= this.document.Length)
                return -1;
            return this.document[this.currentIndex];
        }

        /// <summary>
        /// 文字を取得し、イテレーターを一つ進める
        /// </summary>
        /// <returns>文字。取得できない場合は-1</returns>
        public override int Read()
        {
            int c = this.Peek();
            if(c != -1)
                this.currentIndex++;
            return c;
        }

        /// <summary>
        /// 文字列を読み取りバッファーに書き込む
        /// </summary>
        /// <param name="buffer">バッファー</param>
        /// <param name="index">開始インデックス</param>
        /// <param name="count">カウント</param>
        /// <returns>読み取られた文字数</returns>
        public override int Read(char[] buffer, int index, int count)
        {
            if (this.document == null)
                throw new InvalidOperationException();

            if (buffer == null)
                throw new ArgumentNullException();

            if (this.document.Length < count)
                throw new ArgumentException();

            if (index < 0 || count < 0)
                throw new ArgumentOutOfRangeException();

            if (this.document.Length == 0)
                return 0;

            int actualCount = count;
            if (index + count - 1 > this.document.Length - 1)
                actualCount = this.document.Length - index;

            string str = this.document.ToString(index, actualCount);

            for (int i = 0; i < str.Length; i++)    //ToCharArray()だと戻った時に消えてしまう
                buffer[i] = str[i];

            this.currentIndex = index + actualCount;
            
            return actualCount;
        }

        /// <summary>
        /// オブジェクトを破棄する
        /// </summary>
        /// <param name="disposing">真ならアンマネージドリソースを解放する</param>
        protected override void Dispose(bool disposing)
        {
            this.document = null;
        }
    }
}
