/*
 * Copyright 2009-2010 the Fess Project and the Others.
 *
 * Licensed under the Apache License, Version 2.0 (the "License");
 * you may not use this file except in compliance with the License.
 * You may obtain a copy of the License at
 *
 *     http://www.apache.org/licenses/LICENSE-2.0
 *
 * Unless required by applicable law or agreed to in writing, software
 * distributed under the License is distributed on an "AS IS" BASIS,
 * WITHOUT WARRANTIES OR CONDITIONS OF ANY KIND,
 * either express or implied. See the License for the specific language
 * governing permissions and limitations under the License.
 */
package jp.sf.fess.solr;

import java.io.File;
import java.util.ArrayList;
import java.util.Collection;
import java.util.LinkedHashMap;
import java.util.List;
import java.util.Map;
import java.util.concurrent.ConcurrentHashMap;

import javax.annotation.Resource;

import jp.sf.fess.Constants;
import jp.sf.fess.solr.response.ReplicationResponse;
import jp.sf.fess.util.FessProperties;

import org.apache.solr.client.solrj.SolrRequest;
import org.apache.solr.client.solrj.SolrRequest.METHOD;
import org.apache.solr.client.solrj.SolrServer;
import org.apache.solr.client.solrj.SolrServerException;
import org.apache.solr.client.solrj.response.QueryResponse;
import org.apache.solr.client.solrj.response.SolrPingResponse;
import org.apache.solr.client.solrj.response.UpdateResponse;
import org.apache.solr.common.SolrInputDocument;
import org.apache.solr.common.params.SolrParams;
import org.seasar.framework.util.StringUtil;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class SolrServerGroup {
    private static final Logger logger = LoggerFactory
            .getLogger(SolrServerGroup.class);

    private static final int MAX_LOAD_COUNT = Integer.MAX_VALUE / 2;

    protected LinkedHashMap<String, SolrServer> solrServerMap = new LinkedHashMap<String, SolrServer>();

    /** a max error count */
    public int maxErrorCount = 3;

    /** a max retry count */
    public int maxRetryStatusCheckCount = 5;

    public long retryStatusCheckInterval = 500;

    /** a max retry count */
    public int maxRetryUpdateQueryCount = 3;

    public long retryUpdateQueryInterval = 500;

    /** a max retry count */
    public int maxRetrySelectQueryCount = 3;

    public long retrySelectQueryInterval = 500;

    /** the number of minimum active servers */
    public int minActiveServer = 1;

    protected String groupName;

    @Resource
    protected FessProperties solrServerProperties;

    private volatile Map<String, Integer> solrServerAccessCountMap = new ConcurrentHashMap<String, Integer>();

    private volatile Map<String, Integer> solrServerErrorCountMap = new ConcurrentHashMap<String, Integer>();

    public void addServer(String name, SolrServer solrServer) {
        solrServerMap.put(name, solrServer);
    }

    public Collection<UpdateResponse> add(
            final Collection<SolrInputDocument> docs) {
        // check this group status
        checkStatus();

        return updateQueryCallback(new UpdateProcessCallback<UpdateResponse>() {
            public UpdateResponse callback(SolrServer solrServer)
                    throws Exception {
                return solrServer.add(docs);
            }
        });
    }

    public Collection<UpdateResponse> add(final SolrInputDocument doc) {
        // check this group status
        checkStatus();

        return updateQueryCallback(new UpdateProcessCallback<UpdateResponse>() {
            public UpdateResponse callback(SolrServer solrServer)
                    throws Exception {
                return solrServer.add(doc);
            }
        });
    }

    public Collection<UpdateResponse> addBean(final Object obj) {
        // check this group status
        checkStatus();

        return updateQueryCallback(new UpdateProcessCallback<UpdateResponse>() {
            public UpdateResponse callback(SolrServer solrServer)
                    throws Exception {
                return solrServer.addBean(obj);
            }
        });
    }

    public Collection<UpdateResponse> addBean(final Collection<?> beans) {
        // check this group status
        checkStatus();

        return updateQueryCallback(new UpdateProcessCallback<UpdateResponse>() {
            public UpdateResponse callback(SolrServer solrServer)
                    throws Exception {
                return solrServer.addBeans(beans);
            }
        });
    }

    public Collection<UpdateResponse> commit() {
        // check this group status
        checkStatus();

        return updateQueryCallback(new UpdateProcessCallback<UpdateResponse>() {
            public UpdateResponse callback(SolrServer solrServer)
                    throws Exception {
                return solrServer.commit();
            }
        });
    }

    public Collection<UpdateResponse> commit(final boolean waitFlush,
            final boolean waitSearcher) {
        // check this group status
        checkStatus();

        return updateQueryCallback(new UpdateProcessCallback<UpdateResponse>() {
            public UpdateResponse callback(SolrServer solrServer)
                    throws Exception {
                return solrServer.commit(waitFlush, waitSearcher);
            }
        });
    }

    public Collection<ReplicationResponse> replicate(final File snapshotDir) {
        // check this group status
        checkStatus();

        return updateQueryCallback(new UpdateProcessCallback<ReplicationResponse>() {
            public ReplicationResponse callback(SolrServer solrServer)
                    throws Exception {
                if (solrServer instanceof FessSolrServer) {
                    return ((FessSolrServer) solrServer).replicate(snapshotDir);
                }
                throw new FessSolrException(solrServer.getClass()
                        .getCanonicalName()
                        + " is not supported for a replication.");
            }
        });
    }

    public Collection<UpdateResponse> deleteByQuery(final String query) {
        // check this group status
        checkStatus();

        return updateQueryCallback(new UpdateProcessCallback<UpdateResponse>() {
            public UpdateResponse callback(SolrServer solrServer)
                    throws Exception {
                return solrServer.deleteByQuery(query);
            }
        });
    }

    public Collection<UpdateResponse> deleteById(final String id) {
        // check this group status
        checkStatus();

        return updateQueryCallback(new UpdateProcessCallback<UpdateResponse>() {
            public UpdateResponse callback(SolrServer solrServer)
                    throws Exception {
                return solrServer.deleteById(id);
            }
        });
    }

    public Collection<UpdateResponse> optimize() {
        // check this group status
        checkStatus();

        return updateQueryCallback(new UpdateProcessCallback<UpdateResponse>() {
            public UpdateResponse callback(SolrServer solrServer)
                    throws Exception {
                return solrServer.optimize();
            }
        });
    }

    public Collection<UpdateResponse> optimize(final boolean waitFlush,
            final boolean waitSearcher) {
        // check this group status
        checkStatus();

        return updateQueryCallback(new UpdateProcessCallback<UpdateResponse>() {
            public UpdateResponse callback(SolrServer solrServer)
                    throws Exception {
                return solrServer.optimize(waitFlush, waitSearcher);
            }
        });
    }

    public Collection<UpdateResponse> optimize(final boolean waitFlush,
            final boolean waitSearcher, final int maxSegments) {
        // check this group status
        checkStatus();

        return updateQueryCallback(new UpdateProcessCallback<UpdateResponse>() {
            public UpdateResponse callback(SolrServer solrServer)
                    throws Exception {
                return solrServer
                        .optimize(waitFlush, waitSearcher, maxSegments);
            }
        });
    }

    public Collection<SolrPingResponse> ping() {
        // check this group status
        checkStatus();

        return updateQueryCallback(new UpdateProcessCallback<SolrPingResponse>() {
            public SolrPingResponse callback(SolrServer solrServer)
                    throws Exception {
                return solrServer.ping();
            }
        });
    }

    public QueryResponse query(SolrParams params) {
        return query(params, SolrRequest.METHOD.GET);
    }

    public QueryResponse query(SolrParams params, METHOD method) {
        // check this group status
        checkStatus();

        FessSolrQueryException fsqe = null;
        for (int i = 0; i < maxRetrySelectQueryCount; i++) {
            try {
                return queryInternal(params, method);
            } catch (Exception e) {
                if (fsqe == null) {
                    fsqe = new FessSolrQueryException("Failed query: "
                            + params.toString());
                }
                fsqe.addException(e);
            }
            try {
                Thread.sleep(retrySelectQueryInterval);
            } catch (InterruptedException e1) {
            }
        }
        throw fsqe;
    }

    private QueryResponse queryInternal(SolrParams params, METHOD method) {

        String solrServerName = null;
        SolrServer solrServer = null;
        try {
            int minValue = Integer.MAX_VALUE;
            // get a server which is  
            for (Map.Entry<String, SolrServer> entry : solrServerMap.entrySet()) {
                Integer count = solrServerAccessCountMap.get(entry.getKey());
                if (count == null) {
                    // set count
                    count = 1;
                    solrServerAccessCountMap.put(entry.getKey(), count);
                }
                if (count < minValue || solrServer == null) {
                    String serverStatusKey = groupName + "." + entry.getKey()
                            + ".status";
                    String serverStatus = solrServerProperties.getProperty(
                            serverStatusKey, Constants.ACTIVE);
                    if (StringUtil.isBlank(serverStatus)
                            || serverStatus.equals(Constants.ACTIVE)) {
                        // active
                        minValue = count;
                        solrServerName = entry.getKey();
                        solrServer = entry.getValue();
                    } else if (!serverStatus.equals(Constants.CORRUPTED)
                            && !serverStatus.equals(Constants.INACTIVE)) {
                        // unknown
                        if (logger.isInfoEnabled()) {
                            logger.info("Unknown server status. "
                                    + serverStatusKey + " is " + serverStatus
                                    + ". Change a status to INACTIVE.");
                        }
                        synchronized (solrServerProperties) {
                            solrServerProperties.setProperty(serverStatusKey,
                                    Constants.INACTIVE);
                            solrServerProperties.store();
                        }
                    }
                }
            }

            if (solrServer == null) {
                // If all server is unavailable, server group will be 
                // inactive at the next access
                throw new FessSolrException("No active SolrServer in "
                        + groupName + " group.");
            }

            // update count
            if (minValue > MAX_LOAD_COUNT) {
                // clear all access counts
                solrServerAccessCountMap.clear();
            } else {
                solrServerAccessCountMap.put(solrServerName, minValue + 1);
            }

            QueryResponse queryResponse = solrServer.query(params, method);

            // clear error count
            solrServerErrorCountMap.put(solrServerName, 0);

            return queryResponse;
        } catch (SolrServerException e) {
            if (solrServerName != null) {
                synchronized (solrServerProperties) {
                    Integer errorCount = solrServerErrorCountMap
                            .get(solrServerName);
                    if (errorCount == null) {
                        // set a error count
                        solrServerErrorCountMap.put(solrServerName, 1);
                    } else if (errorCount > maxErrorCount) {
                        // clear error count
                        solrServerErrorCountMap.put(solrServerName, 0);
                        // inactivate a server
                        String serverStatusKey = groupName + "."
                                + solrServerName + ".status";
                        solrServerProperties.setProperty(serverStatusKey,
                                Constants.INACTIVE);
                        solrServerProperties.store();
                        throw new FessSolrException(String.format(
                                "An exception occurs at SolrServer(%s) of %s group. "
                                        + "The retry count is %d(<%d).",
                                solrServerName, groupName, errorCount,
                                maxErrorCount), e);
                    } else {
                        // set a error count
                        solrServerErrorCountMap.put(solrServerName,
                                errorCount + 1);
                    }
                    throw new FessSolrException(String.format(
                            "An exception occurs at SolrServer(%s) of %s group. "
                                    + "This server is used again "
                                    + "if a retry count is not exceeded. "
                                    + "The retry count is %d(<%d).",
                            solrServerName, groupName, errorCount,
                            maxErrorCount), e);
                }
            } else {
                throw new FessSolrException(String.format("Failed query: %s",
                        params.toString()), e);
            }
        }
    }

    protected <T> Collection<T> updateQueryCallback(UpdateProcessCallback<T> upc) {
        List<T> resultList = new ArrayList<T>();
        synchronized (solrServerProperties) {
            for (Map.Entry<String, SolrServer> entry : solrServerMap.entrySet()) {
                String serverStatusKey = groupName + "." + entry.getKey()
                        + ".status";
                String serverStatus = solrServerProperties
                        .getProperty(serverStatusKey);
                if (serverStatus.equals(Constants.ACTIVE)) {
                    executeUpdateQuery(upc, resultList, entry, serverStatusKey,
                            0);
                }
            }

        }
        return resultList;
    }

    private <T> void executeUpdateQuery(UpdateProcessCallback<T> upc,
            List<T> resultList, Map.Entry<String, SolrServer> entry,
            String serverStatusKey, int retryCount) {
        try {
            resultList.add(upc.callback(entry.getValue()));
        } catch (Exception e) {
            logger.warn("Failed to execute an update query for "
                    + serverStatusKey + ". The query is " + upc
                    + ". The retry count is " + retryCount + ".", e);

            if (retryCount > maxRetryUpdateQueryCount) {
                // set to corrupted
                solrServerProperties.setProperty(serverStatusKey,
                        Constants.CORRUPTED);
                solrServerProperties.store();
            } else {
                try {
                    Thread.sleep(retryUpdateQueryInterval);
                } catch (InterruptedException e1) {
                }

                // retry
                executeUpdateQuery(upc, resultList, entry, serverStatusKey,
                        retryCount + 1);
            }
        }
    }

    protected void checkStatus() {
        // if this server group is unavailable, throws FessSolrException.
        if (!isActive(maxRetryStatusCheckCount)) {
            throw new FessSolrException("SolrGroup(" + groupName
                    + ") is not available.");
        }
    }

    protected boolean isActive(int retryCount) {
        // TODO performance

        // check the number of an active server
        int numOfActive = 0;
        for (String serverName : solrServerMap.keySet()) {
            String serverStatusKey = groupName + "." + serverName + ".status";
            String serverStatus = solrServerProperties
                    .getProperty(serverStatusKey);
            if (StringUtil.isBlank(serverStatus)) {
                // status is null
                numOfActive++;
                synchronized (solrServerProperties) {
                    solrServerProperties.setProperty(serverStatusKey,
                            Constants.ACTIVE);
                    solrServerProperties.store();
                }
            } else if (serverStatus.equals(Constants.ACTIVE)) {
                // active
                numOfActive++;
            } else if (!serverStatus.equals(Constants.INACTIVE)
                    && !serverStatus.equals(Constants.CORRUPTED)) {
                // unknown
                // set a status to inactive
                synchronized (solrServerProperties) {
                    solrServerProperties.setProperty(serverStatusKey,
                            Constants.INACTIVE);
                    solrServerProperties.store();
                }
            }
        }

        String groupStatusKey = groupName + ".status";
        String groupStatus = solrServerProperties.getProperty(groupStatusKey,
                Constants.INACTIVE);
        if (numOfActive >= minActiveServer) {
            if (groupStatus.equals(Constants.INACTIVE)) {
                // activate a server group
                synchronized (solrServerProperties) {
                    solrServerProperties.setProperty(groupStatusKey,
                            Constants.ACTIVE);
                    solrServerProperties.store();
                }
            }
            // this server group is active
            return true;
        }

        // The number of an active server is under minActiveServer.
        // This server group is unavailable

        if (!groupStatus.equals(Constants.INACTIVE)) {
            // if a status of this server group is active
            if (retryCount > 0) {
                // check if a server is alive by ping
                checkServersByPing();
                try {
                    Thread.sleep(retryStatusCheckInterval);
                } catch (InterruptedException e) {
                }
                return isActive(retryCount - 1);
            }
            synchronized (solrServerProperties) {
                solrServerProperties.setProperty(groupStatusKey,
                        Constants.INACTIVE);
                solrServerProperties.store();
            }
        }

        // this server group is inactive
        return false;
    }

    private void checkServersByPing() {
        // check a status for all servers 
        synchronized (solrServerProperties) {
            for (Map.Entry<String, SolrServer> entry : solrServerMap.entrySet()) {
                String serverStatusKey = groupName + "." + entry.getKey()
                        + ".status";
                String serverStatus = solrServerProperties.getProperty(
                        serverStatusKey, Constants.ACTIVE);
                if (Constants.INACTIVE.equals(serverStatus)) {
                    // you can activate a server when the status is inactive only..
                    try {
                        SolrPingResponse pingResponse = entry.getValue().ping();
                        if (pingResponse.getStatus() == 0) {
                            // activate a server
                            solrServerProperties.setProperty(serverStatusKey,
                                    Constants.ACTIVE);
                        } else {
                            logger.warn("A solr server (" + serverStatusKey
                                    + ") is still not available. "
                                    + "The ping status is "
                                    + pingResponse.getStatus());
                            // inactivate a server
                            solrServerProperties.setProperty(serverStatusKey,
                                    Constants.INACTIVE);
                        }
                    } catch (Exception e) {
                        logger.warn("A solr server (" + serverStatusKey
                                + ") is not available.", e);
                        solrServerProperties.setProperty(serverStatusKey,
                                Constants.INACTIVE);
                    }
                    solrServerProperties.store();
                }
            }
        }
    }

    public String getGroupName() {
        return groupName;
    }

    public void setGroupName(String groupName) {
        this.groupName = groupName;
    }

    protected static interface UpdateProcessCallback<V> {
        public abstract V callback(SolrServer solrServer) throws Exception;
    }

}
