package jp.sf.fess.ds.impl;

import java.util.ArrayList;
import java.util.List;
import java.util.Map;

import jp.sf.fess.FessSystemException;
import jp.sf.fess.ds.IndexUpdateCallback;
import jp.sf.fess.helper.CrawlingSessionHelper;
import jp.sf.fess.solr.SolrServerGroup;

import org.apache.solr.common.SolrInputDocument;
import org.seasar.framework.container.SingletonS2Container;
import org.slf4j.Logger;
import org.slf4j.LoggerFactory;

public class IndexUpdateCallbackImpl implements IndexUpdateCallback {
    private static final Logger logger = LoggerFactory
            .getLogger(IndexUpdateCallbackImpl.class);

    protected SolrServerGroup solrServerGroup;

    public int maxDocumentCacheSize = 10;

    protected long documentSize = 0;

    protected long commitPerCount = 0;

    protected long executeTime = 0;

    final List<SolrInputDocument> docList = new ArrayList<SolrInputDocument>();

    /* (non-Javadoc)
     * @see jp.sf.fess.ds.impl.IndexUpdateCallback#store(java.util.Map)
     */
    public synchronized boolean store(Map<String, Object> dataMap) {
        long startTime = System.currentTimeMillis();

        if (logger.isDebugEnabled()) {
            logger.debug("Adding " + dataMap);
        }

        // TODO required check
        if (!dataMap.containsKey("url") || dataMap.get("url") == null) {
            throw new FessSystemException("url is null. dataMap=" + dataMap);
        }

        CrawlingSessionHelper crawlingSessionHelper = SingletonS2Container
                .getComponent(CrawlingSessionHelper.class);
        dataMap.put("id", crawlingSessionHelper.generateId(dataMap));

        SolrInputDocument doc = new SolrInputDocument();
        for (Map.Entry<String, Object> entry : dataMap.entrySet()) {
            if ("boost".equals(entry.getKey())) {
                // boost
                float documentBoost = Float
                        .valueOf(entry.getValue().toString());
                doc.setDocumentBoost(documentBoost);
                if (logger.isDebugEnabled()) {
                    logger.debug("Set a document boost (" + documentBoost
                            + ").");
                }
            }
            doc.addField(entry.getKey(), entry.getValue());
        }

        docList.add(doc);
        if (logger.isDebugEnabled()) {
            logger.debug("Added the document. "
                    + "The number of a document cache is " + docList.size()
                    + ".");
        }

        if (docList.size() > maxDocumentCacheSize) {
            sendDocuments();
        }
        documentSize++;
        // commit
        if (commitPerCount > 0 && documentSize % commitPerCount == 0) {
            if (!docList.isEmpty()) {
                sendDocuments();
            }
            commitDocuments();
        }
        if (logger.isDebugEnabled()) {
            logger.debug("The number of an added document is " + documentSize
                    + ".");
        }

        executeTime += System.currentTimeMillis() - startTime;
        return true;
    }

    public void commit() {
        sendDocuments();
        commitDocuments();
    }

    protected void commitDocuments() {
        long execTime = System.currentTimeMillis();
        if (logger.isInfoEnabled()) {
            logger.info("Committing documents. ");
        }
        solrServerGroup.commit();
        if (logger.isInfoEnabled()) {
            logger.info("Committed documents. The execution time is "
                    + (System.currentTimeMillis() - execTime) + "ms.");
        }
    }

    protected void sendDocuments() {
        long execTime = System.currentTimeMillis();
        if (logger.isInfoEnabled()) {
            logger.info("Sending " + docList.size() + " document to a server.");
        }
        solrServerGroup.add(docList);
        if (logger.isInfoEnabled()) {
            logger.info("Sent " + docList.size()
                    + " documents. The execution time is "
                    + (System.currentTimeMillis() - execTime) + "ms.");
        }
        docList.clear();
    }

    public long getDocumentSize() {
        return documentSize;
    }

    public long getExecuteTime() {
        return executeTime;
    }

    public SolrServerGroup getSolrServerGroup() {
        return solrServerGroup;
    }

    public void setSolrServerGroup(SolrServerGroup solrServerGroup) {
        this.solrServerGroup = solrServerGroup;
    }

    public void setCommitPerCount(long commitPerCount) {
        this.commitPerCount = commitPerCount;
    }

}
