/*
 * The MIT License

   Conograph (powder auto-indexing program)

Copyright (c) <2012> <Ryoko Oishi-Tomiyasu, KEK>

Permission is hereby granted, free of charge, to any person obtaining a copy
of this software and associated documentation files (the "Software"), to deal
in the Software without restriction, including without limitation the rights
to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
copies of the Software, and to permit persons to whom the Software is
furnished to do so, subject to the following conditions:

The above copyright notice and this permission notice shall be included in
all copies or substantial portions of the Software.

THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN
THE SOFTWARE.
 *
 */
#ifndef LatticeFigureOfMeritToDisplay_HH_
#define LatticeFigureOfMeritToDisplay_HH_

#include <map>
#include "LatticeFigureOfMeritZeroShift.hh"
#include "../utility_data_structure/range.hh"


// Class for outputting information about a lattice in IGOR file.
class LatticeFigureOfMeritToDisplay
{
private:
	LatticeFigureOfMeritZeroShift m_latfom;
	
	vector<QData> m_qdata;
	vector< multimap<Double, vector<HKL_Q>::const_iterator> > m_associated_hkl_tray;
	vector<HKL_Q> m_cal_hkl_tray;
	
	// The size of the following arrays equals zero or m_figures_of_merit.m_num_ref_figure_of_merit.
	vector< VecDat3<Int4> > m_hkl_to_fit;
	vector<bool> m_fix_or_fit_flag;	// 0:fix, 1:fit.

	void putPeakPosToFit(const ControlParam& cdata, Vec_DP& cal_q_tray, Vec_DP& cal_pos_tray) const;
	void putStandardMillerIndicesToFit(vector< VecDat3<Int4> >& hkl_to_fit,
										vector<bool>& fix_or_fit_flag) const;

public:
	LatticeFigureOfMeritToDisplay();
	~LatticeFigureOfMeritToDisplay(){};

	inline const eCrystalSystem& enumCrystalSystem() const { return m_latfom.enumCrystalSystem(); };
	inline const LatticeFigureOfMeritZeroShift& putLatticeFigureOfMerit() const { return m_latfom; };
	inline const vector<HKL_Q>& putCalMillerIndices() const { return m_cal_hkl_tray; };
	inline const vector< multimap<Double, vector<HKL_Q>::const_iterator> >& putAssociatedMillerIndices() const { return m_associated_hkl_tray; };

	// axis1: axis of the argument lattice constants for Monoclinic(C) or Orthorhombic(C).
	// axis2: axis of the argument lattice constants for Rhombohedral.
	inline ZErrorMessage setLatticeConstantsDegree(const eCrystalSystem& ecys,
													const eABCaxis& axis1,
													const eRHaxis& axis2,
													const VecDat3<Double>& length_axis,
													const VecDat3<Double>& angle_axis) { return m_latfom.setLatticeConstantsDegree(BravaisType(ecys, axis1, axis2), length_axis, angle_axis); };
	inline ZErrorMessage setPeakShiftParamDegree(const ePeakShiftFunctionType& type,
													const Double& wave_length,
													const vector<ZParawError>& peak_shift_param_deg,	// The errors are not used in this method.
													const PeakPosData& pdata);
	inline void reduceLatticeConstants();

	inline void putOptimizedLatticeConstantsDegree(const eABCaxis& axis1,
													const eRHaxis& axis2,
													VecDat3<Double>& length_axis,
													VecDat3<Double>& angle_axis) { m_latfom.putOptimizedLatticeConstantsDegree(axis1, axis2, length_axis, angle_axis); };

	// axis1: axis of the argument lattice constants for Monoclinic(C) or Orthorhombic(C).
	// axis2: axis of the argument lattice constants for Rhombohedral.
	inline void putReducedLatticeConstantsDegree(const eABCaxis& axis1,
													const eRHaxis& axis2,
													VecDat3<Double>& length_axis,
													VecDat3<Double>& angle_axis) { m_latfom.putReducedLatticeConstantsDegree(axis1, axis2, length_axis, angle_axis); };

	inline const ePeakShiftFunctionType& putPeakShiftFunctionType() const { return m_latfom.putPeakShiftFunctionType(); };
	inline vector<ZParawError> putPeakShiftParamDegree() const { return m_latfom.putPeakShiftParamDegree(); };

	inline const vector<QData>& putQDataModifiedWithNewPeakShiftParam() const { return m_qdata; };

	// Return false if Qdata is not set or
	// the number of unindexed reflections is larger max_num_false_peak.
	inline void setFigureOfMerit(const Int4& num_ref_figure_of_merit){ m_latfom.setFigureOfMerit(num_ref_figure_of_merit, m_qdata); };

	// Resets m_associated_hkl_tray and m_cal_hkl_tray.
	void resetMillerIndicesInRange(const Int4& num_fit_data);
	// Resets m_hkl_to_fit and m_fix_or_fit_flag.
	void resetMillerIndicesToFit() { putStandardMillerIndicesToFit(m_hkl_to_fit, m_fix_or_fit_flag); };

	inline ZErrorMessage setFittingIDs(const vector<bool>& arg);
	inline const vector<bool>& putFittingIDs() const { return m_fix_or_fit_flag; };
	inline ZErrorMessage setMillerIndicesToFit(const vector< VecDat3<Int4> >& arg);
	inline const vector< VecDat3<Int4> >& putMillerIndicesToFit() const { return m_hkl_to_fit; };

	// Return peak positions for this lattice candidate.
	void putCalculatedPeakPosInRange(const ControlParam& cdata, Vec_DP& cal_pos_tray) const;

	// Returned value
	// > 1 : Optimization has succeeded at least twice. (The members m_hkl_to_fit and m_fix_or_fit_flag are changed.)
	// > 0 : Optimization has succeeded. (Lattice constants are changed.)
	// 0 : Optimization has failed.
	Int4 fitLatticeParameter(const PeakPosData& pdata, const vector<etype_ID>& fitflag,
								const Int4& Max_ITNUM,
								const Double& limiter);

	// Output indexing results.
	void printIndexingResult(const ControlParam& cdata,
						const PeakPosData& pdata,
						const Int4& label_start0,
						ostream* os) const;

	// For GUI.
	const vector<bool>           &getref_m_fix_or_fit_flag()  const {return m_fix_or_fit_flag;}
	      vector<bool>           &getref_m_fix_or_fit_flag()        {return m_fix_or_fit_flag;}
	const vector<VecDat3<Int4> > &getref_m_hkl_to_fit()       const {return m_hkl_to_fit;}
	      vector<VecDat3<Int4> > &getref_m_hkl_to_fit()             {return m_hkl_to_fit;}
};


inline ZErrorMessage LatticeFigureOfMeritToDisplay::setFittingIDs(const vector<bool>& arg)
{
	if( arg.size() != m_associated_hkl_tray.size() )
	{
		return ZErrorMessage(ZErrorArgmentSize, __FILE__, __LINE__, __FUNCTION__);
	}
	 m_fix_or_fit_flag = arg;
	return ZErrorMessage();
}

inline ZErrorMessage LatticeFigureOfMeritToDisplay::setMillerIndicesToFit(const vector< VecDat3<Int4> >& arg)
{
	if( arg.size() != m_associated_hkl_tray.size() )
	{
		return ZErrorMessage(ZErrorArgmentSize, __FILE__, __LINE__, __FUNCTION__);
	}
	m_hkl_to_fit = arg;
	return ZErrorMessage();
};


inline ZErrorMessage LatticeFigureOfMeritToDisplay::setPeakShiftParamDegree(
		const ePeakShiftFunctionType& type,
		const Double& wave_length,
		const vector<ZParawError>& peak_shift_param_deg,
		const PeakPosData& pdata)
{
	static const Double RadDeg =  PI() / 180.0;

	vector<ZParawError> peak_shift_param_rad = peak_shift_param_deg;
	for(vector<ZParawError>::iterator it=peak_shift_param_rad.begin(); it<peak_shift_param_rad.end(); it++)
	{
		*it *= RadDeg;
	}
	return m_latfom.setPeakShiftParamRadian(VCData::putPeakQData(), type, wave_length, peak_shift_param_rad, pdata,
											VCData::putPeakQData().size(), m_qdata);
};

inline void LatticeFigureOfMeritToDisplay::reduceLatticeConstants()
{
	NRMat<Int4> trans_mat;
	m_latfom.reduceLatticeConstants(trans_mat);

	for(vector<HKL_Q>::iterator it=m_cal_hkl_tray.begin(); it<m_cal_hkl_tray.end(); it++)
	{
		it->HKL() = product_hkl(it->HKL(), trans_mat);
	}

	for(vector< VecDat3<Int4> >::iterator it=m_hkl_to_fit.begin(); it<m_hkl_to_fit.end(); it++)
	{
		*it = product_hkl(*it, trans_mat);
	}
};

#endif /*LatticeFigureOfMeritToDisplay_HH_*/
