<?php
// $Id: rdbobject.php,v 1.2 2003/06/15 12:42:34 haruki Exp $
//
// RdbObject and its Manager
//
// Copyright (C) Haruki Setoyama <pwaf.haruki@planewave.org>
//
// This library is free software; you can redistribute it and/or
// modify it under the terms of the GNU Lesser General Public
// License as published by the Free Software Foundation; either
// version 2.1 of the License, or any later version.
//
// This library is distributed in the hope that it will be useful,
// but WITHOUT ANY WARRANTY; without even the implied warranty of
// MERCHANTABILITY or FITNESS FOR A PARTICULAR PURPOSE.  See the GNU
// Lesser General Public License for more details.
//
// You should have received a copy of the GNU Lesser General Public
// License along with this library; if not, write to the Free Software
// Foundation, Inc., 59 Temple Place, Suite 330, Boston, MA  02111-1307  USA
//
/**
* RdbObject and Manager
* class to manupulate data stored in relational database server
*
* @auther Haruki Setoyama <pwaf.haruki@planewave.org>
* @lisence LGPL
* @since 2003.06
*/
/**
* rdbObject
*
* @access  public
*/
class rdbObject
{
    //
    var $dbRawData = array();
    var $change = array();

    var $relatedData = array();

    var $handler;

    /**
    * reference to manager's variable
    * @access private
    */
    var $element;
    var $pkey;

    function get($varname)
    {
        return $this->element[$varname]->get($this->dbRawData);
    }

    function pkey()
    {
        return $this->get($this->pkey);
    }

    function set($varname, $value)
    {
        return $this->element[$varname]->set($this, $value);
    }

    function reset($varname)
    {
        if(isset($this->relatedData[$varname]))
            $this->relatedData[$varname] = null;
        return $this->element[$varname]->reset($this);
    }

    function getError($varname)
    {
        return $this->element[$varname]->error;
    }

    ///

    function loadRelated($varname)
    {
        $this->relatedData[$varname] = $this->element[$varname]->getRelated($this);
    }

    function &getRelated($varname)
    {
        if(!isset($this->relatedData[$varname]))
        {
            $this->loadRelated($varname);
        }
        return $this->relatedData[$varname];
    }

    function save()
    {
        $handler =& singleton::getInstance($this->handler);
        return $handler->save($this);
    }

    function delete()
    {
        $handler =& singleton::getInstance($this->handler);
        return $handler->delete($this);
    }

}
/**
* rdbObjectDebug
*
* @access  public
*/
class rdbObjectDebug extends rdbObject
{
    function _chack_varname($varname)
    {
        if(!isset($this->element[$varname]))
        {
            trigger_error(
                get_class($this).'::variable name "'.$varname.'" do not exists in '.$this->handler.' '
                , E_USER_WARNING
            );
            return false;
        }
        return true;
    }

    function get($varname)
    {
        if(!$this->_chack_varname($varname)) return null;
        return parent::get($varname);
    }

    function set($varname, $value)
    {
        if(!$this->_chack_varname($varname)) return false;
        return parent::set($varname, $value);
    }

    function reset($varname)
    {
        if(!$this->_chack_varname($varname)) return false;
        return parent::reset($varname);
    }

    function getError($varname)
    {
        if(!$this->_chack_varname($varname)) return false;
        return parent::getError($varname);
    }

    ///

    function loadRelated($varname)
    {
        if(!$this->_chack_varname($varname)) return false;
        parent::loadRelated($varname);
    }

    function &getRelated($varname)
    {
        if(!$this->_chack_varname($varname)) return null;
        return parent::getRelated($varname);
    }
}

/////////////////////////////////////////////////
/**
* define ERROR code
*/
define('RDBO_NO_ERROR',0);
define('RDBO_ERROR_BELOW_MIN',1);
define('RDBO_ERROR_BEYOND_MAX',2);
define('RDBO_ERROR_INVALID',3);
define('RDBO_ERROR_NOT_EXISTS', 4);
/**
* rdbObjectElement.
* base class to manipulate data with type specific way
*
* @access  abstract
*/
class rdbObjectElement
{
    var $option = array();

    var $varname;
    var $column;

    var $error = RDBO_NO_ERROR;


    function get(&$dbRawData)
    {
        return $dbRawData[$this->column];
    }

    function set(&$self, $value)
    {
        if($value === null)
        {
            if(empty($this->option['null'])) return false;
        }
        elseif(! $this->check($value)) return false;

        $self->dbRawData[$this->column] = $value;
        $self->change[$this->varname] = true;
        return true;
    }

    function reset(&$self)
    {
        // $value is ignored
        if(isset($this->option['default']))
            $self->dbRawData[$this->column] = $this->option['default'];
        elseif(!empty($this->option['null']))
            $self->dbRawData[$this->column] = null;
        else
            return false;
        $self->change[$this->varname] = true;
        return true;
    }

    function check(&$value)
    {
        // OVERRIDE this function
        return true;
    }

    function &getRelated($self)
    {
        return null;
    }

    function preSave(&$self)
    {
        if($self->dbRawData[$this->column] === null
            && empty($this->option['null']))
        {
            return false;
        }
        return true;
    }
}

/**
* rdbObjectElement_text.
* for normal text.
*
* @access  public
*/
class rdbObjectElement_text extends rdbObjectElement
{
    function check(&$value)
    {
        if(is_string($value) || is_int($value) || is_float($value) )
        {
            $len = strlen((string) $value);
            if(isset($this->option['min']) && $len < $this->option['min'])
            {
                $this->error = RDBO_ERROR_BELOW_MIN;
                return false;
            }
            if(isset($this->option['max']) && $len > $this->option['max'])
            {
                $this->error = RDBO_ERROR_BEYOND_MAX;
                return false;
            }
            $this->error = RDBO_NO_ERROR;
            return true;
        }
        else
        {
            $this->error = RDBO_ERROR_INVALID;
            return false;
        }
    }
}

/**
* rdbObjectElement_password.
* the password will be convertd by MD5
*
* @access  public
*/
class rdbObjectElement_password extends rdbObjectElement_text
{
    function check(&$value)
    {
        if(! rdbObjectElement_text::check($value)) return false;
        $value = md5($value);
        return true;
    }
}

/**
* rdbObjectElement_array.
* this uses serialize() functon to save.
*
* @access  public
*/
class rdbObjectElement_array extends rdbObjectElement
{
    function check(&$value)
    {
        if(! is_array($value))
        {
            $this->error = RDBO_ERROR_INVALID;
            return false;
        }

        $serialized = serialize($value);
//echo $serialized.'<br>';
        if(isset($this->option['max']))
        {

            if(strlen($serialized) > $this->option['max'])
            {
                $this->error = RDBO_ERROR_BEYOND_MAX;
                return false;
            }
        }
        $value = $serialized;
        return true;
    }

    function get(&$dbRawData)
    {
        return ($dbRawData[$this->column] !== null)
                ? unserialize($dbRawData[$this->column])
                : null;
    }

}

/**
* rdbObjectNumericElement.
* child class of this handles only numeric data.
*
* @access  abstract
*/
class rdbObjectNumericElement extends rdbObjectElement
{

}

/**
* rdbObjectElement_int.
* integer.
*
* @access  public
*/
class rdbObjectElement_int extends rdbObjectNumericElement
{
    function check(&$value)
    {
        if(! is_int($value))
        {
            $this->error = RDBO_ERROR_INVALID;
            return false;
        }
        if(isset($this->option['min']) && $value < $this->option['min'])
        {
            $this->error = RDBO_ERROR_BELOW_MIN;
            return false;
        }
        if(isset($this->option['max']) && $value > $this->option['max'])
        {
            $this->error = RDBO_ERROR_BEYOND_MAX;
             return false;
        }
        $this->error = RDBO_NO_ERROR;
        return true;
    }
}

/**
* rdbObjectElement_numeric.
* general numeric type.
*
* @access  public
*/
class rdbObjectElement_numeric extends rdbObjectNumericElement {

    function check(&$value)
    {
        if(! is_numeric($value))
        {
            $this->error = RDBO_ERROR_INVALID;
            return false;
        }
        if(isset($this->option['decimal']))
        {
            $value = floatval(sprintf('%.'.intval($this->option['decimal']).'f', $value));
        }
        if(isset($this->option['min']) && $value < $this->option['min'])
        {
            $this->error = RDBO_ERROR_BELOW_MIN;
            return false;
        }
        if(isset($this->option['max']) && $value > $this->option['max'])
        {
            $this->error = RDBO_ERROR_BEYOND_MAX;
            return false;
        }
        $this->error = RDBO_NO_ERROR;
        return true;
    }
}

/**
* rdbObjectElement_bool.
* bool. 1 means true, 0 means false.
*
* @access  public
*/
class rdbObjectElement_bool extends rdbObjectNumericElement {

    function check(&$value)
    {
        $value = (bool)$value ? 1: 0;
        return true;
    }
}

/**
* rdbObjectElement_bool.
* bool. 1 means true, 0 means false.
*
* @access  public
*/
class rdbObjectElement_updatetime extends rdbObjectNumericElement {

    function check(&$value)
    {
        $this->error = RDBO_ERROR_INVALID;
        return false;
    }

    function preSave(&$self)
    {
        $self->dbRawData[$this->column] = time();
        $self->change[$this->varname] = true;
        return true;
    }

}

/**
* rdbObjectElement_pkey.
* primary key. positive integer. 0 means 'not saved'.
*
* @access  public
*/
class rdbObjectElement_pkey extends rdbObjectNumericElement
{
    function check(&$value)
    {
        if(! is_int($value))
        {
            $this->error = RDBO_ERROR_INVALID;
            return false;
        }
        if($value < 0)
        {
            $this->error = RDBO_ERROR_BELOW_MIN;
            return false;
        }
        $this->error = RDBO_NO_ERROR;
        return true;
    }
}

/**
* rdbObjectElement_fkey.
* foreign key.
* option['related'] is the handle class name which the key indicates.
*
* @access  public
*/
class rdbObjectElement_fkey extends rdbObjectNumericElement
{
    function set(&$self, $value)
    {
        if(is_a($value, 'rdbObject'))
        {
            $value = $value->pkey();
        }
        if(! parent::set($self, $value)) return false;
        $self->relatedData[$this->varname] = null;
        return true;
    }

    function remove(&$self, $value=null)
    {
        // $value is ignored
        return $this->set($self, 0);
    }

    function check(&$value)
    {
        if(! is_int($value))
        {
            $this->error = RDBO_ERROR_INVALID;
            return false;
        }
        if($value < 0 || ($value == 0 && empty($this->option['zero'])))
        {
            $this->error = RDBO_ERROR_BELOW_MIN;
            return false;
        }
        $this->error = RDBO_NO_ERROR;
        return true;
    }

    function getRelated(&$self)
    {
        $key = $self->dbRawData[$this->column];
        $handler_related =& singleton::getInstance($this->option['related']);
        return $handler_related->openOne($key);
    }
}

/**
* rdbObjectVirtualElement.
* child class of this has no column in the database.
*
* @access  abstract
*/
class rdbObjectVirtualElement extends rdbObjectElement
{
    function check(&$value)
    {
        $this->error = RDBO_ERROR_INVALID;
        return false;
    }

    function deleteRelated($self)
    {
    }

}

/**
* rdbObjectElement_sub
*
* @access  public
*/
class rdbObjectElement_sub extends rdbObjectVirtualElement
{
    function getRelated(&$self)
    {
        $pkey = $self->pkey();
        if(empty($pkey)) //when not saved
        {
            trigger_error(get_class($this).'::getRelated()::no related objects for unseved.'
                                , E_USER_NOTICE);
            return null;
        }
        $h_related =& singleton::getInstance($this->option['related']);
        return $h_related->openWith($this->option['fkey'], $pkey);
    }

    function deleteRelated(&$self)
    {
        $h_related =& singleton::getInstance($this->option['related']);
        $relateds = $h_related->openWith($this->option['fkey'], $self->pkey());
        if(empty($relateds)) return true;
        if(!empty($h_related->element[$this->option['fkey']]->option['zero']))
        {
            foreach($relateds as $related)
            {
                $related->set($this->option['fkey'], 0);
                $related->save();
            }
        }
        else
        {
            foreach($relateds as $related)
            {
                $related->delete();
            }
        }
    }
}

/**
* rdbObjectElement_assoc
*
* @access  public
*/
class rdbObjectElement_assoc extends rdbObjectVirtualElement
{
    function &getRelated(&$self)
    {
        $pkey = $self->pkey();
        $related = singleton::getInstance($this->option['assoc']);
        return $related->openJoin($this->option['key_assoc'], $this->option['key_agent'], $pkey);
    }

    function deleteRelated(&$self)
    {
        $h_agent =& singleton::getInstance($this->option['agent']);
        $agents = $h_cramp->openWith($this->option['key_agent'], $self->pkey());
        if(empty($agents)) return true;
        foreach($agents as $agent)
        {
            $agent->delete();
        }
    }

}
?>