package org.maachang.reflect ;

import java.lang.reflect.Field;

/**
 * 高速リフレクション.
 *
 * @version 2009/03/29
 * @author  masahito suzuki
 * @since   FastReflect 1.00
 */
public abstract class FastReflect {
    private FastReflect() {}
    
    /**
     * クラス情報を取得.
     * @param name 対象のクラス名を設定します.
     * @return Class 対象のクラス情報が返されます.
     * @exception ClassNotFoundException ロードクラス例外.
     */
    public static final Class getClass( String name )
        throws ClassNotFoundException {
        return getClass( null,name ) ;
    }
    
    /**
     * クラス情報を取得.
     * @param loader 対象のクラスローダーを設定します.
     * @param name 対象のクラス名を設定します.
     * @return Class 対象のクラス情報が返されます.
     * @exception ClassNotFoundException ロードクラス例外.
     */
    public static final Class getClass( ClassLoader loader,String name )
        throws ClassNotFoundException {
        FastClassElement em = FastReflectClass.getClass( loader,name ) ;
        if( em == null ) {
            throw new ClassNotFoundException( "指定クラス " + name + " は存在しません" ) ;
        }
        return em.getClassObject() ;
    }
    
    /**
     * コンストラクタ実行.
     * @param name 対象のクラス名を設定します.
     * @return Object 生成されたオブジェクトが返されます.
     * @exception Exception 例外.
     */
    public static final Object newInstance( String name )
        throws Exception {
        return newInstance( null,name,(Object[])null ) ;
    }
    
    /**
     * コンストラクタ実行.
     * @param loader 対象のクラスローダーを設定します.
     * @param name 対象のクラス名を設定します.
     * @return Object 生成されたオブジェクトが返されます.
     * @exception Exception 例外.
     */
    public static final Object newInstance( ClassLoader loader,String name )
        throws Exception {
        return newInstance( loader,name,(Object[])null ) ;
    }
    
    /**
     * コンストラクタ実行.
     * @param name 対象のクラス名を設定します.
     * @param args 対象のコンストラクタ引数を設定します.
     * @return Object 生成されたオブジェクトが返されます.
     * @exception Exception 例外.
     */
    public static final Object newInstance( String name,Object... args )
        throws Exception {
        return newInstance( null,name,args ) ;
    }
    
    /**
     * コンストラクタ実行.
     * @param loader 対象のクラスローダーを設定します.
     * @param name 対象のクラス名を設定します.
     * @param args 対象のコンストラクタ引数を設定します.
     * @return Object 生成されたオブジェクトが返されます.
     * @exception Exception 例外.
     */
    public static final Object newInstance( ClassLoader loader,String name,Object... args )
        throws Exception {
        FastClassElement em = FastReflectClass.getClass( loader,name ) ;
        if( em == null ) {
            throw new ClassNotFoundException( "指定クラス " + name + " は存在しません" ) ;
        }
        Object ret ;
        Class[] types = FastReflectUtil.getParamsType( args ) ;
        while( true ) {
            if( ( ret = em.newInstance( types,args,loader ) ) != null ) {
                return ret ;
            }
            em = FastReflectClass.getClass( loader,em.getSuperClassName() ) ;
            if( em == null ) {
                throw new NoSuchMethodException(
                    "指定クラス " + name + " に対して、対象引数のコンストラクタは存在しません" ) ;
            }
        }
    }
    
    /**
     * コンストラクタ実行.
     * @param loader 対象のクラスローダーを設定します.
     * @param name 対象のクラス名を設定します.
     * @param args 対象のコンストラクタ引数を設定します.
     * @param types 対象のコンストラクタ引数タイプを設定します.
     * @return Object 生成されたオブジェクトが返されます.
     * @exception Exception 例外.
     */
    public static final Object newInstanceTo( ClassLoader loader,String name,Object[] args,Class[] types )
        throws Exception {
        FastClassElement em = FastReflectClass.getClass( loader,name ) ;
        if( em == null ) {
            throw new ClassNotFoundException( "指定クラス " + name + " は存在しません" ) ;
        }
        Object ret ;
        if( args != null && args.length > 0 && types == null ) {
            types = FastReflectUtil.getParamsType( args ) ;
        }
        while( true ) {
            if( ( ret = em.newInstance( types,args,loader ) ) != null ) {
                return ret ;
            }
            em = FastReflectClass.getClass( loader,em.getSuperClassName() ) ;
            if( em == null ) {
                throw new NoSuchMethodException(
                    "指定クラス " + name + " に対して、対象引数のコンストラクタは存在しません" ) ;
            }
        }
    }
    
    /**
     * フィールド設定.
     * @param clazzName 対象のクラス名を設定します.
     * @param target 設定対象のオブジェクトを設定します.<BR>
     *               [null]の場合、staticアクセスで処理します.
     * @param name 対象のフィールド名を設定します.
     * @param value 対象のパラメータ要素を設定します.
     * @exception Exception 例外.
     */
    public static final void setField( String clazzName,Object target,String name,Object value )
        throws Exception {
        setField( null,clazzName,target,name,value ) ;
    }
    
    /**
     * フィールド設定.
     * @param loader 対象のクラスローダーを設定します.
     * @param clazzName 対象のクラス名を設定します.
     * @param target 設定対象のオブジェクトを設定します.<BR>
     *               [null]の場合、staticアクセスで処理します.
     * @param name 対象のフィールド名を設定します.
     * @param value 対象のパラメータ要素を設定します.
     * @exception Exception 例外.
     */
    public static final void setField( ClassLoader loader,String clazzName,Object target,String name,Object value )
        throws Exception {
        FastClassElement em = FastReflectClass.getClass( loader,clazzName ) ;
        if( em == null ) {
            throw new ClassNotFoundException( "指定クラス " + clazzName + " は存在しません" ) ;
        }
        while( true ) {
            Field f = em.getField( ( target == null ),name ) ;
            if( f == null ) {
                em = FastReflectClass.getClass( loader,em.getSuperClassName() ) ;
                if( em == null ) {
                    throw new NoSuchFieldException(
                        "指定フィールド " + name + " はクラス " + clazzName + " に存在しません" ) ;
                }
                continue ;
            }
            f.set( target,value ) ;
            return ;
        }
    }
    
    /**
     * フィールド設定.
     * @param clazz 対象のクラスを設定します.
     * @param target 設定対象のオブジェクトを設定します.<BR>
     *               [null]の場合、staticアクセスで処理します.
     * @param name 対象のフィールド名を設定します.
     * @param value 対象のパラメータ要素を設定します.
     * @exception Exception 例外.
     */
    public static final void setField( Class clazz,Object target,String name,Object value )
        throws Exception {
        setField( null,clazz,target,name,value ) ;
    }
    
    /**
     * フィールド設定.
     * @param loader 対象のクラスローダーを設定します.
     * @param clazz 対象のクラスを設定します.
     * @param target 設定対象のオブジェクトを設定します.<BR>
     *               [null]の場合、staticアクセスで処理します.
     * @param name 対象のフィールド名を設定します.
     * @param value 対象のパラメータ要素を設定します.
     * @exception Exception 例外.
     */
    public static final void setField( ClassLoader loader,Class clazz,Object target,String name,Object value )
        throws Exception {
        String clazzName = clazz.getName() ;
        FastClassElement em = FastReflectClass.getClass( loader,clazzName ) ;
        if( em == null ) {
            throw new ClassNotFoundException( "指定クラス " + clazzName + " は存在しません" ) ;
        }
        while( true ) {
            Field f = em.getField( ( target == null ),name ) ;
            if( f == null ) {
                em = FastReflectClass.getClass( loader,em.getSuperClassName() ) ;
                if( em == null ) {
                    throw new NoSuchFieldException(
                        "指定フィールド " + name + " はクラス " + clazzName + " に存在しません" ) ;
                }
                continue ;
            }
            f.set( target,value ) ;
            return ;
        }
    }
    
    /**
     * フィールド取得.
     * @param clazzName 対象のクラス名を設定します.
     * @param target 設定対象のオブジェクトを設定します.<BR>
     *               [null]の場合、staticアクセスで処理します.
     * @param name 対象のフィールド名を設定します.
     * @return Object フィールドオブジェクト内容が返されます.
     * @exception Exception 例外.
     */
    public static final Object getField( String clazzName,Object target,String name )
        throws Exception {
        return getField( null,clazzName,target,name ) ;
    }
    
    /**
     * フィールド取得.
     * @param loader 対象のクラスローダーを設定します.
     * @param clazzName 対象のクラス名を設定します.
     * @param target 設定対象のオブジェクトを設定します.<BR>
     *               [null]の場合、staticアクセスで処理します.
     * @param name 対象のフィールド名を設定します.
     * @return Object フィールドオブジェクト内容が返されます.
     * @exception Exception 例外.
     */
    public static final Object getField( ClassLoader loader,String clazzName,Object target,String name )
        throws Exception {
        FastClassElement em = FastReflectClass.getClass( loader,clazzName ) ;
        if( em == null ) {
            throw new ClassNotFoundException( "指定クラス " + clazzName + " は存在しません" ) ;
        }
        while( true ) {
            Field f = em.getField( ( target == null ),name ) ;
            if( f == null ) {
                em = FastReflectClass.getClass( loader,em.getSuperClassName() ) ;
                if( em == null ) {
                    throw new NoSuchFieldException(
                        "指定フィールド " + name + " はクラス " + clazzName + " に存在しません" ) ;
                }
                continue ;
            }
            return f.get( target ) ;
        }
    }
    
    /**
     * フィールド取得.
     * @param clazz 対象のクラス名を設定します.
     * @param target 設定対象のオブジェクトを設定します.<BR>
     *               [null]の場合、staticアクセスで処理します.
     * @param name 対象のフィールド名を設定します.
     * @return Object フィールドオブジェクト内容が返されます.
     * @exception Exception 例外.
     */
    public static final Object getField( Class clazz,Object target,String name )
        throws Exception {
        return getField( null,clazz,target,name ) ;
    }
    
    /**
     * フィールド取得.
     * @param loader 対象のクラスローダーを設定します.
     * @param clazz 対象のクラス名を設定します.
     * @param target 設定対象のオブジェクトを設定します.<BR>
     *               [null]の場合、staticアクセスで処理します.
     * @param name 対象のフィールド名を設定します.
     * @return Object フィールドオブジェクト内容が返されます.
     * @exception Exception 例外.
     */
    public static final Object getField( ClassLoader loader,Class clazz,Object target,String name )
        throws Exception {
        String clazzName = clazz.getName() ;
        FastClassElement em = FastReflectClass.getClass( loader,clazzName ) ;
        if( em == null ) {
            throw new ClassNotFoundException( "指定クラス " + clazzName + " は存在しません" ) ;
        }
        while( true ) {
            Field f = em.getField( ( target == null ),name ) ;
            if( f == null ) {
                em = FastReflectClass.getClass( loader,em.getSuperClassName() ) ;
                if( em == null ) {
                    throw new NoSuchFieldException(
                        "指定フィールド " + name + " はクラス " + clazzName + " に存在しません" ) ;
                }
                continue ;
            }
            return f.get( target ) ;
        }
    }
    
    /**
     * メソッド呼び出し.
     * @param clazzName 対象のクラス名を設定します.
     * @param target 設定対象のオブジェクトを設定します.<BR>
     *               [null]の場合、staticアクセスで処理します.
     * @param name 対象のメソッド名を設定します.
     * @return Object 戻り値が返されます.
     * @exception Exception 例外.
     */
    public static final Object invoke( String clazzName,Object target,String name )
        throws Exception {
        return invoke( null,clazzName,target,name,(Object[])null ) ;
    }
    
    /**
     * メソッド呼び出し.
     * @param loader 対象のクラスローダーを設定します.
     * @param clazzName 対象のクラス名を設定します.
     * @param target 設定対象のオブジェクトを設定します.<BR>
     *               [null]の場合、staticアクセスで処理します.
     * @param name 対象のメソッド名を設定します.
     * @return Object 戻り値が返されます.
     * @exception Exception 例外.
     */
    public static final Object invoke( ClassLoader loader,String clazzName,Object target,String name )
        throws Exception {
        return invoke( loader,clazzName,target,name,(Object[])null ) ;
    }
    
    /**
     * メソッド呼び出し.
     * @param clazz 対象のクラスを設定します.
     * @param target 設定対象のオブジェクトを設定します.<BR>
     *               [null]の場合、staticアクセスで処理します.
     * @param name 対象のメソッド名を設定します.
     * @return Object 戻り値が返されます.
     * @exception Exception 例外.
     */
    public static final Object invoke( Class clazz,Object target,String name )
        throws Exception {
        return invoke( null,clazz,target,name,(Object[])null ) ;
    }
    
    /**
     * メソッド呼び出し.
     * @param loader 対象のクラスローダーを設定します.
     * @param clazz 対象のクラスを設定します.
     * @param target 設定対象のオブジェクトを設定します.<BR>
     *               [null]の場合、staticアクセスで処理します.
     * @param name 対象のメソッド名を設定します.
     * @return Object 戻り値が返されます.
     * @exception Exception 例外.
     */
    public static final Object invoke( ClassLoader loader,Class clazz,Object target,String name )
        throws Exception {
        return invoke( loader,clazz,target,name,(Object[])null ) ;
    }
    
    /**
     * メソッド呼び出し.
     * @param clazzName 対象のクラス名を設定します.
     * @param target 設定対象のオブジェクトを設定します.<BR>
     *               [null]の場合、staticアクセスで処理します.
     * @param name 対象のメソッド名を設定します.
     * @param args 対象のメソッドパラメータを設定します.
     * @return Object 戻り値が返されます.
     * @exception Exception 例外.
     */
    public static final Object invoke( String clazzName,Object target,String name,Object... args )
        throws Exception {
        return invoke( null,clazzName,target,name,args ) ;
    }
    
    /**
     * メソッド呼び出し.
     * @param loader 対象のクラスローダーを設定します.
     * @param clazzName 対象のクラス名を設定します.
     * @param target 設定対象のオブジェクトを設定します.<BR>
     *               [null]の場合、staticアクセスで処理します.
     * @param name 対象のメソッド名を設定します.
     * @param args 対象のメソッドパラメータを設定します.
     * @return Object 戻り値が返されます.
     * @exception Exception 例外.
     */
    public static final Object invoke( ClassLoader loader,String clazzName,Object target,String name,Object... args )
        throws Exception {
        FastClassElement em = FastReflectClass.getClass( loader,clazzName ) ;
        if( em == null ) {
            throw new ClassNotFoundException( "指定クラス " + clazzName + " は存在しません" ) ;
        }
        Object[] ret = new Object[ 1 ] ;
        Class[] types = FastReflectUtil.getParamsType( args ) ;
        while( true ) {
            if( em.invokeMethod( ret,target,name,loader,types,args ) ) {
                return ret[0] ;
            }
            String spclazz = em.getSuperClassName() ;
            if( spclazz == null ) {
                throw new NoSuchMethodException(
                        "指定メソッド " + name + " はクラス " + clazzName + " に存在しません" ) ;
            }
            em = FastReflectClass.getClass( loader,spclazz ) ;
        }
    }
    
    /**
     * メソッド呼び出し.
     * @param clazz 対象のクラスを設定します.
     * @param target 設定対象のオブジェクトを設定します.<BR>
     *               [null]の場合、staticアクセスで処理します.
     * @param name 対象のメソッド名を設定します.
     * @param args 対象のメソッドパラメータを設定します.
     * @return Object 戻り値が返されます.
     * @exception Exception 例外.
     */
    public static final Object invoke( Class clazz,Object target,String name,Object... args )
        throws Exception {
        return invoke( null,clazz,target,name,args ) ;
    }
    
    /**
     * メソッド呼び出し.
     * @param loader 対象のクラスローダーを設定します.
     * @param clazz 対象のクラスを設定します.
     * @param target 設定対象のオブジェクトを設定します.<BR>
     *               [null]の場合、staticアクセスで処理します.
     * @param name 対象のメソッド名を設定します.
     * @param args 対象のメソッドパラメータを設定します.
     * @return Object 戻り値が返されます.
     * @exception Exception 例外.
     */
    public static final Object invoke( ClassLoader loader,Class clazz,Object target,String name,Object... args )
        throws Exception {
        String clazzName = clazz.getName() ;
        FastClassElement em = FastReflectClass.getClass( loader,clazzName ) ;
        if( em == null ) {
            throw new ClassNotFoundException( "指定クラス " + clazzName + " は存在しません" ) ;
        }
        Object[] ret = new Object[ 1 ] ;
        Class[] types = FastReflectUtil.getParamsType( args ) ;
        while( true ) {
            if( em.invokeMethod( ret,target,name,loader,types,args ) ) {
                return ret[0] ;
            }
            String spclazz = em.getSuperClassName() ;
            if( spclazz == null ) {
                throw new NoSuchMethodException(
                        "指定メソッド " + name + " はクラス " + clazzName + " に存在しません" ) ;
            }
            em = FastReflectClass.getClass( loader,spclazz ) ;
        }
    }
    
    /**
     * メソッド呼び出し.
     * @param loader 対象のクラスローダーを設定します.
     * @param clazzName 対象のクラス名を設定します.
     * @param target 設定対象のオブジェクトを設定します.<BR>
     *               [null]の場合、staticアクセスで処理します.
     * @param name 対象のメソッド名を設定します.
     * @param args 対象のメソッドパラメータを設定します.
     * @param types 対象のメソッドパラメータタイプを設定します.
     * @return Object 戻り値が返されます.
     * @exception Exception 例外.
     */
    public static final Object invokeTo( ClassLoader loader,String clazzName,Object target,String name,Object[] args,Class[] types )
        throws Exception {
        FastClassElement em = FastReflectClass.getClass( loader,clazzName ) ;
        if( em == null ) {
            throw new ClassNotFoundException( "指定クラス " + clazzName + " は存在しません" ) ;
        }
        if( args != null && args.length > 0 && types == null ) {
            types = FastReflectUtil.getParamsType( args ) ;
        }
        Object[] ret = new Object[ 1 ] ;
        while( true ) {
            if( em.invokeMethod( ret,target,name,loader,types,args ) ) {
                return ret[0] ;
            }
            String spclazz = em.getSuperClassName() ;
            if( spclazz == null ) {
                throw new NoSuchMethodException(
                        "指定メソッド " + name + " はクラス " + clazzName + " に存在しません" ) ;
            }
            em = FastReflectClass.getClass( loader,spclazz ) ;
        }
    }
    
    /**
     * メソッド呼び出し.
     * @param loader 対象のクラスローダーを設定します.
     * @param clazz 対象のクラスを設定します.
     * @param target 設定対象のオブジェクトを設定します.<BR>
     *               [null]の場合、staticアクセスで処理します.
     * @param name 対象のメソッド名を設定します.
     * @param args 対象のメソッドパラメータを設定します.
     * @param types 対象のメソッドパラメータタイプを設定します.
     * @return Object 戻り値が返されます.
     * @exception Exception 例外.
     */
    public static final Object invokeTo( ClassLoader loader,Class clazz,Object target,String name,Object[] args,Class[] types )
        throws Exception {
        String clazzName = clazz.getName() ;
        FastClassElement em = FastReflectClass.getClass( loader,clazzName ) ;
        if( em == null ) {
            throw new ClassNotFoundException( "指定クラス " + clazzName + " は存在しません" ) ;
        }
        if( args != null && args.length > 0 && types == null ) {
            types = FastReflectUtil.getParamsType( args ) ;
        }
        Object[] ret = new Object[ 1 ] ;
        while( true ) {
            if( em.invokeMethod( ret,target,name,loader,types,args ) ) {
                return ret[0] ;
            }
            String spclazz = em.getSuperClassName() ;
            if( spclazz == null ) {
                throw new NoSuchMethodException(
                        "指定メソッド " + name + " はクラス " + clazzName + " に存在しません" ) ;
            }
            em = FastReflectClass.getClass( loader,spclazz ) ;
        }
    }
}
