package org.maachang.service.mail ;

import java.io.BufferedReader;
import java.io.StringReader;
import java.io.StringWriter;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.HashMap;
import java.util.Locale;
import java.util.Properties;

import javax.activation.DataHandler;
import javax.mail.BodyPart;
import javax.mail.Message;
import javax.mail.Multipart;
import javax.mail.Session;
import javax.mail.Transport;
import javax.mail.internet.InternetAddress;
import javax.mail.internet.MimeBodyPart;
import javax.mail.internet.MimeMessage;
import javax.mail.internet.MimeMultipart;
import javax.mail.internet.MimeUtility;
import javax.mail.util.ByteArrayDataSource;

import org.apache.velocity.VelocityContext;
import org.apache.velocity.app.Velocity;
import org.maachang.engine.util.Reflect;

/**
 * メール送信サービス.
 * 
 * @version 2007/10/18
 * @author masahito suzuki
 * @since MaaEngine 1.00
 */
public class SendMailService {
    
    /**
     * メールキャラクターセット.
     */
    private static final String CHARSET = "ISO-2022-JP" ;
    
    /**
     * メールを送信.
     * <BR><BR>
     * 対象の条件でメールを送信します.
     * <BR>
     * @param sendMail 送信条件を設定します.
     * @exception Exception 例外.
     */
    public void sendMail( SendMail sendMail )
        throws Exception {
        if( sendMail == null || sendMail.check() == false ) {
            throw new IllegalArgumentException( "引数は不正です" ) ;
        }
        
        Properties props = new Properties() ;
        props.put( "mail.smtp.host",sendMail.getHost() ) ;
        if( sendMail.getPort() > 0 ) {
            props.put( "mail.smtp.port", String.valueOf( sendMail.getPort() ) ) ;
        }
        
        // タイムアウト設定.
        if( sendMail.getTimeout() > 0L ) {
            props.setProperty( "mail.smtp.connectiontimeout", String.valueOf( sendMail.getTimeout() ) ) ;
            props.setProperty( "mail.smtp.timeout", String.valueOf( sendMail.getTimeout() ) ) ;
        }
        
        // SSL接続.
        if( sendMail.isSsl() == true ) {
            props.setProperty( "mail.smtp.socketFactory.class", "javax.net.ssl.SSLSocketFactory" );
            props.setProperty( "mail.smtp.socketFactory.fallback", "false" );
            if( sendMail.getPort() > 0 ) {
                props.setProperty( "mail.smtp.socketFactory.port", String.valueOf( sendMail.getPort() ) );
            }
        }
        
        Session session = null ;
        
        // 認証あり.
        if( sendMail.isAuth() == true ) {
            props.setProperty( "mail.smtp.auth", "true" ) ;
            session = Session.getInstance( props, new MailAuth( sendMail.getUser(),sendMail.getPasswd() ) ) ;
        }
        // 認証なし.
        else {
            session = Session.getInstance( props, null ) ;
        }
        
        try {
            
            // 送信データ生成.
            Message msg = new MimeMessage( session ) ;
            msg.setHeader("Content-Type","text/plain; charset="+CHARSET);
            msg.setHeader( "Content-Transfer-Encoding", "7bit" ) ;
            msg.setFrom( new InternetAddress( sendMail.getFrom() ) ) ;
            InternetAddress[] address = getTo( sendMail ) ;
            msg.setRecipients( Message.RecipientType.TO, address ) ;
            msg.setSubject( MimeUtility.encodeText( sendMail.getSubject(), "ISO-2022-JP", "B") ) ;
            
            Multipart multipart = new MimeMultipart() ;
            MimeBodyPart textPart = new MimeBodyPart() ;
            textPart.setHeader( "Content-Transfer-Encoding", "7bit" ) ;
            textPart.setText( getSendMessage( sendMail ),CHARSET ) ;
            multipart.addBodyPart( textPart ) ;
            
            // 添付内容ありのメールメッセージ.
            if( sendMail.getAttached() != null && sendMail.getAttached().size() > 0 ) {
                ArrayList<ResultAttached> lst = sendMail.getAttached() ;
                BodyPart attachmentPart = new MimeBodyPart() ;
                int len = lst.size() ;
                for( int i = 0 ; i < len ; i ++ ) {
                    ResultAttached ra = lst.get( i ) ;
                    if( ra != null && ra.getAttached() != null && ra.getAttached().length > 0 ) {
                        ByteArrayDataSource ba = new ByteArrayDataSource(
                            ra.getAttached(),"application/octet-stream" ) ;
                        attachmentPart.setDataHandler( new DataHandler( ba ) ) ;
                        if( ra.getFileName() != null && ra.getFileName().length() > 0 ) {
                            attachmentPart.setFileName(
                                MimeUtility.encodeText( ra.getFileName(), CHARSET, null ) ) ;
                        }
                        multipart.addBodyPart( attachmentPart ) ;
                    }
                }
            }
            msg.setContent( multipart ) ;
            
            // 送信処理.
            msg.setSentDate( new Date() );
            Transport.send(msg) ;
            
        } catch( Exception e ){
            throw e ;
        }
    }
    
    /**
     * 送信先の情報を生成.
     */
    private InternetAddress[] getTo( SendMail sendMail )
        throws Exception {
        int len = sendMail.getTo().size() ;
        InternetAddress[] ret = new InternetAddress[ len ] ;
        for( int i = 0 ; i < len ; i ++ ) {
            ret[ i ] = new InternetAddress( sendMail.getTo().get( i ) ) ;
        }
        return ret ;
    }
    
    /**
     * 送信メッセージを生成.
     */
    private String getSendMessage( SendMail sendMail )
        throws Exception {
        StringWriter w = new StringWriter() ;
        Velocity.evaluate( new VelocityContext( getMap( sendMail ) ),
            w,null,trimBody( sendMail.getBody() ) ) ;
        w.flush() ;
        return w.toString() ;
    }
    
    /**
     * ObjectからHashMapデータを生成.
     */
    private HashMap<String,Object> getMap( SendMail sendMail )
        throws Exception {
        HashMap<String,Object> ret = Reflect.getter( sendMail.getParams() ) ;
        int size = ret.size() ;
        if( size > 0 ) {
            Object[] objs = ret.keySet().toArray() ;
            for( int i = 0 ; i < size ; i ++ ) {
                Object val = ret.get( ( String )objs[ i ] ) ;
                if( val == null ) {
                    ret.put( ( String )objs[ i ],"" ) ;
                }
            }
            // 作成日付を生成.
            SimpleDateFormat fmt = new SimpleDateFormat(
                "yyyy年MM月dd日 - HH時mm分ss秒",Locale.JAPAN ) ;
            ret.put( "createMailSendDate",fmt.format( new Date() ) ) ;
            return ret ;
        }
        return ret ;
    }
    
    /**
     * Bodyを整形.
     */
    private String trimBody( String body )
        throws Exception {
        BufferedReader b = new BufferedReader( new StringReader( body ) ) ;
        StringBuilder buf = new StringBuilder( ( int )( body.length() * 1.25f ) ) ;
        for( ;; ) {
            String s = b.readLine() ;
            if( s == null ) {
                break ;
            }
            buf.append( s ) ;
            buf.append( "\r\n" ) ;
        }
        b.close() ;
        return buf.toString() ;
    }
}

