package org.maachang.engine.validate ;

import java.util.HashMap;

import org.maachang.engine.util.KeyValue;

/**
 * [text]Validate.
 * 
 * @version 2007/10/24
 * @author masahito suzuki
 * @since MaaEngine 1.00
 */
public class TextValidateElement extends AbstractValidateElement implements ValidateElement {
    
    /**
     * パラメータ条件 : チェックタイプ.
     */
    private static final String PARAM_TYPE = "type" ;
    
    /**
     * パラメータ条件 : 長さ最小値.
     */
    private static final String PARAM_MIN_LENGTH = "min" ;
    
    /**
     * パラメータ条件 : 長さ最大値.
     */
    private static final String PARAM_MAX_LENGTH = "max" ;
    
    /**
     * チェックタイプ : 数字のみ.
     */
    private static final String ONLY_NUMBER = "num" ;
    
    /**
     * チェックタイプ : 英字のみ.
     */
    private static final String ONLY_ALPHABET = "alpha" ;
    
    /**
     * チェックタイプ : Asciiのみ.
     */
    private static final String ONLY_ASCII = "ascii" ;
    
    /**
     * チェックタイプ : カナのみ.
     */
    private static final String ONLY_KANA = "kana" ;
    
    /**
     * チェックタイプ : ひらがなのみ.
     */
    private static final String ONLY_HIRA = "hira" ;
    
    
    /**
     * Validate処理を実行.
     * <BR><BR>
     * Validate処理を実行します.
     * <BR>
     * @param check チェック対象内容を設定します.
     * @param args 処理パラメータを設定します.
     * @return boolean [false]の場合、Validateにマッチした条件が存在します.
     * @exception Exception 例外.
     */
    public boolean match( Object check,String[] args )
        throws Exception {
        if( check != null &&
            ( check instanceof String ) == false &&
            ( check instanceof Integer ) == false &&
            ( check instanceof Long ) == false &&
            ( check instanceof Float ) == false &&
            ( check instanceof Double ) == false ) {
            return false ;
        }
        else if( check == null ) {
            check = "" ;
        }
        HashMap<String,KeyValue> keyValue = getUseParams( args ) ;
        String chkString = check.toString() ;
        
        // 文字タイプチェック.
        if( checkType( chkString,keyValue ) == false ) {
            return false ;
        }
        
        // 最小の長さチェック.
        if( minLength( chkString,keyValue ) == false ) {
            return false ;
        }
        
        // 最大の長さチェック.
        if( maxLength( chkString,keyValue ) == false ) {
            return false ;
        }
        return true ;
    }
    
    /**
     * Validate処理対象要素名を取得.
     * <BR><BR>
     * Validate処理対象要素名を取得します.
     * <BR>
     * @return String Validate処理対象要素名が返されます.
     */
    public String getName() {
        return "text" ;
    }
    
    /**
     * カタカナ一覧.
     */
    private static final String KANA = "アィイウウェエォオカガキギクグケゲコゴサザシジスズセゼソゾタダチヂッツヅテデトド"+
        "ナニヌネノハバパヒビピフブプヘベペホボポマミムメモャヤュユョヨラリルレロヮワヰヲンー" ;
    
    /**
     * ひらがな一覧.
     */
    private static final String HIRA = "あぃいぅうぇえぉおかがきぎくぐけげこごさざしじすずせぜそぞただちぢっつづてでとど"+
        "なにぬねのはばぱひびぴふぶぷへべぺほぼぽまみむめもゃやゅゆょよらりるれろゎわゐゑをんー" ;
    
    /**
     * タイプ判別.
     */
    private boolean checkType( String chkString,HashMap<String,KeyValue> keyValue ) {
        KeyValue kv = keyValue.get( PARAM_TYPE ) ;
        if( kv == null ) {
            return true ;
        }
        String type = kv.getValue() ;
        if( chkString == null || chkString.length() <= 0 || type == null ) {
            return true ;
        }
        if( type.startsWith( ONLY_NUMBER ) ) {
            int len = chkString.length() ;
            int cnt = 0 ;
            for( int i = 0 ; i < len ; i ++ ) {
                char c = chkString.charAt( i ) ;
                if( ( c >= '0' && c <= '9' ) == false ) {
                    if( cnt == 0 ) {
                        if( i == 0 && c == '-' ) {
                            cnt = -1 ;
                        }
                        else {
                            return false ;
                        }
                    }
                    else if( cnt != 0 && c != '.' ) {
                        return false ;
                    }
                }
                cnt ++ ;
            }
        }
        else if( type.startsWith( ONLY_ALPHABET ) ) {
            int len = chkString.length() ;
            for( int i = 0 ; i < len ; i ++ ) {
                char c = chkString.charAt( i ) ;
                if( ( ( c >= 'a' && c <= 'z' ) ||
                    ( c >= 'A' && c <= 'Z' ) ) == false ) {
                    return false ;
                }
            }
        }
        else if( type.startsWith( ONLY_ASCII ) ) {
            int len = chkString.length() ;
            for( int i = 0 ; i < len ; i ++ ) {
                char c = chkString.charAt( i ) ;
                if( c > ( char )0x00ff ) {
                    return false ;
                }
            }
        }
        else if( type.startsWith( ONLY_KANA ) ) {
            int len = chkString.length() ;
            for( int i = 0 ; i < len ; i ++ ) {
                String c = chkString.substring( i,i+1 ) ;
                if( KANA.indexOf( c ) == -1 ) {
                    return false ;
                }
            }
        }
        else if( type.startsWith( ONLY_HIRA ) ) {
            int len = chkString.length() ;
            for( int i = 0 ; i < len ; i ++ ) {
                String c = chkString.substring( i,i+1 ) ;
                if( HIRA.indexOf( c ) == -1 ) {
                    return false ;
                }
            }
        }
        return true ;
    }
    
    /**
     * 長さ最小値チェック.
     */
    private boolean minLength( String chkString,HashMap<String,KeyValue> keyValue ) {
        KeyValue kv = keyValue.get( PARAM_MIN_LENGTH ) ;
        if( kv == null ) {
            return true ;
        }
        String min = kv.getValue() ;
        if( min == null ) {
            return true ;
        }
        int len = -1 ;
        try {
            len = Integer.parseInt( min ) ;
        } catch( Exception e ) {
            len = -1 ;
        }
        if( len == -1 ) {
            return true ;
        }
        if( chkString.length() < len ) {
            return false ;
        }
        return true ;
    }
    
    /**
     * 長さ最大値チェック.
     */
    private boolean maxLength( String chkString,HashMap<String,KeyValue> keyValue ) {
        KeyValue kv = keyValue.get( PARAM_MAX_LENGTH ) ;
        if( kv == null ) {
            return true ;
        }
        String max = kv.getValue() ;
        if( max == null ) {
            return true ;
        }
        int len = -1 ;
        try {
            len = Integer.parseInt( max ) ;
        } catch( Exception e ) {
            len = -1 ;
        }
        if( len == -1 ) {
            return true ;
        }
        if( chkString.length() > len ) {
            return false ;
        }
        return true ;
    }
}
