package org.maachang.engine.util;

import java.util.List;

/**
 * 自己参照型リスト.
 * 
 * @version 2007/10/18
 * @author masahito suzuki
 * @since MaaEngine 1.00
 */
public class LinkList<E> {

    protected LinkChild<E> first = null;

    protected LinkChild<E> last = null;

    protected int size = 0;

    /**
     * コンストラクタ.
     */
    public LinkList() {
        first = null;
        last = null;
    }

    /**
     * 情報クリア. <BR>
     * <BR>
     * 情報をクリアします.
     */
    public void clear() {
        first = null;
        last = null;
    }

    /**
     * 情報追加. <BR>
     * <BR>
     * 情報を追加します. <BR>
     * 
     * @param o
     *            追加対象の情報を設定します.
     * @return LinkEntry<E> エントリ情報が返されます.
     */
    public LinkEntry<E> add(E o) {
        if (first == null) {
            first = new LinkChild<E>();
            first.parent = this;
            first.before = null;
            first.next = null;
            first.value = o;
            last = first;
            size = 1;
            return first;
        }
        LinkChild<E> lnk = new LinkChild<E>();
        lnk.parent = this;
        lnk.next = null;
        lnk.before = last;
        lnk.value = o;
        last.next = lnk;
        last = lnk;
        size++;
        return lnk;
    }

    /**
     * 情報設定. <BR>
     * <BR>
     * 指定位置の情報を置き換えます. <BR>
     * 
     * @param no
     *            置き換え位置を設定します.
     * @param o
     *            置き換え情報を設定します.
     * @return LinkEntry<E> エントリ情報が返されます.
     */
    public LinkEntry<E> set(int no, E o) {
        if (first == null) {
            return null;
        }
        LinkChild<E> lnk = first;
        for (int i = 0; i < no; i++) {
            if (lnk.next == null) {
                break;
            }
            lnk = lnk.next;
        }
        lnk.value = o;
        return lnk;
    }

    /**
     * 指定位置の次の位置に情報を追加. <BR>
     * <BR>
     * 指定位置の次の位置に情報を追加します. <BR>
     * 
     * @param no
     *            追加対象の項番を設定します.
     * @param o
     *            追加対象の情報を設定します.
     * @return LinkEntry<E> エントリ情報が返されます.
     */
    public LinkEntry<E> bitween(int no, E o) {
        if (first == null) {
            return add(o);
        }
        if (no <= -1) {
            LinkChild<E> ad = new LinkChild<E>();
            ad.parent = this;
            ad.before = null;
            ad.next = first;
            ad.value = o;
            first.before = ad;
            first = ad;
            size++;
            return ad;
        }
        LinkChild<E> lnk = first;
        for (int i = 0; i < no; i++) {
            if (lnk.next == null) {
                break;
            }
            lnk = lnk.next;
        }
        LinkChild<E> ad = new LinkChild<E>();
        ad.parent = this;
        ad.next = lnk.next;
        if (lnk.next != null) {
            lnk.next.before = ad;
        } else {
            last = ad;
        }
        ad.before = lnk;
        lnk.next = ad;
        ad.value = o;
        size++;
        return ad;
    }

    /**
     * 情報削除. <BR>
     * <BR>
     * 指定位置の情報を削除します. <BR>
     * 
     * @param no
     *            削除対象の項番を設定します.
     * @return E 削除された情報が返されます.
     */
    public E remove(int no) {
        if (first == null) {
            return null;
        }
        LinkChild<E> lnk = first;
        for (int i = 0; i < no; i++) {
            if (lnk.next == null) {
                return null;
            }
            lnk = lnk.next;
        }
        if (lnk.before != null) {
            lnk.before.next = lnk.next;
        } else {
            first = lnk.next;
        }
        if (lnk.next != null) {
            lnk.next.before = lnk.before;
        } else {
            last = lnk.before;
        }
        size--;
        lnk.before = null;
        lnk.next = null;
        return lnk.value;
    }

    /**
     * 一番最初の情報を削除. <BR>
     * <BR>
     * 一番最初の情報を削除します. <BR>
     * 
     * @return E 削除された情報が返されます.
     */
    public E removeFirst() {
        if (first == null) {
            return null;
        }
        E ret = first.value;
        if (first.next != null) {
            LinkChild<E> lnk = first;
            lnk.next.before = null;
            first = lnk.next;
            lnk.before = null;
            lnk.next = null;
        } else {
            first = null;
            last = null;
        }
        size--;
        return ret;
    }

    /**
     * 一番最後の情報を削除. <BR>
     * <BR>
     * 一番最後の情報を削除します. <BR>
     * 
     * @return E 削除された情報が返されます.
     */
    public E removeLast() {
        if (first == null) {
            return null;
        }
        E ret = last.value;
        if (last.before != null) {
            LinkChild<E> lnk = last;
            lnk.before.next = null;
            last = lnk.before;
            lnk.before = null;
            lnk.next = null;
        } else {
            first = null;
            last = null;
        }
        size--;
        return ret;
    }

    /**
     * 情報を取得. <BR>
     * <BR>
     * 指定位置の情報を取得します. <BR>
     * 
     * @param no
     *            対象の項番を設定します.
     * @return E 対象の情報が返されます.
     */
    public E get(int no) {
        if (first == null) {
            return null;
        }
        LinkChild<E> lnk = first;
        for (int i = 0; i < no; i++) {
            if (lnk.next == null) {
                return null;
            }
            lnk = lnk.next;
        }
        if (lnk != null) {
            return lnk.value;
        }
        return null;
    }

    /**
     * 一番初めの情報を取得. <BR>
     * <BR>
     * 一番初めの情報を取得します. <BR>
     * 
     * @return E 対象の情報が返されます.
     */
    public E getFirst() {
        if (first == null) {
            return null;
        }
        return first.value;
    }

    /**
     * 一番最後の情報を取得. <BR>
     * <BR>
     * 一番最後の情報を取得します. <BR>
     * 
     * @return E 対象の情報が返されます.
     */
    public E getLast() {
        if (first == null) {
            return null;
        }
        return last.value;
    }

    /**
     * 格納長を取得. <BR>
     * <BR>
     * 格納長を取得します. <BR>
     * 
     * @return int 格納長が返されます.
     */
    public int size() {
        return size;
    }

    /**
     * 全情報を取得. <BR>
     * <BR>
     * 全部の情報を取得します. <BR>
     * 
     * @return Object[] 全部の情報が返されます.
     */
    public Object[] getAll() {
        if (first == null) {
            return null;
        }
        Object[] ret = new Object[size];
        LinkChild<E> lnk = first;
        for (int i = 0; i < size; i++) {
            ret[i] = lnk.value;
            lnk = lnk.next;
        }
        return ret;
    }

    /**
     * 全情報を取得. <BR>
     * <BR>
     * 全部の情報を取得します. <BR>
     * 
     * @param o
     *            取得情報を格納するオブジェクトを設定します.
     */
    public void getAll(List<E> o) {
        if (o == null || first == null) {
            return;
        }
        LinkChild<E> lnk = first;
        for (;;) {
            o.add(lnk.value);
            lnk = lnk.next;
            if (lnk == null) {
                break;
            }
        }
    }

    /**
     * オブジェクトの情報を文字列で取得. <BR>
     * <BR>
     * オブジェクトの情報を文字列で取得します. <BR>
     * 
     * @return String オブジェクトの情報が返されます.
     */
    public String toString() {
        if (first == null) {
            return "[]";
        }
        StringBuilder buf = new StringBuilder();
        buf.append("[");
        LinkChild<E> lnk = first;
        for (int i = 0;; i++) {
            if (lnk == null) {
                break;
            }
            if (i != 0) {
                buf.append(",");
            }
            if (lnk.value == null) {
                buf.append("");
            } else {
                buf.append(lnk.value);
            }
            lnk = lnk.next;
        }
        buf.append("]");
        return buf.toString();
    }
}

class LinkChild<E> implements LinkEntry<E> {
    private static final long serialVersionUID = -1309678959718560886L;

    public LinkList<E> parent = null;

    public LinkChild<E> before = null;

    public LinkChild<E> next = null;

    public E value = null;

    public void remove() {
        if (parent != null) {
            if (before != null) {
                before.next = next;
            } else {
                parent.first = next;
            }
            if (next != null) {
                next.before = before;
            } else {
                parent.last = before;
            }
            parent.size--;
            before = null;
            next = null;
        }
        parent = null;
    }
}
